// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/environment.h"
#include "base/strings/utf_string_conversions.h"
#include "build/chromeos_buildflags.h"
#include "components/omnibox/browser/autocomplete_input.h"
#include "components/omnibox/browser/mock_autocomplete_provider_client.h"
#include "components/omnibox/browser/omnibox_pedal_provider.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "ui/base/resource/resource_bundle.h"

// Note: Pedals have their own components unit tests, which should be
// the preferred place for testing the classes. The tests here are for
// testing things that depend on Chrome resources, for example the localization
// pak files generated by chrome:packed_resources.

TEST(OmniboxPedals, DataLoadsForAllLocales) {
  // Locale selection is platform sensitive. On Linux, environment is used.
  std::unique_ptr<base::Environment> env = base::Environment::Create();
  MockAutocompleteProviderClient client;

  struct TestCase {
    std::string locale;
    std::vector<std::string> triggers;
  };
  const TestCase test_cases[] = {
      // clang-format off
      // Test cases generated by pedal_processor:
      { "am",
        {
          "አስወግድ መሸጎጫ",
          "አስቀምጥ የይለፍ ቃላት",
          "አስቀምጥ የክሬዲት ካርድ መረጃ",
          "ማንነት የማያሳውቅ መስኮት",
          "ቋንቋ ቀይር ይህ ገጽ",
          "google chrome አዘምን",
        }
      },
      { "ar",
        {
          "إزالة ذاكرة التخزين المؤقت",
          "إدارة بيانات الاعتماد",
          "إدارة معلومات بطاقة الائتمان",
          "علامة تبويب في وضع التصفّح المتخفي",
          "تغيير اللغة هذه الصفحة",
          "google chrome تثبيت",
        }
      },
      { "bg",
        {
          "изчистване",
          "актуализиране идентификацията",
          "актуализиране информацията кредитната карта",
          "стартиране поверително сърфиране",
          "промяна езика тази страница",
          "браузъра актуализиране",
        }
      },
      { "bn",
        {
          "মিটিয়ে দিন ক্যাশে ফাইল",
          "পরিবর্তন করুন ক্রেডেনশিয়াল",
          "পরিবর্তন করুন ক্রেডিট কার্ডের তথ্য",
          "ব্যক্তিগত উইন্ডো",
          "ভাষা পরিবর্তন করুন এই পৃষ্ঠা",
          "ব্রাউজার আপগ্রেড করুন",
        }
      },
      { "ca",
        {
          "suprimeix memòria cau",
          "administrador contrasenyes",
          "actualitzar informació targeta crèdit",
          "finestra d'incògnit",
          "canviar l'idioma aquesta pàgina",
          "google chrome actualitzar versió",
        }
      },
      { "cs",
        {
          "odstranit soubory cookie",
          "aktualizovat přihlašovací údaje",
          "aktualizovat informace o platební kartě",
          "anonymní režim",
          "změnit jazyk tato stránka",
          "google chrome aktualizovat",
        }
      },
      { "da",
        {
          "fjern browseroplysninger",
          "administrator loginoplysninger",
          "administrer",
          "inkognitotilstand",
          "skift sprog denne side",
          "google chrome installer",
        }
      },
      { "de",
        {
          "entfernen browserverlauf",
          "anschauen passwörter",
          "aktualisieren karteninformationen",
          "arbeiten inkognito fenster",
          "sprache ändern diese webseite",
          "google chrome aktualisieren",
        }
      },
      { "el",
        {
          "εκκαθάριση πληροφορίες",
          "διαχειριστής κωδικοί πρόσβασης",
          "επεξεργασία πληροφορίες πιστωτικής κάρτας",
          "παράθυρο για ανώνυμη περιήγηση",
          "αλλαγή γλώσσας αυτή η σελίδα",
          "πρόγραμμα περιήγησης εγκατάσταση",
        }
      },
      { "en",
        {
          "delete information",
          "manager credentials",
          "change credit card information",
          "incognito window",
          "change language this page",
          "google chrome install",
        }
      },
      { "en-GB",
        {
          "delete information",
          "manager credentials",
          "change credit card information",
          "incognito window",
          "change language this page",
          "google chrome install",
        }
      },
      { "es",
        {
          "eliminar información",
          "administrador contraseñas",
          "actualizar información de la tarjeta de crédito",
          "pestaña de incógnito",
          "cambiar idioma esta página",
          "google chrome actualización",
        }
      },
      { "es-419",
        {
          "eliminar información",
          "administrador contraseñas",
          "administrar información de la tarjeta de crédito",
          "pestaña de incógnito",
          "cambiar idioma esta página",
          "google chrome actualización",
        }
      },
      { "et",
        {
          "tühjenda küpsisefailid",
          "värskenda paroolid",
          "värskenda krediitkaardi teave",
          "inkognito vaheleht",
          "muuda keelt see leht",
          "google chrome uuenda versiooni",
        }
      },
      { "fa",
        {
          "حذف کردن حافظه پنهان",
          "به‌روزرسانی کردن اطلاعات کاربری",
          "به‌روزرسانی کردن اطلاعات کارت اعتباری",
          "پنجره ناشناس",
          "ترجمه کردن این صفحه",
          "google chrome به‌روزرسانی کردن",
        }
      },
      { "fi",
        {
          "tyhjennä historiatiedot",
          "hallinnointi kirjautumistiedot",
          "ylläpidä luottokorttien tiedot",
          "yksityinen välilehti",
          "vaihda kieltä tämä sivu",
          "google chrome päivitys",
        }
      },
      { "fil",
        {
          "i-delete impormasyon",
          "pamahalaan mga kredensyal",
          "pamahalaan impormasyon ng credit card",
          "incognito window",
          "baguhin ang wika i-translate ang page",
          "google chrome i-install",
        }
      },
      { "fr",
        {
          "suppression informations",
          "mettre à jour informations de connexion",
          "mettre à jour informations cartes bancaires",
          "sélectionner navigation confidentielle",
          "sélectionner la langue cette page",
          "google chrome mettre à niveau dernière version",
        }
      },
      { "gu",
        {
          "કાઢી નાખો ઇતિહાસ",
          "ફેરફાર કરો લૉગ ઇન વિગત",
          "ફેરફાર કરો ક્રેડિટ કાર્ડની માહિતી",
          "ખાનગી વિંડો",
          "અનુવાદ કરો આ પેજ",
          "બ્રાઉઝર ઇન્સ્ટૉલ કરો",
        }
      },
      { "he",
        {
          "איפוס נתונים קובצי cookie",
          "להציג פרטי כניסה",
          "לעדכן פרטי כרטיס אשראי",
          "כרטיסיית מצב אנונימי",
          "לשנות שפה דף זה",
          "google chrome להתקין",
        }
      },
      { "hi",
        {
          "वाइप करें कैश मेमोरी",
          "प्रबंधित करें क्रेडेंशियल",
          "प्रबंधित करें क्रेडिट कार्ड की जानकारी",
          "गुप्त विंडो",
          "अनुवाद करें यह पेज",
          "ब्राउज़र अपग्रेड करें",
        }
      },
      { "hr",
        {
          "odstranjivanje privremena memorija",
          "ažuriranje vjerodajnice",
          "ažuriranje informacije o kreditnoj kartici",
          "anonimna kartica",
          "promjena jezika ova stranica",
          "google chrome instaliranje",
        }
      },
      { "hu",
        {
          "eltávolítása gyorsítótár",
          "megjelenítése bejelentkezési adatok",
          "módosítása hitelkártya adatok",
          "inkognitó ablak",
          "nyelvének módosítása ezt az oldalt",
          "google chrome frissítése",
        }
      },
      { "id",
        {
          "hapus informasi",
          "pengelola kredensial",
          "kelola informasi kartu kredit",
          "jendela pribadi",
          "terjemahkan halaman ini",
          "google chrome upgrade",
        }
      },
      { "it",
        {
          "cancellare informazioni",
          "visualizzare credenziali",
          "aggiornare informazioni di carte di credito",
          "modalità di navigazione in incognito",
          "cambiare la lingua questa pagina",
          "google chrome eseguire l'upgrade",
        }
      },
      { "ja",
        {
          "クリアキャッシュ",
          "アップデートパスワード",
          "アップデートクレジットカード情報",
          "使い方プライベートブラウジング",
          "トランスレーションこのページ",
          "グーグルクロームアップグレード",
        }
      },
      { "kn",
        {
          "ತೆರವುಗೊಳಿಸಿ ಕುಕೀಗಳು",
          "ಅಪ್‌ಡೇಟ್ ಮಾಡಿ ಪಾಸ್‌ವರ್ಡ್‌ಗಳು",
          "ಅಪ್‌ಡೇಟ್ ಮಾಡಿ ಕ್ರೆಡಿಟ್ ಕಾರ್ಡ್ ಮಾಹಿತಿ",
          "ಅಜ್ಞಾತ ಟ್ಯಾಬ್",
          "ಭಾಷೆಯನ್ನು ಬದಲಾಯಿಸಿ ಈ ಪುಟ",
          "ಬ್ರೌಸರ್ ಅಪ್‌ಗ್ರೇಡ್ ಮಾಡಿ",
        }
      },
      { "ko",
        {
          "완전 삭제 데이터",
          "업데이트 사용자 인증 정보",
          "업데이트 신용카드 정보",
          "비공개 모드",
          "언어 변경 이 페이지",
          "브라우저 업그레이드",
        }
      },
      { "lt",
        {
          "pašalinti informacija",
          "peržiūrėti prisijungimo duomenys",
          "atnaujinti kredito kortelės informacija",
          "inkognito skirtukas",
          "pakeisti kalbą šis puslapis",
          "google chrome atnaujinti",
        }
      },
      { "lv",
        {
          "iztīrīt informācija",
          "pārvaldnieks akreditācijas dati",
          "atjaunināt informācija par kredītkarti",
          "inkognito režīma logs",
          "mainīt valodu šī lapa",
          "google chrome atjaunināt",
        }
      },
      { "ml",
        {
          "നീക്കം ചെയ്യുക കുക്കികൾ",
          "അപ്ഡേറ്റ് ചെയ്യുക ക്രെഡന്‍ഷ്യലുകൾ",
          "അപ്ഡേറ്റ് ചെയ്യുക ക്രെഡിറ്റ് കാർഡ് വിവരങ്ങൾ",
          "സ്വകാര്യ വിൻഡോ",
          "വിവർത്തനം ചെയ്യുക ഈ പേജ്",
          "ബ്രൗസർ അപ്‌ഗ്രേഡ് ചെയ്യുക",
        }
      },
      { "mr",
        {
          "काढून टाका इतिहास",
          "व्यवस्थापित करा क्रेडेंशियल",
          "व्यवस्थापित करा क्रेडिट कार्डाची माहिती",
          "खाजगी विंडो",
          "भाषांतर करा हे पेज",
          "ब्राउझर अपग्रेड करा",
        }
      },
      { "ms",
        {
          "alih keluar maklumat",
          "kemas kini bukti kelayakan",
          "kemas kini maklumat kad kredit",
          "tetingkap inkognito",
          "tukar bahasa halaman ini",
          "penyemak imbas kemas kini",
        }
      },
      { "nl",
        {
          "google chrome verwijderen cachegeheugen",
          "veranderen inloggegevens",
          "bijwerken creditcardinformatie",
          "incognitotabblad",
          "taal wijzigen deze pagina",
          "google chrome installeren",
        }
      },
      { "pl",
        {
          "wyczyść pamięć podręczną przeglądarki",
          "zaktualizuj dane do zalogowania",
          "zaktualizuj informacje o karcie płatniczej",
          "tryb niewidzialny",
          "zmień język tę stronę",
          "google chrome zaktualizuj",
        }
      },
      { "pt-BR",
        {
          "excluir permanentemente informações",
          "atualizar credenciais",
          "atualizar informações do cartão de crédito",
          "modo de navegação anônima",
          "mudar idioma esta página",
          "google chrome fazer upgrade",
        }
      },
      { "pt-PT",
        {
          "eliminar informações",
          "atualizar palavras-passe",
          "atualizar informações do cartão de crédito",
          "separador de navegação anónima",
          "alterar idioma esta página",
          "google chrome atualizar",
        }
      },
      { "ro",
        {
          "elimină informații",
          "actualizează date de conectare",
          "actualizează informațiile cardului de credit",
          "fereastră incognito",
          "schimbă limba această pagină",
          "google chrome actualizează",
        }
      },
      { "ru",
        {
          "почистить информацию",
          "посмотреть учетные данные",
          "управление данные кредитной карты",
          "анонимный просмотр",
          "изменить язык эту страницу",
          "браузер установить обновление",
        }
      },
      { "sk",
        {
          "odstrániť vyrovnávacia pamäť",
          "aktualizovať prihlasovacie údaje",
          "aktualizovať informácie o kreditnej karte",
          "súkromný režim",
          "zmeniť jazyk túto stránku",
          "google chrome aktualizovať",
        }
      },
      { "sl",
        {
          "izbriši predpomnilnik",
          "upravitelj poverilnice",
          "upravljaj podatke o kreditni kartici",
          "zavihek brez beleženja dejavnosti",
          "spremeni jezik to stran",
          "google chrome nadgradi",
        }
      },
      { "sr",
        {
          "избрисати информација",
          "прегледати креденцијала",
          "ажурирати информацијама о кредитним картицама",
          "картица за приватно прегледање",
          "променити језик ове странице",
          "прегледача надограђивање",
        }
      },
      { "sv",
        {
          "ta bort information",
          "uppdatera användaruppgifter",
          "uppdatera kreditkortsinformation",
          "inkognitofönster",
          "ändra språk denna sida",
          "google chrome installera",
        }
      },
      { "sw",
        {
          "ondoa kumbukumbu",
          "badilisha kitambulisho",
          "badilisha maelezo ya kadi ya mikopo",
          "dirisha la faragha",
          "badilisha lugha ukurasa huu",
          "google chrome installera",
        }
      },
      { "ta",
        {
          "வைப் செய் தற்காலிகச் சேமிப்பு",
          "திருத்து அனுமதிச் சான்றுகள்",
          "திருத்து கிரெடிட் கார்டு தகவல்",
          "தனிப்பட்ட பயன்முறை",
          "மொழியை மாற்று மொழிபெயர்க்கவும்",
          "உலாவி மேம்படுத்து",
        }
      },
      { "te",
        {
          "క్లియర్ చేయడం కుక్కీలు",
          "అప్‌డేట్ చేయడం పాస్‌వర్డ్‌లు",
          "అప్‌డేట్ చేయడం క్రెడిట్ కార్డ్ సమాచారం",
          "ప్రైవేట్ ట్యాబ్",
          "భాష మార్చడం ఈ పేజీ",
          "బ్రౌజర్ అప్‌గ్రేడ్ చేయడం",
        }
      },
      { "th",
        {
          "ล้างข้อมูล ประวัติ",
          "เปลี่ยนแปลง ข้อมูลเข้าสู่ระบบ",
          "เปลี่ยนแปลง ข้อมูลบัตรเครดิต",
          "หน้าต่างไม่ระบุตัวตน",
          "เปลี่ยนภาษา หน้านี้",
          "เบราว์เซอร์ ติดตั้ง",
        }
      },
      { "tr",
        {
          "kaldır önbelleği",
          "görüntüle kimlik bilgilerini",
          "değiştir credit card information",
          "gizli pencereye",
          "dili değiştir bu sayfayı",
          "google chrome yeni sürüme geçir",
        }
      },
      { "uk",
        {
          "видалити інформація",
          "переглянути облікові дані",
          "редагувати дані кредитної картки",
          "вікно в режимі анонімного перегляду",
          "змінити мову цієї сторінки",
          "веб-переглядач установити",
        }
      },
      { "vi",
        {
          "dọn sạch bộ nhớ đệm",
          "trình quản lý thông tin đăng nhập",
          "chỉnh sửa thông tin thẻ tín dụng",
          "chế độ riêng tư",
          "thay đổi ngôn ngữ trang này",
          "trình duyệt cập nhật",
        }
      },
      { "zh-CN",
        {
          "删除浏览历史",
          "保存凭据",
          "保存信用卡信息",
          "启动无痕模式",
          "切换语言",
          "google chrome升级",
        }
      },
      { "zh-TW",
        {
          "刪除 歷史記錄",
          "管理員 密碼",
          "儲存 信用卡資訊",
          "私密瀏覽模式",
          "變更語言 這個頁面",
          "google chrome 升級",
        }
      },
      // clang-format on
  };
  AutocompleteInput input;
  input.set_current_url(GURL("https://example.com"));
  for (const TestCase& test_case : test_cases) {
    // Prepare the shared ResourceBundle with data for tested locale.
    env->SetVar("LANG", test_case.locale);
    ui::ResourceBundle::GetSharedInstance().ReloadLocaleResources(
        test_case.locale);

    // Instantiating the provider loads concept data from shared ResourceBundle.
    OmniboxPedalProvider provider(client);

    EXPECT_EQ(provider.FindPedalMatch(input, base::UTF8ToUTF16("")), nullptr);
#if BUILDFLAG(IS_CHROMEOS_ASH)
    // TODO(orinj): Get ChromeOS to use the right dataset, but for now make this
    //  a soft failure so as to not block all other platforms. To ensure this
    //  is not going to cause failure in production, still test that English
    //  triggering functions. Data is there; it works; but warn about locale.
    if (!provider.FindPedalMatch(input,
                                 base::UTF8ToUTF16(test_case.triggers[0]))) {
      EXPECT_NE(
          provider.FindPedalMatch(input, base::UTF8ToUTF16("clear history")),
          nullptr);
      LOG(WARNING) << "ChromeOS using English for locale " << test_case.locale;
      continue;
    }
#endif

    for (const std::string& trigger : test_case.triggers) {
      EXPECT_NE(provider.FindPedalMatch(input, base::UTF8ToUTF16(trigger)),
                nullptr)
          << "locale: " << test_case.locale << std::endl
          << "trigger: " << trigger;
    }
  }
}
