// Created on: 2013-09-26
// Created by: Denis BOGOLEPOV
// Copyright (c) 2013-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <typeinfo>

#include <OpenGl_AspectFace.hxx>
#include <OpenGl_AspectLine.hxx>
#include <OpenGl_AspectMarker.hxx>
#include <OpenGl_AspectText.hxx>
#include <OpenGl_Clipping.hxx>
#include <OpenGl_Context.hxx>
#include <OpenGl_ShaderManager.hxx>
#include <OpenGl_ShaderProgram.hxx>
#include <OpenGl_Workspace.hxx>

IMPLEMENT_STANDARD_HANDLE (OpenGl_SetOfShaderPrograms, Standard_Transient)
IMPLEMENT_STANDARD_RTTIEXT(OpenGl_SetOfShaderPrograms, Standard_Transient)

IMPLEMENT_STANDARD_HANDLE (OpenGl_ShaderManager, Standard_Transient)
IMPLEMENT_STANDARD_RTTIEXT(OpenGl_ShaderManager, Standard_Transient)

namespace
{

#define EOL "\n"

//! Auxiliary function to transform normal
const char THE_FUNC_transformNormal[] =
  EOL"vec3 transformNormal (in vec3 theNormal)"
  EOL"{"
  EOL"  vec4 aResult = occWorldViewMatrixInverseTranspose"
  EOL"               * occModelWorldMatrixInverseTranspose"
  EOL"               * vec4 (theNormal, 0.0);"
  EOL"  return normalize (aResult.xyz);"
  EOL"}";

//! Global shader variable for color definition with lighting enabled.
const char THE_FUNC_lightDef[] =
  EOL"vec3 Ambient;"   //!< Ambient  contribution of light sources
  EOL"vec3 Diffuse;"   //!< Diffuse  contribution of light sources
  EOL"vec3 Specular;"; //!< Specular contribution of light sources

//! Function computes contribution of isotropic point light source
const char THE_FUNC_pointLight[] =
  EOL"void pointLight (in int  theId,"
  EOL"                 in vec3 theNormal,"
  EOL"                 in vec3 theView,"
  EOL"                 in vec3 thePoint,"
  EOL"                 in bool theIsFront)"
  EOL"{"
  EOL"  vec3 aLight = occLight_Position (theId).xyz;"
  EOL"  if (occLight_IsHeadlight (theId) == 0)"
  EOL"  {"
  EOL"    aLight = vec3 (occWorldViewMatrix * occModelWorldMatrix * vec4 (aLight, 1.0));"
  EOL"  }"
  EOL"  aLight -= thePoint;"
  EOL
  EOL"  float aDist = length (aLight);"
  EOL"  aLight = aLight * (1.0 / aDist);"
  EOL
  EOL"  float anAtten = 1.0 / (occLight_ConstAttenuation  (theId)"
  EOL"                       + occLight_LinearAttenuation (theId) * aDist);"
  EOL
  EOL"  vec3 aHalf = normalize (aLight + theView);"
  EOL
  EOL"  vec3  aFaceSideNormal = theIsFront ? theNormal : -theNormal;"
  EOL"  float aNdotL = max (0.0, dot (aFaceSideNormal, aLight));"
  EOL"  float aNdotH = max (0.0, dot (aFaceSideNormal, aHalf ));"
  EOL
  EOL"  float aSpecl = 0.0;"
  EOL"  if (aNdotL > 0.0)"
  EOL"  {"
  EOL"    aSpecl = pow (aNdotH, theIsFront ? occFrontMaterial_Shininess() : occBackMaterial_Shininess());"
  EOL"  }"
  EOL
  EOL"Diffuse  += occLight_Diffuse  (theId).rgb * aNdotL * anAtten;"
  EOL"Specular += occLight_Specular (theId).rgb * aSpecl * anAtten;"
  EOL"}";

//! Function computes contribution of spotlight source
const char THE_FUNC_spotLight[] =
  EOL"void spotLight (in int  theId,"
  EOL"                in vec3 theNormal,"
  EOL"                in vec3 theView,"
  EOL"                in vec3 thePoint,"
  EOL"                in bool theIsFront)"
  EOL"{"
  EOL"  vec3 aLight   = occLight_Position      (theId).xyz;"
  EOL"  vec3 aSpotDir = occLight_SpotDirection (theId).xyz;"
  EOL"  if (occLight_IsHeadlight (theId) == 0)"
  EOL"  {"
  EOL"    aLight   = vec3 (occWorldViewMatrix * occModelWorldMatrix * vec4 (aLight,   1.0));"
  EOL"    aSpotDir = vec3 (occWorldViewMatrix * occModelWorldMatrix * vec4 (aSpotDir, 0.0));"
  EOL"  }"
  EOL"  aLight -= thePoint;"
  EOL
  EOL"  float aDist = length (aLight);"
  EOL"  aLight = aLight * (1.0 / aDist);"
  EOL
  EOL"  aSpotDir = normalize (aSpotDir);"
  // light cone
  EOL"  float aCosA = dot (aSpotDir, -aLight);"
  EOL"  if (aCosA >= 1.0 || aCosA < cos (occLight_SpotCutOff (theId)))"
  EOL"  {"
  EOL"    return;"
  EOL"  }"
  EOL
  EOL"  float anExponent = occLight_SpotExponent (theId);"
  EOL"  float anAtten    = 1.0 / (occLight_ConstAttenuation  (theId)"
  EOL"                          + occLight_LinearAttenuation (theId) * aDist);"
  EOL"  if (anExponent > 0.0)"
  EOL"  {"
  EOL"    anAtten *= pow (aCosA, anExponent * 128.0);"
  EOL"  }"
  EOL
  EOL"  vec3 aHalf = normalize (aLight + theView);"
  EOL
  EOL"  vec3  aFaceSideNormal = theIsFront ? theNormal : -theNormal;"
  EOL"  float aNdotL = max (0.0, dot (aFaceSideNormal, aLight));"
  EOL"  float aNdotH = max (0.0, dot (aFaceSideNormal, aHalf ));"
  EOL
  EOL"  float aSpecl = 0.0;"
  EOL"  if (aNdotL > 0.0)"
  EOL"  {"
  EOL"    aSpecl = pow (aNdotH, theIsFront ? occFrontMaterial_Shininess() : occBackMaterial_Shininess());"
  EOL"  }"
  EOL
  EOL"  Diffuse  += occLight_Diffuse  (theId).rgb * aNdotL * anAtten;"
  EOL"  Specular += occLight_Specular (theId).rgb * aSpecl * anAtten;"
  EOL"}";

//! Function computes contribution of directional light source
const char THE_FUNC_directionalLight[] =
  EOL"void directionalLight (in int  theId,"
  EOL"                       in vec3 theNormal,"
  EOL"                       in vec3 theView,"
  EOL"                       in bool theIsFront)"
  EOL"{"
  EOL"  vec3 aLight = normalize (occLight_Position (theId).xyz);"
  EOL"  if (occLight_IsHeadlight (theId) == 0)"
  EOL"  {"
  EOL"    aLight = vec3 (occWorldViewMatrix * occModelWorldMatrix * vec4 (aLight, 0.0));"
  EOL"  }"
  EOL
  EOL"  vec3 aHalf = normalize (aLight + theView);"
  EOL
  EOL"  vec3  aFaceSideNormal = theIsFront ? theNormal : -theNormal;"
  EOL"  float aNdotL = max (0.0, dot (aFaceSideNormal, aLight));"
  EOL"  float aNdotH = max (0.0, dot (aFaceSideNormal, aHalf ));"
  EOL
  EOL"  float aSpecl = 0.0;"
  EOL"  if (aNdotL > 0.0)"
  EOL"  {"
  EOL"    aSpecl = pow (aNdotH, theIsFront ? occFrontMaterial_Shininess() : occBackMaterial_Shininess());"
  EOL"  }"
  EOL
  EOL"  Diffuse  += occLight_Diffuse  (theId).rgb * aNdotL;"
  EOL"  Specular += occLight_Specular (theId).rgb * aSpecl;"
  EOL"}";

//! Process clipping planes in Fragment Shader.
//! Should be added at the beginning of the main() function.
const char THE_FRAG_CLIP_PLANES[] =
  EOL"  for (int aPlaneIter = 0; aPlaneIter < occClipPlaneCount; ++aPlaneIter)"
  EOL"  {"
  EOL"    vec4 aClipEquation = occClipPlaneEquations[aPlaneIter];"
  EOL"    int  aClipSpace    = occClipPlaneSpaces[aPlaneIter];"
  EOL"    if (aClipSpace == OccEquationCoords_World)"
  EOL"    {"
  EOL"      if (dot (aClipEquation.xyz, PositionWorld.xyz) + aClipEquation.w < 0.0)"
  EOL"      {"
  EOL"        discard;"
  EOL"      }"
  EOL"    }"
  EOL"    else if (aClipSpace == OccEquationCoords_View)"
  EOL"    {"
  EOL"      if (dot (aClipEquation.xyz, Position.xyz) + aClipEquation.w < 0.0)"
  EOL"      {"
  EOL"        discard;"
  EOL"      }"
  EOL"    }"
  EOL"  }";

}

// =======================================================================
// function : OpenGl_ShaderManager
// purpose  : Creates new empty shader manager
// =======================================================================
OpenGl_ShaderManager::OpenGl_ShaderManager (OpenGl_Context* theContext)
: myShadingModel (Visual3d_TOM_VERTEX),
  myContext  (theContext),
  myLastView (NULL)
{
  //
}

// =======================================================================
// function : ~OpenGl_ShaderManager
// purpose  : Releases resources of shader manager
// =======================================================================
OpenGl_ShaderManager::~OpenGl_ShaderManager()
{
  myProgramList.Clear();
}

// =======================================================================
// function : clear
// purpose  :
// =======================================================================
void OpenGl_ShaderManager::clear()
{
  myProgramList.Clear();
  myLightPrograms.Nullify();
  myFlatPrograms = OpenGl_SetOfShaderPrograms();
  myMapOfLightPrograms.Clear();
  myFontProgram.Nullify();
  switchLightPrograms();
}

// =======================================================================
// function : Create
// purpose  : Creates new shader program
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::Create (const Handle(Graphic3d_ShaderProgram)& theProxy,
                                               TCollection_AsciiString&               theShareKey,
                                               Handle(OpenGl_ShaderProgram)&          theProgram)
{
  theProgram.Nullify();
  if (theProxy.IsNull())
  {
    return Standard_False;
  }

  theShareKey = theProxy->GetId();
  if (myContext->GetResource<Handle(OpenGl_ShaderProgram)> (theShareKey, theProgram))
  {
    if (theProgram->Share())
    {
      myProgramList.Append (theProgram);
    }
    return Standard_True;
  }

  theProgram = new OpenGl_ShaderProgram (theProxy);
  if (!theProgram->Initialize (myContext, theProxy->ShaderObjects()))
  {
    theProgram->Release (myContext);
    theShareKey.Clear();
    theProgram.Nullify();
    return Standard_False;
  }

  myProgramList.Append (theProgram);
  myContext->ShareResource (theShareKey, theProgram);
  return Standard_True;
}

// =======================================================================
// function : Unregister
// purpose  : Removes specified shader program from the manager
// =======================================================================
void OpenGl_ShaderManager::Unregister (TCollection_AsciiString&      theShareKey,
                                       Handle(OpenGl_ShaderProgram)& theProgram)
{
  for (OpenGl_ShaderProgramList::Iterator anIt (myProgramList); anIt.More(); anIt.Next())
  {
    if (anIt.Value() == theProgram)
    {
      if (!theProgram->UnShare())
      {
        theShareKey.Clear();
        theProgram.Nullify();
        return;
      }

      myProgramList.Remove (anIt);
      myMaterialStates.UnBind (theProgram);
      break;
    }
  }

  const TCollection_AsciiString anID = theProgram->myProxy->GetId();
  if (anID.IsEmpty())
  {
    myContext->DelayedRelease (theProgram);
    theProgram.Nullify();
  }
  else
  {
    theProgram.Nullify();
    myContext->ReleaseResource (anID, Standard_True);
  }
}

// =======================================================================
// function : ShaderPrograms
// purpose  : Returns list of registered shader programs
// =======================================================================
const OpenGl_ShaderProgramList& OpenGl_ShaderManager::ShaderPrograms() const
{
  return myProgramList;
}

// =======================================================================
// function : Empty
// purpose  : Returns true if no program objects are attached
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::IsEmpty() const
{
  return myProgramList.IsEmpty();
}

// =======================================================================
// function : switchLightPrograms
// purpose  :
// =======================================================================
void OpenGl_ShaderManager::switchLightPrograms()
{
  TCollection_AsciiString aKey (myShadingModel == Visual3d_TOM_FRAGMENT ? "p_" : "g_");
  const OpenGl_ListOfLight* aLights = myLightSourceState.LightSources();
  if (aLights != NULL)
  {
    for (OpenGl_ListOfLight::Iterator aLightIter (*aLights); aLightIter.More(); aLightIter.Next())
    {
      switch (aLightIter.Value().Type)
      {
        case Visual3d_TOLS_AMBIENT:
          break; // skip ambient
        case Visual3d_TOLS_DIRECTIONAL:
          aKey += "d";
          break;
        case Visual3d_TOLS_POSITIONAL:
          aKey += "p";
          break;
        case Visual3d_TOLS_SPOT:
          aKey += "s";
          break;
      }
    }
  }

  if (!myMapOfLightPrograms.Find (aKey, myLightPrograms))
  {
    myLightPrograms = new OpenGl_SetOfShaderPrograms();
    myMapOfLightPrograms.Bind (aKey, myLightPrograms);
  }
}

// =======================================================================
// function : UpdateLightSourceStateTo
// purpose  : Updates state of OCCT light sources
// =======================================================================
void OpenGl_ShaderManager::UpdateLightSourceStateTo (const OpenGl_ListOfLight* theLights)
{
  myLightSourceState.Set (theLights);
  myLightSourceState.Update();
  switchLightPrograms();
}

// =======================================================================
// function : SetShadingModel
// purpose  :
// =======================================================================
void OpenGl_ShaderManager::SetShadingModel (const Visual3d_TypeOfModel theModel)
{
  myShadingModel = theModel;
  switchLightPrograms();
}

// =======================================================================
// function : SetProjectionState
// purpose  : Sets new state of OCCT projection transform
// =======================================================================
void OpenGl_ShaderManager::UpdateProjectionStateTo (const OpenGl_Mat4& theProjectionMatrix)
{
  myProjectionState.Set (theProjectionMatrix);
  myProjectionState.Update();
}

// =======================================================================
// function : SetModelWorldState
// purpose  : Sets new state of OCCT model-world transform
// =======================================================================
void OpenGl_ShaderManager::UpdateModelWorldStateTo (const OpenGl_Mat4& theModelWorldMatrix)
{
  myModelWorldState.Set (theModelWorldMatrix);
  myModelWorldState.Update();
}

// =======================================================================
// function : SetWorldViewState
// purpose  : Sets new state of OCCT world-view transform
// =======================================================================
void OpenGl_ShaderManager::UpdateWorldViewStateTo (const OpenGl_Mat4& theWorldViewMatrix)
{
  myWorldViewState.Set (theWorldViewMatrix);
  myWorldViewState.Update();
}

// =======================================================================
// function : RevertProjectionStateTo
// purpose  : Reverts state of OCCT projection transform
// =======================================================================
void OpenGl_ShaderManager::RevertProjectionStateTo (const OpenGl_Mat4& theProjectionMatrix)
{
  myProjectionState.Set (theProjectionMatrix);
  myProjectionState.Revert();
}

// =======================================================================
// function : RevertModelWorldStateTo
// purpose  : Reverts state of OCCT model-world transform
// =======================================================================
void OpenGl_ShaderManager::RevertModelWorldStateTo (const OpenGl_Mat4& theModelWorldMatrix)
{
  myModelWorldState.Set (theModelWorldMatrix);
  myModelWorldState.Revert();
}

// =======================================================================
// function : RevertWorldViewStateTo
// purpose  : Reverts state of OCCT world-view transform
// =======================================================================
void OpenGl_ShaderManager::RevertWorldViewStateTo (const OpenGl_Mat4& theWorldViewMatrix)
{
  myWorldViewState.Set (theWorldViewMatrix);
  myWorldViewState.Revert();
}

// =======================================================================
// function : LightSourceState
// purpose  : Returns current state of OCCT light sources
// =======================================================================
const OpenGl_LightSourceState& OpenGl_ShaderManager::LightSourceState() const
{
  return myLightSourceState;
}

// =======================================================================
// function : ProjectionState
// purpose  : Returns current state of OCCT projection transform
// =======================================================================
const OpenGl_ProjectionState& OpenGl_ShaderManager::ProjectionState() const
{
  return myProjectionState;
}

// =======================================================================
// function : ModelWorldState
// purpose  : Returns current state of OCCT model-world transform
// =======================================================================
const OpenGl_ModelWorldState& OpenGl_ShaderManager::ModelWorldState() const
{
  return myModelWorldState;
}

// =======================================================================
// function : WorldViewState
// purpose  : Returns current state of OCCT world-view transform
// =======================================================================
const OpenGl_WorldViewState& OpenGl_ShaderManager::WorldViewState() const
{
  return myWorldViewState;
}

//! Packed properties of light source
class OpenGl_ShaderLightParameters
{
public:

  OpenGl_Vec4 Color;
  OpenGl_Vec4 Position;
  OpenGl_Vec4 Direction;
  OpenGl_Vec4 Parameters;

  //! Returns packed (serialized) representation of light source properties
  const OpenGl_Vec4* Packed() const { return reinterpret_cast<const OpenGl_Vec4*> (this); }
  static Standard_Integer NbOfVec4() { return 4; }

};

//! Packed light source type information
class OpenGl_ShaderLightType
{
public:

  Standard_Integer Type;
  Standard_Integer IsHeadlight;

  //! Returns packed (serialized) representation of light source type
  const OpenGl_Vec2i* Packed() const { return reinterpret_cast<const OpenGl_Vec2i*> (this); }
  static Standard_Integer NbOfVec2i() { return 1; }

};

// =======================================================================
// function : PushLightSourceState
// purpose  : Pushes state of OCCT light sources to the program
// =======================================================================
void OpenGl_ShaderManager::PushLightSourceState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (myLightSourceState.Index() == theProgram->ActiveState (OpenGl_LIGHT_SOURCES_STATE)
   || !theProgram->IsValid())
  {
    return;
  }

  OpenGl_ShaderLightType* aLightTypeArray = new OpenGl_ShaderLightType[OpenGLMaxLights];
  for (Standard_Integer aLightIt = 0; aLightIt < OpenGLMaxLights; ++aLightIt)
  {
    aLightTypeArray[aLightIt].Type = -1;
  }

  const Standard_Integer aLightsDefNb = Min (myLightSourceState.LightSources()->Size(), OpenGLMaxLights);
  if (aLightsDefNb < 1)
  {
    theProgram->SetUniform (myContext,
                            theProgram->GetStateLocation (OpenGl_OCC_LIGHT_SOURCE_COUNT),
                            0);
    theProgram->SetUniform (myContext,
                            theProgram->GetStateLocation (OpenGl_OCC_LIGHT_AMBIENT),
                            OpenGl_Vec4 (0.0f, 0.0f, 0.0f, 0.0f));
    theProgram->SetUniform (myContext,
                            theProgram->GetStateLocation (OpenGl_OCC_LIGHT_SOURCE_TYPES),
                            OpenGLMaxLights * OpenGl_ShaderLightType::NbOfVec2i(),
                            aLightTypeArray[0].Packed());
    theProgram->UpdateState (OpenGl_LIGHT_SOURCES_STATE, myLightSourceState.Index());
    delete[] aLightTypeArray;
    return;
  }

  OpenGl_ShaderLightParameters* aLightParamsArray = new OpenGl_ShaderLightParameters[aLightsDefNb];

  OpenGl_Vec4 anAmbient (0.0f, 0.0f, 0.0f, 0.0f);
  Standard_Integer aLightsNb = 0;
  for (OpenGl_ListOfLight::Iterator anIter (*myLightSourceState.LightSources()); anIter.More(); anIter.Next())
  {
    const OpenGl_Light& aLight = anIter.Value();
    if (aLight.Type == Visual3d_TOLS_AMBIENT)
    {
      anAmbient += aLight.Color;
      continue;
    }
    else if (aLightsNb >= OpenGLMaxLights)
    {
      continue;
    }

    OpenGl_ShaderLightType& aLightType = aLightTypeArray[aLightsNb];
    aLightType.Type        = aLight.Type;
    aLightType.IsHeadlight = aLight.IsHeadlight;

    OpenGl_ShaderLightParameters& aLightParams = aLightParamsArray[aLightsNb];
    aLightParams.Color    = aLight.Color;
    aLightParams.Position = aLight.Type == Visual3d_TOLS_DIRECTIONAL
                         ? -aLight.Direction
                         :  aLight.Position;
    if (aLight.Type == Visual3d_TOLS_SPOT)
    {
      aLightParams.Direction = aLight.Direction;
    }
    aLightParams.Parameters = aLight.Params;
    ++aLightsNb;
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_LIGHT_SOURCE_COUNT),
                          aLightsNb);
  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_LIGHT_AMBIENT),
                          anAmbient);
  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_LIGHT_SOURCE_TYPES),
                          OpenGLMaxLights * OpenGl_ShaderLightType::NbOfVec2i(),
                          aLightTypeArray[0].Packed());
  if (aLightsNb > 0)
  {
    theProgram->SetUniform (myContext,
                            theProgram->GetStateLocation (OpenGl_OCC_LIGHT_SOURCE_PARAMS),
                            aLightsNb * OpenGl_ShaderLightParameters::NbOfVec4(),
                            aLightParamsArray[0].Packed());
  }
  delete[] aLightParamsArray;
  delete[] aLightTypeArray;

  theProgram->UpdateState (OpenGl_LIGHT_SOURCES_STATE, myLightSourceState.Index());
}

// =======================================================================
// function : PushProjectionState
// purpose  : Pushes state of OCCT projection transform to the program
// =======================================================================
void OpenGl_ShaderManager::PushProjectionState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (myProjectionState.Index() == theProgram->ActiveState (OpenGl_PROJECTION_STATE))
  {
    return;
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_PROJECTION_MATRIX),
                          myProjectionState.ProjectionMatrix());

  GLint aLocation = theProgram->GetStateLocation (OpenGl_OCC_PROJECTION_MATRIX_INVERSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myProjectionState.ProjectionMatrixInverse());
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_PROJECTION_MATRIX_TRANSPOSE),
                          myProjectionState.ProjectionMatrix(), true);

  aLocation = theProgram->GetStateLocation (OpenGl_OCC_PROJECTION_MATRIX_INVERSE_TRANSPOSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myProjectionState.ProjectionMatrixInverse(), true);
  }

  theProgram->UpdateState (OpenGl_PROJECTION_STATE, myProjectionState.Index());
}

// =======================================================================
// function : PushModelWorldState
// purpose  : Pushes state of OCCT model-world transform to the program
// =======================================================================
void OpenGl_ShaderManager::PushModelWorldState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (myModelWorldState.Index() == theProgram->ActiveState (OpenGl_MODEL_WORLD_STATE))
  {
    return;
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_MODEL_WORLD_MATRIX),
                          myModelWorldState.ModelWorldMatrix());

  GLint aLocation = theProgram->GetStateLocation (OpenGl_OCC_MODEL_WORLD_MATRIX_INVERSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myModelWorldState.ModelWorldMatrixInverse());
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_MODEL_WORLD_MATRIX_TRANSPOSE),
                          myModelWorldState.ModelWorldMatrix(), true);

  aLocation = theProgram->GetStateLocation (OpenGl_OCC_MODEL_WORLD_MATRIX_INVERSE_TRANSPOSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myModelWorldState.ModelWorldMatrixInverse(), true);
  }

  theProgram->UpdateState (OpenGl_MODEL_WORLD_STATE, myModelWorldState.Index());
}

// =======================================================================
// function : PushWorldViewState
// purpose  : Pushes state of OCCT world-view transform to the program
// =======================================================================
void OpenGl_ShaderManager::PushWorldViewState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (myWorldViewState.Index() == theProgram->ActiveState (OpenGl_WORLD_VIEW_STATE))
  {
    return;
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_WORLD_VIEW_MATRIX),
                          myWorldViewState.WorldViewMatrix());

  GLint aLocation = theProgram->GetStateLocation (OpenGl_OCC_WORLD_VIEW_MATRIX_INVERSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myWorldViewState.WorldViewMatrixInverse());
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_WORLD_VIEW_MATRIX_TRANSPOSE),
                          myWorldViewState.WorldViewMatrix(), true);

  aLocation = theProgram->GetStateLocation (OpenGl_OCC_WORLD_VIEW_MATRIX_INVERSE_TRANSPOSE);
  if (aLocation != OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    theProgram->SetUniform (myContext, aLocation, myWorldViewState.WorldViewMatrixInverse(), true);
  }

  theProgram->UpdateState (OpenGl_WORLD_VIEW_STATE, myWorldViewState.Index());
}

// =======================================================================
// function : UpdateClippingState
// purpose  : Updates state of OCCT clipping planes
// =======================================================================
void OpenGl_ShaderManager::UpdateClippingState()
{
  myClippingState.Update();
}

// =======================================================================
// function : RevertClippingState
// purpose  : Reverts state of OCCT clipping planes
// =======================================================================
void OpenGl_ShaderManager::RevertClippingState()
{
  myClippingState.Revert();
}

// =======================================================================
// function : PushClippingState
// purpose  : Pushes state of OCCT clipping planes to the program
// =======================================================================
void OpenGl_ShaderManager::PushClippingState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (myClippingState.Index() == theProgram->ActiveState (OpenGl_CLIP_PLANES_STATE))
  {
    return;
  }

  theProgram->UpdateState (OpenGl_CLIP_PLANES_STATE, myClippingState.Index());
  const GLint aLocEquations = theProgram->GetStateLocation (OpenGl_OCC_CLIP_PLANE_EQUATIONS);
  const GLint aLocSpaces    = theProgram->GetStateLocation (OpenGl_OCC_CLIP_PLANE_SPACES);
  if (aLocEquations == OpenGl_ShaderProgram::INVALID_LOCATION
   && aLocSpaces    == OpenGl_ShaderProgram::INVALID_LOCATION)
  {
    return;
  }

  GLint aPlanesNb = 0;
  for (Graphic3d_SequenceOfHClipPlane::Iterator anIter (myContext->Clipping().Planes());
       anIter.More(); anIter.Next())
  {
    const Handle(Graphic3d_ClipPlane)& aPlane = anIter.Value();
    if (!myContext->Clipping().IsEnabled (aPlane))
    {
      continue;
    }

    ++aPlanesNb;
  }
  if (aPlanesNb < 1)
  {
    return;
  }

  const Standard_Size MAX_CLIP_PLANES = 8;
  OpenGl_Vec4* anEquations = new OpenGl_Vec4[MAX_CLIP_PLANES];
  GLint*       aSpaces     = new GLint      [MAX_CLIP_PLANES];
  GLuint aPlaneId = 0;
  for (Graphic3d_SequenceOfHClipPlane::Iterator anIter (myContext->Clipping().Planes());
       anIter.More(); anIter.Next())
  {
    const Handle(Graphic3d_ClipPlane)& aPlane = anIter.Value();
    if (!myContext->Clipping().IsEnabled (aPlane))
    {
      continue;
    }

    const Graphic3d_ClipPlane::Equation& anEquation = aPlane->GetEquation();
    anEquations[aPlaneId] = OpenGl_Vec4 ((float) anEquation.x(),
                                         (float) anEquation.y(),
                                         (float) anEquation.z(),
                                         (float) anEquation.w());
    aSpaces[aPlaneId] = myContext->Clipping().GetEquationSpace (aPlane);
    ++aPlaneId;
  }

  theProgram->SetUniform (myContext,
                          theProgram->GetStateLocation (OpenGl_OCC_CLIP_PLANE_COUNT),
                          aPlanesNb);
  theProgram->SetUniform (myContext, aLocEquations, MAX_CLIP_PLANES, anEquations);
  theProgram->SetUniform (myContext, aLocSpaces,    MAX_CLIP_PLANES, aSpaces);

  delete[] anEquations;
  delete[] aSpaces;
}

// =======================================================================
// function : UpdateMaterialStateTo
// purpose  : Updates state of OCCT material for specified program
// =======================================================================
void OpenGl_ShaderManager::UpdateMaterialStateTo (const Handle(OpenGl_ShaderProgram)& theProgram,
                                                  const OpenGl_Element*               theAspect)
{
  if (myMaterialStates.IsBound (theProgram))
  {
    OpenGl_MaterialState& aState = myMaterialStates.ChangeFind (theProgram);
    aState.Set (theAspect);
    aState.Update();
  }
  else
  {
    myMaterialStates.Bind       (theProgram, OpenGl_MaterialState (theAspect));
    myMaterialStates.ChangeFind (theProgram).Update();
  }
}

// =======================================================================
// function : ResetMaterialStates
// purpose  : Resets state of OCCT material for all programs
// =======================================================================
void OpenGl_ShaderManager::ResetMaterialStates()
{
  for (OpenGl_ShaderProgramList::Iterator anIt (myProgramList); anIt.More(); anIt.Next())
  {
    anIt.Value()->UpdateState (OpenGl_MATERIALS_STATE, 0);
  }
}

// =======================================================================
// function : MaterialState
// purpose  : Returns state of OCCT material for specified program
// =======================================================================
const OpenGl_MaterialState* OpenGl_ShaderManager::MaterialState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (!myMaterialStates.IsBound (theProgram))
    return NULL;

  return &myMaterialStates.Find (theProgram);
}

namespace
{

static const OpenGl_Vec4 THE_COLOR_BLACK_VEC4 (0.0f, 0.0f, 0.0f, 0.0f);

// =======================================================================
// function : PushAspectFace
// purpose  :
// =======================================================================
static void PushAspectFace (const Handle(OpenGl_Context)&       theCtx,
                            const Handle(OpenGl_ShaderProgram)& theProgram,
                            const OpenGl_AspectFace*            theAspect)
{
  theProgram->SetUniform (theCtx,
                          theProgram->GetStateLocation (OpenGl_OCCT_TEXTURE_ENABLE),
                          (GLint)theAspect->DoTextureMap());
  theProgram->SetUniform (theCtx,
                          theProgram->GetStateLocation (OpenGl_OCCT_ACTIVE_SAMPLER),
                          0 /* GL_TEXTURE0 */);
  theProgram->SetUniform (theCtx,
                          theProgram->GetStateLocation (OpenGl_OCCT_DISTINGUISH_MODE),
                          theAspect->DistinguishingMode());

  OpenGl_Material aParams;
  for (Standard_Integer anIndex = 0; anIndex < 2; ++anIndex)
  {
    const GLint aLoc = theProgram->GetStateLocation (anIndex == 0
                                                   ? OpenGl_OCCT_FRONT_MATERIAL
                                                   : OpenGl_OCCT_BACK_MATERIAL);
    if (aLoc == OpenGl_ShaderProgram::INVALID_LOCATION)
    {
      continue;
    }

    aParams.Init (anIndex == 0 ? theAspect->IntFront() : theAspect->IntBack());
    theProgram->SetUniform (theCtx, aLoc, OpenGl_Material::NbOfVec4(),
                            aParams.Packed());
  }
}

// =======================================================================
// function : PushAspectLine
// purpose  :
// =======================================================================
static void PushAspectLine (const Handle(OpenGl_Context)&       theCtx,
                            const Handle(OpenGl_ShaderProgram)& theProgram,
                            const OpenGl_AspectLine*            theAspect)
{
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_TEXTURE_ENABLE),   TOff);
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_DISTINGUISH_MODE), TOff);

  const OpenGl_Vec4 aDiffuse (theAspect->Color().rgb[0],
                              theAspect->Color().rgb[1],
                              theAspect->Color().rgb[2],
                              theAspect->Color().rgb[3]);
  OpenGl_Vec4 aParams[5];
  aParams[0] = THE_COLOR_BLACK_VEC4;
  aParams[1] = THE_COLOR_BLACK_VEC4;
  aParams[2] = aDiffuse;
  aParams[3] = THE_COLOR_BLACK_VEC4;
  aParams[4].x() = 0.0f; // shininess
  aParams[4].y() = 0.0f; // transparency
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_FRONT_MATERIAL),
                          5, aParams);
}

// =======================================================================
// function : PushAspectText
// purpose  :
// =======================================================================
static void PushAspectText (const Handle(OpenGl_Context)&       theCtx,
                            const Handle(OpenGl_ShaderProgram)& theProgram,
                            const OpenGl_AspectText*            theAspect)
{
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_TEXTURE_ENABLE),   TOn);
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_DISTINGUISH_MODE), TOff);
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_ACTIVE_SAMPLER),   0 /* GL_TEXTURE0 */);

  OpenGl_Vec4 aDiffuse (theAspect->Color().rgb[0],
                        theAspect->Color().rgb[1],
                        theAspect->Color().rgb[2],
                        theAspect->Color().rgb[3]);
  if (theAspect->DisplayType() == Aspect_TODT_DEKALE
   || theAspect->DisplayType() == Aspect_TODT_SUBTITLE)
  {
    aDiffuse = OpenGl_Vec4 (theAspect->SubtitleColor().rgb[0],
                            theAspect->SubtitleColor().rgb[1],
                            theAspect->SubtitleColor().rgb[2],
                            theAspect->SubtitleColor().rgb[3]);
  }

  OpenGl_Vec4 aParams[5];
  aParams[0] = THE_COLOR_BLACK_VEC4;
  aParams[1] = THE_COLOR_BLACK_VEC4;
  aParams[2] = aDiffuse;
  aParams[3] = THE_COLOR_BLACK_VEC4;
  aParams[4].x() = 0.0f; // shininess
  aParams[4].y() = 0.0f; // transparency
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_FRONT_MATERIAL),
                          5, aParams);
}

// =======================================================================
// function : PushAspectMarker
// purpose  :
// =======================================================================
static void PushAspectMarker (const Handle(OpenGl_Context)&       theCtx,
                              const Handle(OpenGl_ShaderProgram)& theProgram,
                              const OpenGl_AspectMarker*          theAspect)
{
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_TEXTURE_ENABLE),   TOn);
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_DISTINGUISH_MODE), TOff);
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_ACTIVE_SAMPLER),   0 /* GL_TEXTURE0 */);

  const OpenGl_Vec4 aDiffuse (theAspect->Color().rgb[0],
                              theAspect->Color().rgb[1],
                              theAspect->Color().rgb[2],
                              theAspect->Color().rgb[3]);
  OpenGl_Vec4 aParams[5];
  aParams[0] = THE_COLOR_BLACK_VEC4;
  aParams[1] = THE_COLOR_BLACK_VEC4;
  aParams[2] = aDiffuse;
  aParams[3] = THE_COLOR_BLACK_VEC4;
  aParams[4].x() = 0.0f; // shininess
  aParams[4].y() = 0.0f; // transparency
  theProgram->SetUniform (theCtx, theProgram->GetStateLocation (OpenGl_OCCT_FRONT_MATERIAL),
                          5, aParams);
}

}; // nameless namespace

// =======================================================================
// function : PushMaterialState
// purpose  : Pushes current state of OCCT material to the program
// =======================================================================
void OpenGl_ShaderManager::PushMaterialState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  if (!myMaterialStates.IsBound (theProgram))
  {
    return;
  }

  const OpenGl_MaterialState& aState = myMaterialStates.Find (theProgram);
  if (aState.Index() == theProgram->ActiveState (OpenGl_MATERIALS_STATE))
  {
    return;
  }

  if (typeid (*aState.Aspect()) == typeid (OpenGl_AspectFace))
  {
    PushAspectFace   (myContext, theProgram, dynamic_cast<const OpenGl_AspectFace*> (aState.Aspect()));
  }
  else if (typeid (*aState.Aspect()) == typeid (OpenGl_AspectLine))
  {
    PushAspectLine   (myContext, theProgram, dynamic_cast<const OpenGl_AspectLine*> (aState.Aspect()));
  }
  else if (typeid (*aState.Aspect()) == typeid (OpenGl_AspectText))
  {
    PushAspectText   (myContext, theProgram, dynamic_cast<const OpenGl_AspectText*> (aState.Aspect()));
  }
  else if (typeid (*aState.Aspect()) == typeid (OpenGl_AspectMarker))
  {
    PushAspectMarker (myContext, theProgram, dynamic_cast<const OpenGl_AspectMarker*> (aState.Aspect()));
  }

  theProgram->UpdateState (OpenGl_MATERIALS_STATE, aState.Index());
}

// =======================================================================
// function : PushWorldViewState
// purpose  : Pushes state of OCCT graphics parameters to the program
// =======================================================================
void OpenGl_ShaderManager::PushState (const Handle(OpenGl_ShaderProgram)& theProgram) const
{
  PushClippingState    (theProgram);
  PushMaterialState    (theProgram);
  PushWorldViewState   (theProgram);
  PushModelWorldState  (theProgram);
  PushProjectionState  (theProgram);
  PushLightSourceState (theProgram);
}

// =======================================================================
// function : prepareStdProgramFont
// purpose  :
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::prepareStdProgramFont()
{
  Handle(Graphic3d_ShaderProgram) aProgramSrc = new Graphic3d_ShaderProgram();
  TCollection_AsciiString aSrcVert =
      EOL"void main()"
      EOL"{"
      EOL"  gl_Position = occProjectionMatrix * occWorldViewMatrix * occModelWorldMatrix * occVertex;"
      EOL"}";

  TCollection_AsciiString aSrcFrag =
      EOL"float getAlpha(void) { return texture2D(occActiveSampler, gl_PointCoord).a; }"
      EOL"void main()"
      EOL"{"
      EOL"  vec4 aColor = occColor;"
      EOL"  aColor.a *= getAlpha();"
      EOL"  if (aColor.a <= 0.285) discard;"
      EOL"  gl_FragColor = aColor;"
      EOL"}";

  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_VERTEX,   aSrcVert));
  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_FRAGMENT, aSrcFrag));
  TCollection_AsciiString aKey;
  if (!Create (aProgramSrc, aKey, myFontProgram))
  {
    myFontProgram = new OpenGl_ShaderProgram(); // just mark as invalid
    return Standard_False;
  }
  return Standard_True;
}

// =======================================================================
// function : prepareStdProgramFlat
// purpose  :
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::prepareStdProgramFlat (Handle(OpenGl_ShaderProgram)& theProgram,
                                                              const Standard_Integer        theBits)
{
  Handle(Graphic3d_ShaderProgram) aProgramSrc = new Graphic3d_ShaderProgram();
  TCollection_AsciiString aSrcVert, aSrcVertExtraOut, aSrcVertExtraMain, aSrcFrag, aSrcFragExtraOut, aSrcFragExtraMain;
  TCollection_AsciiString aSrcFragGetColor     = EOL"vec4 getColor(void) { return occColor; }";
  TCollection_AsciiString aSrcFragMainGetColor = EOL"  gl_FragColor = getColor();";
  if ((theBits & OpenGl_PO_Point) != 0)
  {
  #if defined(GL_ES_VERSION_2_0)
    aSrcVertExtraMain += EOL"  gl_PointSize = occPointSize;";
  #endif
    if ((theBits & OpenGl_PO_TextureA) != 0)
    {
      aSrcFragGetColor =
        EOL"float getAlpha(void) { return texture2D(occActiveSampler, gl_PointCoord).a; }"
        EOL"vec4  getColor(void)"
        EOL"{"
        EOL"  vec4 aColor = occColor;"
        EOL"  aColor.a *= getAlpha();"
        EOL"  return aColor;"
        EOL"}";

      aSrcFragMainGetColor =
        EOL"  vec4 aColor = getColor();"
        EOL"  if (aColor.a <= 0.1) discard;"
        EOL"  gl_FragColor = aColor;";
    }
    else if ((theBits & OpenGl_PO_TextureRGB) != 0)
    {
      aSrcFragGetColor =
        EOL"vec4 getColor(void) { return texture2D(occActiveSampler, gl_PointCoord); }";
      aSrcFragMainGetColor =
        EOL"  vec4 aColor = getColor();"
        EOL"  if (aColor.a <= 0.1) discard;"
        EOL"  gl_FragColor = aColor;";
    }
  }
  if ((theBits & OpenGl_PO_VertColor) != 0)
  {
    aSrcVertExtraOut  += EOL"varying vec4 VertColor;";
    aSrcVertExtraMain += EOL"  VertColor = occVertColor;";
    aSrcFragExtraOut  += EOL"varying vec4 VertColor;";
    aSrcFragGetColor  =  EOL"vec4 getColor(void) { return VertColor; }";
  }
  if ((theBits & OpenGl_PO_ClipPlanes) != 0)
  {
    const char THE_POS_VARY[] =
      EOL"varying vec4 PositionWorld;"
      EOL"varying vec4 Position;";

    aSrcVertExtraOut  += THE_POS_VARY;
    aSrcFragExtraOut  += THE_POS_VARY;
    aSrcVertExtraMain +=
      EOL"  PositionWorld = occModelWorldMatrix * occVertex;"
      EOL"  Position      = occWorldViewMatrix * PositionWorld;";
    aSrcFragExtraMain += THE_FRAG_CLIP_PLANES;
  }

  aSrcVert =
      aSrcVertExtraOut
    + EOL"void main()"
      EOL"{"
    + aSrcVertExtraMain
    + EOL"  gl_Position = occProjectionMatrix * occWorldViewMatrix * occModelWorldMatrix * occVertex;"
      EOL"}";

  aSrcFrag =
      aSrcFragExtraOut
    + aSrcFragGetColor
    + EOL"void main()"
      EOL"{"
    + aSrcFragExtraMain
    + aSrcFragMainGetColor
    + EOL"}";

  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_VERTEX,   aSrcVert));
  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_FRAGMENT, aSrcFrag));

  TCollection_AsciiString aKey;
  if (!Create (aProgramSrc, aKey, theProgram))
  {
    theProgram = new OpenGl_ShaderProgram(); // just mark as invalid
    return Standard_False;
  }
  return Standard_True;
}

// =======================================================================
// function : stdComputeLighting
// purpose  :
// =======================================================================
TCollection_AsciiString OpenGl_ShaderManager::stdComputeLighting()
{
  bool aLightsMap[Visual3d_TOLS_SPOT + 1] = { false, false, false, false };
  TCollection_AsciiString aLightsFunc, aLightsLoop;
  const OpenGl_ListOfLight* aLights = myLightSourceState.LightSources();
  if (aLights != NULL)
  {
    Standard_Integer anIndex = 0;
    for (OpenGl_ListOfLight::Iterator aLightIter (*aLights); aLightIter.More(); aLightIter.Next(), ++anIndex)
    {
      switch (aLightIter.Value().Type)
      {
        case Visual3d_TOLS_AMBIENT:
          --anIndex;
          break; // skip ambient
        case Visual3d_TOLS_DIRECTIONAL:
          aLightsLoop = aLightsLoop + EOL"    directionalLight (" + anIndex + ", theNormal, theView, theIsFront);";
          break;
        case Visual3d_TOLS_POSITIONAL:
          aLightsLoop = aLightsLoop + EOL"    pointLight (" + anIndex + ", theNormal, theView, aPoint, theIsFront);";
          break;
        case Visual3d_TOLS_SPOT:
          aLightsLoop = aLightsLoop + EOL"    spotLight (" + anIndex + ", theNormal, theView, aPoint, theIsFront);";
          break;
      }

      bool& aTypeBit = aLightsMap[aLightIter.Value().Type];
      if (aTypeBit)
      {
        continue;
      }

      aTypeBit = true;
      switch (aLightIter.Value().Type)
      {
        case Visual3d_TOLS_AMBIENT:     break;
        case Visual3d_TOLS_DIRECTIONAL: aLightsFunc += THE_FUNC_directionalLight; break;
        case Visual3d_TOLS_POSITIONAL:  aLightsFunc += THE_FUNC_pointLight;       break;
        case Visual3d_TOLS_SPOT:        aLightsFunc += THE_FUNC_spotLight;        break;
      }
    }
  }

  return TCollection_AsciiString()
    + THE_FUNC_lightDef
    + aLightsFunc
    + EOL
      EOL"vec4 computeLighting (in vec3 theNormal,"
      EOL"                      in vec3 theView,"
      EOL"                      in vec4 thePoint,"
      EOL"                      in bool theIsFront)"
      EOL"{"
      EOL"  Ambient  = occLightAmbient.rgb;"
      EOL"  Diffuse  = vec3 (0.0);"
      EOL"  Specular = vec3 (0.0);"
      EOL"  vec3 aPoint = thePoint.xyz / thePoint.w;"
    + aLightsLoop
    + EOL"  vec4 aMaterialAmbient  = theIsFront ? occFrontMaterial_Ambient()  : occBackMaterial_Ambient();"
      EOL"  vec4 aMaterialDiffuse  = theIsFront ? occFrontMaterial_Diffuse()  : occBackMaterial_Diffuse();"
      EOL"  vec4 aMaterialSpecular = theIsFront ? occFrontMaterial_Specular() : occBackMaterial_Specular();"
      EOL"  vec4 aMaterialEmission = theIsFront ? occFrontMaterial_Emission() : occBackMaterial_Emission();"
      EOL"  return vec4 (Ambient,  1.0) * aMaterialAmbient"
      EOL"       + vec4 (Diffuse,  1.0) * aMaterialDiffuse"
      EOL"       + vec4 (Specular, 1.0) * aMaterialSpecular"
      EOL"                              + aMaterialEmission;"
      EOL"}";
}

// =======================================================================
// function : prepareStdProgramGouraud
// purpose  :
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::prepareStdProgramGouraud (Handle(OpenGl_ShaderProgram)& theProgram,
                                                                 const Standard_Integer        theBits)
{
  Handle(Graphic3d_ShaderProgram) aProgramSrc = new Graphic3d_ShaderProgram();
  TCollection_AsciiString aSrcVert, aSrcVertExtraOut, aSrcVertExtraMain, aSrcFrag, aSrcFragExtraOut, aSrcFragExtraMain;
  if ((theBits & OpenGl_PO_Point) != 0)
  {
  #if defined(GL_ES_VERSION_2_0)
    aSrcVertExtraMain += EOL"  gl_PointSize = occPointSize;";
  #endif
  }
  if ((theBits & OpenGl_PO_ClipPlanes) != 0)
  {
    const char THE_POS_VARY[] =
      EOL"varying vec4 PositionWorld;"
      EOL"varying vec4 Position;";

    aSrcVertExtraOut  += THE_POS_VARY;
    aSrcFragExtraOut  += THE_POS_VARY;
    aSrcVertExtraMain +=
      EOL"  PositionWorld = aPositionWorld;"
      EOL"  Position      = aPosition;";
    aSrcFragExtraMain += THE_FRAG_CLIP_PLANES;
  }

  const TCollection_AsciiString aLights = stdComputeLighting();
  aSrcVert = TCollection_AsciiString()
    + THE_FUNC_transformNormal
    + EOL
    + aLights
    + EOL
      EOL"varying vec4 FrontColor;"
      EOL"varying vec4 BackColor;"
      EOL
    + aSrcVertExtraOut
    + EOL"void main()"
      EOL"{"
      EOL"  vec4 aPositionWorld = occModelWorldMatrix * occVertex;"
      EOL"  vec4 aPosition      = occWorldViewMatrix * aPositionWorld;"
      EOL"  vec3 aNormal        = transformNormal (occNormal);"
      EOL"  vec3 aView          = vec3 (0.0, 0.0, 1.0);"
      EOL"  FrontColor  = computeLighting (normalize (aNormal), normalize (aView), aPosition, true);"
      EOL"  BackColor   = computeLighting (normalize (aNormal), normalize (aView), aPosition, false);"
    + aSrcVertExtraMain
    + EOL"  gl_Position = occProjectionMatrix * occWorldViewMatrix * occModelWorldMatrix * occVertex;"
      EOL"}";

  aSrcFrag = TCollection_AsciiString()
    + EOL"varying vec4 FrontColor;"
      EOL"varying vec4 BackColor;"
    + aSrcFragExtraOut
    + EOL"void main()"
      EOL"{"
    + aSrcFragExtraMain
    + EOL"  gl_FragColor = gl_FrontFacing ? FrontColor : BackColor;"
      EOL"}";

  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_VERTEX,   aSrcVert));
  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_FRAGMENT, aSrcFrag));
  TCollection_AsciiString aKey;
  if (!Create (aProgramSrc, aKey, theProgram))
  {
    theProgram = new OpenGl_ShaderProgram(); // just mark as invalid
    return Standard_False;
  }
  return Standard_True;
}

// =======================================================================
// function : prepareStdProgramPhong
// purpose  :
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::prepareStdProgramPhong (Handle(OpenGl_ShaderProgram)& theProgram,
                                                               const Standard_Integer        theBits)
{
  Handle(Graphic3d_ShaderProgram) aProgramSrc = new Graphic3d_ShaderProgram();
  TCollection_AsciiString aSrcVert, aSrcVertExtraOut, aSrcVertExtraMain, aSrcFrag, aSrcFragExtraMain;
  if ((theBits & OpenGl_PO_Point) != 0)
  {
  #if defined(GL_ES_VERSION_2_0)
    aSrcVertExtraMain += EOL"  gl_PointSize = occPointSize;";
  #endif
  }
  if ((theBits & OpenGl_PO_ClipPlanes) != 0)
  {
    aSrcFragExtraMain += THE_FRAG_CLIP_PLANES;
  }

  aSrcVert = TCollection_AsciiString()
    + THE_FUNC_transformNormal
    + EOL
      EOL"varying vec4 PositionWorld;"
      EOL"varying vec4 Position;"
      EOL"varying vec3 Normal;"
      EOL"varying vec3 View;"
      EOL
    + aSrcVertExtraOut
    + EOL"void main()"
      EOL"{"
      EOL"  PositionWorld = occModelWorldMatrix * occVertex;"
      EOL"  Position      = occWorldViewMatrix * PositionWorld;"
      EOL"  Normal        = transformNormal (occNormal);"
      EOL"  View          = vec3 (0.0, 0.0, 1.0);"
    + aSrcVertExtraMain
    + EOL"  gl_Position = occProjectionMatrix * occWorldViewMatrix * occModelWorldMatrix * occVertex;"
      EOL"}";

  const TCollection_AsciiString aLights = stdComputeLighting();
  aSrcFrag = TCollection_AsciiString()
    + EOL"varying vec4 PositionWorld;"
      EOL"varying vec4 Position;"
      EOL"varying vec3 Normal;"
      EOL"varying vec3 View;"
    + EOL
    + aLights
    + EOL
      EOL"void main()"
      EOL"{"
    + aSrcFragExtraMain
    + EOL"  gl_FragColor = computeLighting (normalize (Normal), normalize (View), Position, gl_FrontFacing);"
      EOL"}";

  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_VERTEX,   aSrcVert));
  aProgramSrc->AttachShader (Graphic3d_ShaderObject::CreateFromSource (Graphic3d_TOS_FRAGMENT, aSrcFrag));
  TCollection_AsciiString aKey;
  if (!Create (aProgramSrc, aKey, theProgram))
  {
    theProgram = new OpenGl_ShaderProgram(); // just mark as invalid
    return Standard_False;
  }
  return Standard_True;
}

// =======================================================================
// function : bindProgramWithState
// purpose  :
// =======================================================================
Standard_Boolean OpenGl_ShaderManager::bindProgramWithState (const Handle(OpenGl_ShaderProgram)& theProgram,
                                                             const OpenGl_Element*               theAspect)
{
  if (!myContext->BindProgram (theProgram))
  {
    return Standard_False;
  }
  theProgram->ApplyVariables (myContext);

  const OpenGl_MaterialState* aMaterialState = MaterialState (theProgram);
  if (aMaterialState == NULL || aMaterialState->Aspect() != theAspect)
  {
    UpdateMaterialStateTo (theProgram, theAspect);
  }

  PushState (theProgram);
  return Standard_True;
}
