// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/sync/test/local_sync_test_server.h"

#include <string>

#include "base/command_line.h"
#include "base/path_service.h"
#include "base/strings/string_number_conversions.h"
#include "base/values.h"
#include "net/test/python_utils.h"
#include "net/test/spawned_test_server/spawned_test_server.h"

namespace syncer {

LocalSyncTestServer::LocalSyncTestServer()
    : LocalTestServer(
          net::SpawnedTestServer::TYPE_HTTP,  // Sync uses the HTTP scheme.
          base::FilePath()),
      xmpp_port_(0) {}

LocalSyncTestServer::LocalSyncTestServer(uint16_t port, uint16_t xmpp_port)
    : LocalTestServer(
          net::SpawnedTestServer::TYPE_HTTP,  // Sync uses the HTTP scheme.
          base::FilePath()),
      xmpp_port_(xmpp_port) {
  SetPort(port);
}

LocalSyncTestServer::~LocalSyncTestServer() {}

bool LocalSyncTestServer::AddCommandLineArguments(
    base::CommandLine* command_line) const {
  // Suppress loading of site packages in order to avoid a version mismatch
  // between pre-installed versions of python libraries.
  // See https://bugs.chromium.org/p/chromium/issues/detail?id=649829
  command_line->AppendSwitch("-S");
  if (!LocalTestServer::AddCommandLineArguments(command_line))
    return false;
  if (xmpp_port_ != 0) {
    std::string xmpp_port_str = base::UintToString(xmpp_port_);
    command_line->AppendArg("--xmpp-port=" + xmpp_port_str);
  }
  return true;
}

bool LocalSyncTestServer::GetTestServerPath(
    base::FilePath* testserver_path) const {
  base::FilePath testserver_dir;
  if (!base::PathService::Get(base::DIR_SOURCE_ROOT, &testserver_dir)) {
    LOG(ERROR) << "Failed to get DIR_SOURCE_ROOT";
    return false;
  }
  testserver_dir = testserver_dir.Append(FILE_PATH_LITERAL("components"))
                       .Append(FILE_PATH_LITERAL("sync"))
                       .Append(FILE_PATH_LITERAL("tools"))
                       .Append(FILE_PATH_LITERAL("testserver"));

  *testserver_path =
      testserver_dir.Append(FILE_PATH_LITERAL("sync_testserver.py"));
  return true;
}

bool LocalSyncTestServer::GetTestScriptPath(
    const base::FilePath::StringType& test_script_name,
    base::FilePath* test_script_path) const {
  base::FilePath testserver_path;
  if (!GetTestServerPath(&testserver_path))
    return false;
  *test_script_path = testserver_path.DirName().Append(test_script_name);
  return true;
}

bool LocalSyncTestServer::SetPythonPath() const {
  if (!LocalTestServer::SetPythonPath())
    return false;

  // Add the net/tools/testserver directory to the path, so that testserver_base
  // can be imported.
  base::FilePath net_testserver_path;
  if (!LocalTestServer::GetTestServerPath(&net_testserver_path)) {
    LOG(ERROR) << "Failed to get net testserver path.";
    return false;
  }
  AppendToPythonPath(net_testserver_path.DirName());

  // Locate the Python code generated by the sync protocol buffers compiler.
  base::FilePath pyproto_dir;
  if (!GetPyProtoPath(&pyproto_dir)) {
    LOG(WARNING) << "Cannot find pyproto dir for generated code. "
                 << "Testserver features that rely on it will not work";
    return true;
  }
  AppendToPythonPath(pyproto_dir.AppendASCII("components")
                         .AppendASCII("sync")
                         .AppendASCII("protocol"));
  return true;
}

}  // namespace syncer
