// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/websockets/websocket_frame.h"

#include <stdint.h>
#include <algorithm>
#include <vector>

#include "base/memory/aligned_memory.h"
#include "base/stl_util.h"
#include "net/base/net_errors.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace net {

namespace {

TEST(WebSocketFrameHeaderTest, FrameLengths) {
  struct TestCase {
    const char* frame_header;
    size_t frame_header_length;
    uint64_t frame_length;
  };
  static const TestCase kTests[] = {
    { "\x81\x00", 2, UINT64_C(0) },
    { "\x81\x7D", 2, UINT64_C(125) },
    { "\x81\x7E\x00\x7E", 4, UINT64_C(126) },
    { "\x81\x7E\xFF\xFF", 4, UINT64_C(0xFFFF) },
    { "\x81\x7F\x00\x00\x00\x00\x00\x01\x00\x00", 10, UINT64_C(0x10000) },
    { "\x81\x7F\x7F\xFF\xFF\xFF\xFF\xFF\xFF\xFF", 10,
      UINT64_C(0x7FFFFFFFFFFFFFFF) }
  };
  static const int kNumTests = base::size(kTests);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketFrameHeader header(WebSocketFrameHeader::kOpCodeText);
    header.final = true;
    header.payload_length = kTests[i].frame_length;

    std::vector<char> expected_output(
        kTests[i].frame_header,
        kTests[i].frame_header + kTests[i].frame_header_length);
    std::vector<char> output(expected_output.size());
    EXPECT_EQ(static_cast<int>(expected_output.size()),
              WriteWebSocketFrameHeader(
                  header, NULL, &output.front(), output.size()));
    EXPECT_EQ(expected_output, output);
  }
}

TEST(WebSocketFrameHeaderTest, FrameLengthsWithMasking) {
  static const char kMaskingKey[] = "\xDE\xAD\xBE\xEF";
  static_assert(
      base::size(kMaskingKey) - 1 == WebSocketFrameHeader::kMaskingKeyLength,
      "incorrect masking key size");

  struct TestCase {
    const char* frame_header;
    size_t frame_header_length;
    uint64_t frame_length;
  };
  static const TestCase kTests[] = {
    { "\x81\x80\xDE\xAD\xBE\xEF", 6, UINT64_C(0) },
    { "\x81\xFD\xDE\xAD\xBE\xEF", 6, UINT64_C(125) },
    { "\x81\xFE\x00\x7E\xDE\xAD\xBE\xEF", 8, UINT64_C(126) },
    { "\x81\xFE\xFF\xFF\xDE\xAD\xBE\xEF", 8, UINT64_C(0xFFFF) },
    { "\x81\xFF\x00\x00\x00\x00\x00\x01\x00\x00\xDE\xAD\xBE\xEF", 14,
      UINT64_C(0x10000) },
    { "\x81\xFF\x7F\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xDE\xAD\xBE\xEF", 14,
      UINT64_C(0x7FFFFFFFFFFFFFFF) }
  };
  static const int kNumTests = base::size(kTests);

  WebSocketMaskingKey masking_key;
  std::copy(kMaskingKey,
            kMaskingKey + WebSocketFrameHeader::kMaskingKeyLength,
            masking_key.key);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketFrameHeader header(WebSocketFrameHeader::kOpCodeText);
    header.final = true;
    header.masked = true;
    header.payload_length = kTests[i].frame_length;

    std::vector<char> expected_output(
        kTests[i].frame_header,
        kTests[i].frame_header + kTests[i].frame_header_length);
    std::vector<char> output(expected_output.size());
    EXPECT_EQ(static_cast<int>(expected_output.size()),
              WriteWebSocketFrameHeader(
                  header, &masking_key, &output.front(), output.size()));
    EXPECT_EQ(expected_output, output);
  }
}

TEST(WebSocketFrameHeaderTest, FrameOpCodes) {
  struct TestCase {
    const char* frame_header;
    size_t frame_header_length;
    WebSocketFrameHeader::OpCode opcode;
  };
  static const TestCase kTests[] = {
    { "\x80\x00", 2, WebSocketFrameHeader::kOpCodeContinuation },
    { "\x81\x00", 2, WebSocketFrameHeader::kOpCodeText },
    { "\x82\x00", 2, WebSocketFrameHeader::kOpCodeBinary },
    { "\x88\x00", 2, WebSocketFrameHeader::kOpCodeClose },
    { "\x89\x00", 2, WebSocketFrameHeader::kOpCodePing },
    { "\x8A\x00", 2, WebSocketFrameHeader::kOpCodePong },
    // These are undefined opcodes, but the builder should accept them anyway.
    { "\x83\x00", 2, 0x3 },
    { "\x84\x00", 2, 0x4 },
    { "\x85\x00", 2, 0x5 },
    { "\x86\x00", 2, 0x6 },
    { "\x87\x00", 2, 0x7 },
    { "\x8B\x00", 2, 0xB },
    { "\x8C\x00", 2, 0xC },
    { "\x8D\x00", 2, 0xD },
    { "\x8E\x00", 2, 0xE },
    { "\x8F\x00", 2, 0xF }
  };
  static const int kNumTests = base::size(kTests);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketFrameHeader header(kTests[i].opcode);
    header.final = true;
    header.payload_length = 0;

    std::vector<char> expected_output(
        kTests[i].frame_header,
        kTests[i].frame_header + kTests[i].frame_header_length);
    std::vector<char> output(expected_output.size());
    EXPECT_EQ(static_cast<int>(expected_output.size()),
              WriteWebSocketFrameHeader(
                  header, NULL, &output.front(), output.size()));
    EXPECT_EQ(expected_output, output);
  }
}

TEST(WebSocketFrameHeaderTest, FinalBitAndReservedBits) {
  struct TestCase {
    const char* frame_header;
    size_t frame_header_length;
    bool final;
    bool reserved1;
    bool reserved2;
    bool reserved3;
  };
  static const TestCase kTests[] = {
    { "\x81\x00", 2, true, false, false, false },
    { "\x01\x00", 2, false, false, false, false },
    { "\xC1\x00", 2, true, true, false, false },
    { "\xA1\x00", 2, true, false, true, false },
    { "\x91\x00", 2, true, false, false, true },
    { "\x71\x00", 2, false, true, true, true },
    { "\xF1\x00", 2, true, true, true, true }
  };
  static const int kNumTests = base::size(kTests);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketFrameHeader header(WebSocketFrameHeader::kOpCodeText);
    header.final = kTests[i].final;
    header.reserved1 = kTests[i].reserved1;
    header.reserved2 = kTests[i].reserved2;
    header.reserved3 = kTests[i].reserved3;
    header.payload_length = 0;

    std::vector<char> expected_output(
        kTests[i].frame_header,
        kTests[i].frame_header + kTests[i].frame_header_length);
    std::vector<char> output(expected_output.size());
    EXPECT_EQ(static_cast<int>(expected_output.size()),
              WriteWebSocketFrameHeader(
                  header, NULL, &output.front(), output.size()));
    EXPECT_EQ(expected_output, output);
  }
}

TEST(WebSocketFrameHeaderTest, InsufficientBufferSize) {
  struct TestCase {
    uint64_t payload_length;
    bool masked;
    size_t expected_header_size;
  };
  static const TestCase kTests[] = {
    { UINT64_C(0), false, 2u },
    { UINT64_C(125), false, 2u },
    { UINT64_C(126), false, 4u },
    { UINT64_C(0xFFFF), false, 4u },
    { UINT64_C(0x10000), false, 10u },
    { UINT64_C(0x7FFFFFFFFFFFFFFF), false, 10u },
    { UINT64_C(0), true, 6u },
    { UINT64_C(125), true, 6u },
    { UINT64_C(126), true, 8u },
    { UINT64_C(0xFFFF), true, 8u },
    { UINT64_C(0x10000), true, 14u },
    { UINT64_C(0x7FFFFFFFFFFFFFFF), true, 14u }
  };
  static const int kNumTests = base::size(kTests);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketFrameHeader header(WebSocketFrameHeader::kOpCodeText);
    header.final = true;
    header.opcode = WebSocketFrameHeader::kOpCodeText;
    header.masked = kTests[i].masked;
    header.payload_length = kTests[i].payload_length;

    char dummy_buffer[14];
    // Set an insufficient size to |buffer_size|.
    EXPECT_EQ(
        ERR_INVALID_ARGUMENT,
        WriteWebSocketFrameHeader(
            header, NULL, dummy_buffer, kTests[i].expected_header_size - 1));
  }
}

TEST(WebSocketFrameTest, MaskPayload) {
  struct TestCase {
    const char* masking_key;
    uint64_t frame_offset;
    const char* input;
    const char* output;
    size_t data_length;
  };
  static const TestCase kTests[] = {
    { "\xDE\xAD\xBE\xEF", 0, "FooBar", "\x98\xC2\xD1\xAD\xBF\xDF", 6 },
    { "\xDE\xAD\xBE\xEF", 1, "FooBar", "\xEB\xD1\x80\x9C\xCC\xCC", 6 },
    { "\xDE\xAD\xBE\xEF", 2, "FooBar", "\xF8\x80\xB1\xEF\xDF\x9D", 6 },
    { "\xDE\xAD\xBE\xEF", 3, "FooBar", "\xA9\xB1\xC2\xFC\x8E\xAC", 6 },
    { "\xDE\xAD\xBE\xEF", 4, "FooBar", "\x98\xC2\xD1\xAD\xBF\xDF", 6 },
    { "\xDE\xAD\xBE\xEF", 42, "FooBar", "\xF8\x80\xB1\xEF\xDF\x9D", 6 },
    { "\xDE\xAD\xBE\xEF", 0, "", "", 0 },
    { "\xDE\xAD\xBE\xEF", 0, "\xDE\xAD\xBE\xEF", "\x00\x00\x00\x00", 4 },
    { "\xDE\xAD\xBE\xEF", 0, "\x00\x00\x00\x00", "\xDE\xAD\xBE\xEF", 4 },
    { "\x00\x00\x00\x00", 0, "FooBar", "FooBar", 6 },
    { "\xFF\xFF\xFF\xFF", 0, "FooBar", "\xB9\x90\x90\xBD\x9E\x8D", 6 },
  };
  static const int kNumTests = base::size(kTests);

  for (int i = 0; i < kNumTests; ++i) {
    WebSocketMaskingKey masking_key;
    std::copy(kTests[i].masking_key,
              kTests[i].masking_key + WebSocketFrameHeader::kMaskingKeyLength,
              masking_key.key);
    std::vector<char> frame_data(kTests[i].input,
                                 kTests[i].input + kTests[i].data_length);
    std::vector<char> expected_output(kTests[i].output,
                                      kTests[i].output + kTests[i].data_length);
    MaskWebSocketFramePayload(masking_key,
                              kTests[i].frame_offset,
                              frame_data.empty() ? NULL : &frame_data.front(),
                              frame_data.size());
    EXPECT_EQ(expected_output, frame_data);
  }
}

// Check that all combinations of alignment, frame offset and chunk size work
// correctly for MaskWebSocketFramePayload(). This is mainly used to ensure that
// vectorisation optimisations don't break anything. We could take a "white box"
// approach and only test the edge cases, but since the exhaustive "black box"
// approach runs in acceptable time, we don't have to take the risk of being
// clever.
//
// This brute-force approach runs in O(N^3) time where N is the size of the
// maximum vector size we want to test again. This might need reconsidering if
// MaskWebSocketFramePayload() is ever optimised for a dedicated vector
// architecture.
TEST(WebSocketFrameTest, MaskPayloadAlignment) {
  // This reflects what might be implemented in the future, rather than
  // the current implementation. FMA3 and FMA4 support 256-bit vector ops.
  static const size_t kMaxVectorSizeInBits = 256;
  static const size_t kMaxVectorSize = kMaxVectorSizeInBits / 8;
  static const size_t kMaxVectorAlignment = kMaxVectorSize;
  static const size_t kMaskingKeyLength =
      WebSocketFrameHeader::kMaskingKeyLength;
  static const size_t kScratchBufferSize =
      kMaxVectorAlignment + kMaxVectorSize * 2;
  static const char kTestMask[] = "\xd2\xba\x5a\xbe";
  // We use 786 bits of random input to reduce the risk of correlated errors.
  static const char kTestInput[] = {
    "\x3d\x77\x1d\x1b\x19\x8c\x48\xa3\x19\x6d\xf7\xcc\x39\xe7\x57\x0b"
    "\x69\x8c\xda\x4b\xfc\xac\x2c\xd3\x49\x96\x6e\x8a\x7b\x5a\x32\x76"
    "\xd0\x11\x43\xa0\x89\xfc\x76\x2b\x10\x2f\x4c\x7b\x4f\xa6\xdd\xe4"
    "\xfc\x8e\xd8\x72\xcf\x7e\x37\xcd\x31\xcd\xc1\xc0\x89\x0c\xa7\x4c"
    "\xda\xa8\x4b\x75\xa1\xcb\xa9\x77\x19\x4d\x6e\xdf\xc8\x08\x1c\xb6"
    "\x6d\xfb\x38\x04\x44\xd5\xba\x57\x9f\x76\xb0\x2e\x07\x91\xe6\xa8"
  };
  static const size_t kTestInputSize = base::size(kTestInput) - 1;
  static const char kTestOutput[] = {
    "\xef\xcd\x47\xa5\xcb\x36\x12\x1d\xcb\xd7\xad\x72\xeb\x5d\x0d\xb5"
    "\xbb\x36\x80\xf5\x2e\x16\x76\x6d\x9b\x2c\x34\x34\xa9\xe0\x68\xc8"
    "\x02\xab\x19\x1e\x5b\x46\x2c\x95\xc2\x95\x16\xc5\x9d\x1c\x87\x5a"
    "\x2e\x34\x82\xcc\x1d\xc4\x6d\x73\xe3\x77\x9b\x7e\x5b\xb6\xfd\xf2"
    "\x08\x12\x11\xcb\x73\x71\xf3\xc9\xcb\xf7\x34\x61\x1a\xb2\x46\x08"
    "\xbf\x41\x62\xba\x96\x6f\xe0\xe9\x4d\xcc\xea\x90\xd5\x2b\xbc\x16"
  };
  static_assert(base::size(kTestInput) == base::size(kTestOutput),
                "output and input arrays should have the same length");
  std::unique_ptr<char, base::AlignedFreeDeleter> scratch(static_cast<char*>(
      base::AlignedAlloc(kScratchBufferSize, kMaxVectorAlignment)));
  WebSocketMaskingKey masking_key;
  std::copy(kTestMask, kTestMask + kMaskingKeyLength, masking_key.key);
  for (size_t frame_offset = 0; frame_offset < kMaskingKeyLength;
       ++frame_offset) {
    for (size_t alignment = 0; alignment < kMaxVectorAlignment; ++alignment) {
      char* const aligned_scratch = scratch.get() + alignment;
      const size_t aligned_len = std::min(kScratchBufferSize - alignment,
                                          kTestInputSize - frame_offset);
      for (size_t chunk_size = 1; chunk_size < kMaxVectorSize; ++chunk_size) {
        memcpy(aligned_scratch, kTestInput + frame_offset, aligned_len);
        for (size_t chunk_start = 0; chunk_start < aligned_len;
             chunk_start += chunk_size) {
          const size_t this_chunk_size =
              std::min(chunk_size, aligned_len - chunk_start);
          MaskWebSocketFramePayload(masking_key,
                                    frame_offset + chunk_start,
                                    aligned_scratch + chunk_start,
                                    this_chunk_size);
        }
        // Stop the test if it fails, since we don't want to spew thousands of
        // failures.
        ASSERT_TRUE(std::equal(aligned_scratch,
                               aligned_scratch + aligned_len,
                               kTestOutput + frame_offset))
            << "Output failed to match for frame_offset=" << frame_offset
            << ", alignment=" << alignment << ", chunk_size=" << chunk_size;
      }
    }
  }
}

// "IsKnownDataOpCode" is currently implemented in an "obviously correct"
// manner, but we test is anyway in case it changes to a more complex
// implementation in future.
TEST(WebSocketFrameHeaderTest, IsKnownDataOpCode) {
  // Make the test less verbose.
  typedef WebSocketFrameHeader Frame;

  // Known opcode, is used for data frames
  EXPECT_TRUE(Frame::IsKnownDataOpCode(Frame::kOpCodeContinuation));
  EXPECT_TRUE(Frame::IsKnownDataOpCode(Frame::kOpCodeText));
  EXPECT_TRUE(Frame::IsKnownDataOpCode(Frame::kOpCodeBinary));

  // Known opcode, is used for control frames
  EXPECT_FALSE(Frame::IsKnownDataOpCode(Frame::kOpCodeClose));
  EXPECT_FALSE(Frame::IsKnownDataOpCode(Frame::kOpCodePing));
  EXPECT_FALSE(Frame::IsKnownDataOpCode(Frame::kOpCodePong));

  // Check that unused opcodes return false
  EXPECT_FALSE(Frame::IsKnownDataOpCode(Frame::kOpCodeDataUnused));
  EXPECT_FALSE(Frame::IsKnownDataOpCode(Frame::kOpCodeControlUnused));

  // Check that opcodes with the 4 bit set return false
  EXPECT_FALSE(Frame::IsKnownDataOpCode(0x6));
  EXPECT_FALSE(Frame::IsKnownDataOpCode(0xF));

  // Check that out-of-range opcodes return false
  EXPECT_FALSE(Frame::IsKnownDataOpCode(-1));
  EXPECT_FALSE(Frame::IsKnownDataOpCode(0xFF));
}

// "IsKnownControlOpCode" is implemented in an "obviously correct" manner but
// might be optimised in future.
TEST(WebSocketFrameHeaderTest, IsKnownControlOpCode) {
  // Make the test less verbose.
  typedef WebSocketFrameHeader Frame;

  // Known opcode, is used for data frames
  EXPECT_FALSE(Frame::IsKnownControlOpCode(Frame::kOpCodeContinuation));
  EXPECT_FALSE(Frame::IsKnownControlOpCode(Frame::kOpCodeText));
  EXPECT_FALSE(Frame::IsKnownControlOpCode(Frame::kOpCodeBinary));

  // Known opcode, is used for control frames
  EXPECT_TRUE(Frame::IsKnownControlOpCode(Frame::kOpCodeClose));
  EXPECT_TRUE(Frame::IsKnownControlOpCode(Frame::kOpCodePing));
  EXPECT_TRUE(Frame::IsKnownControlOpCode(Frame::kOpCodePong));

  // Check that unused opcodes return false
  EXPECT_FALSE(Frame::IsKnownControlOpCode(Frame::kOpCodeDataUnused));
  EXPECT_FALSE(Frame::IsKnownControlOpCode(Frame::kOpCodeControlUnused));

  // Check that opcodes with the 4 bit set return false
  EXPECT_FALSE(Frame::IsKnownControlOpCode(0x6));
  EXPECT_FALSE(Frame::IsKnownControlOpCode(0xF));

  // Check that out-of-range opcodes return false
  EXPECT_FALSE(Frame::IsKnownControlOpCode(-1));
  EXPECT_FALSE(Frame::IsKnownControlOpCode(0xFF));
}

}  // namespace

}  // namespace net
