/*****************************************************************************
 *   GATB : Genome Assembly Tool Box
 *   Copyright (C) 2014  R.Chikhi, G.Rizk, E.Drezen
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as
 *  published by the Free Software Foundation, either version 3 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

/**\page snippets_page How to use the library ?
 *
 * \subpage compilation           \n
 * \subpage new_project           \n
 * \subpage snippets_graph        \n
 * \subpage snippets_bank         \n
 * \subpage snippets_kmer         \n
 * \subpage snippets_iterators    \n
 * \subpage snippets_multithread  \n
 * \subpage snippets_storage      \n
 * \subpage snippets_tools        \n
 *
 *************************************************************************************
 *************************************************************************************
 * \page compilation Instructions to compile GATB-Core
 ************************************************************************************
 *
 * \section compilation_library Compile the library (and other artifacts)
 *
 * The gatb::core library is generated by the cmake tool.
 *
 * In the following, we will call GATB the main directory of the gatb core project (ie.
 * the root directory of the source archive).
 *
 * You have then to do the following:
 *
 * \code
 * cd <some_directory>
 * git clone https://github.com/GATB/gatb-core.git
 * cd gatb-core/gatb-core 
 * mkdir build ; cd build ; cmake ..
 * make -j8   (adjust -j8 to the number of cores of your system)
 * \endcode
 *
 * You can force compilation in debug mode by calling cmake in a specific way:
 * \code
 * cmake -D CMAKE_BUILD_TYPE=Debug ..
 * \endcode
 *
 * Several artifacts are generated:
 *
 *  - the gatb core library is available in the generated <tt>build/lib</tt> directory
 *  - the gatb core binaries are available in the generated <tt>build/bin</tt> directory
 *
 * you can type 'make help' to know all the available targets.
 *
 * If CppUnit is installed, a unit tests binary should be generated; you can launch it with
 * \code
 * cd <some_directory>/gatb-core/gatb-core/build/bin
 * 
 * export CPPUNIT_VERBOSE=1
 *
 * # Launch the full test suite
 * ./gatb-core-cppunit all  <gatb-core-home>/gatb-core/test/db
 * \endcode
 * You can use the exit status code of the command to know the success status (0 for success).
 *
 * Use
 * \code
 * ./gatb-core-cppunit -h
 * \endcode
 *
 * to get more information on how to use this testing program.
 *
 * Note that one may set the environment variable CPPUNIT_VERBOSE to 1 to known which tests pass.
 *
 * By default, gatb::core supports kmer sizes up to 128. In fact, it has 4 different implementations
 * of kmers, one for KSIZE_1=32, KSIZE_2=64, KSIZE_3=96 and KSIZE_4=128. If you need bigger kmers
 * sizes, you can compile this way:
 * \code
 * cmake -Dk1=32 -Dk2=64 -Dk3=96 -Dk4=256 ..
 * \endcode
 *
 *
 ************************************************************************************
 * \section compilation_snippets Compile the code snippets
 *
 * A directory named <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples">examples</a> holds some snippets that show how to use services provided by the library.
 *
 * In order to compile them, you will first need to compile the full library (see above).
 *
 * A simple way to generate the snippets is to type:
 * \code
 * cmake -DGATB_CORE_INCLUDE_EXAMPLES=True -DCMAKE_BUILD_TYPE=Debug ..   [1]
 * make examples
 * \endcode
 *
 * [1] some code snippets use assert(), so it is required to compile in Debug mode.
 *
 * You can also compile a single snippet using its name, such as:
 * \code
 * make bank1
 * \endcode
 *
 * Then, you can have a look at the header of each snippet source file to review how
 * to use them. You can also have a look at <a href="https://github.com/GATB/gatb-core/blob/master/gatb-core/test/jenkins/test-snippets-debian7-64bits-gcc-4.7.sh">this script</a> we used on a Jenkins platform
 * to automatically run and test all these snippets; in that script, you'll see how
 * they are started (i.e. what is the command-line to use).
 *
 * Some documentation about these code snippets is also available <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples">here</a>.
 *
 ************************************************************************************
 * \section use_eclipse Use Eclipse c++ IDE or Xcode IDE to work with GATB-Core
 *
 * If you are interested in using an IDE to develop c++ source codes relying on the
 * GATB-Core library, we have written this manual to explain how to setup Eclipse
 * c++ framework or Xcode one:
 *
 * <a href="https://gatb.inria.fr/use-eclipse-to-develop-gatb-core-softwares/">https://gatb.inria.fr/use-eclipse-to-develop-gatb-core-softwares/</a>
 * <a href="https://gatb.inria.fr/use-xcode-to-develop-gatb-core-softwares/">https://gatb.inria.fr/use-xcode-to-develop-gatb-core-softwares/</a>
 *
 * You also have the opportunity to easily work on <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples">code snippets</a> directly from Eclipse/Xcode. Please, refer to the above links.
 *
 *************************************************************************************
 * \page new_project Quick project creation
 ************************************************************************************
 *
 * \section project_create Create a new Project
 *
 * You use GATB-CORE by creating a new tool project, with the following script:
 * 
 * \code
 *     ./[path-to-gatb-core]/scripts/NewProject/NewProject.sh -d directory -n toolName
 * \endcode
 * 
 * where 'directory' is the directory where the project will be created and 'toolName' is the name of the project.
 * The script will automatically creates the full path directory/toolName to deploy a self-contained tool.
 *  
 * The directory where the project is created has no link to any external resources. You can therefore
 * move it anywhere you want.
 * 
 * Such a project can be a start point for building applications based on GATB-CORE. 
 *
 * \section project_organisation Project organisation
 * 
 * By default, the following part will be included in the project:
 *   - a CMakeLists.txt file used for building the project
 *   - a 'scripts' directory holding a script to automatically package the tool 
 *   - a 'tools' directory holding a default source code using GATB-Core
 *   - a 'thirdparty' directory holding the gatb-core resources
 *   - a 'doc' directory
 *     
 * The 'thirdparty' directory is only available for tool created outside the GATB-Tools repository.
 * Tools located within GATB-Tools rely on a common GATB-Core sub-module already available in this repository.
 * 
 * It is advised to use:
 * 
 *   - 'scripts' directory to hold any scripts this tool relies on
 *   - 'doc' directory to hold tool's documentation
 *   - 'thirdparty' directory to hold any third-party librairies this tool relies on
 *     
 * \section project_source Project source code
 *
 * It is worth noting that 'tools' directory is organised using sub-folders; by default, there is
 * at least one such sub-folder called 'toolName'. It holds the source code of the tool. 
 * 
 * However, when considering a more complex software, it could be nice to setup several "sub-tools", 
 * each of them making a particular feature. In that case, you can easily create several "sub-tool" 
 * folders inside "tools", each of them having a "src" folder containing the source code, as well as 
 * a "main.cpp", for each feature. 
 * 
 * Using this organisation has a big advantage: the provided CMakeLists.txt is aware of 
 * that, so you do not have to edit the CMake file when you add a new "sub-tool". As a real 
 * example, you can have a look at the DiscoSNP software.
 *
 * \section project_compilation Project building
 * 
 * To build your project, you should do the following
 *     
 *     cd [path-to-tool-home]
 *     mkdir build;  cd build;  cmake ..;  make -j8
 *     
 * Then, you should get a binary holding the name of the project within 'build/tools'.
 * 
 * The first compilation should take some time since the GATB-CORE library is generated.
 * 
 * Note that all the needed material is included in the generated project (in particular
 * the gatb::core library and the header files), so you can move it wherever you want.
 * Note also that you won't have to do any tricky configuration to build the binary; you
 * just have to do cmake+make and you will have your binaries in the build/tools directory.
 * 
 * \section project_version Project version
 * 
 * You can set a version number to your project by editing the beginning of CMakeLists.txt file located
 * in the tool main directory or by running cmake like this: 
 * \code
 * cmake -DMAJOR=1 -DMINOR=1 -DPATCH=0 ..
 * \endcode
 * 
 * \section project_packaging Project packaging
 * 
 * You can prepare your tool for distribution using:
 * \code    
 *     ./[path-to-tool-home]/scripts/package_tool.sh -M X -m Y -p Z
 * \endcode
 *
 *  With X, Y and Z being major, minor and patch release numbers, respectively.
 *
 * Then, you should get two 'tar.gz' files within 'build', one containing the binary release
 * and the other the source codes.
 * 
 * Note: the script re-built the entire tool from its sources to ensure a clean build process.
 *
* \section use_eclipse2 Use Eclipse c++ IDE to work on a GATB-TOOL
 *
 * If you are interested in using an IDE to develop c++ source codes relying on the
 * GATB-Core library, we have written this manual to explain how to setup Eclipse
 * c++ framework:
 *
 * <a href="https://gatb.inria.fr/use-eclipse-to-develop-gatb-core-softwares/">https://gatb.inria.fr/use-eclipse-to-develop-gatb-core-softwares/</a>
 *
 ************************************************************************************
 ************************************************************************************
 * \page snippets_iterators Iterators snippets
 ************************************************************************************
  * \tableofcontents
 *************************************************************************************
 * \section snippets_iterators_intro Presentation
 * 
 * This page presents some code snippets related to the use of Iterators.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/tools">gatb-core/gatb-core/examples/tools</a>.
 *
 * \section snippets_iterators_snippet1 Iterate a list
 *
 * This snippet shows how to iterate a STL list with our iterator design.
 *
 * Code is from example iterators1.cpp:
 *\snippet iterators1.cpp  snippet1
 *
 ************************************************************************************
 * \section snippets_iterators_snippet2 Iterate the Cartesian product of two lists
 *
 * This snippet shows how to iterate the Cartesian product of two lists:
 *
 *  1) Declare two iterators \n
 *  2) Declare one Cartesian iterator configured with the two iterators \n
 *  3) Iterate the Cartesian iterator. \n
 *
 * The current item of the iteration is a pair, so one should retrieve the couple of
 * values with methods 'first' and 'second'.
 *
 * Code is from example iterators2.cpp:
 *\snippet iterators2.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 ************************************************************************************
 * \section snippets_iterators_snippet7 Iterate two lists by pairs
 *
 * This snippet shows how to iterate two iterators at the same time, providing pairs
 * of items at each iteration.
 *
 * A usage of such an iterator is to iterate two paired ends banks.
 *
 * Code is from example iterators7.cpp:
 *\snippet iterators7.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 ************************************************************************************
 * \section snippets_iterators_snippet3 Truncating an iteration
 *
 * This snippet shows how to truncate the iteration of some iterator.
 *
 * Code is from example iterators3.cpp:
 *\snippet iterators3.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 *
 ************************************************************************************
 * \section snippets_iterators_snippet4 Iterate a list with progress feedback
 *
 * This snippet shows how to iterate a STL list and being notified as a listener about
 * its progression.
 *
 * The idea is to use a SubjectIterator instance that refers the actual iterator we want
 * to iterate.
 *
 * Then, it is possible to subscribe some callback function (here as a functor) to the
 * SubjectIterator instance.
 *
 * The listener will then receive at regular interval the number of currently iterated
 * items.
 *
 * Code is from example iterators4.cpp:
 *\snippet iterators4.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 ************************************************************************************
 * \section snippets_iterators_snippet5 Iterate a list with progress feedback (simple)
 *
 * This snippet is the same as before but here we use a default console progress bar.
 * In most case, it allows to avoid an explicit listener configuration.
 *
 * Code is from example iterators5.cpp:
 *\snippet iterators5.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 ************************************************************************************
 * \section snippets_iterators_snippet6 Iterate a list and filter out some items
 *
 * This snippet shows how to iterate a STL list while filtering out some items that
 * don't check some condition.
 *
 * Code is from example iterators6.cpp:
 *\snippet iterators6.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 ************************************************************************************
 * \section snippets_iterators_snippet8 Mixing iterators
 *
 * This snippet shows how mix several iterators. Note again that the iteration loop
 * is still the same.
 *
 * Code is from example iterators8.cpp:
 *\snippet iterators8.cpp  snippet1
 * [go back to \ref snippets_iterators "top"]
 *
 *
 ************************************************************************************
 ************************************************************************************
 * \page snippets_multithread Multithreading snippets
 ************************************************************************************
 *
 * \tableofcontents
 *
 * *************************************************************************************
 *
 * \section snippets_multithread_intro Presentation
 *
 * This page presents some code snippets related to the use of Multithreading API.
 *
 * Some of the snippets presented below <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">can be used online</a>.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/tools">gatb-core/gatb-core/examples/tools</a>.
 *
 * \section snippets_multithread_snippet1 Iteration in a multithreaded fashion
 *
 * This snippet shows how to iterate some Iterator object (here a range of integers)
 * with N threads in order to speed up the iteration.
 *
 * This snippet introduces the Dispatcher class and shows how to simply use it for
 * parallelizing one iteration.
 *
 * <b>Note: this approach can work only if the items can be iterated and processed independently
 * from each other.</b>
 *
 * Code is from example multithreading1.cpp:
 *\snippet multithreading1.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 *************************************************************************************
 *
 * \section snippets_multithread_snippet2 Multithreaded iteration and shared resources
 *
 * This snippet shows how to parallelize an iteration and how several threads can modify
 * a common resource throughout the iteration.
 *
 * The important point here is to understand that shared resources must be modified
 * cautiously by different threads running at the same time.
 *
 * Code is from example multithreading2.cpp:
 *\snippet multithreading2.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 *************************************************************************************
 *
 * \section snippets_multithread_snippet3 Multithreaded iteration with synchronization of a shared resource
 *
 * Here, our shared resource is a file, so we can't use intrinsic instruction like we
 * did before for integer addition.
 *
 * We need some general synchronization mechanism that will ensure that a portion of code
 * can be executed only by one thread at one time.
 *
 * Code is from example multithreading3.cpp:
 *\snippet multithreading3.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 *
 *************************************************************************************
 *
 * \section snippets_multithread_snippet4 Multithreaded iteration with synchronization of a shared resource (bis)
 *
 * This snippet is similar to the previous one. It only shows how to use the LocalSynchronizer
 * class to simply lock/unlock the containing instruction block.
 *
 * This is useful for avoiding classical deadlock bugs when one forgets to unlock a
 * synchronizer.
 *
 * Code is from example multithreading4.cpp:
 *\snippet multithreading4.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 *************************************************************************************
 *
 * \section snippets_multithread_snippet5 Multithreaded iteration without shared resources management
 *
 * This snippet introduces the ThreadObject class designed to avoid concurrent accesses
 * issues.
 *
 * Instead of working on a single shared resource, threads use local resources during
 * the iteration and then, a final aggregation of the local resources is done after
 * the iteration.
 *
 * Such an approach skips the need of synchronization mechanisms when threads directly
 * uses a single shared resource. This may be interesting since synchronization mechanisms
 * may introduce time overheads.
 *
 * Code is from example multithreading5.cpp:
 *\snippet multithreading5.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 *************************************************************************************
 *
 * \section snippets_multithread_snippet6 Multithreaded iteration of a bank
 *
 * This snippet shows how to iterate sequences of a bank and counts how many A,C,G,T it
 * contains. The interesting part is to see that the Bank class can create Iterator instances
 * that can be iterated through a Dispatcher instance.
 *
 * Note: iterating a bank from a disk makes a lot of I/O, so parallelizing such an iteration
 * may not lead to significant better performance. However, if the snippet is launched
 * once, the bank (if not too big) may be in the RAM cache, so it is interesting to
 * relaunch the snippet with varying number of cores and see how execution time evolves.
 *
 * Code is from example multithreading6.cpp:
 *\snippet multithreading6.cpp  snippet1
 * [go back to \ref snippets_multithread "top"]
 *
 ************************************************************************************
 ************************************************************************************
 * \page snippets_bank Bank snippets
 ************************************************************************************
 *
 * \tableofcontents
 *
 ************************************************************************************
 *
 * \section snippets_bank_intro Presentation
 *
 * This page presents some code snippets related to the use of Bank API.
 *
 * Some of the snippets presented below <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">can be used online</a>.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/bank">gatb-core/gatb-core/examples/bank</a>.
 *
 * \section snippets_bank_snippet1 Parsing a single FASTA bank without checks
 *
 * This snippet shows how to read one FASTA bank in a simple way. No check is done about
 * the correctness of the FASTA bank file path.
 *
 * Some information of each iterated sequence are diplayed as output.
 *
 * Code is from example bank1.cpp:
 *\snippet bank1.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 ************************************************************************************
 * \section snippets_bank_snippet2 Parsing several banks
 *
 * This snippet shows how to read one ore more banks in a simple way. The idea is to use
 * the Bank::open method that analyzes the provided uri and get the correct IBank handle for
 * one or more banks. For instance, one can run this snippet with:
 *    - bank2  reads1.fa,reads2.fa,reads3.fa
 *
 * Some information of each iterated sequence are diplayed as output.
 *
 * Code is from example bank2.cpp:
 *\snippet bank2.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 *************************************************************************************
 * \section snippets_bank_snippet3 Parsing a FASTA bank in a different way
 *
 * This snippet shows how to read one or more FASTA banks in a "push" model;
 * it means that the sequence iterator calls some function for each sequence.
 *
 * This is another way to iterate items and opposite to the "pull" model where the
 * iterator is called to provide the current item, instead of calling some function to
 * do as we do in this sample.
 *
 * Code is from example bank3.cpp:
 *\snippet bank3.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 **************************************************************************************
 * \section snippets_bank_snippet4 Parsing a FASTA bank and getting progress information
 *
 * This snippet shows how to create an iterator on something (here sequences from a FASTA
 * file) and encapsulate it with another iterator that adds the possibility to notify some
 * listener every 10 iterated sequences (used here for showing some progression during the
 * iteration).
 *
 * Note: the "notifying" iterator is generic and could be reused to send progress notification
 * with any kind of iterator, not only on sequences.
 *
 * Code is from example bank4.cpp:
 *\snippet bank4.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 **************************************************************************************
 * \section snippets_bank_snippet5 Parsing a FASTA bank and getting percentage progress information
 *
 * This snippet shows how to read one or more FASTA banks and get a percentage progress
 * information during the iteration.
 *
 * We use there a ProgressIterator on Sequence. By default, we got progress information with
 * remaining time estimation and resources usage (cpu and memory).
 * Note that the ProgressIterator class has a second template parameter that can provide other
 * progress information, try for instance:
 *      - ProgressIterator<Sequence,ProgressTimer>
 *
 * Code is from example bank5.cpp:
 *\snippet bank5.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet6 Output a FASTA file with data line of given size
 *
 * This snippet provides a small utility for cutting lines of data with a given size.
 *
 * Code is from example bank6.cpp:
 *\snippet bank6.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet7 Filter sequences with a given data size
 *
 * This snippet shows how to parse a bank with a functor used to filter out some items.
 *
 * Code is from example bank7.cpp:
 *\snippet bank7.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet7b Filter sequences using Phred quality
 *
 * This snippet shows how to parse a FastQ file and filtering out reads by Phred quality.
 *
 * Code is from example bank24.cpp:
 *\snippet bank24.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet8 Conversion of a FASTA bank to a binary format
 *
 * This snippet shows how to parse a nucleic bank and convert it to a binary format.
 *
 * Code is from example bank8.cpp:
 *\snippet bank8.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet9 Conversion of a bank with some filtering
 *
 * This snippet shows how to parse a bank, check whether the sequences match a criteria
 * and dump the matching sequences into an output FASTA bank.s
 *
 * Code is from example bank9.cpp:
 *\snippet bank9.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet10 Split a bank
 *
 * This snippet shows how to split an input bank into parts. It creates an output album
 * bank by using the BankAlbum class. Example of use:
 * \code
 * bank10 -in reads.fa -max-size 10000000
 * \endcode
 * This example will create a directory 'reads_S1000000' where we can find:
 *      - album.txt   =>  album file containing all reads_X split parts
 *      - reads_0
 *      - reads_1
 *      - ...
 *      - reads_N
 *
 * All the 'reads_X' files are about 1000000 bytes. Now, the generated album.txt file
 * can be used as a bank and could be the input bank of the snippet bank14 for instance
 * (and we should get the same results as using directly the 'reads.fa' bank)
 *
 * Code is from example bank10.cpp:
 *\snippet bank10.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet11 Iterating a bank whose sequences are kmers
 *
 * This snippet shows how to iterate a BankKmers. Such a bank iterates all possible
 * kmers for a given kmers size.
 *
 * Each sequence is saved into an output FASTA file.
 *
 * Code is from example bank11.cpp:
 *\snippet bank11.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet12 Extracting sequences with specific ids with FilterIterator
 *
 * This snippet shows how to extract sequences from a bank. A sequence is kept if its
 * index in its bank belongs to a set (provided as an input file holding one index per line)
 *
 * It uses the FilterIterator feature.
 *
 * Code is from example bank12.cpp:
 *\snippet bank12.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet13 Extracting sequences with too many N characters
 *
 * This snippet shows how to extract sequences that don't have too many N.
 *
 * It uses the FilterIterator feature.
 *
 * Code is from example bank13.cpp:
 *\snippet bank13.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet14  Computing statistics on a bank
 *
 * This snippet shows how to read a bank and get statistics on it.
 *
 * Code is from example bank14.cpp:
 *\snippet bank14.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 **************************************************************************************
 * \section snippets_bank_snippet18  Iterating paired end banks
 *
 * This snippet shows how to read two banks in the same time, and each item is a pair
 * of items of bank1 and bank2.
 *
 * This is good example how to reads paired end banks.
 *
 * Code is from example bank18.cpp:
 *\snippet bank18.cpp  snippet1
 * [go back to \ref snippets_bank "top"]
 *
 * \n
 *
 ************************************************************************************
 ************************************************************************************
 * \page snippets_kmer Kmer snippets
 ************************************************************************************
 *
 * \tableofcontents
 *
 *************************************************************************************
 *
 * \section snippets_kmer_intro Presentation
 *
 * This page presents some code snippets related to the use of k-mer API.
 *
 * Some of the snippets presented below <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">can be used online</a>.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/kmer">gatb-core/gatb-core/examples/kmer</a>.
 *
 * \section snippets_kmer_snippet1 Using a kmer model
 *
 * This snippet shows how to create kmer models.
 *
 * Code is from example kmer1.cpp:
 *\snippet kmer1.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet2 Computing kmers with a model
 *
 * This snippet shows how to get kmers from a model. Here, we can see that we can
 * use 3 different kinds of models, giving different kinds of kmers:
 *  - ModelDirect : direct kmers
 *  - ModelCanonical : minimum value of the direct kmer and its reverse complement
 *  - ModelMinimizer : provides also a minimizer for the kmer
 *
 * The snippet shows different methods usable for each kind of model.
 *
 * Code is from example kmer2.cpp:
 *\snippet kmer2.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet3 Iterating kmers from a sequence
 *
 * This snippet shows how to iterate the kmers from a sequence, for a given model.
 *
 * Code is from example kmer3.cpp:
 *\snippet kmer3.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet4 Iterating kmers from one or several banks
 *
 * This snippet shows how to iterate the kmers from banks. In particular, we use two
 * iterators and two loops:
 *      - outer loop on sequences of the bank
 *      - inner loop on kmer on the current sequence from the outer loop
 *
 * Code is from example kmer4.cpp:
 *\snippet kmer4.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet5 Computing statistics about minimizers
 *
 * This snippet shows iterate the kmers of an input bank and computes some statistics
 * about the iterated minimizers.
 *
 * It also computes the following distribution : number of times a read has X different
 * minimizers (in other words, the number of super kmers).
 *
 * Code is from example kmer5.cpp:
 *\snippet kmer5.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet6 Checking span of kmers model
 *
 * This snippet shows what is legal in terms of kmers size.
 *
 * Actually, a kmers model has a type with a 'span' template. This span represents the
 * maximum kmer size reachable for this type. For instance, a span of 32 allows up to
 * kmer size of 31, a span of 64 up to 63, etc...
 *
 * The 'span' value must be one of the project defined constants: KSIZE_1, KSIZE_2, KSIZE_3
 * and KSIZE_4. By default, we have KSIZE_n = 32*n. Note that it is possible to compile
 * GATB-CORE with a bigger value of KSIZE_4=128. See \ref compilation.
 *
 * It is important to understand that each one of the four span values defines a specific
 * kmers model with a specific integer type that represents the kmers values. For instance:
 *  - KSIZE_1=32 implies that we need 64 bits integers (2 bits per nucleotide), which is available
 *  as a native type on 64 bits architecture
 *  - KSIZE_2=64 implies that we need 128 bits integers which may (or not) be available as native
 *  integer type
 *  - for KSIZE_3 and KSIZE_4, we need to switch to specific large integer representations that
 *  are no more native on the system, which implies bigger computation times.
 *
 * Code is from example kmer6.cpp:
 *\snippet kmer6.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet7 Other kind of statistics about minimizers
 *
 * This snippet shows how to iterate the kmers of an input bank and computes some statistics
 * about the iterated minimizers.
 *
 * Code is from example kmer7.cpp:
 *\snippet kmer7.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet8  Setting custom minimizers definition
 *
 * This snippet shows how to configure custom minimizer definition through a functor.
 *
 * Code is from example kmer8.cpp:
 *\snippet kmer8.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet9  Counting kmers from a bank with the sorting count algorithm
 *
 * This snippet shows how to use the SortingCountAlgorithm class for counting kmers in
 * a bank
 *
 * Code is from example kmer9.cpp:
 *\snippet kmer9.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_kmer_snippet10  Reading a file generated by the sorting count algorithm
 *
 * This snippet shows how to read the output of the SortingCountAlgorithm.
 *
 * We use two ways for reading the couples [kmer,abundance]
 *  - we read all the couples with a single iterator
 *  - we read each solid collection and use an iterator on it
 *
 * Code is from example kmer10.cpp:
 *\snippet kmer10.cpp  snippet1
 * [go back to \ref snippets_kmer "top"]
 * \n
 *
 *
 *
 ************************************************************************************
 *************************************************************************************
 * \page snippets_graph De Bruijn graph snippets
 *************************************************************************************
 *************************************************************************************
 *
 *\tableofcontents
 *
 * \section snippets_dbg_intro Presentation
 *
 * This page presents some code snippets related to the use of De Bruijn graph API.
 *
 * Some of the snippets presented below <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">can be used online</a>.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/debruijn">gatb-core/gatb-core/examples/debruijn</a>.
 *
 ************************************************************************************
 * \section snippets_dbg_1  Build / Load De Bruijn graphs
 ************************************************************************************
 * \subsection snippets_kmer_dbg_1  Building a De Bruijn graph from command line options
 *
 * This snippet shows how to create a Graph object thanks to command line options with at
 * least a mandatory FASTA file URI.
 *
 * The first thing to do is to get a parser that analyzes the command line options
 * from (argc,argv). Such a parser can be retrieved with a static method from Graph class.
 *
 * Then, the parsed options can be provided to the Graph::create method and then we get
 * a Graph object on which we can do anything we want.
 *
 * The only mandatory option is '-in fastafile'. All other options have default values if not
 * set through command line.
 *
 * In this snippet, we dump information about the Graph object building with Graph::getInfo method.
 *
 * \remark This snippet essentially does the same job as the \b dbgh5 tool provided by the
 * gatb-core project: it takes a set of reads (as a FASTA file) and generates the corresponding
 * De Bruijn graph as a HDF5 file.
 *
 * Code is from example debruijn1.cpp:
 *\snippet debruijn1.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_2  Building a De Bruijn graph from a command-line-like string
 *
 * Like the previous snippet, we create a Graph object with command line options, but
 * here the options are directly provided as a string.
 *
 * Code is from example debruijn2.cpp:
 * \snippet debruijn2.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_3  Building a De Bruijn graph from a bank object
 *
 * Here, we create a Graph object by providing a bank object, more precisely a IBank
 * object.
 *
 * It is therefore possible to provide a Bank instance (ie a FASTA bank), or
 * another kind of bank that implements the IBank interface.
 *
 * Note in the example that we can provide additional options after the bank object.
 *
 * Code is from example debruijn3.cpp:
 * \snippet debruijn3.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_4  Building a De Bruijn graph from a fake bank object
 *
 * Like the previous snippet, we create a Graph object by providing a bank object, but
 * here this is a 'fake' bank built "on the fly".
 *
 * Such banks are often useful for testing purposes.
 *
 * In such a case, the output file for the graph will be named "noname", unless a specific
 * name is set through the command line option "-out mygraph".
 *
 * Code is from example debruijn4.cpp:
 * \snippet debruijn4.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_5  Load a De Bruijn graph from a graph file
 *
 * Once we have built a graph, it is saved as a file (likely a HDF5 file).
 *
 * This snippet shows how to load such a file to get a Graph object.
 *
 * Code is from example debruijn5.cpp:
 * \snippet debruijn5.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_dbg_2  Iterating nodes
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_6  Iterate the nodes of a De Bruijn graph
 *
 * This snippet shows how to iterate all the nodes of a graph (the graph is loaded
 * from a graph file).
 *
 * The idea is to get an iterator from the graph and use it to get each node of the graph.
 *
 * Here, the nodes are retrieved regardless of any edges between them.
 *
 * Code is from example debruijn6.cpp:
 * \snippet debruijn6.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_7  Iterate the nodes of a De Bruijn graph in a multithread way
 *
 * As the previous example, this snippet shows how to iterate all the nodes of a graph.
 *
 * The difference here is that the iteration is parallelized, using all possible available
 * cores, which should speed up the iteration.
 *
 * WARNING ! don't forget this is parallel execution, so you have be careful if you want to
 * modify the same object in different thread execution contexts.
 *
 * Note: lambda expressions are used here to have code conciseness (which suppose to use
 * an up-to-date compiler). You can have some information at
 * http://stackoverflow.com/questions/7627098/what-is-a-lambda-expression-in-c11
 *
 * Code is from example debruijn7.cpp:
 * \snippet debruijn7.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_dbg_3  Neighborhoods
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_9  Working with neighborhoods of nodes in the De Bruijn graph
 *
 * This snippet shows how to use some methods related to neighbors in a graph.
 *
 * We use a fake bank with one sequence of size 5 and use a kmer size of 4, so we will have 2 possible
 * nodes in the graph.
 *
 * We iterate these two nodes, and for one of them, we ask for its neighbors with the Graph::successors method.
 * We can then check that the retrieved neighbor is the correct one by analyzing the node string representations.
 *
 * In this example, we use the successors method, but note it is possible to get the incoming neighbors with the
 * Graph::predecessors method. By the way, it is possible to know the in and out degree of a given node with the two
 * corresponding methods Graph::indegree and Graph::outdegree
 *
 * Code is from example debruijn9.cpp:
 * \snippet debruijn9.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_10  Working with neighborhoods of nodes in the De Bruijn graph (continued)
 *
 * This snippet shows how to use some methods related to neighbors in a graph.
 *
 * We do the same work as the previous example. The only difference is that we retrieve the neighbors as
 * Edge objects rather than Node objects, so we will have the full information about each transition between
 * the source node and the retrieved neighbors (including the transition nucleotide for instance).
 *
 * In particular, we use some of the Edge attributes (Edge::to, Edge::nt)
 *
 * Code is from example debruijn10.cpp:
 * \snippet debruijn10.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_11  Working with a specific neighbor for a specific node
 *
 * This snippet shows how to get a specific neighbor for a specific node.
 *
 * Sometimes, it is interesting to ask the graph for only one neighbor for a given node.
 * It may happen when one has already got neighbors information through a Graph::neighbors
 * call and memorized only the transition nucleotides for valid neighbors.
 *
 * The Graph::neighbor fulfills this need. This method has two forms, with or without check
 * to graph membership, according to performance considerations.
 *
 * Code is from example debruijn11.cpp:
 * \snippet debruijn11.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_dbg_4  Branching nodes
 ************************************************************************************
 *
 * In the De Bruijn graph, we can define two types of nodes:
 *  - a node N is 'simple'    <=> indegree(N)==1 && outdegree(N)
 *  - a node N is 'branching' <=> N is not simple
 *
 * Branching nodes set is an important subset in the whole nodes set of the De Bruijn graph, so it is of most
 * interest to have some graph methods that work on such nodes. In particular, we can:
 *  - iterate all the branching nodes of the De Bruijn graph
 *  - get the branching neighbors of some node
 *
 * \remarks - With this definition, a branching node may have 0 outcoming neighbors or 0 incoming neighbors.
 *
 * \remarks - Since we are considering assembly matters, we should have few branching nodes compared to the simple
 * nodes.
 *
 * \subsection snippets_kmer_dbg_8  Iterate the branching nodes of a De Bruijn graph
 *
 * This snippet shows how to iterate the branching nodes of a graph (the graph is loaded
 * from a graph file).
 *
 * The idea is to get an iterator from the graph and use it to get each branching node of the graph.
 *
 * Code is from example debruijn8.cpp:
 * \snippet debruijn8.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_16  Working with branching neighbors of a node
 *
 * This snippet shows how to get the branching neighbors of a node. Such neighbors are
 * computed as follow:
 *      - the immediate neighbors of the node are retrieved
 *      - a simple path is done from each neighbor in order to reach the first non simple node
 *
 * Here, we use directly the Graph::successors<BranchingNode> method that encapsulates
 * this behavior.
 *
 * Code is from example debruijn16.cpp:
 * \snippet debruijn16.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_17  Working with branching neighbors of a node (continued)
 *
 * This snippet shows how to get the branching neighbors of a node.
 *
 * It is similar to the previous snippet. The difference here is that we retrieve BranchingEdge
 * objects. A BranchingEdge object is made of:
 *  - the source branching node
 *  - the target branching node
 *  - the direction of the neighbors (in/out coming)
 *  - the nucleotide of the transition between the initial branching node and the first neighbor
 *    on the simple path between the two branching nodes.
 *  - the number of transitions that link the two branching nodes.
 *
 * Code is from example debruijn17.cpp:
 * \snippet debruijn17.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 * ************************************************************************************
 * \section snippets_dbg_14  Simple path
 ************************************************************************************
 *
 * \subsection snippets_kmer_dbg_14  Iterating simple path from a node
 *
 * As we saw previously, a simple node is defined as having indegree==1 and outdegree==1.
 * It is often useful to iterate successive simple nodes in order to build some path in the De Bruijn graph.
 *
 * This snippet shows how to iterate such a simple path. Here, the iterated items are the
 * successive nodes of the path.
 *
 * Code is from example debruijn14.cpp:
 * \snippet debruijn14.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 *************************************************************************************
 * \subsection snippets_kmer_dbg_15  Iterating simple path from a node (continued)
 *
 * Like the previous example, this snippet shows how to iterate a simple path.
 * Here, the iterated items are the successive edges of the path. If we note E an edge
 * in this path, we will have:
 *  - outdegree(E.from)==1  &&  indegree(E.to)==1
 *
 * Code is from example debruijn15.cpp:
 * \snippet debruijn15.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_dbg_6  Miscellanous
 *************************************************************************************
 * \subsection snippets_kmer_dbg_12  Playing with node strands
 *
 * A Node object is fully defined by a kmer value and a strand that disambiguates how to
 * interpret the kmer.
 *
 * It is often required to change the reading strand of a node. This can be done with the
 * Graph::reverse method.
 *
 * Code is from example debruijn12.cpp:
 * \snippet debruijn12.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *************************************************************************************
 * \subsection snippets_kmer_dbg_13  Playing with fake nodes
 *
 * Sometimes, it is useful to build "fake" nodes from a simple sequence, without having
 * a graph holding true data.
 *
 * It is possible to get an empty Graph object (its kmer size must be nevertheless specified),
 * and then use the Graph::buildNode to get a node based on a Data object.
 *
 * Code is from example debruijn13.cpp:
 * \snippet debruijn13.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 * \n
 *
 ************************************************************************************
 * \section snippets_dbg_7  To go further...
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_18  Getting branching nodes statistics in a parallel way
 *
 * This example is a little bit harder than the previous ones. Its purpose is to show how
 * to use the graph API for extracting some information. In particular, we will try to
 * take care about time execution by using available cores.
 *
 * The idea here is to compute information about branching nodes and their in and out degrees.
 * For instance, we want to know how many branching nodes have indegree==2 and outdegree=3.
 * We compute therefore the number of branching nodes having indegree==X and outdegree==Y,
 * with X and Y in [0..4] and the constraint that we can't have X==Y==1 (otherwise the node
 * wouldn't be a branching one).
 *
 * We can do it easily by using the methods:
 *  - Graph::successors<BranchingNode>
 *  - Graph::predecessors<BranchingNode>
 *
 * Moreover, we want to do it in a parallel way in order to speed up the computation. The idea is
 * to get an iterator over the branching nodes and iterate it through a Dispatcher object; such a
 * dispatcher will create as many threads as wanted and will feed each threads with branching nodes.
 * Note that this scheme can work here because the processing of one branching node is independant of
 * the others.
 *
 * We need also some container for the in/out degrees statistics. A natural choice is to use a map,
 * with the key being a unique identifier for a couple (indegree/outdegree) and the value the number
 * of occurrences for the key. The idea is to use one map per thread and to merge the N maps into a global
 * one after the parallel iteration of the branching nodes. We use here a ThreadObject object
 * that allows to do it in a simple way. This object clones N time the global map and each clone is used in
 * a specific thread. The ThreadObject class allows to hide many cumbersome points for the parallelization.
 *
 * In this example, we also use progress notification feature (with ProgressIterator) in order to have
 * some user feedback during the iteration of the branching nodes.
 *
 * Code is from example debruijn18.cpp:
 * \snippet debruijn18.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 *
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_19  Computing connected components of the branching nodes sub graph
 *
 * Code is from example debruijn19.cpp:
 * \snippet debruijn19.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 *
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_24  Generate dot file from a graph
 *
 * This snippet generates a dot file from a de Bruijn graph. It is then possible to
 * generate a pdf output of the graph (dot -Tpdf graph.dot -o graph.pdf)
 *
 * Code is from example debruijn24.cpp:
 * \snippet debruijn24.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 *
 *************************************************************************************
 *
 * \subsection snippets_kmer_dbg_26  Retrieve extra information from a node
 *
 * This snippet shows how to use the 'queryAbundance' method that returns the occurrences
 * number of the node in the initial set of reads.
 *
 * Code is from example debruijn26.cpp:
 * \snippet debruijn26.cpp  snippet1
 * [go back to \ref snippets_graph "top"]
 *
 * ************************************************************************************
 ************************************************************************************
 * \page snippets_storage Storage snippets
 ************************************************************************************
 *
 * \tableofcontents
 *
 * \section snippets_storage_intro Presentation
 *
 * This page presents some code snippets related to the use of persistency API.
 *
 * Some of the snippets presented below <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">can be used online</a>.
 *
 * Additional snippets are available in directory: <a href="https://github.com/GATB/gatb-core/tree/master/gatb-core/examples/storage">gatb-core/gatb-core/examples/storage</a>.
 *
 *************************************************************************************
 *
 * \section snippets_storage_snippet1 Create and save a collection with a Storage object
 *
 * This snippet shows how to use a Storage object for creating a collection of integers.
 * We use the HDF5 format, so we can control the result of our snippet with HDF5 tools.
 *
 * Code is from example storage1.cpp:
 *\snippet storage1.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet2 Create and save two collections with a Storage object
 *
 * This snippet shows how to use a Storage object for creating collections of integers.
 * We use the HDF5 format, so we can control the result of our snippet with HDF5 tools.
 *
 * Code is from example storage2.cpp:
 *\snippet storage2.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet3 Load a collection from a Storage object
 *
 * This snippet shows how to load a Storage object and get a saved collection from it.
 *
 * Code is from example storage3.cpp:
 *\snippet storage3.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet4 Load collections from a Storage object
 *
 * This snippet shows how to load a Storage object and get saved collections from it.
 * Note that we use lambda expressions in this example.
 *
 * Code is from example storage4.cpp:
 *\snippet storage4.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet6 Iterate solid kmers from a HDF5 file
 *
 * This snippet shows how to use a HDF5 Storage object holding solid kmers and iterate
 * the kmers.
 *
 * It also uses a Model instance in order to convert the solid kmers values into the
 * corresponding nucleotides sequence.
 *
 * The input file is likely to have been generated by dbgh5 for instance, or by dsk.
 *
 * If you want to know the structure of the HDF5 file, you can use the h5dump utility,
 * for instance:  h5dump -H file.h5
 *
 * Code is from example storage6.cpp:
 *\snippet storage6.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet7 Associate metadata to HDF5 collections
 *
 * This snippet shows how to associate metadata to HDF5 collections.
 *
 * You can dump such values with h5dump:
 *      h5dump -a myIntegers/myData foo.h5
 *
 * Code is from example storage7.cpp:
 *\snippet storage7.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 *************************************************************************************
 * \section snippets_storage_snippet8 Using C++ like streams with HDF5
 *
 * This snippet shows how to use binary input/output streams with HDF5. There are two
 * types:
 *  - Storage::ostream : used for saving binary data into a HDF5 collection
 *  - Storage::istream : used for retrieving binary data from a HDF5 collection
 *
 * Code is from example storage8.cpp:
 *\snippet storage8.cpp  snippet1
 * [go back to \ref snippets_storage "top"]
 *
 ************************************************************************************
 * \page snippets_tools  Tools snippets
 ************************************************************************************
 *
 * \tableofcontents
 *
 *************************************************************************************
 *
 * These snippets provide several miscellaneous tools that ease the developper's life.
 *
 *************************************************************************************
 * \section snippets_tools_snippet1  Using the Tool class for quick tool development
 *
 * This snippet shows how to quickly create a new tool by subclassing the Tool class.
 *
 * You can try it <a href="http://gatb-core.gforge.inria.fr/training?trail=Expert">online here</a>.
 *
 * Using the Tool class will give to use some useful features for developing our toy tool.
 *
 * Code is from example ToyTool.cpp:
 *\snippet ToyTool.cpp  snippet1
 *
 */
