/*
 * pam_sshauth: PAM module for authentication via a remote ssh server.
 * Copyright (C) 2010-2013 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <config.h>
#include <libssh2.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/select.h>
#include <unistd.h>
#include <arpa/inet.h>

#include <security/pam_modules.h>
#include <security/pam_ext.h>

#include "pam_sshauth.h"

static void
kbd_callback (const char *name, int name_len,
	      const char *instruction, int instruction_len, int num_prompts,
	      const LIBSSH2_USERAUTH_KBDINT_PROMPT * prompts,
	      LIBSSH2_USERAUTH_KBDINT_RESPONSE * responses, void **abstract)
{
  pam_handle_t *pamh = *abstract;
  int i;

  /*
   * Get any instructions the ssh session has generted,
   * and send them to the user via the pam message system.
   */

  if (instruction_len > 0)
    {
      send_pam_msg (pamh, PAM_TEXT_INFO, instruction);
    }

  /*
   * Loop through the prompts that ssh has given us, and ask the
   * user via pam prompts for the answers.
   */

  for (i = 0; i < num_prompts; i++)
    {
      int style = prompts[i].echo ? PAM_PROMPT_ECHO_ON : PAM_PROMPT_ECHO_OFF;
      int pam_retval;
      char *buf, *response;

      if ((buf = malloc (prompts[i].length + 1)) == NULL)
	{
	  return;
	}
      strncpy (buf, prompts[i].text, prompts[i].length);
      *(buf + prompts[i].length) = '\0';
      pam_retval = pam_prompt (pamh, style, &response, "%s", buf);
      free (buf);
      if (pam_retval != PAM_SUCCESS)
	{
	  return;
	}

      responses[i].text = strdup (response);
      responses[i].length = strlen (response);

      if (pam_set_item (pamh, PAM_AUTHTOK, response) != PAM_SUCCESS)
	{
	  return;
	}
    }
}


int
auth_interactive (pam_handle_t * pamh, const char *username,
		  LIBSSH2_SESSION * session)
{
  return libssh2_userauth_keyboard_interactive (session, username,
						&kbd_callback);
}

/*
 * auth_pw ()
 *
 * conduct an ssh simple password based authentication
 */

int
auth_pw (pam_handle_t * pamh, const char *username, LIBSSH2_SESSION * session)
{
  int ssh_result = SSH_AUTH_ERROR;
  char *password = NULL;

  /*
   * try_first_pass works with simple password authentication.
   */

  if (try_first_pass)
    {
      if (pam_get_item (pamh, PAM_AUTHTOK, (const void **) &password) !=
	  PAM_SUCCESS)
	{
	  pam_syslog (pamh, LOG_ERR,
		      "Couldn't obtain PAM_AUTHTOK from the pam stack.");
	  password = NULL;
	}
    }

  if (password == NULL)
    {
      if (pam_prompt (pamh, PAM_PROMPT_ECHO_OFF, &password, "Password:") !=
	  PAM_SUCCESS)
	{
	  pam_syslog (pamh, LOG_ERR,
		      "Couldn't obtain password from pam_prompt.");
	  return SSH_AUTH_ERROR;
	}
    }

  ssh_result = libssh2_userauth_password (session, username, password);
  if (ssh_result == SSH_AUTH_SUCCESS)
    {
      /*
       * The very last response we've gotten should be the password.  Store it
       * as the AUTHTOK
       */

      if (!try_first_pass
	  && pam_set_item (pamh, PAM_AUTHTOK, password) != PAM_SUCCESS)
	{
	  pam_syslog (pamh, LOG_ERR,
		      "Couldn't store password as PAM_AUTHTOK.");
	  return SSH_AUTH_ERROR;
	}

      return ssh_result;
    }
  else
    {
      char *errmsg;
      int len;

      libssh2_session_last_error (session, &errmsg, &len, 0);

      send_pam_msg (pamh, PAM_TEXT_INFO, errmsg);
      return ssh_result;
    }
}

/*
 * auth_publickey
 *
 * Authenticate via publickey
 */

int
auth_publickey (pam_handle_t * pamh, const char *username,
		LIBSSH2_SESSION * session)
{
  int ssh_result = SSH_AUTH_ERROR;
  LIBSSH2_AGENT *agent = NULL;
  struct libssh2_agent_publickey *identity = NULL;
  struct libssh2_agent_publickey *prev_identity = NULL;

  /*
   * Authenticate depending on the method available.
   * Try public key first.
   */

  /* Connect to the ssh-agent */
  agent = libssh2_agent_init (session);

  if (!agent)
    {
      pam_debug (pamh, "Failure initializing ssh-agent support.");
      goto fail;
    }

  if (libssh2_agent_connect (agent))
    {
      pam_debug (pamh, "Failure connecting to ssh-agent.");
      goto fail;
    }

  if (libssh2_agent_list_identities (agent))
    {
      pam_debug (pamh, "Failure requesting identities to ssh-agent.");
      goto fail;
    }

  while (1)
    {
      ssh_result =
	libssh2_agent_get_identity (agent, &identity, prev_identity);

      if (ssh_result == 1)
	break;

      if (ssh_result < 0)
	{
	  pam_debug (pamh,
		     "Failure obtaining identity from ssh-agent support.");
	  ssh_result = 1;
	  goto fail;
	}

      if (libssh2_agent_userauth (agent, username, identity))
	{
	  pam_debug (pamh, "Publickey authentication failed!");
	}
      else
	{
	  pam_debug (pamh, "Publickey Authentication succeeded!");
	  break;
	}
      prev_identity = identity;
    }

  if (ssh_result)
    {
      pam_debug (pamh, "Couldn't continue authentication.");
      goto fail;
    }

  /* We're authenticated now. */
  if (ssh_result == SSH_AUTH_SUCCESS)
    {
      libssh2_agent_disconnect (agent);
    }

fail:
  return ssh_result;
}
