/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#include <core_resource.hxx>
#include "migrationerror.hxx"
#include "migrationlog.hxx"
#include <strings.hrc>

#include <comphelper/anytostring.hxx>
#include <osl/diagnose.h>
#include <rtl/ustrbuf.hxx>

#include <vector>
#include <map>
#include <algorithm>

namespace dbmm
{
    // LibraryEntry
    struct LibraryEntry
    {
        ScriptType      eType;
        OUString sOldName;
        OUString sNewName;

        LibraryEntry( const ScriptType& _eType, const OUString& _rOldName, const OUString& _rNewName )
            :eType( _eType )
            ,sOldName( _rOldName )
            ,sNewName( _rNewName )
        {
        }
    };

    // DocumentEntry
    struct DocumentEntry
    {
        SubDocumentType                 eType;
        OUString                 sName;
        std::vector< LibraryEntry >   aMovedLibraries;

        DocumentEntry()
            :eType( eForm )
            ,sName()
            ,aMovedLibraries()
        {
        }

        DocumentEntry( const SubDocumentType _eType, const OUString& _rName )
            :eType( _eType )
            ,sName( _rName )
        {
        }
    };

    // DocumentLogs
    typedef std::map< DocumentID, DocumentEntry > DocumentLogs;

    // ErrorLog
    typedef std::vector< MigrationError >   ErrorLog;

    // MigrationLog_Data
    struct MigrationLog_Data
    {
        OUString sBackupLocation;
        DocumentLogs    aDocumentLogs;
        ErrorLog        aFailures;
        ErrorLog        aWarnings;
    };

    // MigrationLog
    MigrationLog::MigrationLog()
        :m_pData( new MigrationLog_Data )
    {
    }

    MigrationLog::~MigrationLog()
    {
    }

    void MigrationLog::logFailure( const MigrationError& _rError )
    {
        m_pData->aFailures.push_back( _rError );
    }

    void MigrationLog::logRecoverable( const MigrationError& _rError )
    {
        m_pData->aWarnings.push_back( _rError );
    }

    bool MigrationLog::hadFailure() const
    {
        return !m_pData->aFailures.empty();
    }

    void MigrationLog::backedUpDocument( const OUString& _rNewDocumentLocation )
    {
        m_pData->sBackupLocation = _rNewDocumentLocation;
    }

    DocumentID MigrationLog::startedDocument( const SubDocumentType _eType, const OUString& _rName )
    {
#if OSL_DEBUG_LEVEL > 0
        bool bAlreadyKnown = false;
        for (auto const& documentLog : m_pData->aDocumentLogs)
        {
            bAlreadyKnown = ( documentLog.second.eType == _eType ) && ( documentLog.second.sName == _rName );
            if (bAlreadyKnown)
                break;
        }
        OSL_ENSURE( !bAlreadyKnown, "MigrationLog::startedDocument: document is already known!" );
#endif

        DocumentID nID = static_cast<DocumentID>( m_pData->aDocumentLogs.size() + 1 );
        while ( m_pData->aDocumentLogs.find( nID ) != m_pData->aDocumentLogs.end() )
            ++nID;

        m_pData->aDocumentLogs[ nID ] = DocumentEntry( _eType, _rName );

        return nID;
    }

    void MigrationLog::movedLibrary( const DocumentID _nDocID, const ScriptType _eScriptType,
            const OUString& _rOriginalLibName, const OUString& _rNewLibName )
    {
        OSL_ENSURE( m_pData->aDocumentLogs.find( _nDocID ) != m_pData->aDocumentLogs.end(),
            "MigrationLog::movedLibrary: document is not known!" );

        DocumentEntry& rDocEntry = m_pData->aDocumentLogs[ _nDocID ];
        rDocEntry.aMovedLibraries.emplace_back( _eScriptType, _rOriginalLibName, _rNewLibName );
    }

    void MigrationLog::finishedDocument( const DocumentID _nDocID )
    {
        OSL_ENSURE( m_pData->aDocumentLogs.find( _nDocID ) != m_pData->aDocumentLogs.end(),
            "MigrationLog::finishedDocument: document is not known!" );

        // nothing to do here
    }

    OUString MigrationLog::getNewLibraryName( DocumentID _nDocID, ScriptType _eScriptType,
        const OUString& _rOriginalLibName ) const
    {
        DocumentLogs::const_iterator docPos = m_pData->aDocumentLogs.find( _nDocID );
        if ( docPos == m_pData->aDocumentLogs.end() )
        {
            OSL_FAIL( "MigrationLog::getNewLibraryName: document is not known!" );
            return OUString();
        }

        const DocumentEntry& rDocEntry( docPos->second );
        for (auto const& elem : rDocEntry.aMovedLibraries)
        {
            if  (   ( _eScriptType == elem.eType )
                &&  ( _rOriginalLibName == elem.sOldName )
                )
                return elem.sNewName;
        }

        OSL_FAIL( "MigrationLog::getNewLibraryName: doc is known, but library isn't!" );
        return OUString();
    }

    namespace
    {
        void lcl_appendErrorDescription( OUStringBuffer& _inout_rBuffer, const MigrationError& _rError )
        {
            const sal_Char* pAsciiErrorDescription( nullptr );
            std::vector< OUString > aParameterNames;
            switch ( _rError.eType )
            {
            case ERR_OPENING_SUB_DOCUMENT_FAILED:
                pAsciiErrorDescription = "opening '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_CLOSING_SUB_DOCUMENT_FAILED:
                pAsciiErrorDescription = "closing '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_STORAGE_COMMIT_FAILED:
                pAsciiErrorDescription = "committing the changes for document '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_STORING_DATABASEDOC_FAILED:
                pAsciiErrorDescription = "storing the database document failed";
                break;

            case ERR_COLLECTING_DOCUMENTS_FAILED:
                pAsciiErrorDescription = "collecting the forms/reports of the database document failed";
                break;

            case ERR_UNEXPECTED_LIBSTORAGE_ELEMENT:
                pAsciiErrorDescription = "unexpected #lib# storage element in document '#doc#', named '#element#'";
                aParameterNames.emplace_back("#doc#");
                aParameterNames.emplace_back("#libstore#");
                aParameterNames.emplace_back("#element#");
                break;

            case ERR_CREATING_DBDOC_SCRIPT_STORAGE_FAILED:
                pAsciiErrorDescription = "creating the database document's storage for #scripttype# scripts failed";
                aParameterNames.emplace_back("#scripttype#");
                break;

            case ERR_COMMITTING_SCRIPT_STORAGES_FAILED:
                pAsciiErrorDescription = "saving the #scripttype# scripts for document '#doc#' failed";
                aParameterNames.emplace_back("#scripttype#");
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_GENERAL_SCRIPT_MIGRATION_FAILURE:
                pAsciiErrorDescription = "general error while migrating #scripttype# scripts of document '#doc#'";
                aParameterNames.emplace_back("#scripttype#");
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_GENERAL_MACRO_MIGRATION_FAILURE:
                pAsciiErrorDescription = "general error during macro migration of document '#doc#'";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_UNKNOWN_SCRIPT_TYPE:
                pAsciiErrorDescription = "unknown script type: #type#";
                aParameterNames.emplace_back("#type#");
                break;

            case ERR_UNKNOWN_SCRIPT_LANGUAGE:
                pAsciiErrorDescription = "unknown script language: #lang#";
                aParameterNames.emplace_back("#lang#");
                break;

            case ERR_UNKNOWN_SCRIPT_NAME_FORMAT:
                pAsciiErrorDescription = "unknown script name format: #script#";
                aParameterNames.emplace_back("#script#");
                break;

            case ERR_SCRIPT_TRANSLATION_FAILURE:
                pAsciiErrorDescription = "analyzing/translating the script URL failed; script type: #type#; script: #code#";
                aParameterNames.emplace_back("#type#");
                aParameterNames.emplace_back("#code#");
                break;

            case ERR_INVALID_SCRIPT_DESCRIPTOR_FORMAT:
                pAsciiErrorDescription = "invalid script descriptor format";
                break;

            case ERR_ADJUSTING_DOCUMENT_EVENTS_FAILED:
                pAsciiErrorDescription = "adjusting events for document '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_ADJUSTING_DIALOG_EVENTS_FAILED:
                pAsciiErrorDescription = "adjusting events for dialog #lib#.#dlg# in document '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                aParameterNames.emplace_back("#lib#");
                aParameterNames.emplace_back("#dlg#");
                break;

            case ERR_ADJUSTING_FORMCOMP_EVENTS_FAILED:
                pAsciiErrorDescription = "adjusting form component events for '#doc#' failed";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_BIND_SCRIPT_STORAGE_FAILED:
                pAsciiErrorDescription = "binding to the script storage failed for document '#doc#'";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_REMOVE_SCRIPTS_STORAGE_FAILED:
                pAsciiErrorDescription = "removing a scripts storage failed for document '#doc#'";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_DOCUMENT_BACKUP_FAILED:
                pAsciiErrorDescription = "backing up the document to #location# failed";
                aParameterNames.emplace_back("#location#");
                break;

            case ERR_UNKNOWN_SCRIPT_FOLDER:
                pAsciiErrorDescription = "unknown script folder '#name#' in document '#doc#'";
                aParameterNames.emplace_back("#doc#");
                aParameterNames.emplace_back("#name#");
                break;

            case ERR_EXAMINING_SCRIPTS_FOLDER_FAILED:
                pAsciiErrorDescription = "examining the 'Scripts' folder failed for document '#doc#'";
                aParameterNames.emplace_back("#doc#");
                break;

            case ERR_PASSWORD_VERIFICATION_FAILED:
                pAsciiErrorDescription = "password verification failed for document '#doc#', #libtype# library '#name#'";
                aParameterNames.emplace_back("#doc#");
                aParameterNames.emplace_back("#libtype#");
                aParameterNames.emplace_back("#name#");
                break;

            case ERR_NEW_STYLE_REPORT:
                pAsciiErrorDescription = "#doc# could not be processed, since you don't have the Report Builder feature installed.";
                aParameterNames.emplace_back("#doc#");
                break;

                // do *not* add a default case here: Without a default, some compilers will warn you when
                // you miss a newly-introduced enum value here
            }
            OSL_ENSURE( pAsciiErrorDescription, "lcl_appendErrorDescription: no error message!" );
            if ( pAsciiErrorDescription )
            {
                OUString sSubstituted( OUString::createFromAscii( pAsciiErrorDescription ) );
                OSL_ENSURE( aParameterNames.size() == _rError.aErrorDetails.size(),
                    "lcl_appendErrorDescription: unexpected number of error message parameters!" );

                for ( size_t i=0; i < std::min( aParameterNames.size(), _rError.aErrorDetails.size() ); ++i )
                {
                    sSubstituted = sSubstituted.replaceFirst(
                        aParameterNames[i], _rError.aErrorDetails[i]);
                }

                _inout_rBuffer.append( sSubstituted );
            }
        }

        void lcl_describeErrors( OUStringBuffer& _rBuffer, const ErrorLog& _rErrors, const char* pHeadingResId )
        {
            _rBuffer.append( "=== " );
            _rBuffer.append( DBA_RES(pHeadingResId) );
            _rBuffer.append( " ===\n" );

            OUString sException( DBA_RES( STR_EXCEPTION ) );

            for (auto const& error : _rErrors)
            {
                _rBuffer.append( '-' );
                _rBuffer.append( ' ' );
                lcl_appendErrorDescription(_rBuffer, error);
                _rBuffer.append( '\n' );

                if ( !error.aCaughtException.hasValue() )
                    continue;

                _rBuffer.append( sException );
                _rBuffer.append( ::comphelper::anyToString( error.aCaughtException ) );
                _rBuffer.append( '\n' );
                _rBuffer.append( '\n' );
            }
        }
    }

    bool MigrationLog::movedAnyLibrary( const DocumentID _nDocID )
    {
        DocumentLogs::const_iterator docPos = m_pData->aDocumentLogs.find( _nDocID );
        if ( docPos == m_pData->aDocumentLogs.end() )
        {
            OSL_FAIL( "MigrationLog::movedAnyLibrary: document is not known!" );
            return false;
        }
        return !docPos->second.aMovedLibraries.empty();
    }

    OUString MigrationLog::getCompleteLog() const
    {
        OUStringBuffer aBuffer;

        if ( !m_pData->sBackupLocation.isEmpty() )
        {
            OUString sBackedUp( DBA_RES( STR_SAVED_COPY_TO ) );
            sBackedUp = sBackedUp.replaceAll( "$location$", m_pData->sBackupLocation );

            aBuffer.append( "=== " + DBA_RES( STR_DATABASE_DOCUMENT )
                    + " ===\n" + sBackedUp + "\n\n");
        }

        if ( !m_pData->aFailures.empty() )
        {
            lcl_describeErrors( aBuffer, m_pData->aFailures
                , STR_ERRORS );
        }
        else
        {
            OUString sMovedLibTemplate( DBA_RES( STR_MOVED_LIBRARY ) );

            for (auto const& documentLog : m_pData->aDocumentLogs)
            {
                const DocumentEntry& rDoc( documentLog.second );

                if ( rDoc.aMovedLibraries.empty() )
                    continue;

                OUString sDocTitle( DBA_RES( rDoc.eType == eForm ? STR_FORM : STR_REPORT ) );
                sDocTitle = sDocTitle.replaceAll( "$name$", rDoc.sName );

                aBuffer.append( "=== " + sDocTitle + " ===\n" );

                for (auto const& elem : rDoc.aMovedLibraries)
                {
                    OUString sMovedLib( sMovedLibTemplate );
                    sMovedLib = sMovedLib.replaceAll( "$type$", getScriptTypeDisplayName( elem.eType ) );
                    sMovedLib = sMovedLib.replaceAll( "$old$", elem.sOldName );
                    sMovedLib = sMovedLib.replaceAll( "$new$", elem.sNewName );

                    aBuffer.append( sMovedLib + "\n" );
                }

                aBuffer.append( '\n' );
            }
        }

        if ( !m_pData->aWarnings.empty() )
        {
            lcl_describeErrors( aBuffer, m_pData->aWarnings, STR_WARNINGS );
        }

        return aBuffer.makeStringAndClear();
    }

} // namespace dbmm

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
