/* 
 * Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Implementation of the mforms splitter control for Windows.
 */

#include "stdafx.h"
#include "wf_splitter.h"

using namespace MySQL::Forms;
using namespace System::Windows::Forms;

//--------------------------------------------------------------------------------------------------

SplitterImpl::SplitterImpl(::mforms::Splitter *self)
  : ViewImpl(self)
{
  pendingSplitterDistance= -1;
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_position(int position)
{
  SplitContainer^ container= get_control<SplitContainer>();
  if (container->IsHandleCreated && container->Visible)
    container->SplitterDistance= position;
  else
    pendingSplitterDistance= position;
}

//--------------------------------------------------------------------------------------------------

int SplitterImpl::get_position()
{
  SplitContainer^ container= get_control<SplitContainer>();
  if (pendingSplitterDistance == -1)
    return container->SplitterDistance;
  else
    return pendingSplitterDistance;
}

//--------------------------------------------------------------------------------------------------

bool SplitterImpl::create(::mforms::Splitter *self, bool horizontal)
{
  SplitterImpl^ splitter= gcnew SplitterImpl(self);

  if (splitter != nullptr)
  {
    SplitContainer^ container= ViewImpl::create<SplitContainer>(self, splitter);
    container->BackColor = Color::Transparent;
    container->Paint += gcnew PaintEventHandler(&SplitterImpl::DoPaint);
    container->SplitterMoved += gcnew SplitterEventHandler(&SplitterImpl::DoSplitterMoved);
    container->Size= Size(100, 100);
    container->TabStop = false;

    // Note: parameter @horizontal means the layout direction, not that of the splitter line.
    //       It is important to set the splitter distance to a good value or the split container
    //       throws an exception (why doesn't it simply set to the allowed range instead?).
    container->SplitterDistance= horizontal ? container->Height / 2 : container->Width / 2;
    container->Orientation= horizontal? Orientation::Vertical : Orientation::Horizontal;
    container->SplitterWidth = 6;

    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

/**
 * Adds a new child window to the splitter. Behavior is as follows:
 * - If the left panel is empty add to this.
 * - If not and the right panel is empty add to this.
 * - If no panel is empty then we have an error.
 */
void SplitterImpl::add(::mforms::Splitter *self, ::mforms::View *child, int min_size)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    ViewImpl^ view= (ViewImpl^)ObjectImpl::FromUnmanaged(child);
    Control^ control= view->get_control<Control>();

    SplitContainer^ container= splitter->get_control<SplitContainer>();
    if (container != nullptr && control != nullptr)
    {
      if (container->Panel1->Controls->Count == 0)
      {
        container->Panel1->Controls->Add(control);
        container->Panel1MinSize= min_size;
        control->Dock= DockStyle::Fill;
        child->set_resize_mode(::mforms::ResizeNone);
      }
      else
        if (container->Panel2->Controls->Count == 0)
        {
          container->Panel2->Controls->Add(control);
          container->Panel2MinSize= min_size;
          control->Dock= DockStyle::Fill;
          child->set_resize_mode(::mforms::ResizeNone);
        }
        else
          throw std::logic_error("mforms splitter error: adding more than 2 child controls is not allowed.");

      // Hide the panel that has no child control.
      container->Panel1Collapsed = container->Panel1->Controls->Count == 0;
      container->Panel2Collapsed = container->Panel2->Controls->Count == 0;
    }
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::remove(::mforms::Splitter *self, ::mforms::View *child)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    ViewImpl^ view= (ViewImpl^)ObjectImpl::FromUnmanaged(child);
    Control^ control= view->get_control<Control>();

    // Since there can only be one child window on each panel we do a Clear() instead removing only the
    // individual control, fixing so possibly invalid setups (if they ever occur).
    SplitContainer^ container= splitter->get_control<SplitContainer>();
    if (container->Panel1->Controls->Contains(control))
    {
      // Child is on the first panel. Remove it from there and move the child from panel 2 to panel 1
      // (if there is one).
      container->Panel1->Controls->Clear();
      if (container->Panel2->Controls->Count > 0)
      {
        Control^ second_control= container->Panel2->Controls[0];
        container->Panel2->Controls->Clear();
        container->Panel1->Controls->Add(second_control);
      }
    }
    else
      if (container->Panel2->Controls->Contains(control))
        container->Panel2->Controls->Clear();
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_position(::mforms::Splitter *self, int position)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
    splitter->set_position(position);
}

//--------------------------------------------------------------------------------------------------

int SplitterImpl::get_position(::mforms::Splitter *self)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
    return splitter->get_position();
  return 0;
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::set_expanded(::mforms::Splitter *self, bool first, bool expand)
{
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(self);

  if (splitter != nullptr)
  {
    SplitContainer^ container= splitter->get_control<SplitContainer>();
    if (first)
      container->Panel1Collapsed = !expand;
    else
      container->Panel2Collapsed = !expand;
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * We don't use the paint event for drawing but to apply a pending splitter position setting
 * as it is way too unreliable to set it directly (depends on visibility state, size, created handles etc.).
 */
void SplitterImpl::DoPaint(System::Object ^sender, PaintEventArgs ^e)
{
  SplitContainer^ container= dynamic_cast<SplitContainer^>(sender);
  mforms::Splitter* backend= ViewImpl::get_backend_control<mforms::Splitter>(container);
  SplitterImpl^ splitter= (SplitterImpl^) ObjectImpl::FromUnmanaged(backend);
  if (splitter->pendingSplitterDistance > -1)
  {
    container->SplitterDistance= splitter->pendingSplitterDistance;
    splitter->pendingSplitterDistance= -1;
  }
}

//--------------------------------------------------------------------------------------------------

void SplitterImpl::DoSplitterMoved(System::Object^ sender, SplitterEventArgs^ e)
{
  SplitContainer^ container= dynamic_cast<SplitContainer^>(sender);
  mforms::Splitter* backend= ViewImpl::get_backend_control<mforms::Splitter>(container);
  if (backend != NULL)
    backend->position_changed();
}

//--------------------------------------------------------------------------------------------------
