// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef DEVICE_BATTERY_BATTERY_STATUS_SERVICE_H_
#define DEVICE_BATTERY_BATTERY_STATUS_SERVICE_H_

#include "base/callback_list.h"
#include "base/macros.h"
#include "base/memory/scoped_ptr.h"
#include "base/memory/singleton.h"
#include "device/battery/battery_export.h"
#include "device/battery/battery_status.mojom.h"

namespace base {
class SingleThreadTaskRunner;
}

namespace device {
class BatteryStatusManager;

class DEVICE_BATTERY_EXPORT BatteryStatusService {
 public:
  typedef base::Callback<void(const BatteryStatus&)> BatteryUpdateCallback;
  typedef base::CallbackList<void(const BatteryStatus&)>
      BatteryUpdateCallbackList;
  typedef BatteryUpdateCallbackList::Subscription BatteryUpdateSubscription;

  // Returns the BatteryStatusService singleton.
  static BatteryStatusService* GetInstance();

  // Adds a callback to receive battery status updates.  Must be called on the
  // main thread. The callback itself will be called on the main thread as well.
  // NOTE: The callback may be run before AddCallback returns!
  scoped_ptr<BatteryUpdateSubscription> AddCallback(
      const BatteryUpdateCallback& callback);

  // Gracefully clean-up.
  void Shutdown();

  // Injects a custom battery status manager for testing purposes.
  void SetBatteryManagerForTesting(
      scoped_ptr<BatteryStatusManager> test_battery_manager);

  // Returns callback to invoke when battery is changed. Used for testing.
  const BatteryUpdateCallback& GetUpdateCallbackForTesting() const;

 private:
  friend struct base::DefaultSingletonTraits<BatteryStatusService>;
  friend class BatteryStatusServiceTest;

  BatteryStatusService();
  virtual ~BatteryStatusService();

  // Updates current battery status and sends new status to interested
  // render processes. Can be called on any thread via a callback.
  void NotifyConsumers(const BatteryStatus& status);
  void NotifyConsumersOnMainThread(const BatteryStatus& status);
  void ConsumersChanged();

  scoped_refptr<base::SingleThreadTaskRunner> main_thread_task_runner_;
  scoped_ptr<BatteryStatusManager> battery_fetcher_;
  BatteryUpdateCallbackList callback_list_;
  BatteryUpdateCallback update_callback_;
  BatteryStatus status_;
  bool status_updated_;
  bool is_shutdown_;

  DISALLOW_COPY_AND_ASSIGN(BatteryStatusService);
};

}  // namespace device

#endif  // DEVICE_BATTERY_BATTERY_STATUS_SERVICE_H_
