% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genomeToX.R
\name{genomeToTranscript}
\alias{genomeToTranscript}
\title{Map genomic coordinates to transcript coordinates}
\usage{
genomeToTranscript(x, db)
}
\arguments{
\item{x}{\code{GRanges} object with the genomic coordinates that should be
mapped.}

\item{db}{\code{EnsDb} object.}
}
\value{
An \code{IRangesList} with length equal to \code{length(x)}. Each element providing
the mapping(s) to position within any encoded transcripts at the respective
genomic location as an \code{IRanges} object. An \code{IRanges} with negative start
coordinates is returned, if the provided genomic coordinates are not
completely within the genomic coordinates of an exon.

The ID of the exon and its rank (index of the exon in the transcript) are
provided in the result's \code{IRanges} metadata columns as well as the genomic
position of \code{x}.
}
\description{
\code{genomeToTranscript} maps genomic coordinates to positions within the
transcript (if at the provided genomic position a transcript is encoded).
The function does only support mapping of genomic coordinates that are
completely within the genomic region at which an exon is encoded. If the
genomic region crosses the exon boundary an empty \code{IRanges} is returned.
See examples for details.
}
\details{
The function first retrieves all exons overlapping the provided genomic
coordinates and identifies then exons that are fully containing the
coordinates in \code{x}. The transcript-relative coordinates are calculated based
on the relative position of the provided genomic coordinates in this exon.
}
\note{
The function throws a warning and returns an empty \code{IRanges} object if the
genomic coordinates can not be mapped to a transcript.
}
\examples{

library(EnsDb.Hsapiens.v86)

## Subsetting the EnsDb object to chromosome X only to speed up execution
## time of examples
edbx <- filter(EnsDb.Hsapiens.v86, filter = ~ seq_name == "X")

## Define a genomic region and calculate within-transcript coordinates
gnm <- GRanges("X:107716399-107716401")

res <- genomeToTranscript(gnm, edbx)
## Result is an IRanges object with the start and end coordinates within
## each transcript that has an exon at the genomic range.
res

## An IRanges with negative coordinates is returned if at the provided
## position no exon is present. Below we use the same coordinates but
## specify that the coordinates are on the forward (+) strand
gnm <- GRanges("X:107716399-107716401:+")
genomeToTranscript(gnm, edbx)

## Next we provide multiple genomic positions.
gnm <- GRanges("X", IRanges(start = c(644635, 107716399, 107716399),
    end = c(644639, 107716401, 107716401)), strand = c("*", "*", "+"))

## The result of the mapping is an IRangesList each element providing the
## within-transcript coordinates for each input region
genomeToTranscript(gnm, edbx)
}
\seealso{
Other coordinate mapping functions: \code{\link{cdsToTranscript}},
  \code{\link{genomeToProtein}},
  \code{\link{proteinToGenome}},
  \code{\link{proteinToTranscript}},
  \code{\link{transcriptToCds}},
  \code{\link{transcriptToGenome}},
  \code{\link{transcriptToProtein}}
}
\author{
Johannes Rainer
}
\concept{coordinate mapping functions}
