/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, you can obtain one at http://mozilla.org/MPL/2.0/. */

/* globals gFolderDisplay, gFolderTreeView, MsgOpenNewWindowForMessage, MsgOpenSelectedMessages  */

add_task(async () => {
  let extension = ExtensionTestUtils.loadExtension({
    async background() {
      function waitForEvent() {
        return new Promise(resolve => {
          let listener = (...args) => {
            browser.messageDisplay.onMessageDisplayed.removeListener(listener);
            resolve(args);
          };
          browser.messageDisplay.onMessageDisplayed.addListener(listener);
        });
      }

      let [{ id: firstTabId, displayedFolder }] = await browser.mailTabs.query({
        active: true,
        currentWindow: true,
      });
      let { messages } = await browser.messages.list(displayedFolder);

      // Test that selecting a different message fires the event.
      let eventListener = waitForEvent();
      browser.test.sendMessage("show message 1");
      let [tabId, message] = await eventListener;
      browser.test.assertEq(firstTabId, tabId);
      browser.test.assertEq(messages[1].subject, message.subject);

      message = await browser.messageDisplay.getDisplayedMessage(tabId);
      browser.test.assertEq(messages[1].subject, message.subject);

      // ... and again, for good measure.
      eventListener = waitForEvent();
      browser.test.sendMessage("show message 2");
      [tabId, message] = await eventListener;
      browser.test.assertEq(firstTabId, tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      message = await browser.messageDisplay.getDisplayedMessage(tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      // Test that opening a message in a new tab fires the event.
      eventListener = waitForEvent();
      browser.test.sendMessage("open message tab");
      [tabId, message] = await eventListener;
      browser.test.assertTrue(firstTabId != tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      message = await browser.messageDisplay.getDisplayedMessage(tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      // Test that the first tab is not displaying a message.
      message = await browser.messageDisplay.getDisplayedMessage(firstTabId);
      browser.test.assertEq(null, message);

      // Closing the tab should return us to the first tab, and fires the
      // event. It doesn't have to be this way, it just is.
      eventListener = waitForEvent();
      browser.tabs.remove(tabId);
      [tabId, message] = await eventListener;
      browser.test.assertEq(firstTabId, tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      message = await browser.messageDisplay.getDisplayedMessage(tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      // Test that opening a message in a new window fires the event.
      eventListener = waitForEvent();
      browser.test.sendMessage("open message window");
      [tabId, message] = await eventListener;
      browser.test.assertTrue(firstTabId != tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      message = await browser.messageDisplay.getDisplayedMessage(tabId);
      browser.test.assertEq(messages[2].subject, message.subject);

      // Close the window.
      browser.tabs.remove(tabId);

      browser.test.notifyPass("finished");
    },
    manifest: {
      permissions: ["accountsRead", "messagesRead"],
    },
  });

  let account = createAccount();
  let rootFolder = account.incomingServer.rootFolder;
  rootFolder.createSubfolder("test1", null);
  let subFolders = {};
  for (let folder of rootFolder.subFolders) {
    subFolders[folder.name] = folder;
  }
  createMessages(subFolders.test1, 5);

  gFolderTreeView.selectFolder(subFolders.test1);
  gFolderDisplay.selectViewIndex(0);
  // For some unknown reason, this test doesn't like switching to the main
  // window if the thread tree has focus. Make sure it doesn't have focus.
  document.getElementById("folderTree").focus();

  await extension.startup();

  await extension.awaitMessage("show message 1");
  await new Promise(resolve => setTimeout(resolve));
  gFolderDisplay.selectViewIndex(1);

  await extension.awaitMessage("show message 2");
  await new Promise(resolve => setTimeout(resolve));
  gFolderDisplay.selectViewIndex(2);

  await extension.awaitMessage("open message tab");
  await new Promise(resolve => setTimeout(resolve));
  MsgOpenSelectedMessages();

  await extension.awaitMessage("open message window");
  await new Promise(resolve => setTimeout(resolve));
  MsgOpenNewWindowForMessage();

  await extension.awaitFinish("finished");
  await extension.unload();
});
