-- lib/tongue/codex.lua
--
-- Lua I18N library 'Tongue' - Language Codexes
--
-- Copyright 2016 Daniel Silverstone <dsilvers@digital-scurf.org>
--
-- For licence terms, see COPYING
--

--- Language Codexes comprise zero or more language packs which together
-- represent a multilingual translation for a single domain.
--
-- Codexes fundamentally consist of a number of loaders, some logic for
-- managing the packs, and a mechanism for creating new blank packs on demand.
--
-- @module tongue.codex

local codex = {}

--- Tongue Language Codex.
--
-- A Tongue Language Codex is the entry point for applications into Tongue.
-- Codexes are given loaders which can load language packs into the codex and
-- they automatically invoke the loaders when needed, to reduce unnecessary
-- disc access.
--
-- @type codex

--- Add a loader to the codex.
--
-- Loaders are added to a list.  The first loader added to a codex will be
-- expected to construct the language pack, later loaders will be given the
-- language pack to add to.  If the loader determines that it needs to create a
-- language pack which depends on another language pack, then it can request
-- the pack from the codex.  No protection is given to this causing loops, that
-- is up to the loaders.
--
-- @tparam function loader The loader function to add to the codex.
-- @function add_loader
function codex:add_loader(loader)
   self.loaders[#self.loaders + 1] = loader
end

--- Retrieve the given language pack from the codex.
--
-- Note, if the language pack has not yet been loaded, or if new loaders have
-- been added since the language pack was loaded, then loaders may be invoked
-- to create or augment the language pack before it is returned.
--
-- @tparam string lang The base language of the desired pack.
-- @tparam ?string sublang The (optional) sub-language of the desired pack.
-- @treturn langpack The language pack from the codex for the given language.
-- @function get_langpack
function codex:get_langpack(lang, sublang)
   local langentry = lang .. (sublang and ("_" .. sublang) or "")
   -- If we lack a packentry, make it
   if not self.packs[langentry] then
      self.packs[langentry] = { loaders_used = 0,
				pack = self:_newPack(lang, sublang) }
   end
   local packentry = self.packs[langentry]
   -- If our packentry is insufficiently loaded, load more
   while packentry.loaders_used < #self.loaders do
      local loader_n = packentry.loaders_used + 1
      local loader = self.loaders[loader_n]
      loader(codex, packentry.pack, langentry)
      packentry.loaders_used = loader_n
   end
   -- And finally return the loaded pack
   return packentry.pack
end

--- Expand a translation using the codex.
--
-- Note, this can cause loaders to be invoked if the language in question has
-- not yet been loaded.
--
-- @tparam string lang The base language of the desired expansion
-- @tparam ?string sublang The (optional) sub-language of the desired expansion
-- @tparam string token The token to expand
-- @tparam table args The token's arguments for expansion
-- @function expand
function codex:expand(lang, sublang, token, args)
   local pack = self:get_langpack(lang, sublang)
   return pack:expand(token, args or {})
end

local codex_mt = {__index=codex}

---
-- @section tongue.codex

--- Create a new language codex.
--
-- Codexes require only one important function to be created -- namely a
-- function which when called with language and sub-language will create an
-- instance of `langpack`.  If you do not pass a function to do this, then
-- Tongue's default behaviour will be to parent sub-languages to their base
-- language, and otherwise to not parent languages.  This means, for example,
-- en/GB will be parented to en.
--
-- @tparam ?function newpack Function for creating a new langpack instance.
-- @treturn codex
-- @function create
local function createCodex(newpack)
   if not newpack then
      function newpack(codex, lang, sublang)
	 local langpack = require "tongue.langpack"
	 local parent = nil
	 if sublang then
	    parent = codex:get_langpack(lang, nil)
	 end
	 return langpack.create(lang, sublang, parent)
      end
   end
   return setmetatable({_newPack=newpack, packs={}, loaders={}}, codex_mt)
end

return {
   create = createCodex,
}
