const builtin = @import("builtin");
const inf = math.inf;
const math = std.math;
const fmax = math.floatMax;
const fmin = math.floatMin;
const imax = math.maxInt;
const imin = math.minInt;
const nan = math.nan;
const next = math.nextAfter;
const std = @import("std");
const tmin = math.floatTrueMin;

const Gpr = switch (builtin.cpu.arch) {
    else => unreachable,
    .x86 => u32,
    .x86_64 => u64,
};
const Sse = if (std.Target.x86.featureSetHas(builtin.cpu.features, .avx))
    @Vector(32, u8)
else
    @Vector(16, u8);

inline fn runtime(comptime Type: type, comptime value: Type) Type {
    if (@inComptime()) return value;
    return struct {
        var variable: Type = value;
    }.variable;
}

fn Scalar(comptime Type: type) type {
    return switch (@typeInfo(Type)) {
        else => Type,
        .vector => |info| info.child,
    };
}
fn AddOneBit(comptime Type: type) type {
    const ResultScalar = switch (@typeInfo(Scalar(Type))) {
        .int => |int| @Type(.{ .int = .{ .signedness = int.signedness, .bits = 1 + int.bits } }),
        .float => Scalar(Type),
        else => @compileError(@typeName(Type)),
    };
    return switch (@typeInfo(Type)) {
        else => ResultScalar,
        .vector => |vector| @Vector(vector.len, ResultScalar),
    };
}
fn DoubleBits(comptime Type: type) type {
    const ResultScalar = switch (@typeInfo(Scalar(Type))) {
        .int => |int| @Type(.{ .int = .{ .signedness = int.signedness, .bits = int.bits * 2 } }),
        .float => Scalar(Type),
        else => @compileError(@typeName(Type)),
    };
    return switch (@typeInfo(Type)) {
        else => ResultScalar,
        .vector => |vector| @Vector(vector.len, ResultScalar),
    };
}
fn RoundBitsUp(comptime Type: type, comptime multiple: u16) type {
    const ResultScalar = switch (@typeInfo(Scalar(Type))) {
        .int => |int| @Type(.{ .int = .{ .signedness = int.signedness, .bits = std.mem.alignForward(u16, int.bits, multiple) } }),
        .float => Scalar(Type),
        else => @compileError(@typeName(Type)),
    };
    return switch (@typeInfo(Type)) {
        else => ResultScalar,
        .vector => |vector| @Vector(vector.len, ResultScalar),
    };
}
// inline to avoid a runtime `@splat`
inline fn splat(comptime Type: type, scalar: Scalar(Type)) Type {
    return switch (@typeInfo(Type)) {
        else => scalar,
        .vector => @splat(scalar),
    };
}
// inline to avoid a runtime `@select`
inline fn select(cond: anytype, lhs: anytype, rhs: @TypeOf(lhs)) @TypeOf(lhs) {
    return switch (@typeInfo(@TypeOf(cond))) {
        .bool => if (cond) lhs else rhs,
        .vector => @select(Scalar(@TypeOf(lhs)), cond, lhs, rhs),
        else => @compileError(@typeName(@TypeOf(cond))),
    };
}
fn sign(rhs: anytype) switch (@typeInfo(@TypeOf(rhs))) {
    else => bool,
    .vector => |vector| @Vector(vector.len, bool),
} {
    const ScalarInt = @Type(.{ .int = .{
        .signedness = .unsigned,
        .bits = @bitSizeOf(Scalar(@TypeOf(rhs))),
    } });
    const VectorInt = switch (@typeInfo(@TypeOf(rhs))) {
        else => ScalarInt,
        .vector => |vector| @Vector(vector.len, ScalarInt),
    };
    return @as(VectorInt, @bitCast(rhs)) & splat(VectorInt, @as(ScalarInt, 1) << @bitSizeOf(ScalarInt) - 1) != splat(VectorInt, 0);
}
fn boolAnd(lhs: anytype, rhs: @TypeOf(lhs)) @TypeOf(lhs) {
    switch (@typeInfo(@TypeOf(lhs))) {
        .bool => return lhs and rhs,
        .vector => |vector| switch (vector.child) {
            bool => {
                const Bits = @Type(.{ .int = .{ .signedness = .unsigned, .bits = vector.len } });
                const lhs_bits: Bits = @bitCast(lhs);
                const rhs_bits: Bits = @bitCast(rhs);
                return @bitCast(lhs_bits & rhs_bits);
            },
            else => {},
        },
        else => {},
    }
    @compileError("unsupported boolAnd type: " ++ @typeName(@TypeOf(lhs)));
}
fn boolOr(lhs: anytype, rhs: @TypeOf(lhs)) @TypeOf(lhs) {
    switch (@typeInfo(@TypeOf(lhs))) {
        .bool => return lhs or rhs,
        .vector => |vector| switch (vector.child) {
            bool => {
                const Bits = @Type(.{ .int = .{ .signedness = .unsigned, .bits = vector.len } });
                const lhs_bits: Bits = @bitCast(lhs);
                const rhs_bits: Bits = @bitCast(rhs);
                return @bitCast(lhs_bits | rhs_bits);
            },
            else => {},
        },
        else => {},
    }
    @compileError("unsupported boolOr type: " ++ @typeName(@TypeOf(lhs)));
}

const Compare = enum { strict, relaxed, approx, approx_int };
// noinline for a more helpful stack trace
noinline fn checkExpected(expected: anytype, actual: @TypeOf(expected), comptime compare: Compare) !void {
    const Expected = @TypeOf(expected);
    const unexpected = unexpected: switch (@typeInfo(Scalar(Expected))) {
        else => expected != actual,
        .float => switch (compare) {
            .strict, .relaxed => {
                const unequal = boolAnd(expected != actual, boolOr(expected == expected, actual == actual));
                break :unexpected switch (compare) {
                    .strict => boolOr(unequal, sign(expected) != sign(actual)),
                    .relaxed => unequal,
                    .approx, .approx_int => comptime unreachable,
                };
            },
            .approx, .approx_int => {
                const epsilon = math.floatEps(Scalar(Expected));
                const tolerance = @sqrt(epsilon);
                break :unexpected @abs(expected - actual) > @max(
                    @abs(expected) * splat(Expected, tolerance),
                    splat(Expected, switch (compare) {
                        .strict, .relaxed => comptime unreachable,
                        .approx => tolerance,
                        .approx_int => 1,
                    }),
                );
            },
        },
    };
    if (switch (@typeInfo(Expected)) {
        else => unexpected,
        .vector => @reduce(.Or, unexpected),
    }) return error.Unexpected;
}
test checkExpected {
    if (checkExpected(nan(f16), nan(f16), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(nan(f16), -nan(f16), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f16, 0.0), @as(f16, 0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f16, -0.0), @as(f16, -0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f16, -0.0), @as(f16, 0.0), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f16, 0.0), @as(f16, -0.0), .strict) != error.Unexpected) return error.Unexpected;

    if (checkExpected(nan(f32), nan(f32), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(nan(f32), -nan(f32), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f32, 0.0), @as(f32, 0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f32, -0.0), @as(f32, -0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f32, -0.0), @as(f32, 0.0), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f32, 0.0), @as(f32, -0.0), .strict) != error.Unexpected) return error.Unexpected;

    if (checkExpected(nan(f64), nan(f64), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(nan(f64), -nan(f64), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f64, 0.0), @as(f64, 0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f64, -0.0), @as(f64, -0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f64, -0.0), @as(f64, 0.0), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f64, 0.0), @as(f64, -0.0), .strict) != error.Unexpected) return error.Unexpected;

    if (checkExpected(nan(f80), nan(f80), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(nan(f80), -nan(f80), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f80, 0.0), @as(f80, 0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f80, -0.0), @as(f80, -0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f80, -0.0), @as(f80, 0.0), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f80, 0.0), @as(f80, -0.0), .strict) != error.Unexpected) return error.Unexpected;

    if (checkExpected(nan(f128), nan(f128), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(nan(f128), -nan(f128), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f128, 0.0), @as(f128, 0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f128, -0.0), @as(f128, -0.0), .strict) == error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f128, -0.0), @as(f128, 0.0), .strict) != error.Unexpected) return error.Unexpected;
    if (checkExpected(@as(f128, 0.0), @as(f128, -0.0), .strict) != error.Unexpected) return error.Unexpected;
}

fn unary(comptime op: anytype, comptime opts: struct {
    libc_name: ?[]const u8 = null,
    compare: Compare = .relaxed,
}) type {
    return struct {
        // noinline so that `mem_arg` is on the stack
        noinline fn testArgKinds(
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            comptime Type: type,
            comptime imm_arg: Type,
            mem_arg: Type,
        ) !void {
            const expected = expected: {
                if (opts.libc_name) |libc_name| libc: {
                    const libc_func = @extern(*const fn (Scalar(Type)) callconv(.c) Scalar(Type), .{
                        .name = switch (Scalar(Type)) {
                            f16 => "__" ++ libc_name ++ "h",
                            f32 => libc_name ++ "f",
                            f64 => libc_name,
                            f80 => "__" ++ libc_name ++ "x",
                            f128 => libc_name ++ "q",
                            else => break :libc,
                        },
                    });
                    switch (@typeInfo(Type)) {
                        else => break :expected libc_func(imm_arg),
                        .vector => |vector| {
                            var res: Type = undefined;
                            inline for (0..vector.len) |i| res[i] = libc_func(imm_arg[i]);
                            break :expected res;
                        },
                    }
                }
                break :expected comptime op(Type, imm_arg);
            };
            var reg_arg = mem_arg;
            _ = .{&reg_arg};
            try checkExpected(expected, op(Type, reg_arg), opts.compare);
            try checkExpected(expected, op(Type, mem_arg), opts.compare);
            if (opts.libc_name == null) try checkExpected(expected, op(Type, imm_arg), opts.compare);
        }
        // noinline for a more helpful stack trace
        noinline fn testArgs(comptime Type: type, comptime imm_arg: Type) !void {
            try testArgKinds(
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                Type,
                imm_arg,
                imm_arg,
            );
        }
        fn testIntTypes() !void {
            try testArgs(i1, undefined);
            try testArgs(u1, undefined);
            try testArgs(i2, undefined);
            try testArgs(u2, undefined);
            try testArgs(i3, undefined);
            try testArgs(u3, undefined);
            try testArgs(i4, undefined);
            try testArgs(u4, undefined);
            try testArgs(i5, undefined);
            try testArgs(u5, undefined);
            try testArgs(i7, undefined);
            try testArgs(u7, undefined);
            try testArgs(i8, undefined);
            try testArgs(u8, undefined);
            try testArgs(i9, undefined);
            try testArgs(u9, undefined);
            try testArgs(i15, undefined);
            try testArgs(u15, undefined);
            try testArgs(i16, undefined);
            try testArgs(u16, undefined);
            try testArgs(i17, undefined);
            try testArgs(u17, undefined);
            try testArgs(i31, undefined);
            try testArgs(u31, undefined);
            try testArgs(i32, undefined);
            try testArgs(u32, undefined);
            try testArgs(i33, undefined);
            try testArgs(u33, undefined);
            try testArgs(i63, undefined);
            try testArgs(u63, undefined);
            try testArgs(i64, undefined);
            try testArgs(u64, undefined);
            try testArgs(i65, undefined);
            try testArgs(u65, undefined);
            try testArgs(i95, undefined);
            try testArgs(u95, undefined);
            try testArgs(i96, undefined);
            try testArgs(u96, undefined);
            try testArgs(i97, undefined);
            try testArgs(u97, undefined);
            try testArgs(i127, undefined);
            try testArgs(u127, undefined);
            try testArgs(i128, undefined);
            try testArgs(u128, undefined);
            try testArgs(i129, undefined);
            try testArgs(u129, undefined);
            try testArgs(i159, undefined);
            try testArgs(u159, undefined);
            try testArgs(i160, undefined);
            try testArgs(u160, undefined);
            try testArgs(i161, undefined);
            try testArgs(u161, undefined);
            try testArgs(i191, undefined);
            try testArgs(u191, undefined);
            try testArgs(i192, undefined);
            try testArgs(u192, undefined);
            try testArgs(i193, undefined);
            try testArgs(u193, undefined);
            try testArgs(i223, undefined);
            try testArgs(u223, undefined);
            try testArgs(i224, undefined);
            try testArgs(u224, undefined);
            try testArgs(i225, undefined);
            try testArgs(u225, undefined);
            try testArgs(i255, undefined);
            try testArgs(u255, undefined);
            try testArgs(i256, undefined);
            try testArgs(u256, undefined);
            try testArgs(i257, undefined);
            try testArgs(u257, undefined);
            try testArgs(i511, undefined);
            try testArgs(u511, undefined);
            try testArgs(i512, undefined);
            try testArgs(u512, undefined);
            try testArgs(i513, undefined);
            try testArgs(u513, undefined);
            try testArgs(i1023, undefined);
            try testArgs(u1023, undefined);
            try testArgs(i1024, undefined);
            try testArgs(u1024, undefined);
            try testArgs(i1025, undefined);
            try testArgs(u1025, undefined);
        }
        fn testInts() !void {
            try testArgs(i1, -1);
            try testArgs(i1, 0);
            try testArgs(u1, 0);
            try testArgs(u1, 1 << 0);

            try testArgs(i2, -1 << 1);
            try testArgs(i2, -1);
            try testArgs(i2, 0);
            try testArgs(u2, 0);
            try testArgs(u2, 1 << 0);
            try testArgs(u2, 1 << 1);

            try testArgs(i3, -1 << 2);
            try testArgs(i3, -1);
            try testArgs(i3, 0);
            try testArgs(u3, 0);
            try testArgs(u3, 1 << 0);
            try testArgs(u3, 1 << 1);
            try testArgs(u3, 1 << 2);

            try testArgs(i4, -1 << 3);
            try testArgs(i4, -1);
            try testArgs(i4, 0);
            try testArgs(u4, 0);
            try testArgs(u4, 1 << 0);
            try testArgs(u4, 1 << 1);
            try testArgs(u4, 1 << 2);
            try testArgs(u4, 1 << 3);

            try testArgs(i5, -1 << 4);
            try testArgs(i5, -1);
            try testArgs(i5, 0);
            try testArgs(u5, 0);
            try testArgs(u5, 1 << 0);
            try testArgs(u5, 1 << 1);
            try testArgs(u5, 1 << 3);
            try testArgs(u5, 1 << 4);

            try testArgs(i7, -1 << 6);
            try testArgs(i7, -1);
            try testArgs(i7, 0);
            try testArgs(u7, 0);
            try testArgs(u7, 1 << 0);
            try testArgs(u7, 1 << 1);
            try testArgs(u7, 1 << 5);
            try testArgs(u7, 1 << 6);

            try testArgs(i8, -1 << 7);
            try testArgs(i8, -1);
            try testArgs(i8, 0);
            try testArgs(u8, 0);
            try testArgs(u8, 1 << 0);
            try testArgs(u8, 1 << 1);
            try testArgs(u8, 1 << 6);
            try testArgs(u8, 1 << 7);

            try testArgs(i9, -1 << 8);
            try testArgs(i9, -1);
            try testArgs(i9, 0);
            try testArgs(u9, 0);
            try testArgs(u9, 1 << 0);
            try testArgs(u9, 1 << 1);
            try testArgs(u9, 1 << 7);
            try testArgs(u9, 1 << 8);

            try testArgs(i15, -1 << 14);
            try testArgs(i15, -1);
            try testArgs(i15, 0);
            try testArgs(u15, 0);
            try testArgs(u15, 1 << 0);
            try testArgs(u15, 1 << 1);
            try testArgs(u15, 1 << 13);
            try testArgs(u15, 1 << 14);

            try testArgs(i16, -1 << 15);
            try testArgs(i16, -1);
            try testArgs(i16, 0);
            try testArgs(u16, 0);
            try testArgs(u16, 1 << 0);
            try testArgs(u16, 1 << 1);
            try testArgs(u16, 1 << 14);
            try testArgs(u16, 1 << 15);

            try testArgs(i17, -1 << 16);
            try testArgs(i17, -1);
            try testArgs(i17, 0);
            try testArgs(u17, 0);
            try testArgs(u17, 1 << 0);
            try testArgs(u17, 1 << 1);
            try testArgs(u17, 1 << 15);
            try testArgs(u17, 1 << 16);

            try testArgs(i31, -1 << 30);
            try testArgs(i31, -1);
            try testArgs(i31, 0);
            try testArgs(u31, 0);
            try testArgs(u31, 1 << 0);
            try testArgs(u31, 1 << 1);
            try testArgs(u31, 1 << 29);
            try testArgs(u31, 1 << 30);

            try testArgs(i32, -1 << 31);
            try testArgs(i32, -1);
            try testArgs(i32, 0);
            try testArgs(u32, 0);
            try testArgs(u32, 1 << 0);
            try testArgs(u32, 1 << 1);
            try testArgs(u32, 1 << 30);
            try testArgs(u32, 1 << 31);

            try testArgs(i33, -1 << 32);
            try testArgs(i33, -1);
            try testArgs(i33, 0);
            try testArgs(u33, 0);
            try testArgs(u33, 1 << 0);
            try testArgs(u33, 1 << 1);
            try testArgs(u33, 1 << 31);
            try testArgs(u33, 1 << 32);

            try testArgs(i63, -1 << 62);
            try testArgs(i63, -1);
            try testArgs(i63, 0);
            try testArgs(u63, 0);
            try testArgs(u63, 1 << 0);
            try testArgs(u63, 1 << 1);
            try testArgs(u63, 1 << 61);
            try testArgs(u63, 1 << 62);

            try testArgs(i64, -1 << 63);
            try testArgs(i64, -1);
            try testArgs(i64, 0);
            try testArgs(u64, 0);
            try testArgs(u64, 1 << 0);
            try testArgs(u64, 1 << 1);
            try testArgs(u64, 1 << 62);
            try testArgs(u64, 1 << 63);

            try testArgs(i65, -1 << 64);
            try testArgs(i65, -1);
            try testArgs(i65, 0);
            try testArgs(u65, 0);
            try testArgs(u65, 1 << 0);
            try testArgs(u65, 1 << 1);
            try testArgs(u65, 1 << 63);
            try testArgs(u65, 1 << 64);

            try testArgs(i95, -1 << 94);
            try testArgs(i95, -1);
            try testArgs(i95, 0);
            try testArgs(u95, 0);
            try testArgs(u95, 1 << 0);
            try testArgs(u95, 1 << 1);
            try testArgs(u95, 1 << 93);
            try testArgs(u95, 1 << 94);

            try testArgs(i96, -1 << 95);
            try testArgs(i96, -1);
            try testArgs(i96, 0);
            try testArgs(u96, 0);
            try testArgs(u96, 1 << 0);
            try testArgs(u96, 1 << 1);
            try testArgs(u96, 1 << 94);
            try testArgs(u96, 1 << 95);

            try testArgs(i97, -1 << 96);
            try testArgs(i97, -1);
            try testArgs(i97, 0);
            try testArgs(u97, 0);
            try testArgs(u97, 1 << 0);
            try testArgs(u97, 1 << 1);
            try testArgs(u97, 1 << 95);
            try testArgs(u97, 1 << 96);

            try testArgs(i127, -1 << 126);
            try testArgs(i127, -1);
            try testArgs(i127, 0);
            try testArgs(u127, 0);
            try testArgs(u127, 1 << 0);
            try testArgs(u127, 1 << 1);
            try testArgs(u127, 1 << 125);
            try testArgs(u127, 1 << 126);

            try testArgs(i128, -1 << 127);
            try testArgs(i128, -1);
            try testArgs(i128, 0);
            try testArgs(u128, 0);
            try testArgs(u128, 1 << 0);
            try testArgs(u128, 1 << 1);
            try testArgs(u128, 1 << 126);
            try testArgs(u128, 1 << 127);

            try testArgs(i129, -1 << 128);
            try testArgs(i129, -1);
            try testArgs(i129, 0);
            try testArgs(u129, 0);
            try testArgs(u129, 1 << 0);
            try testArgs(u129, 1 << 1);
            try testArgs(u129, 1 << 127);
            try testArgs(u129, 1 << 128);

            try testArgs(i159, -1 << 158);
            try testArgs(i159, -1);
            try testArgs(i159, 0);
            try testArgs(u159, 0);
            try testArgs(u159, 1 << 0);
            try testArgs(u159, 1 << 1);
            try testArgs(u159, 1 << 157);
            try testArgs(u159, 1 << 158);

            try testArgs(i160, -1 << 159);
            try testArgs(i160, -1);
            try testArgs(i160, 0);
            try testArgs(u160, 0);
            try testArgs(u160, 1 << 0);
            try testArgs(u160, 1 << 1);
            try testArgs(u160, 1 << 158);
            try testArgs(u160, 1 << 159);

            try testArgs(i161, -1 << 160);
            try testArgs(i161, -1);
            try testArgs(i161, 0);
            try testArgs(u161, 0);
            try testArgs(u161, 1 << 0);
            try testArgs(u161, 1 << 1);
            try testArgs(u161, 1 << 159);
            try testArgs(u161, 1 << 160);

            try testArgs(i191, -1 << 190);
            try testArgs(i191, -1);
            try testArgs(i191, 0);
            try testArgs(u191, 0);
            try testArgs(u191, 1 << 0);
            try testArgs(u191, 1 << 1);
            try testArgs(u191, 1 << 189);
            try testArgs(u191, 1 << 190);

            try testArgs(i192, -1 << 191);
            try testArgs(i192, -1);
            try testArgs(i192, 0);
            try testArgs(u192, 0);
            try testArgs(u192, 1 << 0);
            try testArgs(u192, 1 << 1);
            try testArgs(u192, 1 << 190);
            try testArgs(u192, 1 << 191);

            try testArgs(i193, -1 << 192);
            try testArgs(i193, -1);
            try testArgs(i193, 0);
            try testArgs(u193, 0);
            try testArgs(u193, 1 << 0);
            try testArgs(u193, 1 << 1);
            try testArgs(u193, 1 << 191);
            try testArgs(u193, 1 << 192);

            try testArgs(i223, -1 << 222);
            try testArgs(i223, -1);
            try testArgs(i223, 0);
            try testArgs(u223, 0);
            try testArgs(u223, 1 << 0);
            try testArgs(u223, 1 << 1);
            try testArgs(u223, 1 << 221);
            try testArgs(u223, 1 << 222);

            try testArgs(i224, -1 << 223);
            try testArgs(i224, -1);
            try testArgs(i224, 0);
            try testArgs(u224, 0);
            try testArgs(u224, 1 << 0);
            try testArgs(u224, 1 << 1);
            try testArgs(u224, 1 << 222);
            try testArgs(u224, 1 << 223);

            try testArgs(i225, -1 << 224);
            try testArgs(i225, -1);
            try testArgs(i225, 0);
            try testArgs(u225, 0);
            try testArgs(u225, 1 << 0);
            try testArgs(u225, 1 << 1);
            try testArgs(u225, 1 << 223);
            try testArgs(u225, 1 << 224);

            try testArgs(i255, -1 << 254);
            try testArgs(i255, -1);
            try testArgs(i255, 0);
            try testArgs(u255, 0);
            try testArgs(u255, 1 << 0);
            try testArgs(u255, 1 << 1);
            try testArgs(u255, 1 << 253);
            try testArgs(u255, 1 << 254);

            try testArgs(i256, -1 << 255);
            try testArgs(i256, -1);
            try testArgs(i256, 0);
            try testArgs(u256, 0);
            try testArgs(u256, 1 << 0);
            try testArgs(u256, 1 << 1);
            try testArgs(u256, 1 << 254);
            try testArgs(u256, 1 << 255);

            try testArgs(i257, -1 << 256);
            try testArgs(i257, -1);
            try testArgs(i257, 0);
            try testArgs(u257, 0);
            try testArgs(u257, 1 << 0);
            try testArgs(u257, 1 << 1);
            try testArgs(u257, 1 << 255);
            try testArgs(u257, 1 << 256);

            try testArgs(i383, -1 << 382);
            try testArgs(i383, -1);
            try testArgs(i383, 0);
            try testArgs(u383, 0);
            try testArgs(u383, 1 << 0);
            try testArgs(u383, 1 << 1);
            try testArgs(u383, 1 << 381);
            try testArgs(u383, 1 << 382);

            try testArgs(i384, -1 << 383);
            try testArgs(i384, -1);
            try testArgs(i384, 0);
            try testArgs(u384, 0);
            try testArgs(u384, 1 << 0);
            try testArgs(u384, 1 << 1);
            try testArgs(u384, 1 << 382);
            try testArgs(u384, 1 << 383);

            try testArgs(i385, -1 << 384);
            try testArgs(i385, -1);
            try testArgs(i385, 0);
            try testArgs(u385, 0);
            try testArgs(u385, 1 << 0);
            try testArgs(u385, 1 << 1);
            try testArgs(u385, 1 << 383);
            try testArgs(u385, 1 << 384);

            try testArgs(i511, -1 << 510);
            try testArgs(i511, -1);
            try testArgs(i511, 0);
            try testArgs(u511, 0);
            try testArgs(u511, 1 << 0);
            try testArgs(u511, 1 << 1);
            try testArgs(u511, 1 << 509);
            try testArgs(u511, 1 << 510);

            try testArgs(i512, -1 << 511);
            try testArgs(i512, -1);
            try testArgs(i512, 0);
            try testArgs(u512, 0);
            try testArgs(u512, 1 << 0);
            try testArgs(u512, 1 << 1);
            try testArgs(u512, 1 << 510);
            try testArgs(u512, 1 << 511);

            try testArgs(i513, -1 << 512);
            try testArgs(i513, -1);
            try testArgs(i513, 0);
            try testArgs(u513, 0);
            try testArgs(u513, 1 << 0);
            try testArgs(u513, 1 << 1);
            try testArgs(u513, 1 << 511);
            try testArgs(u513, 1 << 512);

            try testArgs(i1023, -1 << 1022);
            try testArgs(i1023, -1);
            try testArgs(i1023, 0);
            try testArgs(u1023, 0);
            try testArgs(u1023, 1 << 0);
            try testArgs(u1023, 1 << 1);
            try testArgs(u1023, 1 << 1021);
            try testArgs(u1023, 1 << 1022);

            try testArgs(i1024, -1 << 1023);
            try testArgs(i1024, -1);
            try testArgs(i1024, 0);
            try testArgs(u1024, 0);
            try testArgs(u1024, 1 << 0);
            try testArgs(u1024, 1 << 1);
            try testArgs(u1024, 1 << 1022);
            try testArgs(u1024, 1 << 1023);

            try testArgs(i1025, -1 << 1024);
            try testArgs(i1025, -1);
            try testArgs(i1025, 0);
            try testArgs(u1025, 0);
            try testArgs(u1025, 1 << 0);
            try testArgs(u1025, 1 << 1);
            try testArgs(u1025, 1 << 1023);
            try testArgs(u1025, 1 << 1024);
        }
        fn testFloatTypes() !void {
            try testArgs(f16, undefined);
            try testArgs(f32, undefined);
            try testArgs(f64, undefined);
            try testArgs(f80, undefined);
            try testArgs(f128, undefined);
        }
        fn testFloats() !void {
            try testArgs(f16, -nan(f16));
            try testArgs(f16, -inf(f16));
            try testArgs(f16, -fmax(f16));
            try testArgs(f16, -1e1);
            try testArgs(f16, -1e0);
            try testArgs(f16, -1e-1);
            try testArgs(f16, -fmin(f16));
            try testArgs(f16, -tmin(f16));
            try testArgs(f16, -0.0);
            try testArgs(f16, 0.0);
            try testArgs(f16, tmin(f16));
            try testArgs(f16, fmin(f16));
            try testArgs(f16, 1e-1);
            try testArgs(f16, 1e0);
            try testArgs(f16, 1e1);
            try testArgs(f16, fmax(f16));
            try testArgs(f16, inf(f16));
            try testArgs(f16, nan(f16));

            try testArgs(f32, -nan(f32));
            try testArgs(f32, -inf(f32));
            try testArgs(f32, -fmax(f32));
            try testArgs(f32, -1e1);
            try testArgs(f32, -1e0);
            try testArgs(f32, -1e-1);
            try testArgs(f32, -fmin(f32));
            try testArgs(f32, -tmin(f32));
            try testArgs(f32, -0.0);
            try testArgs(f32, 0.0);
            try testArgs(f32, tmin(f32));
            try testArgs(f32, fmin(f32));
            try testArgs(f32, 1e-1);
            try testArgs(f32, 1e0);
            try testArgs(f32, 1e1);
            try testArgs(f32, fmax(f32));
            try testArgs(f32, inf(f32));
            try testArgs(f32, nan(f32));

            try testArgs(f64, -nan(f64));
            try testArgs(f64, -inf(f64));
            try testArgs(f64, -fmax(f64));
            try testArgs(f64, -1e1);
            try testArgs(f64, -1e0);
            try testArgs(f64, -1e-1);
            try testArgs(f64, -fmin(f64));
            try testArgs(f64, -tmin(f64));
            try testArgs(f64, -0.0);
            try testArgs(f64, 0.0);
            try testArgs(f64, tmin(f64));
            try testArgs(f64, fmin(f64));
            try testArgs(f64, 1e-1);
            try testArgs(f64, 1e0);
            try testArgs(f64, 1e1);
            try testArgs(f64, fmax(f64));
            try testArgs(f64, inf(f64));
            try testArgs(f64, nan(f64));

            try testArgs(f80, -nan(f80));
            try testArgs(f80, -inf(f80));
            try testArgs(f80, -fmax(f80));
            try testArgs(f80, -1e1);
            try testArgs(f80, -1e0);
            try testArgs(f80, -1e-1);
            try testArgs(f80, -fmin(f80));
            try testArgs(f80, -tmin(f80));
            try testArgs(f80, -0.0);
            try testArgs(f80, 0.0);
            try testArgs(f80, tmin(f80));
            try testArgs(f80, fmin(f80));
            try testArgs(f80, 1e-1);
            try testArgs(f80, 1e0);
            try testArgs(f80, 1e1);
            try testArgs(f80, fmax(f80));
            try testArgs(f80, inf(f80));
            try testArgs(f80, nan(f80));

            try testArgs(f128, -nan(f128));
            try testArgs(f128, -inf(f128));
            try testArgs(f128, -fmax(f128));
            try testArgs(f128, -1e1);
            try testArgs(f128, -1e0);
            try testArgs(f128, -1e-1);
            try testArgs(f128, -fmin(f128));
            try testArgs(f128, -tmin(f128));
            try testArgs(f128, -0.0);
            try testArgs(f128, 0.0);
            try testArgs(f128, tmin(f128));
            try testArgs(f128, fmin(f128));
            try testArgs(f128, 1e-1);
            try testArgs(f128, 1e0);
            try testArgs(f128, 1e1);
            try testArgs(f128, fmax(f128));
            try testArgs(f128, inf(f128));
            try testArgs(f128, nan(f128));
        }
        fn testIntVectorTypes() !void {
            try testArgs(@Vector(3, i1), undefined);
            try testArgs(@Vector(3, u1), undefined);
            try testArgs(@Vector(3, i2), undefined);
            try testArgs(@Vector(3, u2), undefined);
            try testArgs(@Vector(3, i3), undefined);
            try testArgs(@Vector(3, u3), undefined);
            try testArgs(@Vector(3, i4), undefined);
            try testArgs(@Vector(1, i4), undefined);
            try testArgs(@Vector(2, i4), undefined);
            try testArgs(@Vector(4, i4), undefined);
            try testArgs(@Vector(8, i4), undefined);
            try testArgs(@Vector(16, i4), undefined);
            try testArgs(@Vector(32, i4), undefined);
            try testArgs(@Vector(64, i4), undefined);
            try testArgs(@Vector(128, i4), undefined);
            try testArgs(@Vector(256, i4), undefined);
            try testArgs(@Vector(3, u4), undefined);
            try testArgs(@Vector(1, u4), undefined);
            try testArgs(@Vector(2, u4), undefined);
            try testArgs(@Vector(4, u4), undefined);
            try testArgs(@Vector(8, u4), undefined);
            try testArgs(@Vector(16, u4), undefined);
            try testArgs(@Vector(32, u4), undefined);
            try testArgs(@Vector(64, u4), undefined);
            try testArgs(@Vector(128, u4), undefined);
            try testArgs(@Vector(256, u4), undefined);
            try testArgs(@Vector(3, i5), undefined);
            try testArgs(@Vector(3, u5), undefined);
            try testArgs(@Vector(3, i7), undefined);
            try testArgs(@Vector(3, u7), undefined);
            try testArgs(@Vector(3, i8), undefined);
            try testArgs(@Vector(1, i8), undefined);
            try testArgs(@Vector(2, i8), undefined);
            try testArgs(@Vector(4, i8), undefined);
            try testArgs(@Vector(8, i8), undefined);
            try testArgs(@Vector(16, i8), undefined);
            try testArgs(@Vector(32, i8), undefined);
            try testArgs(@Vector(64, i8), undefined);
            try testArgs(@Vector(128, i8), undefined);
            try testArgs(@Vector(3, u8), undefined);
            try testArgs(@Vector(1, u8), undefined);
            try testArgs(@Vector(2, u8), undefined);
            try testArgs(@Vector(4, u8), undefined);
            try testArgs(@Vector(8, u8), undefined);
            try testArgs(@Vector(16, u8), undefined);
            try testArgs(@Vector(32, u8), undefined);
            try testArgs(@Vector(64, u8), undefined);
            try testArgs(@Vector(128, u8), undefined);
            try testArgs(@Vector(3, i9), undefined);
            try testArgs(@Vector(3, u9), undefined);
            try testArgs(@Vector(3, i15), undefined);
            try testArgs(@Vector(3, u15), undefined);
            try testArgs(@Vector(3, i16), undefined);
            try testArgs(@Vector(1, i16), undefined);
            try testArgs(@Vector(2, i16), undefined);
            try testArgs(@Vector(4, i16), undefined);
            try testArgs(@Vector(8, i16), undefined);
            try testArgs(@Vector(16, i16), undefined);
            try testArgs(@Vector(32, i16), undefined);
            try testArgs(@Vector(64, i16), undefined);
            try testArgs(@Vector(3, u16), undefined);
            try testArgs(@Vector(1, u16), undefined);
            try testArgs(@Vector(2, u16), undefined);
            try testArgs(@Vector(4, u16), undefined);
            try testArgs(@Vector(8, u16), undefined);
            try testArgs(@Vector(16, u16), undefined);
            try testArgs(@Vector(32, u16), undefined);
            try testArgs(@Vector(64, u16), undefined);
            try testArgs(@Vector(3, i17), undefined);
            try testArgs(@Vector(3, u17), undefined);
            try testArgs(@Vector(3, i31), undefined);
            try testArgs(@Vector(3, u31), undefined);
            try testArgs(@Vector(3, i32), undefined);
            try testArgs(@Vector(1, i32), undefined);
            try testArgs(@Vector(2, i32), undefined);
            try testArgs(@Vector(4, i32), undefined);
            try testArgs(@Vector(8, i32), undefined);
            try testArgs(@Vector(16, i32), undefined);
            try testArgs(@Vector(32, i32), undefined);
            try testArgs(@Vector(3, u32), undefined);
            try testArgs(@Vector(1, u32), undefined);
            try testArgs(@Vector(2, u32), undefined);
            try testArgs(@Vector(4, u32), undefined);
            try testArgs(@Vector(8, u32), undefined);
            try testArgs(@Vector(16, u32), undefined);
            try testArgs(@Vector(32, u32), undefined);
            try testArgs(@Vector(3, i33), undefined);
            try testArgs(@Vector(3, u33), undefined);
            try testArgs(@Vector(3, i63), undefined);
            try testArgs(@Vector(3, u63), undefined);
            try testArgs(@Vector(3, i64), undefined);
            try testArgs(@Vector(1, i64), undefined);
            try testArgs(@Vector(2, i64), undefined);
            try testArgs(@Vector(4, i64), undefined);
            try testArgs(@Vector(8, i64), undefined);
            try testArgs(@Vector(16, i64), undefined);
            try testArgs(@Vector(3, u64), undefined);
            try testArgs(@Vector(1, u64), undefined);
            try testArgs(@Vector(2, u64), undefined);
            try testArgs(@Vector(4, u64), undefined);
            try testArgs(@Vector(8, u64), undefined);
            try testArgs(@Vector(16, u64), undefined);
            try testArgs(@Vector(3, i65), undefined);
            try testArgs(@Vector(3, u65), undefined);
            try testArgs(@Vector(3, i127), undefined);
            try testArgs(@Vector(3, u127), undefined);
            try testArgs(@Vector(3, i128), undefined);
            try testArgs(@Vector(1, i128), undefined);
            try testArgs(@Vector(2, i128), undefined);
            try testArgs(@Vector(4, i128), undefined);
            try testArgs(@Vector(8, i128), undefined);
            try testArgs(@Vector(3, u128), undefined);
            try testArgs(@Vector(1, u128), undefined);
            try testArgs(@Vector(2, u128), undefined);
            try testArgs(@Vector(4, u128), undefined);
            try testArgs(@Vector(8, u128), undefined);
            try testArgs(@Vector(3, i129), undefined);
            try testArgs(@Vector(3, u129), undefined);
            try testArgs(@Vector(3, i191), undefined);
            try testArgs(@Vector(3, u191), undefined);
            try testArgs(@Vector(3, i192), undefined);
            try testArgs(@Vector(1, i192), undefined);
            try testArgs(@Vector(2, i192), undefined);
            try testArgs(@Vector(4, i192), undefined);
            try testArgs(@Vector(3, u192), undefined);
            try testArgs(@Vector(1, u192), undefined);
            try testArgs(@Vector(2, u192), undefined);
            try testArgs(@Vector(4, u192), undefined);
            try testArgs(@Vector(3, i193), undefined);
            try testArgs(@Vector(3, u193), undefined);
            try testArgs(@Vector(3, i255), undefined);
            try testArgs(@Vector(3, u255), undefined);
            try testArgs(@Vector(3, i256), undefined);
            try testArgs(@Vector(1, i256), undefined);
            try testArgs(@Vector(2, i256), undefined);
            try testArgs(@Vector(4, i256), undefined);
            try testArgs(@Vector(3, u256), undefined);
            try testArgs(@Vector(1, u256), undefined);
            try testArgs(@Vector(2, u256), undefined);
            try testArgs(@Vector(4, u256), undefined);
            try testArgs(@Vector(3, i257), undefined);
            try testArgs(@Vector(3, u257), undefined);
            try testArgs(@Vector(3, i511), undefined);
            try testArgs(@Vector(3, u511), undefined);
            try testArgs(@Vector(3, i512), undefined);
            try testArgs(@Vector(1, i512), undefined);
            try testArgs(@Vector(2, i512), undefined);
            try testArgs(@Vector(3, u512), undefined);
            try testArgs(@Vector(1, u512), undefined);
            try testArgs(@Vector(2, u512), undefined);
            try testArgs(@Vector(3, i513), undefined);
            try testArgs(@Vector(3, u513), undefined);
            try testArgs(@Vector(3, i1023), undefined);
            try testArgs(@Vector(3, u1023), undefined);
            try testArgs(@Vector(3, i1024), undefined);
            try testArgs(@Vector(1, i1024), undefined);
            try testArgs(@Vector(3, u1024), undefined);
            try testArgs(@Vector(1, u1024), undefined);
            try testArgs(@Vector(3, i1025), undefined);
            try testArgs(@Vector(3, u1025), undefined);
        }
        fn testIntVectors() !void {
            try testArgs(@Vector(3, i1), .{ -1 << 0, -1, 0 });
            try testArgs(@Vector(3, u1), .{ 0, 1, 1 << 0 });

            try testArgs(@Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u2), .{ 0, 1, 1 << 1 });

            try testArgs(@Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u3), .{ 0, 1, 1 << 2 });

            try testArgs(@Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(1, i4), .{
                -0x2,
            });
            try testArgs(@Vector(2, i4), .{
                -0x7, 0x4,
            });
            try testArgs(@Vector(4, i4), .{
                -0x3, 0x4, 0x2, -0x2,
            });
            try testArgs(@Vector(8, i4), .{
                -0x6, 0x3, 0x4, 0x3, 0x4, -0x8, -0x3, -0x5,
            });
            try testArgs(@Vector(16, i4), .{
                -0x3, 0x5, 0x4, -0x1, 0x2, 0x7, 0x1, 0x0, -0x2, 0x6, -0x1, -0x3, 0x5, -0x3, 0x3, -0x7,
            });
            try testArgs(@Vector(32, i4), .{
                -0x4, -0x2, 0x6, 0x6, -0x5, -0x8, -0x8, 0x7, -0x5, -0x5, 0x4, 0x5, -0x6, -0x1, 0x2, 0x0, -0x1, 0x3, 0x5, 0x1, -0x4, 0x2, -0x8, -0x6, -0x1, 0x3, 0x1, -0x8, 0x5, -0x6, 0x0, 0x2,
            });
            try testArgs(@Vector(64, i4), .{
                -0x2, 0x6,  -0x5, 0x2,  0x6, -0x5, 0x1,  -0x6, -0x6, 0x3, -0x5, 0x5, 0x0,  0x3, -0x6, -0x2, 0x0, -0x5, -0x2, -0x7, 0x6,  0x6, -0x6, 0x5, -0x1, 0x1, -0x5, 0x4,  -0x1, 0x2,  0x5,  0x0,
                0x6,  -0x1, -0x3, -0x1, 0x0, 0x0,  -0x2, -0x5, 0x7,  0x4, -0x7, 0x4, -0x8, 0x2, -0x1, -0x5, 0x4, -0x6, -0x3, 0x6,  -0x6, 0x5, 0x0,  0x6, -0x3, 0x3, -0x4, -0x4, 0x3,  -0x6, -0x5, -0x3,
            });
            try testArgs(@Vector(128, i4), .{
                -0x2, 0x7,  -0x7, 0x5,  0x4,  -0x8, -0x4, 0x2,  -0x6, 0x6,  0x3,  0x4,  -0x6, -0x3, 0x1,  -0x3, 0x4,  -0x4, 0x0, -0x5, 0x4,  -0x2, 0x4,  -0x6, 0x4,  0x7,  -0x6, 0x3,  -0x6, 0x5,  0x7,  -0x7,
                -0x8, 0x0,  0x2,  -0x6, -0x4, 0x5,  -0x2, -0x6, 0x2,  -0x3, -0x8, -0x3, -0x1, 0x4,  0x7,  -0x2, 0x7,  -0x3, 0x5, 0x3,  -0x6, 0x5,  -0x2, -0x5, -0x1, 0x5,  -0x6, -0x2, -0x5, -0x4, -0x7, -0x3,
                -0x4, -0x4, 0x6,  -0x8, -0x2, 0x3,  0x1,  0x7,  0x1,  -0x2, -0x7, -0x2, -0x8, -0x6, -0x6, 0x0,  -0x3, -0x4, 0x3, -0x5, -0x3, -0x5, 0x6,  0x5,  -0x7, -0x8, -0x5, -0x6, -0x2, -0x5, 0x5,  -0x5,
                0x0,  -0x6, -0x3, 0x0,  0x7,  0x6,  -0x6, -0x7, -0x4, -0x5, 0x3,  0x2,  0x7,  -0x3, -0x2, 0x4,  -0x4, -0x5, 0x6, 0x1,  0x7,  -0x5, -0x6, 0x0,  0x0,  -0x8, 0x4,  -0x1, -0x7, 0x0,  0x0,  0x5,
            });
            try testArgs(@Vector(256, i4), .{
                -0x7, 0x4,  0x7,  -0x5, 0x6,  -0x2, 0x6,  -0x5, 0x5,  0x5,  0x3,  -0x3, -0x5, 0x0,  0x5,  0x1,  0x4,  -0x1, 0x4,  -0x8, -0x4, -0x8, 0x2,  -0x8, 0x3,  0x1,  -0x7, -0x3, -0x1, 0x5,  -0x5, -0x8,
                -0x3, -0x3, -0x5, 0x6,  0x0,  0x4,  -0x3, -0x5, 0x0,  0x5,  -0x1, -0x3, -0x4, -0x3, 0x6,  -0x3, -0x1, 0x5,  -0x3, -0x3, 0x0,  0x3,  -0x2, -0x1, -0x5, 0x3,  0x2,  -0x8, 0x7,  -0x8, 0x6,  0x4,
                -0x5, -0x4, 0x5,  0x5,  0x6,  -0x3, 0x2,  -0x4, 0x3,  0x7,  0x6,  -0x2, -0x8, -0x1, -0x8, 0x2,  0x4,  0x1,  0x2,  -0x1, 0x5,  0x1,  0x3,  0x1,  0x3,  -0x5, 0x3,  -0x5, -0x5, 0x5,  -0x6, -0x7,
                0x0,  0x0,  -0x3, 0x6,  0x0,  0x5,  0x3,  0x0,  0x0,  -0x1, -0x6, -0x4, 0x5,  -0x8, -0x4, -0x3, -0x3, 0x2,  -0x5, -0x4, 0x4,  0x5,  -0x6, -0x3, 0x2,  0x5,  -0x7, -0x6, 0x3,  0x7,  -0x2, 0x6,
                0x2,  0x3,  0x7,  0x3,  0x2,  -0x5, 0x4,  0x5,  -0x4, -0x7, 0x2,  0x2,  -0x5, 0x7,  -0x3, -0x8, 0x2,  -0x4, 0x2,  0x4,  0x5,  -0x7, 0x7,  -0x6, 0x4,  -0x8, -0x1, 0x7,  0x0,  -0x4, 0x6,  -0x8,
                -0x5, 0x4,  -0x5, 0x1,  0x6,  -0x8, -0x1, -0x3, -0x5, 0x7,  0x1,  0x0,  -0x3, 0x4,  -0x5, -0x7, -0x5, 0x2,  0x0,  -0x1, -0x4, 0x0,  0x5,  0x6,  -0x3, -0x4, -0x2, 0x4,  -0x1, -0x8, 0x0,  0x6,
                0x7,  0x1,  0x5,  0x2,  -0x4, -0x7, -0x3, -0x3, -0x8, -0x8, -0x3, -0x4, 0x5,  -0x5, -0x2, -0x2, 0x1,  0x1,  0x1,  -0x8, 0x5,  0x4,  0x5,  0x6,  0x3,  0x0,  -0x2, -0x1, 0x4,  -0x4, -0x5, 0x0,
                -0x7, -0x8, -0x2, 0x1,  0x5,  0x4,  0x5,  -0x7, 0x3,  0x2,  0x2,  0x5,  -0x3, 0x7,  -0x4, 0x0,  -0x3, -0x2, -0x5, 0x1,  0x1,  -0x4, -0x4, 0x1,  -0x8, -0x3, 0x6,  -0x8, -0x2, 0x5,  0x7,  -0x3,
            });

            try testArgs(@Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(1, u4), .{
                0xb,
            });
            try testArgs(@Vector(2, u4), .{
                0x3, 0x4,
            });
            try testArgs(@Vector(4, u4), .{
                0x9, 0x2, 0xf, 0xe,
            });
            try testArgs(@Vector(8, u4), .{
                0x8, 0x1, 0xb, 0x1, 0xf, 0x5, 0x9, 0x6,
            });
            try testArgs(@Vector(16, u4), .{
                0xb, 0x6, 0x0, 0x7, 0x8, 0x5, 0x6, 0x9, 0xe, 0xb, 0x3, 0xa, 0xb, 0x5, 0x8, 0xc,
            });
            try testArgs(@Vector(32, u4), .{
                0xe, 0x6, 0xe, 0xa, 0xb, 0x4, 0xa, 0xb, 0x1, 0x3, 0xb, 0xc, 0x0, 0xb, 0x9, 0x4, 0xd, 0xa, 0xd, 0xd, 0x4, 0x8, 0x8, 0x6, 0xb, 0xe, 0x9, 0x6, 0xc, 0xd, 0x5, 0xd,
            });
            try testArgs(@Vector(64, u4), .{
                0x1, 0xc, 0xe, 0x9, 0x9, 0xf, 0x3, 0xf, 0x9, 0x9, 0x5, 0x3, 0xb, 0xd, 0xd, 0xf, 0x1, 0x2, 0xf, 0x9, 0x4, 0x4, 0x8, 0x9, 0x2, 0x9, 0x8, 0xe, 0x8, 0xa, 0x4, 0x3,
                0x4, 0xc, 0xb, 0x6, 0x4, 0x0, 0xa, 0x5, 0x1, 0xa, 0x4, 0xe, 0xa, 0x7, 0xd, 0x0, 0x4, 0xe, 0xe, 0x7, 0x7, 0xa, 0x4, 0x5, 0x6, 0xc, 0x6, 0x2, 0x6, 0xa, 0xe, 0xa,
            });
            try testArgs(@Vector(128, u4), .{
                0xd, 0x5, 0x6, 0xe, 0x3, 0x3, 0x3, 0xe, 0xd, 0xd, 0x9, 0x0, 0x0, 0xe, 0xa, 0x9, 0x8, 0x7, 0xb, 0x5, 0x7, 0xf, 0xb, 0x8, 0x0, 0xf, 0xb, 0x3, 0xa, 0x2, 0xb, 0xc,
                0x1, 0x1, 0xc, 0x8, 0x8, 0x6, 0x9, 0x1, 0xb, 0x0, 0x2, 0xb, 0x2, 0x2, 0x7, 0x6, 0x1, 0x1, 0xb, 0x4, 0x6, 0x4, 0x7, 0xc, 0xd, 0xc, 0xa, 0x8, 0x1, 0x7, 0x8, 0xa,
                0x9, 0xa, 0x1, 0x8, 0x1, 0x7, 0x9, 0x4, 0x5, 0x9, 0xd, 0x0, 0xa, 0xf, 0x3, 0x3, 0x9, 0x2, 0xf, 0x5, 0xb, 0x8, 0x6, 0xb, 0xf, 0x5, 0x8, 0x3, 0x9, 0xf, 0x6, 0x8,
                0xc, 0x8, 0x3, 0x4, 0xa, 0xe, 0xc, 0x1, 0xe, 0x9, 0x1, 0x8, 0xf, 0x6, 0xc, 0xc, 0x6, 0xf, 0x6, 0xd, 0xb, 0x9, 0xc, 0x3, 0xd, 0xa, 0x6, 0x8, 0x4, 0xa, 0x6, 0x9,
            });
            try testArgs(@Vector(256, u4), .{
                0x6, 0xc, 0xe, 0x3, 0x8, 0x2, 0xb, 0xd, 0x3, 0xa, 0x3, 0x8, 0xb, 0x8, 0x3, 0x0, 0xb, 0x5, 0x1, 0x3, 0x2, 0x2, 0xf, 0xc, 0x5, 0x1, 0x3, 0xb, 0x1, 0xc, 0x2, 0xd,
                0xa, 0x8, 0x1, 0xc, 0xb, 0xa, 0x3, 0x1, 0xe, 0x4, 0xf, 0xb, 0xd, 0x8, 0xf, 0xa, 0xc, 0xb, 0xb, 0x0, 0xa, 0xc, 0xf, 0xe, 0x8, 0xd, 0x9, 0x3, 0xa, 0xe, 0x8, 0x7,
                0x5, 0xa, 0x0, 0xe, 0x0, 0xd, 0x2, 0x2, 0x9, 0x4, 0x8, 0x9, 0x0, 0x4, 0x4, 0x8, 0xe, 0x1, 0xf, 0x1, 0x9, 0x3, 0xf, 0xc, 0xa, 0x0, 0x3, 0x2, 0x4, 0x1, 0x2, 0x3,
                0xf, 0x2, 0x7, 0xb, 0x5, 0x0, 0xd, 0x3, 0x4, 0xf, 0xa, 0x3, 0xc, 0x2, 0x5, 0xe, 0x7, 0x5, 0xd, 0x7, 0x9, 0x0, 0xd, 0x7, 0x9, 0xd, 0x5, 0x7, 0xf, 0xd, 0xb, 0x4,
                0x9, 0x6, 0xf, 0xb, 0x1, 0xb, 0x6, 0xb, 0xf, 0x7, 0xf, 0x0, 0x4, 0x7, 0x5, 0xa, 0x8, 0x1, 0xf, 0x9, 0x9, 0x0, 0x6, 0xb, 0x1, 0x2, 0x4, 0x3, 0x2, 0x0, 0x7, 0x0,
                0x6, 0x7, 0xf, 0x1, 0xe, 0xa, 0x8, 0x2, 0x9, 0xc, 0x1, 0x5, 0x7, 0x1, 0xb, 0x0, 0x1, 0x3, 0xd, 0x3, 0x0, 0x1, 0xa, 0x0, 0x3, 0x7, 0x1, 0x2, 0xb, 0xc, 0x2, 0x9,
                0x8, 0x8, 0x7, 0x0, 0xd, 0x5, 0x1, 0x5, 0x7, 0x7, 0x2, 0x3, 0x8, 0x7, 0xc, 0x8, 0xf, 0xa, 0xf, 0xf, 0x3, 0x2, 0x0, 0x4, 0x7, 0x5, 0x6, 0xd, 0x6, 0x3, 0xa, 0x4,
                0x1, 0x1, 0x2, 0xc, 0x3, 0xe, 0x2, 0xc, 0x7, 0x6, 0xe, 0xf, 0xb, 0x8, 0x6, 0x6, 0x9, 0x0, 0x4, 0xb, 0xe, 0x4, 0x2, 0x7, 0xf, 0xc, 0x0, 0x6, 0xd, 0xa, 0xe, 0xc,
            });

            try testArgs(@Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u5), .{ 0, 1, 1 << 4 });

            try testArgs(@Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u7), .{ 0, 1, 1 << 6 });

            try testArgs(@Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(1, i8), .{
                0x71,
            });
            try testArgs(@Vector(2, i8), .{
                -0x50, -0x43,
            });
            try testArgs(@Vector(4, i8), .{
                -0x09, -0x19, -0x15, -0x5d,
            });
            try testArgs(@Vector(8, i8), .{
                -0x4f, -0x55, -0x5b, -0x23, -0x76, 0x36, 0x6f, -0x63,
            });
            try testArgs(@Vector(16, i8), .{
                0x24, -0x03, 0x2e, 0x7b, 0x68, 0x29, 0x6c, 0x7f, -0x2f, -0x3b, -0x11, -0x3c, -0x2e, 0x27, -0x45, 0x45,
            });
            try testArgs(@Vector(32, i8), .{
                0x70, 0x33, -0x28, -0x38, -0x3b, 0x44,  -0x1d, 0x7d,  -0x48, 0x3c,  0x61, -0x09, -0x49, 0x15,  0x0a, -0x5a,
                0x78, 0x11, -0x07, -0x23, 0x4a,  -0x72, 0x25,  -0x17, -0x51, -0x04, 0x55, 0x20,  -0x80, -0x3d, 0x59, -0x39,
            });
            try testArgs(@Vector(64, i8), .{
                0x4f, 0x40,  -0x62, -0x4f, 0x37, -0x06, -0x33, 0x4d,  -0x10, 0x55,  0x24,  -0x76, 0x1d,  0x2b,  -0x54, -0x0f,
                0x21, -0x4c, -0x74, -0x07, 0x23, -0x5a, -0x21, -0x4a, -0x7c, -0x16, -0x20, -0x2e, 0x0a,  0x15,  0x03,  0x44,
                0x19, -0x27, 0x3e,  0x61,  0x6e, -0x76, 0x2a,  0x74,  -0x21, 0x34,  -0x69, -0x18, -0x21, -0x61, -0x34, -0x02,
                0x5e, -0x36, -0x79, -0x0f, 0x26, 0x6e,  0x5f,  0x52,  -0x0f, -0x64, 0x1a,  0x74,  -0x37, 0x00,  -0x47, -0x57,
            });
            try testArgs(@Vector(128, i8), .{
                -0x38, -0x19, 0x51,  0x09,  0x76,  -0x3b, -0x33, 0x39,  0x67,  0x51,  0x10,  0x77,  0x24,  0x21,  0x6f,  -0x1a,
                0x4e,  -0x69, 0x2e,  -0x78, -0x06, 0x5c,  0x17,  0x2e,  -0x0e, -0x2e, 0x09,  0x2a,  -0x5f, -0x40, -0x64, 0x3f,
                0x4a,  -0x77, -0x54, 0x38,  0x6b,  0x1f,  -0x04, 0x40,  0x27,  -0x0c, 0x65,  -0x46, 0x49,  -0x69, -0x53, 0x64,
                0x13,  -0x33, 0x3a,  -0x10, -0x15, 0x7f,  -0x1c, 0x5e,  -0x22, 0x2f,  -0x75, 0x77,  0x22,  0x6b,  -0x32, -0x55,
                0x18,  0x19,  0x2c,  -0x27, -0x03, 0x4f,  0x07,  0x0b,  0x44,  -0x21, 0x79,  0x55,  -0x65, 0x1d,  -0x29, 0x2f,
                0x4a,  0x6f,  -0x40, -0x57, -0x2f, 0x42,  0x52,  0x68,  -0x2a, -0x6b, 0x6f,  -0x49, -0x32, 0x52,  0x1e,  -0x60,
                -0x80, 0x53,  0x5e,  0x73,  -0x1e, 0x2d,  -0x46, -0x27, 0x4b,  0x57,  0x1f,  0x6a,  -0x65, 0x5f,  -0x2b, -0x03,
                -0x3a, -0x76, -0x51, 0x20,  0x04,  -0x0a, 0x2b,  -0x04, -0x1e, -0x18, -0x2d, 0x53,  -0x58, -0x69, 0x16,  0x19,
            });

            try testArgs(@Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(1, u8), .{
                0x33,
            });
            try testArgs(@Vector(2, u8), .{
                0x66, 0x87,
            });
            try testArgs(@Vector(4, u8), .{
                0x9d, 0xcb, 0x30, 0x7b,
            });
            try testArgs(@Vector(8, u8), .{
                0x4b, 0x35, 0x3f, 0x5c, 0xa5, 0x91, 0x23, 0x6d,
            });
            try testArgs(@Vector(16, u8), .{
                0xb7, 0x57, 0x27, 0x29, 0x58, 0xf8, 0xc9, 0x6c, 0xbe, 0x41, 0xf4, 0xd7, 0x4d, 0x01, 0xf0, 0x37,
            });
            try testArgs(@Vector(32, u8), .{
                0x5f, 0x61, 0x34, 0xe8, 0x37, 0x12, 0xba, 0x5a, 0x85, 0xf3, 0x3e, 0xa2, 0x0f, 0xd0, 0x65, 0xae,
                0xed, 0xf5, 0xe8, 0x65, 0x61, 0x28, 0x4a, 0x27, 0x2e, 0x01, 0x40, 0x8c, 0xe3, 0x36, 0x5d, 0xb6,
            });
            try testArgs(@Vector(64, u8), .{
                0xb0, 0x19, 0x5c, 0xc2, 0x3b, 0x16, 0x70, 0xad, 0x26, 0x45, 0xf2, 0xe1, 0x4f, 0x0f, 0x01, 0x72,
                0x7f, 0x1f, 0x07, 0x9e, 0xee, 0x9b, 0xb3, 0x38, 0x50, 0xf3, 0x56, 0x73, 0xd0, 0xd1, 0xee, 0xe3,
                0xeb, 0xf3, 0x1b, 0xe0, 0x77, 0x78, 0x75, 0xc6, 0x19, 0xe4, 0x69, 0xaa, 0x73, 0x08, 0xcd, 0x0c,
                0xf9, 0xed, 0x94, 0xf8, 0x79, 0x86, 0x63, 0x31, 0xbf, 0xd1, 0xe3, 0x17, 0x2b, 0xb9, 0xa1, 0x72,
            });
            try testArgs(@Vector(128, u8), .{
                0x2e, 0x93, 0x87, 0x09, 0x4f, 0x68, 0x14, 0xab, 0x3f, 0x04, 0x86, 0xc1, 0x95, 0xe8, 0x74, 0x11,
                0x57, 0x25, 0xe1, 0x88, 0xc0, 0x96, 0x33, 0x99, 0x15, 0x86, 0x2c, 0x84, 0x2e, 0xd7, 0x57, 0x21,
                0xd3, 0x18, 0xd5, 0x0e, 0xb4, 0x60, 0xe2, 0x08, 0xce, 0xbc, 0xd5, 0x4d, 0x8f, 0x59, 0x01, 0x67,
                0x71, 0x0a, 0x74, 0x48, 0xef, 0x39, 0x49, 0x7e, 0xa8, 0x39, 0x34, 0x75, 0x95, 0x3b, 0x38, 0xea,
                0x60, 0xd7, 0xed, 0x8f, 0xbb, 0xc0, 0x7d, 0xc2, 0x79, 0x2d, 0xbf, 0xa5, 0x64, 0xf4, 0x09, 0x86,
                0xfb, 0x29, 0xfe, 0xc7, 0xff, 0x62, 0x1a, 0x6f, 0xf8, 0xbd, 0xfe, 0xa4, 0xac, 0x24, 0xcf, 0x56,
                0x82, 0x69, 0x81, 0x0d, 0xc1, 0x51, 0x8d, 0x85, 0xf4, 0x00, 0xe7, 0x25, 0xab, 0xa5, 0x33, 0x45,
                0x66, 0x2e, 0x33, 0xc8, 0xf3, 0x35, 0x16, 0x7d, 0x1f, 0xc9, 0xf7, 0x44, 0xab, 0x66, 0x28, 0x0d,
            });

            try testArgs(@Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u9), .{ 0, 1, 1 << 8 });

            try testArgs(@Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u15), .{ 0, 1, 1 << 14 });

            try testArgs(@Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(1, i16), .{
                -0x015a,
            });
            try testArgs(@Vector(2, i16), .{
                -0x1c2f, 0x5ce8,
            });
            try testArgs(@Vector(4, i16), .{
                0x1212, 0x5bfc, -0x20ea, 0x0993,
            });
            try testArgs(@Vector(8, i16), .{
                0x4d55, -0x0dfb, -0x7921, 0x7e20, 0x74a5, -0x7371, -0x08e0, 0x7f23,
            });
            try testArgs(@Vector(16, i16), .{
                0x2354, -0x048a, -0x3ef9, 0x29d4, 0x4e5e, -0x3da9, -0x0cc4, -0x0377,
                0x4d44, 0x4384,  -0x1e46, 0x0bf1, 0x3151, -0x57c6, -0x367e, -0x7ae5,
            });
            try testArgs(@Vector(32, i16), .{
                0x5b5a, -0x54c4, -0x2089, -0x448d, 0x38e8,  -0x36a5, -0x0a8f, 0x06e0,
                0x09d9, 0x3877,  0x33c8,  0x5d3a,  0x018b,  0x29c9,  0x6f59,  -0x4078,
                0x6be4, -0x249e, 0x43b3,  -0x0389, 0x545e,  0x6ed7,  0x6636,  0x587d,
                0x55b0, -0x608b, 0x72e0,  0x4dfd,  -0x051d, 0x7433,  -0x7fc2, 0x2de3,
            });
            try testArgs(@Vector(64, i16), .{
                0x7834,  -0x43f9, -0x1cb3, -0x05f2, 0x25b5,  0x55f2,  0x4cfb,  -0x58bb,
                0x7292,  -0x082e, -0x5a6e, 0x1fc8,  -0x1f49, 0x7e3c,  0x4aa5,  -0x617e,
                0x2fab,  -0x2b96, 0x7474,  -0x6644, -0x5484, -0x278e, -0x6a0e, -0x5210,
                0x1adf,  -0x2799, 0x61e0,  -0x733c, -0x6bcc, -0x6fe2, -0x4e91, 0x5d01,
                0x3745,  0x24eb,  0x6c89,  0x4a94,  -0x7339, 0x4907,  -0x4f8f, -0x7e39,
                0x1a32,  0x65ca,  -0x6c27, -0x3269, 0x107b,  0x1c53,  -0x5529, 0x5232,
                -0x26ec, 0x4442,  -0x63f5, -0x174a, 0x3033,  -0x7363, 0x58be,  0x239f,
                0x7f7b,  -0x437d, -0x6df6, 0x0a7b,  0x3faa,  -0x1d75, -0x7426, 0x1274,
            });

            try testArgs(@Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(1, u16), .{
                0x4da6,
            });
            try testArgs(@Vector(2, u16), .{
                0x04d7, 0x50c6,
            });
            try testArgs(@Vector(4, u16), .{
                0x4c06, 0xd71f, 0x4d8f, 0xe0a4,
            });
            try testArgs(@Vector(8, u16), .{
                0xee9a, 0x881d, 0x31fb, 0xd3f7, 0x2c74, 0x6949, 0x4e04, 0x53d7,
            });
            try testArgs(@Vector(16, u16), .{
                0xeafe, 0x9a7b, 0x0d6f, 0x18cb, 0xaf8f, 0x8ee4, 0xa47e, 0xd39a,
                0x6572, 0x9c53, 0xf36e, 0x982e, 0x41c1, 0x8682, 0xf5dc, 0x7e01,
            });
            try testArgs(@Vector(32, u16), .{
                0xdfb3, 0x7de6, 0xd9ed, 0xb42e, 0x95ac, 0x9b5b, 0x0422, 0xdfcd,
                0x6196, 0x4dbe, 0x1818, 0x8816, 0x75e7, 0xc9b0, 0x92f7, 0x1f71,
                0xe584, 0x576c, 0x043a, 0x0f31, 0xfc4c, 0x2c87, 0x6b02, 0x0229,
                0x25b7, 0x53cd, 0x9bab, 0x866b, 0x9008, 0xf0f3, 0xeb21, 0x88e2,
            });
            try testArgs(@Vector(64, u16), .{
                0x084c, 0x445f, 0xce89, 0xd3ee, 0xb399, 0x315d, 0x8ef8, 0x4f6f,
                0xf9af, 0xcbc4, 0x0332, 0xcd55, 0xa4dc, 0xbc38, 0x6e33, 0x8ead,
                0xd15a, 0x5057, 0x58ef, 0x657a, 0xe9f0, 0x1418, 0x2b62, 0x3387,
                0x1c15, 0x04e1, 0x0276, 0x3783, 0xad9c, 0xea9a, 0x0e5e, 0xe803,
                0x2ee7, 0x0cf1, 0x30f1, 0xb12a, 0x381b, 0x353d, 0xf637, 0xf853,
                0x2ac1, 0x7ce8, 0x6a50, 0xcbb8, 0xc9b8, 0x9b25, 0xd1e9, 0xeff0,
                0xc0a2, 0x8e51, 0xde7a, 0x4e58, 0x5685, 0xeb3f, 0xd29b, 0x66ed,
                0x3dd5, 0xcb59, 0x6003, 0xf710, 0x943a, 0x7276, 0xe547, 0xe48f,
            });

            try testArgs(@Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u17), .{ 0, 1, 1 << 16 });

            try testArgs(@Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u31), .{ 0, 1, 1 << 30 });

            try testArgs(@Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(1, i32), .{
                -0x27f49dce,
            });
            try testArgs(@Vector(2, i32), .{
                0x24641ec7, 0x436c5bd2,
            });
            try testArgs(@Vector(4, i32), .{
                0x59e5eff1, -0x46b5b8db, -0x1029efa7, -0x1937fe73,
            });
            try testArgs(@Vector(8, i32), .{
                0x0ca01401,  -0x46b2bc0c, 0x51e5dee7, -0x74edfde8,
                -0x0ab09a6a, -0x5a51a88b, 0x18c28bc2, 0x63d79966,
            });
            try testArgs(@Vector(16, i32), .{
                0x3900e6c8, 0x2408c2bb, 0x5e01bc6e,  -0x0eb8c400,
                0x4c0dc6c2, 0x6c75e7f5, -0x66632ca8, 0x0e978daf,
                0x61ffe725, 0x720253e4, -0x6f6c38c1, -0x3302e60a,
                0x43f53c92, 0x5a3c1075, 0x7044a110,  0x18e41ad8,
            });
            try testArgs(@Vector(32, i32), .{
                0x3a5c2b01,  0x2a52d9fa,  -0x5843fc47, 0x6c493c7d,
                -0x47937cb1, -0x3ad95ec4, 0x71cf5e7b,  -0x3b6719c2,
                0x06bace17,  -0x6ccda5ed, 0x42b9ed04,  0x6be2b287,
                -0x7cf56523, -0x3c98e2e4, 0x1e7db6c0,  -0x7e668ad2,
                -0x6c245ecf, -0x09842450, -0x403a4335, -0x7a68e9b7,
                0x0036cf57,  -0x251edb4e, -0x67ec3abf, -0x183f0333,
                -0x4b46723c, -0x1e5383d6, 0x188c1de3,  0x400b3648,
                -0x4b21d9d3, 0x61635257,  0x179eb187,  0x31cd8376,
            });

            try testArgs(@Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(1, u32), .{
                0x17e2805c,
            });
            try testArgs(@Vector(2, u32), .{
                0xdb6aadc5, 0xb1ff3754,
            });
            try testArgs(@Vector(4, u32), .{
                0xf7897b31, 0x342e1af9, 0x190fd76b, 0x283b5374,
            });
            try testArgs(@Vector(8, u32), .{
                0x81a0bd16, 0xc55da94e, 0x910f7e7c, 0x078d5ef7,
                0x0bdb1e4a, 0xf1a96e99, 0xcdd729b5, 0xe6966a1c,
            });
            try testArgs(@Vector(16, u32), .{
                0xfee812db, 0x29eacbed, 0xaed48136, 0x3053de13,
                0xbbda20df, 0x6faa274a, 0xe0b5ec3a, 0x1878b0dc,
                0x98204475, 0x810d8d05, 0x1e6996b6, 0xc543826a,
                0x53b47d8c, 0xc72c3142, 0x12f7e1f9, 0xf6782e54,
            });
            try testArgs(@Vector(32, u32), .{
                0xf0cf30d3, 0xe3c587b8, 0xcee44739, 0xe4a0bd72,
                0x41d44cce, 0x6d7c4259, 0xd85580a5, 0xec4b02d7,
                0xa366483d, 0x2d7b59d4, 0xe9c0ace4, 0x82cb441c,
                0xa23958ba, 0x04a70148, 0x3f0d20a3, 0xf9e21e37,
                0x009fce8b, 0x4a34a229, 0xf09c35cf, 0xc0977d4d,
                0xcc4d4647, 0xa30f1363, 0x27a65b14, 0xe572c785,
                0x8f42e320, 0x2b2cdeca, 0x11205bd4, 0x739d26aa,
                0xcbcc2df0, 0x5f7a3649, 0xbde1b7aa, 0x180a169f,
            });

            try testArgs(@Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u33), .{ 0, 1, 1 << 32 });

            try testArgs(@Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u63), .{ 0, 1, 1 << 62 });

            try testArgs(@Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(1, i64), .{
                0x29113011488d8b65,
            });
            try testArgs(@Vector(2, i64), .{
                -0x3f865dcdfd831d03, -0x35512d15095445d6,
            });
            try testArgs(@Vector(4, i64), .{
                0x6f37a9484440251e, 0x2757e5e2b77e6ef3,
                0x4903a91bd2993d0b, 0x162244ba22371f62,
            });
            try testArgs(@Vector(8, i64), .{
                -0x46e2340c765175c1, -0x031ee2297e6cc8b3,
                -0x2627434d4b4fb796, 0x525e1ef31b6daa46,
                0x72d8eaaea07fa5ea,  0x2a8c0c36da019448,
                -0x5419ebf5cd514cde, -0x618c56a881057ac4,
            });
            try testArgs(@Vector(16, i64), .{
                0x36b4a703d084c774,  0x07a500f0d603a4d5,
                -0x27387989d2450cdd, 0x02073880984d74c8,
                -0x18d1593e36724417, -0x79df283cc6f403d8,
                0x36838a7c54da5f2b,  -0x2bf76c1666a1b768,
                -0x6ace0d64a2757edc, 0x41442e9979a0ab64,
                0x002612bfdf419826,  0x1128ba5648d22fe8,
                0x49b0f67e0abb8f3b,  0x6bf3e9ac37f73cf3,
                -0x5c89f516258c7e77, 0x6b345f04e60d2e56,
            });

            try testArgs(@Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(1, u64), .{
                0x7d2e439abb0edba7,
            });
            try testArgs(@Vector(2, u64), .{
                0x3749ee5a2d237b9f, 0x6d8f4c3e1378f389,
            });
            try testArgs(@Vector(4, u64), .{
                0x03c127040e10d52b, 0xa86fe019072e27eb,
                0x0a554a47b709cdba, 0xf4342cc597e196c3,
            });
            try testArgs(@Vector(8, u64), .{
                0xea455c104375a055, 0x5c35d9d945edb2fa,
                0xc11b73d9d9d546fc, 0x2a9d63aae838dd5b,
                0xed6603f1f5d574b3, 0x2f37b354c81c1e56,
                0xbe7f5e2476bc76bd, 0xb0c88eacfffa9a8f,
            });
            try testArgs(@Vector(16, u64), .{
                0x2258fc04b31f8dbe, 0x3a2e5483003a10d8,
                0xebf24b31c0460510, 0x15d5b4c09b53ffa5,
                0x05abf6e744b17cc6, 0x9747b483f2d159fe,
                0x4616d8b2c8673125, 0x8ae3f91d422447eb,
                0x18da2f101a9e9776, 0x77a1197fb0441007,
                0x4ba480c8ec2dd10b, 0xeb99b9c0a1725278,
                0xd9d0acc5084ecdf0, 0xa0a23317fff4f515,
                0x0901c59a9a6a408b, 0x7c77ca72e25df033,
            });

            try testArgs(@Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u65), .{ 0, 1, 1 << 64 });

            try testArgs(@Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u127), .{ 0, 1, 1 << 126 });

            try testArgs(@Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(1, i128), .{
                -0x2b0b1462b44785f39d1b7d763ec7bdb2,
            });
            try testArgs(@Vector(2, i128), .{
                -0x2faebe898a6fe60fbc6aadc3623431b7,
                0x5e596259e7b2588860d2b470ba751ace,
            });
            try testArgs(@Vector(4, i128), .{
                -0x624cb7e74cf789c06121809a3a5b51ba,
                0x23af4553d4d64672795c2b949635426f,
                -0x0b598b1f94876757fb13f2198e902b13,
                0x1daf732f50654d8211d464fda4fc030c,
            });
            try testArgs(@Vector(8, i128), .{
                -0x03c7df38daee9bc9a2c659a1a124ef10,
                0x657a590c91905c4021c28b0d6e42304a,
                -0x3f5176206dadc974d10e6fcbd67f3d29,
                0x066310ace384b1bc3549c71113b96b8a,
                -0x6c0201f66583206fcea7b7fe11889644,
                -0x5cc4d2a368002b380b25415be83f8218,
                0x11156c91b97a6a93427009efebcb2c31,
                -0x4221b5249ed0686c2ff2d5cab9f1c362,
            });

            try testArgs(@Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(1, u128), .{
                0x809f29e7fbafadc01145e1732590e7d9,
            });
            try testArgs(@Vector(2, u128), .{
                0x5150ac3438aacd0d51132cc2723b2995,
                0x151be9c47ad29cf719cf8358dd40165c,
            });
            try testArgs(@Vector(4, u128), .{
                0x4bae22df929f2f7cb9bd84deaad3e7a8,
                0x1ed46b2d6e1f3569f56b2ac33d8bc1cb,
                0xae93ea459d2ccfd5fb794e6d5c31aabb,
                0xb1177136acf099f550b70949ac202ec4,
            });
            try testArgs(@Vector(8, u128), .{
                0x7cd78db6baed6bfdf8c5265136c4e0fd,
                0xa41b8984c6bbde84640068194b7eba98,
                0xd33102778f2ae1a48d1e9bf8801bbbf0,
                0x0d59f6de003513a60055c86cbce2c200,
                0x825579d90012afddfbf04851c0748561,
                0xc2647c885e9d6f0ee1f5fac5da8ef7f5,
                0xcb4bbc1f81aa8ee68aa4dc140745687b,
                0x4ff10f914f74b46c694407f5bf7c7836,
            });

            try testArgs(@Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u129), .{ 0, 1, 1 << 128 });

            try testArgs(@Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u191), .{ 0, 1, 1 << 190 });

            try testArgs(@Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(1, i192), .{
                0x0206223e53631dfaf431066cf5ac30dd203bb8c7baa0cec7,
            });
            try testArgs(@Vector(2, i192), .{
                0x187a65fa29d1981dacf927e6a8e435481cfdcba6b63b781b,
                -0x0f53cb01d7662de0d19fa0b250e5bbc6edf7d3dd152f0dc3,
            });
            try testArgs(@Vector(4, i192), .{
                -0x3a456cd0eab663b34d5b6ad15933a31623aacb913adb8e41,
                -0x03376d57e9c495ac4ea623e1bf427ae22dcef26e4833da33,
                -0x28a90cfee819450e3000f3f2694a7dba2c02311996e01073,
                0x46c6cae4281780acd6a0322c3f4f8b63c3741da31b20a3cd,
            });

            try testArgs(@Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(1, u192), .{
                0xe7baafcb9781626a77571b0539b9471a60c97d6c02106c8b,
            });
            try testArgs(@Vector(2, u192), .{
                0xbc9510913ed09e2c2aa50ffab9f1bc7b303a87f36e232a83,
                0x1f37bee446d7712d1ad457c47a66812cb926198d052aee65,
            });
            try testArgs(@Vector(4, u192), .{
                0xdca6a7cfc19c69efc34022062a8ca36f2569ab3dce001202,
                0xd25a4529e621c9084181fdb6917c6a32eccc58b63601b35d,
                0x0a258afd6debbaf8c158f1caa61fed63b31871d13f51b43d,
                0x6b40a178674fcb82c623ac322f851623d5e993dac97a219a,
            });

            try testArgs(@Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u193), .{ 0, 1, 1 << 192 });

            try testArgs(@Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u255), .{ 0, 1, 1 << 254 });

            try testArgs(@Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(1, i256), .{
                0x59a12bff854679d9b3c6d1d195333d9f748dd1e2a7ad28f24f611a208bf91ed3,
            });
            try testArgs(@Vector(2, i256), .{
                0x6b266e98bd5e7e66ba90f2e1cb2ff555ac755efdbe0946313660c58b46c589bb,
                -0x4ab426d26f53253ae3b2fb412d9649fc8071db22605e528f918b9a3ee9d2a832,
            });
            try testArgs(@Vector(4, i256), .{
                -0x3a64f67fddd0859c0f3b063fc12b13b1865447b87d1740de51358421f50553b5,
                -0x7c364fc0218f1cab29425b1a4c9cbdbf0c676375bee8079b135ce40de3557c0b,
                0x368d25dc3eab1b00decd18679b29b7f4d95314161bd3ee687f2896e8cd525311,
                -0x6d9aacd172a363bf2d53ea497c289fd35e62c2484329c208e10a91b4cea88111,
            });

            try testArgs(@Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(1, u256), .{
                0x230413bb481fa3a997796acf282010c560d1942e7339fd584a0f15a90c83fbda,
            });
            try testArgs(@Vector(2, u256), .{
                0x3ad569f8d91fdbc9da8ec0e933565919f2feb90b996c90c352b461aa0908e62d,
                0x0f109696d64647983f1f757042515510729ad1350e862cbf38cb73b5cf99f0f7,
            });
            try testArgs(@Vector(4, u256), .{
                0x1717c6ded4ac6de282d59f75f068da47d5a47a30f2c5053d2d59e715f9d28b97,
                0x3087189ce7540e2e0028b80af571ebc6353a00b2917f243a869ed29ecca0adaa,
                0x1507c6a9d104684bf503cdb08841cf91adab4644306bd67aafff5326604833ce,
                0x857e134ff9179733c871295b25f824bd3eb562977bad30890964fa0cdc15bb07,
            });

            try testArgs(@Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u257), .{ 0, 1, 1 << 256 });

            try testArgs(@Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u511), .{ 0, 1, 1 << 510 });

            try testArgs(@Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(1, i512), .{
                -0x235b5d838cdf67b9eb6d7eeb518fb63cff402a74c687927feb363b5040556b8d32c55e565cc2fe33cb4dcc37e8fd1c92989522c11b6c186d11400d17e40d35b5,
            });
            try testArgs(@Vector(2, i512), .{
                -0x5f5ff44fec38adc4c9c8bc8de00acf01fcc62bc55d07033f4e788d4f3825382e1e39f6bd69dff328eec9a89486ebaaaffd9ab69d28eb7d952be4ef250cff6de1,
                -0x403e0fd866e1598ad928ecd234005debd527483375f5e7e79eee3a129868354acb5b74e42de9f297f81062d04ea41adc158e542ab04770dd039d527cffb81845,
            });

            try testArgs(@Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(1, u512), .{
                0xa3ff51a609f1370e5eeb96b05169bf7469e465cf76ac5b4ea8ffd166c1ba3cd94f2dedf0d647a1fe424f3a06e6d7940f03e257f28100970b00bd5528c52b9ae6,
            });
            try testArgs(@Vector(2, u512), .{
                0xc6d43cd46ae31ab71f9468a895c83bf17516c6b2f1c9b04b9aa113bf7fe1b789eb7d95fcf951f12a9a6f2124589551efdd8c00f528b366a7bfb852faf8f3da53,
                0xc9099d2bdf8d1a0d30485ec6db4a24cbc0d89a863de30e18313ee1d66f71dd2d26235caaa703286cf4a2b51e1a12ef96d2d944c66c0bd3f0d72dd4cf0fc8100e,
            });

            try testArgs(@Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u513), .{ 0, 1, 1 << 512 });

            try testArgs(@Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u1023), .{ 0, 1, 1 << 1022 });

            try testArgs(@Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(1, i1024), .{
                0x10eee350e115375812126750b24255ca76fdee619b64261c354af58bd4a29af6e2448ccda4d84e1b2fbf76d3710cf1b5e62b1360c3b63e104d0755fa264d6c171f8f7a3292d7859b08a5dff60e9ad8ba9dcdd7e6098eb70be7a27a0cbcc6480661330c21299b2960fac954ee4480f3a2cc1ca5a492e1e75084c079ba701cd7ab,
            });

            try testArgs(@Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(1, u1024), .{
                0xc6cfaa6571139552e1f067402dfc131d9b9a58aafda97198a78764b05138fb68cf26f085b7652f3d5ae0e56aa21732f296a581bb411d4a73795c213de793489fa49b173b9f5c089aa6295ff1fcdc14d491a05035b45d08fc35cd67a83d887a02b8db512f07518132e0ba56533c7d6fbe958255eddf5649bd8aba288c0dd84a25,
            });

            try testArgs(@Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u1025), .{ 0, 1, 1 << 1024 });
        }
        fn testFloatVectorTypes() !void {
            try testArgs(@Vector(1, f16), undefined);
            try testArgs(@Vector(2, f16), undefined);
            try testArgs(@Vector(4, f16), undefined);
            try testArgs(@Vector(8, f16), undefined);
            try testArgs(@Vector(16, f16), undefined);
            try testArgs(@Vector(32, f16), undefined);
            try testArgs(@Vector(64, f16), undefined);

            try testArgs(@Vector(1, f32), undefined);
            try testArgs(@Vector(2, f32), undefined);
            try testArgs(@Vector(4, f32), undefined);
            try testArgs(@Vector(8, f32), undefined);
            try testArgs(@Vector(16, f32), undefined);
            try testArgs(@Vector(32, f32), undefined);

            try testArgs(@Vector(1, f64), undefined);
            try testArgs(@Vector(2, f64), undefined);
            try testArgs(@Vector(4, f64), undefined);
            try testArgs(@Vector(8, f64), undefined);
            try testArgs(@Vector(16, f64), undefined);

            try testArgs(@Vector(1, f80), undefined);
            try testArgs(@Vector(2, f80), undefined);
            try testArgs(@Vector(4, f80), undefined);
            try testArgs(@Vector(8, f80), undefined);

            try testArgs(@Vector(1, f128), undefined);
            try testArgs(@Vector(2, f128), undefined);
            try testArgs(@Vector(4, f128), undefined);
            try testArgs(@Vector(8, f128), undefined);
        }
        fn testFloatVectors() !void {
            try testArgs(@Vector(1, f16), .{
                -0x1.17cp-12,
            });
            try testArgs(@Vector(2, f16), .{
                0x1.47cp9, 0x1.3acp9,
            });
            try testArgs(@Vector(4, f16), .{
                0x1.ab4p0, -0x1.7fcp-7, -0x1.1cp0, -0x1.f14p12,
            });
            try testArgs(@Vector(8, f16), .{
                -0x1.8d8p8, 0x1.83p10, -0x1.5ap-1, -0x1.d78p13, -0x1.608p12, 0x1.e8p-9, -0x1.688p-10, -0x1.738p9,
            });
            try testArgs(@Vector(16, f16), .{
                0x1.da8p-1, -0x1.ed4p-10, -0x1.dc8p1,  0x1.b78p-14, nan(f16),    0x1.9d8p8,   nan(f16),     0x1.d5p13,
                -0x1.2dp13, 0x1.6c4p12,   0x1.a9cp-11, -0x1.0ecp8,  0x0.4ccp-14, -0x1.0a8p-6, -0x1.5bcp-14, 0x1.6d8p-9,
            });
            try testArgs(@Vector(32, f16), .{
                0x1.d5cp-6,  -0x1.a98p5,  0x1.49cp5,   -0x1.e4p-1,  -0x1.21p-13, -0x1.c94p-1, -0x1.adcp-5, -0x1.524p-1,
                -0x1.0d8p-3, -0x1.5c4p-2, 0x1.f84p-2,  0x1.664p1,   -0x1.f64p13, -0x1.bf4p4,  -0x1.4b8p0,  -0x0.f64p-14,
                -0x1.3f8p1,  0x1.098p2,   -0x1.a44p8,  0x1.048p13,  0x1.fd4p-11, 0x1.18p-9,   -0x1.504p2,  0x1.d04p7,
                -nan(f16),   0x1.a94p2,   0x0.5e8p-14, -0x1.7acp-7, 0x1.4c8p-3,  0x1.518p-4,  nan(f16),    0x1.8f8p10,
            });
            try testArgs(@Vector(64, f16), .{
                -0x1.c2p2,   0x0.2fcp-14,  0x1.de8p0,    -0x1.714p2,   0x1.f9p-7,    -0x1.11cp-13, -0x1.558p10, -0x1.2acp-7,
                0x1.348p14,  0x1.2dcp7,    -0x1.8acp-12, -0x1.2cp2,    0x1.868p1,    -0x1.1f8p-14, 0x1.638p7,   -0x1.734p-5,
                0x0.b98p-14, -0x1.7f4p-12, -0x1.38cp15,  0x1.50cp15,   0x1.91cp8,    0x1.cb4p-1,   0x1.fc4p-13, 0x1.9a4p0,
                0x1.18p-4,   0x1.60cp10,   0x1.6fp-12,   0x1.b48p6,    0x1.37cp-11,  0x1.424p7,    0x1.44cp13,  0x1.aep5,
                0x1.968p14,  0x1.e8p13,    -0x1.bp2,     -0x1.644p5,   0x1.de4p-8,   -0x1.5b4p-14, -0x1.4ap1,   -0x1.868p9,
                -0x1.d14p0,  0x1.d7cp15,   0x1.3c8p14,   0x1.2ccp-14,  -0x1.ee4p8,   0x1.49p-3,    0x1.35cp12,  0x1.d34p6,
                0x1.7acp3,   -0x1.fa4p2,   0x1.7b4p13,   -0x1.cf4p-12, -0x1.ebcp-10, -0x1.5p-3,    0x1.4bp-6,   0x1.83p12,
                -0x1.f9cp-8, -0x1.43p-8,   -0x1.99p-1,   -0x1.dacp3,   -0x1.728p-4,  -0x1.03cp4,   0x1.604p-2,  -0x1.0ep13,
            });

            try testArgs(@Vector(1, f32), .{
                -0x1.17cp-12,
            });
            try testArgs(@Vector(2, f32), .{
                -0x1.a3123ap90, -0x1.4a2ec6p-54,
            });
            try testArgs(@Vector(4, f32), .{
                -0x1.8a41p77, -0x1.7c54e2p-61, -0x1.498556p-41, 0x1.d77c22p-20,
            });
            try testArgs(@Vector(8, f32), .{
                0x1.943da4p-86, 0x1.528792p95,  -0x1.9c9bfap-26, -0x1.8df936p-90,
                -0x1.6a70cep56, 0x1.626638p-48, 0x1.7bb2bap-57,  -0x1.ac5104p94,
            });
            try testArgs(@Vector(16, f32), .{
                0x1.157044p115, -0x1.416c04p-111, 0x1.a8f164p-104, 0x1.9b6678p84,
                -0x1.9d065cp9,  -0x1.e8c4b4p126,  -0x1.ddb968p84,  -0x1.fec8c8p74,
                0x1.64ffb2p59,  0x1.548922p20,    0x1.7270fcp22,   -0x1.abac68p33,
                0x1.faabfp33,   -0x1.8aee82p55,   0x1.1bf8fp75,    0x1.33c46ap-66,
            });
            try testArgs(@Vector(32, f32), .{
                -0x1.039b68p37,   -0x1.34de4ap-74, -0x1.05d78ap-76, -0x1.be0f5ap-47,
                0x1.032204p-38,   0x1.ef8e2ap-78,  -0x1.b013ecp-80, 0x1.71fe4cp99,
                0x1.abdadap-14,   0x1.56a9a8p-48,  -0x1.8bbd7ep9,   0x1.edd308p-72,
                -0x1.92fafcp-121, -0x1.50812p19,   0x1.f4ddc4p28,   -0x1.6f0b12p-50,
                -0x1.12ab02p127,  0x1.24df48p21,   -0x1.993c3p-14,  -0x1.4cc476p-112,
                0x1.13d9a8p-40,   0x1.a6e652p-9,   -0x1.9c730cp-21, -0x1.a75aaap-70,
                -0x1.39e632p-111, 0x1.8e8da8p-45,  0x1.b5652cp31,   0x1.258366p44,
                0x1.d473aap92,    -0x1.951b64p9,   0x1.542edp15,    -0x0.f6222ap-126,
            });

            try testArgs(@Vector(1, f64), .{
                -0x1.0114613df6f97p816,
            });
            try testArgs(@Vector(2, f64), .{
                -0x1.8404dad72003cp720, -0x1.6b14b40bcf3b7p-176,
            });
            try testArgs(@Vector(4, f64), .{
                -0x1.04e1acbfddd9cp681, -0x1.ed553cc056da7p-749,
                0x1.3d3f703a0c893p-905, 0x1.0b35633fa78fp691,
            });
            try testArgs(@Vector(8, f64), .{
                -0x1.901a2a60f0562p-301, -0x1.2516175ad61ecp-447,
                0x1.e7b12124846bfp564,   0x1.9291384bd7259p209,
                -0x1.a7bf62f803c98p900,  0x1.4e2e26257bb3p987,
                -0x1.413ca9a32d894p811,  0x1.61b1dd9432e95p479,
            });
            try testArgs(@Vector(16, f64), .{
                -0x1.8fc7286d95f54p-235,  -0x1.796a7ea8372b6p-837,
                -0x1.8c0f930539acbp-98,   -0x1.ec80dfbf0b931p-430,
                -0x1.e3d80c640652fp-1019, 0x1.8241238fb542fp161,
                -0x1.e1f1a79d50263p137,   -0x1.9ac5cb2771c28p-791,
                0x1.4d8f00fe881e7p-401,   -0x1.87fbd7bfd99d7p346,
                -0x1.a8a7cc575335ep1017,  0x1.37bb88dc3fd8bp-355,
                0x1.9d53d346c0e65p929,    -0x1.bbae3d0229c34p289,
                -0x1.cb8ef994d5ce5p25,    0x1.ba20af512616ap50,
            });

            try testArgs(@Vector(1, f80), .{
                -0x1.a2e9410a7dfedabp-2324,
            });
            try testArgs(@Vector(2, f80), .{
                -0x1.a2e9410a7dfedabp-2324,
                0x1.2b17da3b9746885p-8665,
            });
            try testArgs(@Vector(4, f80), .{
                -0x1.c488fedb7ab646cep-13007,
                0x1.e914deaccaa50016p2073,
                -0x1.d1c7ae8ec3c9df86p10642,
                -0x1.2da1658f337fa01p9893,
            });
            try testArgs(@Vector(8, f80), .{
                -0x1.bed8a74c43750656p890,
                -0x1.7bf57f38004ac976p8481,
                -0x1.9cdc10ac0657d328p7884,
                0x1.c86f61883da149fp12293,
                -0x1.528d6957df6bfdd8p14125,
                -0x1.5ebb4006d0243bfep14530,
                -0x1.94b9b18636d12402p-1845,
                -0x1.25439a6d68add188p5962,
            });

            try testArgs(@Vector(1, f128), .{
                -0x1.d1e6fc3b1e66632e7b79051a47dap14300,
            });
            try testArgs(@Vector(2, f128), .{
                0x1.84b3ac8ffe5893b2c6af8d68de9dp-83,
                -0x1.438ca2c8a0d8e3ee9062d351c46ep-10235,
            });
            try testArgs(@Vector(4, f128), .{
                0x1.04eb03882d4fd1b090e714d3e5ep806,
                -0x1.4082b29f7c26e701764c915642ffp-6182,
                -0x1.b6f1e8565e5040415110f18b519ap13383,
                0x1.1c29f8c162cead9061c5797ea15ap11957,
            });
            try testArgs(@Vector(8, f128), .{
                -0x1.53d7f00cd204d80e5ff5bb665773p11218,
                -0x1.4daa1c81cffe28e8fa5cd703c287p2362,
                -0x1.cc6a71c3ad4560871efdbd025cd7p-8116,
                -0x1.87f8553cf8772fb6b78e7df3e3bap14523,
                -0x1.14b6880f6678f86dfb543dde1c6ep2105,
                0x1.9d2d4398414da9d857e76e8fd7ccp-13668,
                0x1.a37f07af240ded458d103c022064p-1158,
                0x1.425d53e6bd6070b847e5da1ed593p1394,
            });
        }
    };
}

fn cast(comptime op: anytype, comptime opts: struct { compare: Compare = .relaxed }) type {
    return struct {
        // noinline so that `mem_arg` is on the stack
        noinline fn testArgKinds(
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            comptime Result: type,
            comptime Type: type,
            comptime imm_arg: Type,
            mem_arg: Type,
        ) !void {
            const expected = comptime op(Result, Type, imm_arg, imm_arg);
            var reg_arg = mem_arg;
            _ = .{&reg_arg};
            try checkExpected(expected, op(Result, Type, reg_arg, imm_arg), opts.compare);
            try checkExpected(expected, op(Result, Type, mem_arg, imm_arg), opts.compare);
            try checkExpected(expected, op(Result, Type, imm_arg, imm_arg), opts.compare);
        }
        // noinline for a more helpful stack trace
        noinline fn testArgs(comptime Result: type, comptime Type: type, comptime imm_arg: Type) !void {
            try testArgKinds(
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                Result,
                Type,
                imm_arg,
                imm_arg,
            );
        }
        fn testSameSignednessInts() !void {
            try testArgs(i8, i1, -1);
            try testArgs(i8, i1, 0);
            try testArgs(i16, i1, -1);
            try testArgs(i16, i1, 0);
            try testArgs(i32, i1, -1);
            try testArgs(i32, i1, 0);
            try testArgs(i64, i1, -1);
            try testArgs(i64, i1, 0);
            try testArgs(i128, i1, -1);
            try testArgs(i128, i1, 0);
            try testArgs(i256, i1, -1);
            try testArgs(i256, i1, 0);
            try testArgs(i512, i1, -1);
            try testArgs(i512, i1, 0);
            try testArgs(i1024, i1, -1);
            try testArgs(i1024, i1, 0);
            try testArgs(u8, u1, 0);
            try testArgs(u8, u1, 1 << 0);
            try testArgs(u16, u1, 0);
            try testArgs(u16, u1, 1 << 0);
            try testArgs(u32, u1, 0);
            try testArgs(u32, u1, 1 << 0);
            try testArgs(u64, u1, 0);
            try testArgs(u64, u1, 1 << 0);
            try testArgs(u128, u1, 0);
            try testArgs(u128, u1, 1 << 0);
            try testArgs(u256, u1, 0);
            try testArgs(u256, u1, 1 << 0);
            try testArgs(u512, u1, 0);
            try testArgs(u512, u1, 1 << 0);
            try testArgs(u1024, u1, 0);
            try testArgs(u1024, u1, 1 << 0);

            try testArgs(i8, i2, -1 << 1);
            try testArgs(i8, i2, -1);
            try testArgs(i8, i2, 0);
            try testArgs(i16, i2, -1 << 1);
            try testArgs(i16, i2, -1);
            try testArgs(i16, i2, 0);
            try testArgs(i32, i2, -1 << 1);
            try testArgs(i32, i2, -1);
            try testArgs(i32, i2, 0);
            try testArgs(i64, i2, -1 << 1);
            try testArgs(i64, i2, -1);
            try testArgs(i64, i2, 0);
            try testArgs(i128, i2, -1 << 1);
            try testArgs(i128, i2, -1);
            try testArgs(i128, i2, 0);
            try testArgs(i256, i2, -1 << 1);
            try testArgs(i256, i2, -1);
            try testArgs(i256, i2, 0);
            try testArgs(i512, i2, -1 << 1);
            try testArgs(i512, i2, -1);
            try testArgs(i512, i2, 0);
            try testArgs(i1024, i2, -1 << 1);
            try testArgs(i1024, i2, -1);
            try testArgs(i1024, i2, 0);
            try testArgs(u8, u2, 0);
            try testArgs(u8, u2, 1 << 0);
            try testArgs(u8, u2, 1 << 1);
            try testArgs(u16, u2, 0);
            try testArgs(u16, u2, 1 << 0);
            try testArgs(u16, u2, 1 << 1);
            try testArgs(u32, u2, 0);
            try testArgs(u32, u2, 1 << 0);
            try testArgs(u32, u2, 1 << 1);
            try testArgs(u64, u2, 0);
            try testArgs(u64, u2, 1 << 0);
            try testArgs(u64, u2, 1 << 1);
            try testArgs(u128, u2, 0);
            try testArgs(u128, u2, 1 << 0);
            try testArgs(u128, u2, 1 << 1);
            try testArgs(u256, u2, 0);
            try testArgs(u256, u2, 1 << 0);
            try testArgs(u256, u2, 1 << 1);
            try testArgs(u512, u2, 0);
            try testArgs(u512, u2, 1 << 0);
            try testArgs(u512, u2, 1 << 1);
            try testArgs(u1024, u2, 0);
            try testArgs(u1024, u2, 1 << 0);
            try testArgs(u1024, u2, 1 << 1);

            try testArgs(i8, i3, -1 << 2);
            try testArgs(i8, i3, -1);
            try testArgs(i8, i3, 0);
            try testArgs(i16, i3, -1 << 2);
            try testArgs(i16, i3, -1);
            try testArgs(i16, i3, 0);
            try testArgs(i32, i3, -1 << 2);
            try testArgs(i32, i3, -1);
            try testArgs(i32, i3, 0);
            try testArgs(i64, i3, -1 << 2);
            try testArgs(i64, i3, -1);
            try testArgs(i64, i3, 0);
            try testArgs(i128, i3, -1 << 2);
            try testArgs(i128, i3, -1);
            try testArgs(i128, i3, 0);
            try testArgs(i256, i3, -1 << 2);
            try testArgs(i256, i3, -1);
            try testArgs(i256, i3, 0);
            try testArgs(i512, i3, -1 << 2);
            try testArgs(i512, i3, -1);
            try testArgs(i512, i3, 0);
            try testArgs(i1024, i3, -1 << 2);
            try testArgs(i1024, i3, -1);
            try testArgs(i1024, i3, 0);
            try testArgs(u8, u3, 0);
            try testArgs(u8, u3, 1 << 0);
            try testArgs(u8, u3, 1 << 2);
            try testArgs(u16, u3, 0);
            try testArgs(u16, u3, 1 << 0);
            try testArgs(u16, u3, 1 << 2);
            try testArgs(u32, u3, 0);
            try testArgs(u32, u3, 1 << 0);
            try testArgs(u32, u3, 1 << 2);
            try testArgs(u64, u3, 0);
            try testArgs(u64, u3, 1 << 0);
            try testArgs(u64, u3, 1 << 2);
            try testArgs(u128, u3, 0);
            try testArgs(u128, u3, 1 << 0);
            try testArgs(u128, u3, 1 << 2);
            try testArgs(u256, u3, 0);
            try testArgs(u256, u3, 1 << 0);
            try testArgs(u256, u3, 1 << 2);
            try testArgs(u512, u3, 0);
            try testArgs(u512, u3, 1 << 0);
            try testArgs(u512, u3, 1 << 2);
            try testArgs(u1024, u3, 0);
            try testArgs(u1024, u3, 1 << 0);
            try testArgs(u1024, u3, 1 << 2);

            try testArgs(i8, i4, -1 << 3);
            try testArgs(i8, i4, -1);
            try testArgs(i8, i4, 0);
            try testArgs(i16, i4, -1 << 3);
            try testArgs(i16, i4, -1);
            try testArgs(i16, i4, 0);
            try testArgs(i32, i4, -1 << 3);
            try testArgs(i32, i4, -1);
            try testArgs(i32, i4, 0);
            try testArgs(i64, i4, -1 << 3);
            try testArgs(i64, i4, -1);
            try testArgs(i64, i4, 0);
            try testArgs(i128, i4, -1 << 3);
            try testArgs(i128, i4, -1);
            try testArgs(i128, i4, 0);
            try testArgs(i256, i4, -1 << 3);
            try testArgs(i256, i4, -1);
            try testArgs(i256, i4, 0);
            try testArgs(i512, i4, -1 << 3);
            try testArgs(i512, i4, -1);
            try testArgs(i512, i4, 0);
            try testArgs(i1024, i4, -1 << 3);
            try testArgs(i1024, i4, -1);
            try testArgs(i1024, i4, 0);
            try testArgs(u8, u4, 0);
            try testArgs(u8, u4, 1 << 0);
            try testArgs(u8, u4, 1 << 3);
            try testArgs(u16, u4, 0);
            try testArgs(u16, u4, 1 << 0);
            try testArgs(u16, u4, 1 << 3);
            try testArgs(u32, u4, 0);
            try testArgs(u32, u4, 1 << 0);
            try testArgs(u32, u4, 1 << 3);
            try testArgs(u64, u4, 0);
            try testArgs(u64, u4, 1 << 0);
            try testArgs(u64, u4, 1 << 3);
            try testArgs(u128, u4, 0);
            try testArgs(u128, u4, 1 << 0);
            try testArgs(u128, u4, 1 << 3);
            try testArgs(u256, u4, 0);
            try testArgs(u256, u4, 1 << 0);
            try testArgs(u256, u4, 1 << 3);
            try testArgs(u512, u4, 0);
            try testArgs(u512, u4, 1 << 0);
            try testArgs(u512, u4, 1 << 3);
            try testArgs(u1024, u4, 0);
            try testArgs(u1024, u4, 1 << 0);
            try testArgs(u1024, u4, 1 << 3);

            try testArgs(i8, i5, -1 << 4);
            try testArgs(i8, i5, -1);
            try testArgs(i8, i5, 0);
            try testArgs(i16, i5, -1 << 4);
            try testArgs(i16, i5, -1);
            try testArgs(i16, i5, 0);
            try testArgs(i32, i5, -1 << 4);
            try testArgs(i32, i5, -1);
            try testArgs(i32, i5, 0);
            try testArgs(i64, i5, -1 << 4);
            try testArgs(i64, i5, -1);
            try testArgs(i64, i5, 0);
            try testArgs(i128, i5, -1 << 4);
            try testArgs(i128, i5, -1);
            try testArgs(i128, i5, 0);
            try testArgs(i256, i5, -1 << 4);
            try testArgs(i256, i5, -1);
            try testArgs(i256, i5, 0);
            try testArgs(i512, i5, -1 << 4);
            try testArgs(i512, i5, -1);
            try testArgs(i512, i5, 0);
            try testArgs(i1024, i5, -1 << 4);
            try testArgs(i1024, i5, -1);
            try testArgs(i1024, i5, 0);
            try testArgs(u8, u5, 0);
            try testArgs(u8, u5, 1 << 0);
            try testArgs(u8, u5, 1 << 4);
            try testArgs(u16, u5, 0);
            try testArgs(u16, u5, 1 << 0);
            try testArgs(u16, u5, 1 << 4);
            try testArgs(u32, u5, 0);
            try testArgs(u32, u5, 1 << 0);
            try testArgs(u32, u5, 1 << 4);
            try testArgs(u64, u5, 0);
            try testArgs(u64, u5, 1 << 0);
            try testArgs(u64, u5, 1 << 4);
            try testArgs(u128, u5, 0);
            try testArgs(u128, u5, 1 << 0);
            try testArgs(u128, u5, 1 << 4);
            try testArgs(u256, u5, 0);
            try testArgs(u256, u5, 1 << 0);
            try testArgs(u256, u5, 1 << 4);
            try testArgs(u512, u5, 0);
            try testArgs(u512, u5, 1 << 0);
            try testArgs(u512, u5, 1 << 4);
            try testArgs(u1024, u5, 0);
            try testArgs(u1024, u5, 1 << 0);
            try testArgs(u1024, u5, 1 << 4);

            try testArgs(i8, i7, -1 << 6);
            try testArgs(i8, i7, -1);
            try testArgs(i8, i7, 0);
            try testArgs(i16, i7, -1 << 6);
            try testArgs(i16, i7, -1);
            try testArgs(i16, i7, 0);
            try testArgs(i32, i7, -1 << 6);
            try testArgs(i32, i7, -1);
            try testArgs(i32, i7, 0);
            try testArgs(i64, i7, -1 << 6);
            try testArgs(i64, i7, -1);
            try testArgs(i64, i7, 0);
            try testArgs(i128, i7, -1 << 6);
            try testArgs(i128, i7, -1);
            try testArgs(i128, i7, 0);
            try testArgs(i256, i7, -1 << 6);
            try testArgs(i256, i7, -1);
            try testArgs(i256, i7, 0);
            try testArgs(i512, i7, -1 << 6);
            try testArgs(i512, i7, -1);
            try testArgs(i512, i7, 0);
            try testArgs(i1024, i7, -1 << 6);
            try testArgs(i1024, i7, -1);
            try testArgs(i1024, i7, 0);
            try testArgs(u8, u7, 0);
            try testArgs(u8, u7, 1 << 0);
            try testArgs(u8, u7, 1 << 6);
            try testArgs(u16, u7, 0);
            try testArgs(u16, u7, 1 << 0);
            try testArgs(u16, u7, 1 << 6);
            try testArgs(u32, u7, 0);
            try testArgs(u32, u7, 1 << 0);
            try testArgs(u32, u7, 1 << 6);
            try testArgs(u64, u7, 0);
            try testArgs(u64, u7, 1 << 0);
            try testArgs(u64, u7, 1 << 6);
            try testArgs(u128, u7, 0);
            try testArgs(u128, u7, 1 << 0);
            try testArgs(u128, u7, 1 << 6);
            try testArgs(u256, u7, 0);
            try testArgs(u256, u7, 1 << 0);
            try testArgs(u256, u7, 1 << 6);
            try testArgs(u512, u7, 0);
            try testArgs(u512, u7, 1 << 0);
            try testArgs(u512, u7, 1 << 6);
            try testArgs(u1024, u7, 0);
            try testArgs(u1024, u7, 1 << 0);
            try testArgs(u1024, u7, 1 << 6);

            try testArgs(i8, i8, -1 << 7);
            try testArgs(i8, i8, -1);
            try testArgs(i8, i8, 0);
            try testArgs(i16, i8, -1 << 7);
            try testArgs(i16, i8, -1);
            try testArgs(i16, i8, 0);
            try testArgs(i32, i8, -1 << 7);
            try testArgs(i32, i8, -1);
            try testArgs(i32, i8, 0);
            try testArgs(i64, i8, -1 << 7);
            try testArgs(i64, i8, -1);
            try testArgs(i64, i8, 0);
            try testArgs(i128, i8, -1 << 7);
            try testArgs(i128, i8, -1);
            try testArgs(i128, i8, 0);
            try testArgs(i256, i8, -1 << 7);
            try testArgs(i256, i8, -1);
            try testArgs(i256, i8, 0);
            try testArgs(i512, i8, -1 << 7);
            try testArgs(i512, i8, -1);
            try testArgs(i512, i8, 0);
            try testArgs(i1024, i8, -1 << 7);
            try testArgs(i1024, i8, -1);
            try testArgs(i1024, i8, 0);
            try testArgs(u8, u8, 0);
            try testArgs(u8, u8, 1 << 0);
            try testArgs(u8, u8, 1 << 7);
            try testArgs(u16, u8, 0);
            try testArgs(u16, u8, 1 << 0);
            try testArgs(u16, u8, 1 << 7);
            try testArgs(u32, u8, 0);
            try testArgs(u32, u8, 1 << 0);
            try testArgs(u32, u8, 1 << 7);
            try testArgs(u64, u8, 0);
            try testArgs(u64, u8, 1 << 0);
            try testArgs(u64, u8, 1 << 7);
            try testArgs(u128, u8, 0);
            try testArgs(u128, u8, 1 << 0);
            try testArgs(u128, u8, 1 << 7);
            try testArgs(u256, u8, 0);
            try testArgs(u256, u8, 1 << 0);
            try testArgs(u256, u8, 1 << 7);
            try testArgs(u512, u8, 0);
            try testArgs(u512, u8, 1 << 0);
            try testArgs(u512, u8, 1 << 7);
            try testArgs(u1024, u8, 0);
            try testArgs(u1024, u8, 1 << 0);
            try testArgs(u1024, u8, 1 << 7);

            try testArgs(i8, i9, -1 << 8);
            try testArgs(i8, i9, -1);
            try testArgs(i8, i9, 0);
            try testArgs(i16, i9, -1 << 8);
            try testArgs(i16, i9, -1);
            try testArgs(i16, i9, 0);
            try testArgs(i32, i9, -1 << 8);
            try testArgs(i32, i9, -1);
            try testArgs(i32, i9, 0);
            try testArgs(i64, i9, -1 << 8);
            try testArgs(i64, i9, -1);
            try testArgs(i64, i9, 0);
            try testArgs(i128, i9, -1 << 8);
            try testArgs(i128, i9, -1);
            try testArgs(i128, i9, 0);
            try testArgs(i256, i9, -1 << 8);
            try testArgs(i256, i9, -1);
            try testArgs(i256, i9, 0);
            try testArgs(i512, i9, -1 << 8);
            try testArgs(i512, i9, -1);
            try testArgs(i512, i9, 0);
            try testArgs(i1024, i9, -1 << 8);
            try testArgs(i1024, i9, -1);
            try testArgs(i1024, i9, 0);
            try testArgs(u8, u9, 0);
            try testArgs(u8, u9, 1 << 0);
            try testArgs(u8, u9, 1 << 8);
            try testArgs(u16, u9, 0);
            try testArgs(u16, u9, 1 << 0);
            try testArgs(u16, u9, 1 << 8);
            try testArgs(u32, u9, 0);
            try testArgs(u32, u9, 1 << 0);
            try testArgs(u32, u9, 1 << 8);
            try testArgs(u64, u9, 0);
            try testArgs(u64, u9, 1 << 0);
            try testArgs(u64, u9, 1 << 8);
            try testArgs(u128, u9, 0);
            try testArgs(u128, u9, 1 << 0);
            try testArgs(u128, u9, 1 << 8);
            try testArgs(u256, u9, 0);
            try testArgs(u256, u9, 1 << 0);
            try testArgs(u256, u9, 1 << 8);
            try testArgs(u512, u9, 0);
            try testArgs(u512, u9, 1 << 0);
            try testArgs(u512, u9, 1 << 8);
            try testArgs(u1024, u9, 0);
            try testArgs(u1024, u9, 1 << 0);
            try testArgs(u1024, u9, 1 << 8);

            try testArgs(i8, i15, -1 << 14);
            try testArgs(i8, i15, -1);
            try testArgs(i8, i15, 0);
            try testArgs(i16, i15, -1 << 14);
            try testArgs(i16, i15, -1);
            try testArgs(i16, i15, 0);
            try testArgs(i32, i15, -1 << 14);
            try testArgs(i32, i15, -1);
            try testArgs(i32, i15, 0);
            try testArgs(i64, i15, -1 << 14);
            try testArgs(i64, i15, -1);
            try testArgs(i64, i15, 0);
            try testArgs(i128, i15, -1 << 14);
            try testArgs(i128, i15, -1);
            try testArgs(i128, i15, 0);
            try testArgs(i256, i15, -1 << 14);
            try testArgs(i256, i15, -1);
            try testArgs(i256, i15, 0);
            try testArgs(i512, i15, -1 << 14);
            try testArgs(i512, i15, -1);
            try testArgs(i512, i15, 0);
            try testArgs(i1024, i15, -1 << 14);
            try testArgs(i1024, i15, -1);
            try testArgs(i1024, i15, 0);
            try testArgs(u8, u15, 0);
            try testArgs(u8, u15, 1 << 0);
            try testArgs(u8, u15, 1 << 14);
            try testArgs(u16, u15, 0);
            try testArgs(u16, u15, 1 << 0);
            try testArgs(u16, u15, 1 << 14);
            try testArgs(u32, u15, 0);
            try testArgs(u32, u15, 1 << 0);
            try testArgs(u32, u15, 1 << 14);
            try testArgs(u64, u15, 0);
            try testArgs(u64, u15, 1 << 0);
            try testArgs(u64, u15, 1 << 14);
            try testArgs(u128, u15, 0);
            try testArgs(u128, u15, 1 << 0);
            try testArgs(u128, u15, 1 << 14);
            try testArgs(u256, u15, 0);
            try testArgs(u256, u15, 1 << 0);
            try testArgs(u256, u15, 1 << 14);
            try testArgs(u512, u15, 0);
            try testArgs(u512, u15, 1 << 0);
            try testArgs(u512, u15, 1 << 14);
            try testArgs(u1024, u15, 0);
            try testArgs(u1024, u15, 1 << 0);
            try testArgs(u1024, u15, 1 << 14);

            try testArgs(i8, i16, -1 << 15);
            try testArgs(i8, i16, -1);
            try testArgs(i8, i16, 0);
            try testArgs(i16, i16, -1 << 15);
            try testArgs(i16, i16, -1);
            try testArgs(i16, i16, 0);
            try testArgs(i32, i16, -1 << 15);
            try testArgs(i32, i16, -1);
            try testArgs(i32, i16, 0);
            try testArgs(i64, i16, -1 << 15);
            try testArgs(i64, i16, -1);
            try testArgs(i64, i16, 0);
            try testArgs(i128, i16, -1 << 15);
            try testArgs(i128, i16, -1);
            try testArgs(i128, i16, 0);
            try testArgs(i256, i16, -1 << 15);
            try testArgs(i256, i16, -1);
            try testArgs(i256, i16, 0);
            try testArgs(i512, i16, -1 << 15);
            try testArgs(i512, i16, -1);
            try testArgs(i512, i16, 0);
            try testArgs(i1024, i16, -1 << 15);
            try testArgs(i1024, i16, -1);
            try testArgs(i1024, i16, 0);
            try testArgs(u8, u16, 0);
            try testArgs(u8, u16, 1 << 0);
            try testArgs(u8, u16, 1 << 15);
            try testArgs(u16, u16, 0);
            try testArgs(u16, u16, 1 << 0);
            try testArgs(u16, u16, 1 << 15);
            try testArgs(u32, u16, 0);
            try testArgs(u32, u16, 1 << 0);
            try testArgs(u32, u16, 1 << 15);
            try testArgs(u64, u16, 0);
            try testArgs(u64, u16, 1 << 0);
            try testArgs(u64, u16, 1 << 15);
            try testArgs(u128, u16, 0);
            try testArgs(u128, u16, 1 << 0);
            try testArgs(u128, u16, 1 << 15);
            try testArgs(u256, u16, 0);
            try testArgs(u256, u16, 1 << 0);
            try testArgs(u256, u16, 1 << 15);
            try testArgs(u512, u16, 0);
            try testArgs(u512, u16, 1 << 0);
            try testArgs(u512, u16, 1 << 15);
            try testArgs(u1024, u16, 0);
            try testArgs(u1024, u16, 1 << 0);
            try testArgs(u1024, u16, 1 << 15);

            try testArgs(i8, i17, -1 << 16);
            try testArgs(i8, i17, -1);
            try testArgs(i8, i17, 0);
            try testArgs(i16, i17, -1 << 16);
            try testArgs(i16, i17, -1);
            try testArgs(i16, i17, 0);
            try testArgs(i32, i17, -1 << 16);
            try testArgs(i32, i17, -1);
            try testArgs(i32, i17, 0);
            try testArgs(i64, i17, -1 << 16);
            try testArgs(i64, i17, -1);
            try testArgs(i64, i17, 0);
            try testArgs(i128, i17, -1 << 16);
            try testArgs(i128, i17, -1);
            try testArgs(i128, i17, 0);
            try testArgs(i256, i17, -1 << 16);
            try testArgs(i256, i17, -1);
            try testArgs(i256, i17, 0);
            try testArgs(i512, i17, -1 << 16);
            try testArgs(i512, i17, -1);
            try testArgs(i512, i17, 0);
            try testArgs(i1024, i17, -1 << 16);
            try testArgs(i1024, i17, -1);
            try testArgs(i1024, i17, 0);
            try testArgs(u8, u17, 0);
            try testArgs(u8, u17, 1 << 0);
            try testArgs(u8, u17, 1 << 16);
            try testArgs(u16, u17, 0);
            try testArgs(u16, u17, 1 << 0);
            try testArgs(u16, u17, 1 << 16);
            try testArgs(u32, u17, 0);
            try testArgs(u32, u17, 1 << 0);
            try testArgs(u32, u17, 1 << 16);
            try testArgs(u64, u17, 0);
            try testArgs(u64, u17, 1 << 0);
            try testArgs(u64, u17, 1 << 16);
            try testArgs(u128, u17, 0);
            try testArgs(u128, u17, 1 << 0);
            try testArgs(u128, u17, 1 << 16);
            try testArgs(u256, u17, 0);
            try testArgs(u256, u17, 1 << 0);
            try testArgs(u256, u17, 1 << 16);
            try testArgs(u512, u17, 0);
            try testArgs(u512, u17, 1 << 0);
            try testArgs(u512, u17, 1 << 16);
            try testArgs(u1024, u17, 0);
            try testArgs(u1024, u17, 1 << 0);
            try testArgs(u1024, u17, 1 << 16);

            try testArgs(i8, i31, -1 << 30);
            try testArgs(i8, i31, -1);
            try testArgs(i8, i31, 0);
            try testArgs(i16, i31, -1 << 30);
            try testArgs(i16, i31, -1);
            try testArgs(i16, i31, 0);
            try testArgs(i32, i31, -1 << 30);
            try testArgs(i32, i31, -1);
            try testArgs(i32, i31, 0);
            try testArgs(i64, i31, -1 << 30);
            try testArgs(i64, i31, -1);
            try testArgs(i64, i31, 0);
            try testArgs(i128, i31, -1 << 30);
            try testArgs(i128, i31, -1);
            try testArgs(i128, i31, 0);
            try testArgs(i256, i31, -1 << 30);
            try testArgs(i256, i31, -1);
            try testArgs(i256, i31, 0);
            try testArgs(i512, i31, -1 << 30);
            try testArgs(i512, i31, -1);
            try testArgs(i512, i31, 0);
            try testArgs(i1024, i31, -1 << 30);
            try testArgs(i1024, i31, -1);
            try testArgs(i1024, i31, 0);
            try testArgs(u8, u31, 0);
            try testArgs(u8, u31, 1 << 0);
            try testArgs(u8, u31, 1 << 30);
            try testArgs(u16, u31, 0);
            try testArgs(u16, u31, 1 << 0);
            try testArgs(u16, u31, 1 << 30);
            try testArgs(u32, u31, 0);
            try testArgs(u32, u31, 1 << 0);
            try testArgs(u32, u31, 1 << 30);
            try testArgs(u64, u31, 0);
            try testArgs(u64, u31, 1 << 0);
            try testArgs(u64, u31, 1 << 30);
            try testArgs(u128, u31, 0);
            try testArgs(u128, u31, 1 << 0);
            try testArgs(u128, u31, 1 << 30);
            try testArgs(u256, u31, 0);
            try testArgs(u256, u31, 1 << 0);
            try testArgs(u256, u31, 1 << 30);
            try testArgs(u512, u31, 0);
            try testArgs(u512, u31, 1 << 0);
            try testArgs(u512, u31, 1 << 30);
            try testArgs(u1024, u31, 0);
            try testArgs(u1024, u31, 1 << 0);
            try testArgs(u1024, u31, 1 << 30);

            try testArgs(i8, i32, -1 << 31);
            try testArgs(i8, i32, -1);
            try testArgs(i8, i32, 0);
            try testArgs(i16, i32, -1 << 31);
            try testArgs(i16, i32, -1);
            try testArgs(i16, i32, 0);
            try testArgs(i32, i32, -1 << 31);
            try testArgs(i32, i32, -1);
            try testArgs(i32, i32, 0);
            try testArgs(i64, i32, -1 << 31);
            try testArgs(i64, i32, -1);
            try testArgs(i64, i32, 0);
            try testArgs(i128, i32, -1 << 31);
            try testArgs(i128, i32, -1);
            try testArgs(i128, i32, 0);
            try testArgs(i256, i32, -1 << 31);
            try testArgs(i256, i32, -1);
            try testArgs(i256, i32, 0);
            try testArgs(i512, i32, -1 << 31);
            try testArgs(i512, i32, -1);
            try testArgs(i512, i32, 0);
            try testArgs(i1024, i32, -1 << 31);
            try testArgs(i1024, i32, -1);
            try testArgs(i1024, i32, 0);
            try testArgs(u8, u32, 0);
            try testArgs(u8, u32, 1 << 0);
            try testArgs(u8, u32, 1 << 31);
            try testArgs(u16, u32, 0);
            try testArgs(u16, u32, 1 << 0);
            try testArgs(u16, u32, 1 << 31);
            try testArgs(u32, u32, 0);
            try testArgs(u32, u32, 1 << 0);
            try testArgs(u32, u32, 1 << 31);
            try testArgs(u64, u32, 0);
            try testArgs(u64, u32, 1 << 0);
            try testArgs(u64, u32, 1 << 31);
            try testArgs(u128, u32, 0);
            try testArgs(u128, u32, 1 << 0);
            try testArgs(u128, u32, 1 << 31);
            try testArgs(u256, u32, 0);
            try testArgs(u256, u32, 1 << 0);
            try testArgs(u256, u32, 1 << 31);
            try testArgs(u512, u32, 0);
            try testArgs(u512, u32, 1 << 0);
            try testArgs(u512, u32, 1 << 31);
            try testArgs(u1024, u32, 0);
            try testArgs(u1024, u32, 1 << 0);
            try testArgs(u1024, u32, 1 << 31);

            try testArgs(i8, i33, -1 << 32);
            try testArgs(i8, i33, -1);
            try testArgs(i8, i33, 0);
            try testArgs(i16, i33, -1 << 32);
            try testArgs(i16, i33, -1);
            try testArgs(i16, i33, 0);
            try testArgs(i32, i33, -1 << 32);
            try testArgs(i32, i33, -1);
            try testArgs(i32, i33, 0);
            try testArgs(i64, i33, -1 << 32);
            try testArgs(i64, i33, -1);
            try testArgs(i64, i33, 0);
            try testArgs(i128, i33, -1 << 32);
            try testArgs(i128, i33, -1);
            try testArgs(i128, i33, 0);
            try testArgs(i256, i33, -1 << 32);
            try testArgs(i256, i33, -1);
            try testArgs(i256, i33, 0);
            try testArgs(i512, i33, -1 << 32);
            try testArgs(i512, i33, -1);
            try testArgs(i512, i33, 0);
            try testArgs(i1024, i33, -1 << 32);
            try testArgs(i1024, i33, -1);
            try testArgs(i1024, i33, 0);
            try testArgs(u8, u33, 0);
            try testArgs(u8, u33, 1 << 0);
            try testArgs(u8, u33, 1 << 32);
            try testArgs(u16, u33, 0);
            try testArgs(u16, u33, 1 << 0);
            try testArgs(u16, u33, 1 << 32);
            try testArgs(u32, u33, 0);
            try testArgs(u32, u33, 1 << 0);
            try testArgs(u32, u33, 1 << 32);
            try testArgs(u64, u33, 0);
            try testArgs(u64, u33, 1 << 0);
            try testArgs(u64, u33, 1 << 32);
            try testArgs(u128, u33, 0);
            try testArgs(u128, u33, 1 << 0);
            try testArgs(u128, u33, 1 << 32);
            try testArgs(u256, u33, 0);
            try testArgs(u256, u33, 1 << 0);
            try testArgs(u256, u33, 1 << 32);
            try testArgs(u512, u33, 0);
            try testArgs(u512, u33, 1 << 0);
            try testArgs(u512, u33, 1 << 32);
            try testArgs(u1024, u33, 0);
            try testArgs(u1024, u33, 1 << 0);
            try testArgs(u1024, u33, 1 << 32);

            try testArgs(i8, i63, -1 << 62);
            try testArgs(i8, i63, -1);
            try testArgs(i8, i63, 0);
            try testArgs(i16, i63, -1 << 62);
            try testArgs(i16, i63, -1);
            try testArgs(i16, i63, 0);
            try testArgs(i32, i63, -1 << 62);
            try testArgs(i32, i63, -1);
            try testArgs(i32, i63, 0);
            try testArgs(i64, i63, -1 << 62);
            try testArgs(i64, i63, -1);
            try testArgs(i64, i63, 0);
            try testArgs(i128, i63, -1 << 62);
            try testArgs(i128, i63, -1);
            try testArgs(i128, i63, 0);
            try testArgs(i256, i63, -1 << 62);
            try testArgs(i256, i63, -1);
            try testArgs(i256, i63, 0);
            try testArgs(i512, i63, -1 << 62);
            try testArgs(i512, i63, -1);
            try testArgs(i512, i63, 0);
            try testArgs(i1024, i63, -1 << 62);
            try testArgs(i1024, i63, -1);
            try testArgs(i1024, i63, 0);
            try testArgs(u8, u63, 0);
            try testArgs(u8, u63, 1 << 0);
            try testArgs(u8, u63, 1 << 62);
            try testArgs(u16, u63, 0);
            try testArgs(u16, u63, 1 << 0);
            try testArgs(u16, u63, 1 << 62);
            try testArgs(u32, u63, 0);
            try testArgs(u32, u63, 1 << 0);
            try testArgs(u32, u63, 1 << 62);
            try testArgs(u64, u63, 0);
            try testArgs(u64, u63, 1 << 0);
            try testArgs(u64, u63, 1 << 62);
            try testArgs(u128, u63, 0);
            try testArgs(u128, u63, 1 << 0);
            try testArgs(u128, u63, 1 << 62);
            try testArgs(u256, u63, 0);
            try testArgs(u256, u63, 1 << 0);
            try testArgs(u256, u63, 1 << 62);
            try testArgs(u512, u63, 0);
            try testArgs(u512, u63, 1 << 0);
            try testArgs(u512, u63, 1 << 62);
            try testArgs(u1024, u63, 0);
            try testArgs(u1024, u63, 1 << 0);
            try testArgs(u1024, u63, 1 << 62);

            try testArgs(i8, i64, -1 << 63);
            try testArgs(i8, i64, -1);
            try testArgs(i8, i64, 0);
            try testArgs(i16, i64, -1 << 63);
            try testArgs(i16, i64, -1);
            try testArgs(i16, i64, 0);
            try testArgs(i32, i64, -1 << 63);
            try testArgs(i32, i64, -1);
            try testArgs(i32, i64, 0);
            try testArgs(i64, i64, -1 << 63);
            try testArgs(i64, i64, -1);
            try testArgs(i64, i64, 0);
            try testArgs(i128, i64, -1 << 63);
            try testArgs(i128, i64, -1);
            try testArgs(i128, i64, 0);
            try testArgs(i256, i64, -1 << 63);
            try testArgs(i256, i64, -1);
            try testArgs(i256, i64, 0);
            try testArgs(i512, i64, -1 << 63);
            try testArgs(i512, i64, -1);
            try testArgs(i512, i64, 0);
            try testArgs(i1024, i64, -1 << 63);
            try testArgs(i1024, i64, -1);
            try testArgs(i1024, i64, 0);
            try testArgs(u8, u64, 0);
            try testArgs(u8, u64, 1 << 0);
            try testArgs(u8, u64, 1 << 63);
            try testArgs(u16, u64, 0);
            try testArgs(u16, u64, 1 << 0);
            try testArgs(u16, u64, 1 << 63);
            try testArgs(u32, u64, 0);
            try testArgs(u32, u64, 1 << 0);
            try testArgs(u32, u64, 1 << 63);
            try testArgs(u64, u64, 0);
            try testArgs(u64, u64, 1 << 0);
            try testArgs(u64, u64, 1 << 63);
            try testArgs(u128, u64, 0);
            try testArgs(u128, u64, 1 << 0);
            try testArgs(u128, u64, 1 << 63);
            try testArgs(u256, u64, 0);
            try testArgs(u256, u64, 1 << 0);
            try testArgs(u256, u64, 1 << 63);
            try testArgs(u512, u64, 0);
            try testArgs(u512, u64, 1 << 0);
            try testArgs(u512, u64, 1 << 63);
            try testArgs(u1024, u64, 0);
            try testArgs(u1024, u64, 1 << 0);
            try testArgs(u1024, u64, 1 << 63);

            try testArgs(i8, i65, -1 << 64);
            try testArgs(i8, i65, -1);
            try testArgs(i8, i65, 0);
            try testArgs(i16, i65, -1 << 64);
            try testArgs(i16, i65, -1);
            try testArgs(i16, i65, 0);
            try testArgs(i32, i65, -1 << 64);
            try testArgs(i32, i65, -1);
            try testArgs(i32, i65, 0);
            try testArgs(i64, i65, -1 << 64);
            try testArgs(i64, i65, -1);
            try testArgs(i64, i65, 0);
            try testArgs(i128, i65, -1 << 64);
            try testArgs(i128, i65, -1);
            try testArgs(i128, i65, 0);
            try testArgs(i256, i65, -1 << 64);
            try testArgs(i256, i65, -1);
            try testArgs(i256, i65, 0);
            try testArgs(i512, i65, -1 << 64);
            try testArgs(i512, i65, -1);
            try testArgs(i512, i65, 0);
            try testArgs(i1024, i65, -1 << 64);
            try testArgs(i1024, i65, -1);
            try testArgs(i1024, i65, 0);
            try testArgs(u8, u65, 0);
            try testArgs(u8, u65, 1 << 0);
            try testArgs(u8, u65, 1 << 64);
            try testArgs(u16, u65, 0);
            try testArgs(u16, u65, 1 << 0);
            try testArgs(u16, u65, 1 << 64);
            try testArgs(u32, u65, 0);
            try testArgs(u32, u65, 1 << 0);
            try testArgs(u32, u65, 1 << 64);
            try testArgs(u64, u65, 0);
            try testArgs(u64, u65, 1 << 0);
            try testArgs(u64, u65, 1 << 64);
            try testArgs(u128, u65, 0);
            try testArgs(u128, u65, 1 << 0);
            try testArgs(u128, u65, 1 << 64);
            try testArgs(u256, u65, 0);
            try testArgs(u256, u65, 1 << 0);
            try testArgs(u256, u65, 1 << 64);
            try testArgs(u512, u65, 0);
            try testArgs(u512, u65, 1 << 0);
            try testArgs(u512, u65, 1 << 64);
            try testArgs(u1024, u65, 0);
            try testArgs(u1024, u65, 1 << 0);
            try testArgs(u1024, u65, 1 << 64);

            try testArgs(i8, i95, -1 << 94);
            try testArgs(i8, i95, -1);
            try testArgs(i8, i95, 0);
            try testArgs(i16, i95, -1 << 94);
            try testArgs(i16, i95, -1);
            try testArgs(i16, i95, 0);
            try testArgs(i32, i95, -1 << 94);
            try testArgs(i32, i95, -1);
            try testArgs(i32, i95, 0);
            try testArgs(i64, i95, -1 << 94);
            try testArgs(i64, i95, -1);
            try testArgs(i64, i95, 0);
            try testArgs(i128, i95, -1 << 94);
            try testArgs(i128, i95, -1);
            try testArgs(i128, i95, 0);
            try testArgs(i256, i95, -1 << 94);
            try testArgs(i256, i95, -1);
            try testArgs(i256, i95, 0);
            try testArgs(i512, i95, -1 << 94);
            try testArgs(i512, i95, -1);
            try testArgs(i512, i95, 0);
            try testArgs(i1024, i95, -1 << 94);
            try testArgs(i1024, i95, -1);
            try testArgs(i1024, i95, 0);
            try testArgs(u8, u95, 0);
            try testArgs(u8, u95, 1 << 0);
            try testArgs(u8, u95, 1 << 94);
            try testArgs(u16, u95, 0);
            try testArgs(u16, u95, 1 << 0);
            try testArgs(u16, u95, 1 << 94);
            try testArgs(u32, u95, 0);
            try testArgs(u32, u95, 1 << 0);
            try testArgs(u32, u95, 1 << 94);
            try testArgs(u64, u95, 0);
            try testArgs(u64, u95, 1 << 0);
            try testArgs(u64, u95, 1 << 94);
            try testArgs(u128, u95, 0);
            try testArgs(u128, u95, 1 << 0);
            try testArgs(u128, u95, 1 << 94);
            try testArgs(u256, u95, 0);
            try testArgs(u256, u95, 1 << 0);
            try testArgs(u256, u95, 1 << 94);
            try testArgs(u512, u95, 0);
            try testArgs(u512, u95, 1 << 0);
            try testArgs(u512, u95, 1 << 94);
            try testArgs(u1024, u95, 0);
            try testArgs(u1024, u95, 1 << 0);
            try testArgs(u1024, u95, 1 << 94);

            try testArgs(i8, i97, -1 << 96);
            try testArgs(i8, i97, -1);
            try testArgs(i8, i97, 0);
            try testArgs(i16, i97, -1 << 96);
            try testArgs(i16, i97, -1);
            try testArgs(i16, i97, 0);
            try testArgs(i32, i97, -1 << 96);
            try testArgs(i32, i97, -1);
            try testArgs(i32, i97, 0);
            try testArgs(i64, i97, -1 << 96);
            try testArgs(i64, i97, -1);
            try testArgs(i64, i97, 0);
            try testArgs(i128, i97, -1 << 96);
            try testArgs(i128, i97, -1);
            try testArgs(i128, i97, 0);
            try testArgs(i256, i97, -1 << 96);
            try testArgs(i256, i97, -1);
            try testArgs(i256, i97, 0);
            try testArgs(i512, i97, -1 << 96);
            try testArgs(i512, i97, -1);
            try testArgs(i512, i97, 0);
            try testArgs(i1024, i97, -1 << 96);
            try testArgs(i1024, i97, -1);
            try testArgs(i1024, i97, 0);
            try testArgs(u8, u97, 0);
            try testArgs(u8, u97, 1 << 0);
            try testArgs(u8, u97, 1 << 96);
            try testArgs(u16, u97, 0);
            try testArgs(u16, u97, 1 << 0);
            try testArgs(u16, u97, 1 << 96);
            try testArgs(u32, u97, 0);
            try testArgs(u32, u97, 1 << 0);
            try testArgs(u32, u97, 1 << 96);
            try testArgs(u64, u97, 0);
            try testArgs(u64, u97, 1 << 0);
            try testArgs(u64, u97, 1 << 96);
            try testArgs(u128, u97, 0);
            try testArgs(u128, u97, 1 << 0);
            try testArgs(u128, u97, 1 << 96);
            try testArgs(u256, u97, 0);
            try testArgs(u256, u97, 1 << 0);
            try testArgs(u256, u97, 1 << 96);
            try testArgs(u512, u97, 0);
            try testArgs(u512, u97, 1 << 0);
            try testArgs(u512, u97, 1 << 96);
            try testArgs(u1024, u97, 0);
            try testArgs(u1024, u97, 1 << 0);
            try testArgs(u1024, u97, 1 << 96);

            try testArgs(i8, i127, -1 << 126);
            try testArgs(i8, i127, -1);
            try testArgs(i8, i127, 0);
            try testArgs(i16, i127, -1 << 126);
            try testArgs(i16, i127, -1);
            try testArgs(i16, i127, 0);
            try testArgs(i32, i127, -1 << 126);
            try testArgs(i32, i127, -1);
            try testArgs(i32, i127, 0);
            try testArgs(i64, i127, -1 << 126);
            try testArgs(i64, i127, -1);
            try testArgs(i64, i127, 0);
            try testArgs(i128, i127, -1 << 126);
            try testArgs(i128, i127, -1);
            try testArgs(i128, i127, 0);
            try testArgs(i256, i127, -1 << 126);
            try testArgs(i256, i127, -1);
            try testArgs(i256, i127, 0);
            try testArgs(i512, i127, -1 << 126);
            try testArgs(i512, i127, -1);
            try testArgs(i512, i127, 0);
            try testArgs(i1024, i127, -1 << 126);
            try testArgs(i1024, i127, -1);
            try testArgs(i1024, i127, 0);
            try testArgs(u8, u127, 0);
            try testArgs(u8, u127, 1 << 0);
            try testArgs(u8, u127, 1 << 126);
            try testArgs(u16, u127, 0);
            try testArgs(u16, u127, 1 << 0);
            try testArgs(u16, u127, 1 << 126);
            try testArgs(u32, u127, 0);
            try testArgs(u32, u127, 1 << 0);
            try testArgs(u32, u127, 1 << 126);
            try testArgs(u64, u127, 0);
            try testArgs(u64, u127, 1 << 0);
            try testArgs(u64, u127, 1 << 126);
            try testArgs(u128, u127, 0);
            try testArgs(u128, u127, 1 << 0);
            try testArgs(u128, u127, 1 << 126);
            try testArgs(u256, u127, 0);
            try testArgs(u256, u127, 1 << 0);
            try testArgs(u256, u127, 1 << 126);
            try testArgs(u512, u127, 0);
            try testArgs(u512, u127, 1 << 0);
            try testArgs(u512, u127, 1 << 126);
            try testArgs(u1024, u127, 0);
            try testArgs(u1024, u127, 1 << 0);
            try testArgs(u1024, u127, 1 << 126);

            try testArgs(i8, i128, -1 << 127);
            try testArgs(i8, i128, -1);
            try testArgs(i8, i128, 0);
            try testArgs(i16, i128, -1 << 127);
            try testArgs(i16, i128, -1);
            try testArgs(i16, i128, 0);
            try testArgs(i32, i128, -1 << 127);
            try testArgs(i32, i128, -1);
            try testArgs(i32, i128, 0);
            try testArgs(i64, i128, -1 << 127);
            try testArgs(i64, i128, -1);
            try testArgs(i64, i128, 0);
            try testArgs(i128, i128, -1 << 127);
            try testArgs(i128, i128, -1);
            try testArgs(i128, i128, 0);
            try testArgs(i256, i128, -1 << 127);
            try testArgs(i256, i128, -1);
            try testArgs(i256, i128, 0);
            try testArgs(i512, i128, -1 << 127);
            try testArgs(i512, i128, -1);
            try testArgs(i512, i128, 0);
            try testArgs(i1024, i128, -1 << 127);
            try testArgs(i1024, i128, -1);
            try testArgs(i1024, i128, 0);
            try testArgs(u8, u128, 0);
            try testArgs(u8, u128, 1 << 0);
            try testArgs(u8, u128, 1 << 127);
            try testArgs(u16, u128, 0);
            try testArgs(u16, u128, 1 << 0);
            try testArgs(u16, u128, 1 << 127);
            try testArgs(u32, u128, 0);
            try testArgs(u32, u128, 1 << 0);
            try testArgs(u32, u128, 1 << 127);
            try testArgs(u64, u128, 0);
            try testArgs(u64, u128, 1 << 0);
            try testArgs(u64, u128, 1 << 127);
            try testArgs(u128, u128, 0);
            try testArgs(u128, u128, 1 << 0);
            try testArgs(u128, u128, 1 << 127);
            try testArgs(u256, u128, 0);
            try testArgs(u256, u128, 1 << 0);
            try testArgs(u256, u128, 1 << 127);
            try testArgs(u512, u128, 0);
            try testArgs(u512, u128, 1 << 0);
            try testArgs(u512, u128, 1 << 127);
            try testArgs(u1024, u128, 0);
            try testArgs(u1024, u128, 1 << 0);
            try testArgs(u1024, u128, 1 << 127);

            try testArgs(i8, i129, -1 << 128);
            try testArgs(i8, i129, -1);
            try testArgs(i8, i129, 0);
            try testArgs(i16, i129, -1 << 128);
            try testArgs(i16, i129, -1);
            try testArgs(i16, i129, 0);
            try testArgs(i32, i129, -1 << 128);
            try testArgs(i32, i129, -1);
            try testArgs(i32, i129, 0);
            try testArgs(i64, i129, -1 << 128);
            try testArgs(i64, i129, -1);
            try testArgs(i64, i129, 0);
            try testArgs(i128, i129, -1 << 128);
            try testArgs(i128, i129, -1);
            try testArgs(i128, i129, 0);
            try testArgs(i256, i129, -1 << 128);
            try testArgs(i256, i129, -1);
            try testArgs(i256, i129, 0);
            try testArgs(i512, i129, -1 << 128);
            try testArgs(i512, i129, -1);
            try testArgs(i512, i129, 0);
            try testArgs(i1024, i129, -1 << 128);
            try testArgs(i1024, i129, -1);
            try testArgs(i1024, i129, 0);
            try testArgs(u8, u129, 0);
            try testArgs(u8, u129, 1 << 0);
            try testArgs(u8, u129, 1 << 128);
            try testArgs(u16, u129, 0);
            try testArgs(u16, u129, 1 << 0);
            try testArgs(u16, u129, 1 << 128);
            try testArgs(u32, u129, 0);
            try testArgs(u32, u129, 1 << 0);
            try testArgs(u32, u129, 1 << 128);
            try testArgs(u64, u129, 0);
            try testArgs(u64, u129, 1 << 0);
            try testArgs(u64, u129, 1 << 128);
            try testArgs(u128, u129, 0);
            try testArgs(u128, u129, 1 << 0);
            try testArgs(u128, u129, 1 << 128);
            try testArgs(u256, u129, 0);
            try testArgs(u256, u129, 1 << 0);
            try testArgs(u256, u129, 1 << 128);
            try testArgs(u512, u129, 0);
            try testArgs(u512, u129, 1 << 0);
            try testArgs(u512, u129, 1 << 128);
            try testArgs(u1024, u129, 0);
            try testArgs(u1024, u129, 1 << 0);
            try testArgs(u1024, u129, 1 << 128);

            try testArgs(i8, i255, -1 << 254);
            try testArgs(i8, i255, -1);
            try testArgs(i8, i255, 0);
            try testArgs(i16, i255, -1 << 254);
            try testArgs(i16, i255, -1);
            try testArgs(i16, i255, 0);
            try testArgs(i32, i255, -1 << 254);
            try testArgs(i32, i255, -1);
            try testArgs(i32, i255, 0);
            try testArgs(i64, i255, -1 << 254);
            try testArgs(i64, i255, -1);
            try testArgs(i64, i255, 0);
            try testArgs(i128, i255, -1 << 254);
            try testArgs(i128, i255, -1);
            try testArgs(i128, i255, 0);
            try testArgs(i256, i255, -1 << 254);
            try testArgs(i256, i255, -1);
            try testArgs(i256, i255, 0);
            try testArgs(i512, i255, -1 << 254);
            try testArgs(i512, i255, -1);
            try testArgs(i512, i255, 0);
            try testArgs(i1024, i255, -1 << 254);
            try testArgs(i1024, i255, -1);
            try testArgs(i1024, i255, 0);
            try testArgs(u8, u255, 0);
            try testArgs(u8, u255, 1 << 0);
            try testArgs(u8, u255, 1 << 254);
            try testArgs(u16, u255, 0);
            try testArgs(u16, u255, 1 << 0);
            try testArgs(u16, u255, 1 << 254);
            try testArgs(u32, u255, 0);
            try testArgs(u32, u255, 1 << 0);
            try testArgs(u32, u255, 1 << 254);
            try testArgs(u64, u255, 0);
            try testArgs(u64, u255, 1 << 0);
            try testArgs(u64, u255, 1 << 254);
            try testArgs(u128, u255, 0);
            try testArgs(u128, u255, 1 << 0);
            try testArgs(u128, u255, 1 << 254);
            try testArgs(u256, u255, 0);
            try testArgs(u256, u255, 1 << 0);
            try testArgs(u256, u255, 1 << 254);
            try testArgs(u512, u255, 0);
            try testArgs(u512, u255, 1 << 0);
            try testArgs(u512, u255, 1 << 254);
            try testArgs(u1024, u255, 0);
            try testArgs(u1024, u255, 1 << 0);
            try testArgs(u1024, u255, 1 << 254);

            try testArgs(i8, i256, -1 << 255);
            try testArgs(i8, i256, -1);
            try testArgs(i8, i256, 0);
            try testArgs(i16, i256, -1 << 255);
            try testArgs(i16, i256, -1);
            try testArgs(i16, i256, 0);
            try testArgs(i32, i256, -1 << 255);
            try testArgs(i32, i256, -1);
            try testArgs(i32, i256, 0);
            try testArgs(i64, i256, -1 << 255);
            try testArgs(i64, i256, -1);
            try testArgs(i64, i256, 0);
            try testArgs(i128, i256, -1 << 255);
            try testArgs(i128, i256, -1);
            try testArgs(i128, i256, 0);
            try testArgs(i256, i256, -1 << 255);
            try testArgs(i256, i256, -1);
            try testArgs(i256, i256, 0);
            try testArgs(i512, i256, -1 << 255);
            try testArgs(i512, i256, -1);
            try testArgs(i512, i256, 0);
            try testArgs(i1024, i256, -1 << 255);
            try testArgs(i1024, i256, -1);
            try testArgs(i1024, i256, 0);
            try testArgs(u8, u256, 0);
            try testArgs(u8, u256, 1 << 0);
            try testArgs(u8, u256, 1 << 255);
            try testArgs(u16, u256, 0);
            try testArgs(u16, u256, 1 << 0);
            try testArgs(u16, u256, 1 << 255);
            try testArgs(u32, u256, 0);
            try testArgs(u32, u256, 1 << 0);
            try testArgs(u32, u256, 1 << 255);
            try testArgs(u64, u256, 0);
            try testArgs(u64, u256, 1 << 0);
            try testArgs(u64, u256, 1 << 255);
            try testArgs(u128, u256, 0);
            try testArgs(u128, u256, 1 << 0);
            try testArgs(u128, u256, 1 << 255);
            try testArgs(u256, u256, 0);
            try testArgs(u256, u256, 1 << 0);
            try testArgs(u256, u256, 1 << 255);
            try testArgs(u512, u256, 0);
            try testArgs(u512, u256, 1 << 0);
            try testArgs(u512, u256, 1 << 255);
            try testArgs(u1024, u256, 0);
            try testArgs(u1024, u256, 1 << 0);
            try testArgs(u1024, u256, 1 << 255);

            try testArgs(i8, i257, -1 << 256);
            try testArgs(i8, i257, -1);
            try testArgs(i8, i257, 0);
            try testArgs(i16, i257, -1 << 256);
            try testArgs(i16, i257, -1);
            try testArgs(i16, i257, 0);
            try testArgs(i32, i257, -1 << 256);
            try testArgs(i32, i257, -1);
            try testArgs(i32, i257, 0);
            try testArgs(i64, i257, -1 << 256);
            try testArgs(i64, i257, -1);
            try testArgs(i64, i257, 0);
            try testArgs(i128, i257, -1 << 256);
            try testArgs(i128, i257, -1);
            try testArgs(i128, i257, 0);
            try testArgs(i256, i257, -1 << 256);
            try testArgs(i256, i257, -1);
            try testArgs(i256, i257, 0);
            try testArgs(i512, i257, -1 << 256);
            try testArgs(i512, i257, -1);
            try testArgs(i512, i257, 0);
            try testArgs(i1024, i257, -1 << 256);
            try testArgs(i1024, i257, -1);
            try testArgs(i1024, i257, 0);
            try testArgs(u8, u257, 0);
            try testArgs(u8, u257, 1 << 0);
            try testArgs(u8, u257, 1 << 256);
            try testArgs(u16, u257, 0);
            try testArgs(u16, u257, 1 << 0);
            try testArgs(u16, u257, 1 << 256);
            try testArgs(u32, u257, 0);
            try testArgs(u32, u257, 1 << 0);
            try testArgs(u32, u257, 1 << 256);
            try testArgs(u64, u257, 0);
            try testArgs(u64, u257, 1 << 0);
            try testArgs(u64, u257, 1 << 256);
            try testArgs(u128, u257, 0);
            try testArgs(u128, u257, 1 << 0);
            try testArgs(u128, u257, 1 << 256);
            try testArgs(u256, u257, 0);
            try testArgs(u256, u257, 1 << 0);
            try testArgs(u256, u257, 1 << 256);
            try testArgs(u512, u257, 0);
            try testArgs(u512, u257, 1 << 0);
            try testArgs(u512, u257, 1 << 256);
            try testArgs(u1024, u257, 0);
            try testArgs(u1024, u257, 1 << 0);
            try testArgs(u1024, u257, 1 << 256);

            try testArgs(i8, i511, -1 << 510);
            try testArgs(i8, i511, -1);
            try testArgs(i8, i511, 0);
            try testArgs(i16, i511, -1 << 510);
            try testArgs(i16, i511, -1);
            try testArgs(i16, i511, 0);
            try testArgs(i32, i511, -1 << 510);
            try testArgs(i32, i511, -1);
            try testArgs(i32, i511, 0);
            try testArgs(i64, i511, -1 << 510);
            try testArgs(i64, i511, -1);
            try testArgs(i64, i511, 0);
            try testArgs(i128, i511, -1 << 510);
            try testArgs(i128, i511, -1);
            try testArgs(i128, i511, 0);
            try testArgs(i256, i511, -1 << 510);
            try testArgs(i256, i511, -1);
            try testArgs(i256, i511, 0);
            try testArgs(i512, i511, -1 << 510);
            try testArgs(i512, i511, -1);
            try testArgs(i512, i511, 0);
            try testArgs(i1024, i511, -1 << 510);
            try testArgs(i1024, i511, -1);
            try testArgs(i1024, i511, 0);
            try testArgs(u8, u511, 0);
            try testArgs(u8, u511, 1 << 0);
            try testArgs(u8, u511, 1 << 510);
            try testArgs(u16, u511, 0);
            try testArgs(u16, u511, 1 << 0);
            try testArgs(u16, u511, 1 << 510);
            try testArgs(u32, u511, 0);
            try testArgs(u32, u511, 1 << 0);
            try testArgs(u32, u511, 1 << 510);
            try testArgs(u64, u511, 0);
            try testArgs(u64, u511, 1 << 0);
            try testArgs(u64, u511, 1 << 510);
            try testArgs(u128, u511, 0);
            try testArgs(u128, u511, 1 << 0);
            try testArgs(u128, u511, 1 << 510);
            try testArgs(u256, u511, 0);
            try testArgs(u256, u511, 1 << 0);
            try testArgs(u256, u511, 1 << 510);
            try testArgs(u512, u511, 0);
            try testArgs(u512, u511, 1 << 0);
            try testArgs(u512, u511, 1 << 510);
            try testArgs(u1024, u511, 0);
            try testArgs(u1024, u511, 1 << 0);
            try testArgs(u1024, u511, 1 << 510);

            try testArgs(i8, i512, -1 << 511);
            try testArgs(i8, i512, -1);
            try testArgs(i8, i512, 0);
            try testArgs(i16, i512, -1 << 511);
            try testArgs(i16, i512, -1);
            try testArgs(i16, i512, 0);
            try testArgs(i32, i512, -1 << 511);
            try testArgs(i32, i512, -1);
            try testArgs(i32, i512, 0);
            try testArgs(i64, i512, -1 << 511);
            try testArgs(i64, i512, -1);
            try testArgs(i64, i512, 0);
            try testArgs(i128, i512, -1 << 511);
            try testArgs(i128, i512, -1);
            try testArgs(i128, i512, 0);
            try testArgs(i256, i512, -1 << 511);
            try testArgs(i256, i512, -1);
            try testArgs(i256, i512, 0);
            try testArgs(i512, i512, -1 << 511);
            try testArgs(i512, i512, -1);
            try testArgs(i512, i512, 0);
            try testArgs(i1024, i512, -1 << 511);
            try testArgs(i1024, i512, -1);
            try testArgs(i1024, i512, 0);
            try testArgs(u8, u512, 0);
            try testArgs(u8, u512, 1 << 0);
            try testArgs(u8, u512, 1 << 511);
            try testArgs(u16, u512, 0);
            try testArgs(u16, u512, 1 << 0);
            try testArgs(u16, u512, 1 << 511);
            try testArgs(u32, u512, 0);
            try testArgs(u32, u512, 1 << 0);
            try testArgs(u32, u512, 1 << 511);
            try testArgs(u64, u512, 0);
            try testArgs(u64, u512, 1 << 0);
            try testArgs(u64, u512, 1 << 511);
            try testArgs(u128, u512, 0);
            try testArgs(u128, u512, 1 << 0);
            try testArgs(u128, u512, 1 << 511);
            try testArgs(u256, u512, 0);
            try testArgs(u256, u512, 1 << 0);
            try testArgs(u256, u512, 1 << 511);
            try testArgs(u512, u512, 0);
            try testArgs(u512, u512, 1 << 0);
            try testArgs(u512, u512, 1 << 511);
            try testArgs(u1024, u512, 0);
            try testArgs(u1024, u512, 1 << 0);
            try testArgs(u1024, u512, 1 << 511);

            try testArgs(i8, i513, -1 << 512);
            try testArgs(i8, i513, -1);
            try testArgs(i8, i513, 0);
            try testArgs(i16, i513, -1 << 512);
            try testArgs(i16, i513, -1);
            try testArgs(i16, i513, 0);
            try testArgs(i32, i513, -1 << 512);
            try testArgs(i32, i513, -1);
            try testArgs(i32, i513, 0);
            try testArgs(i64, i513, -1 << 512);
            try testArgs(i64, i513, -1);
            try testArgs(i64, i513, 0);
            try testArgs(i128, i513, -1 << 512);
            try testArgs(i128, i513, -1);
            try testArgs(i128, i513, 0);
            try testArgs(i256, i513, -1 << 512);
            try testArgs(i256, i513, -1);
            try testArgs(i256, i513, 0);
            try testArgs(i512, i513, -1 << 512);
            try testArgs(i512, i513, -1);
            try testArgs(i512, i513, 0);
            try testArgs(i1024, i513, -1 << 512);
            try testArgs(i1024, i513, -1);
            try testArgs(i1024, i513, 0);
            try testArgs(u8, u513, 0);
            try testArgs(u8, u513, 1 << 0);
            try testArgs(u8, u513, 1 << 512);
            try testArgs(u16, u513, 0);
            try testArgs(u16, u513, 1 << 0);
            try testArgs(u16, u513, 1 << 512);
            try testArgs(u32, u513, 0);
            try testArgs(u32, u513, 1 << 0);
            try testArgs(u32, u513, 1 << 512);
            try testArgs(u64, u513, 0);
            try testArgs(u64, u513, 1 << 0);
            try testArgs(u64, u513, 1 << 512);
            try testArgs(u128, u513, 0);
            try testArgs(u128, u513, 1 << 0);
            try testArgs(u128, u513, 1 << 512);
            try testArgs(u256, u513, 0);
            try testArgs(u256, u513, 1 << 0);
            try testArgs(u256, u513, 1 << 512);
            try testArgs(u512, u513, 0);
            try testArgs(u512, u513, 1 << 0);
            try testArgs(u512, u513, 1 << 512);
            try testArgs(u1024, u513, 0);
            try testArgs(u1024, u513, 1 << 0);
            try testArgs(u1024, u513, 1 << 512);

            try testArgs(i8, i1023, -1 << 1022);
            try testArgs(i8, i1023, -1);
            try testArgs(i8, i1023, 0);
            try testArgs(i16, i1023, -1 << 1022);
            try testArgs(i16, i1023, -1);
            try testArgs(i16, i1023, 0);
            try testArgs(i32, i1023, -1 << 1022);
            try testArgs(i32, i1023, -1);
            try testArgs(i32, i1023, 0);
            try testArgs(i64, i1023, -1 << 1022);
            try testArgs(i64, i1023, -1);
            try testArgs(i64, i1023, 0);
            try testArgs(i128, i1023, -1 << 1022);
            try testArgs(i128, i1023, -1);
            try testArgs(i128, i1023, 0);
            try testArgs(i256, i1023, -1 << 1022);
            try testArgs(i256, i1023, -1);
            try testArgs(i256, i1023, 0);
            try testArgs(i512, i1023, -1 << 1022);
            try testArgs(i512, i1023, -1);
            try testArgs(i512, i1023, 0);
            try testArgs(i1024, i1023, -1 << 1022);
            try testArgs(i1024, i1023, -1);
            try testArgs(i1024, i1023, 0);
            try testArgs(u8, u1023, 0);
            try testArgs(u8, u1023, 1 << 0);
            try testArgs(u8, u1023, 1 << 1022);
            try testArgs(u16, u1023, 0);
            try testArgs(u16, u1023, 1 << 0);
            try testArgs(u16, u1023, 1 << 1022);
            try testArgs(u32, u1023, 0);
            try testArgs(u32, u1023, 1 << 0);
            try testArgs(u32, u1023, 1 << 1022);
            try testArgs(u64, u1023, 0);
            try testArgs(u64, u1023, 1 << 0);
            try testArgs(u64, u1023, 1 << 1022);
            try testArgs(u128, u1023, 0);
            try testArgs(u128, u1023, 1 << 0);
            try testArgs(u128, u1023, 1 << 1022);
            try testArgs(u256, u1023, 0);
            try testArgs(u256, u1023, 1 << 0);
            try testArgs(u256, u1023, 1 << 1022);
            try testArgs(u512, u1023, 0);
            try testArgs(u512, u1023, 1 << 0);
            try testArgs(u512, u1023, 1 << 1022);
            try testArgs(u1024, u1023, 0);
            try testArgs(u1024, u1023, 1 << 0);
            try testArgs(u1024, u1023, 1 << 1022);

            try testArgs(i8, i1024, -1 << 1023);
            try testArgs(i8, i1024, -1);
            try testArgs(i8, i1024, 0);
            try testArgs(i16, i1024, -1 << 1023);
            try testArgs(i16, i1024, -1);
            try testArgs(i16, i1024, 0);
            try testArgs(i32, i1024, -1 << 1023);
            try testArgs(i32, i1024, -1);
            try testArgs(i32, i1024, 0);
            try testArgs(i64, i1024, -1 << 1023);
            try testArgs(i64, i1024, -1);
            try testArgs(i64, i1024, 0);
            try testArgs(i128, i1024, -1 << 1023);
            try testArgs(i128, i1024, -1);
            try testArgs(i128, i1024, 0);
            try testArgs(i256, i1024, -1 << 1023);
            try testArgs(i256, i1024, -1);
            try testArgs(i256, i1024, 0);
            try testArgs(i512, i1024, -1 << 1023);
            try testArgs(i512, i1024, -1);
            try testArgs(i512, i1024, 0);
            try testArgs(i1024, i1024, -1 << 1023);
            try testArgs(i1024, i1024, -1);
            try testArgs(i1024, i1024, 0);
            try testArgs(u8, u1024, 0);
            try testArgs(u8, u1024, 1 << 0);
            try testArgs(u8, u1024, 1 << 1023);
            try testArgs(u16, u1024, 0);
            try testArgs(u16, u1024, 1 << 0);
            try testArgs(u16, u1024, 1 << 1023);
            try testArgs(u32, u1024, 0);
            try testArgs(u32, u1024, 1 << 0);
            try testArgs(u32, u1024, 1 << 1023);
            try testArgs(u64, u1024, 0);
            try testArgs(u64, u1024, 1 << 0);
            try testArgs(u64, u1024, 1 << 1023);
            try testArgs(u128, u1024, 0);
            try testArgs(u128, u1024, 1 << 0);
            try testArgs(u128, u1024, 1 << 1023);
            try testArgs(u256, u1024, 0);
            try testArgs(u256, u1024, 1 << 0);
            try testArgs(u256, u1024, 1 << 1023);
            try testArgs(u512, u1024, 0);
            try testArgs(u512, u1024, 1 << 0);
            try testArgs(u512, u1024, 1 << 1023);
            try testArgs(u1024, u1024, 0);
            try testArgs(u1024, u1024, 1 << 0);
            try testArgs(u1024, u1024, 1 << 1023);

            try testArgs(i8, i1025, -1 << 1024);
            try testArgs(i8, i1025, -1);
            try testArgs(i8, i1025, 0);
            try testArgs(i16, i1025, -1 << 1024);
            try testArgs(i16, i1025, -1);
            try testArgs(i16, i1025, 0);
            try testArgs(i32, i1025, -1 << 1024);
            try testArgs(i32, i1025, -1);
            try testArgs(i32, i1025, 0);
            try testArgs(i64, i1025, -1 << 1024);
            try testArgs(i64, i1025, -1);
            try testArgs(i64, i1025, 0);
            try testArgs(i128, i1025, -1 << 1024);
            try testArgs(i128, i1025, -1);
            try testArgs(i128, i1025, 0);
            try testArgs(i256, i1025, -1 << 1024);
            try testArgs(i256, i1025, -1);
            try testArgs(i256, i1025, 0);
            try testArgs(i512, i1025, -1 << 1024);
            try testArgs(i512, i1025, -1);
            try testArgs(i512, i1025, 0);
            try testArgs(i1024, i1025, -1 << 1024);
            try testArgs(i1024, i1025, -1);
            try testArgs(i1024, i1025, 0);
            try testArgs(u8, u1025, 0);
            try testArgs(u8, u1025, 1 << 0);
            try testArgs(u8, u1025, 1 << 1024);
            try testArgs(u16, u1025, 0);
            try testArgs(u16, u1025, 1 << 0);
            try testArgs(u16, u1025, 1 << 1024);
            try testArgs(u32, u1025, 0);
            try testArgs(u32, u1025, 1 << 0);
            try testArgs(u32, u1025, 1 << 1024);
            try testArgs(u64, u1025, 0);
            try testArgs(u64, u1025, 1 << 0);
            try testArgs(u64, u1025, 1 << 1024);
            try testArgs(u128, u1025, 0);
            try testArgs(u128, u1025, 1 << 0);
            try testArgs(u128, u1025, 1 << 1024);
            try testArgs(u256, u1025, 0);
            try testArgs(u256, u1025, 1 << 0);
            try testArgs(u256, u1025, 1 << 1024);
            try testArgs(u512, u1025, 0);
            try testArgs(u512, u1025, 1 << 0);
            try testArgs(u512, u1025, 1 << 1024);
            try testArgs(u1024, u1025, 0);
            try testArgs(u1024, u1025, 1 << 0);
            try testArgs(u1024, u1025, 1 << 1024);
        }
        fn testInts() !void {
            try testSameSignednessInts();

            try testArgs(u8, i1, -1);
            try testArgs(u8, i1, 0);
            try testArgs(u16, i1, -1);
            try testArgs(u16, i1, 0);
            try testArgs(u32, i1, -1);
            try testArgs(u32, i1, 0);
            try testArgs(u64, i1, -1);
            try testArgs(u64, i1, 0);
            try testArgs(u128, i1, -1);
            try testArgs(u128, i1, 0);
            try testArgs(u256, i1, -1);
            try testArgs(u256, i1, 0);
            try testArgs(u512, i1, -1);
            try testArgs(u512, i1, 0);
            try testArgs(u1024, i1, -1);
            try testArgs(u1024, i1, 0);
            try testArgs(i8, u1, 0);
            try testArgs(i8, u1, 1 << 0);
            try testArgs(i16, u1, 0);
            try testArgs(i16, u1, 1 << 0);
            try testArgs(i32, u1, 0);
            try testArgs(i32, u1, 1 << 0);
            try testArgs(i64, u1, 0);
            try testArgs(i64, u1, 1 << 0);
            try testArgs(i128, u1, 0);
            try testArgs(i128, u1, 1 << 0);
            try testArgs(i256, u1, 0);
            try testArgs(i256, u1, 1 << 0);
            try testArgs(i512, u1, 0);
            try testArgs(i512, u1, 1 << 0);
            try testArgs(i1024, u1, 0);
            try testArgs(i1024, u1, 1 << 0);

            try testArgs(u8, i2, -1 << 1);
            try testArgs(u8, i2, -1);
            try testArgs(u8, i2, 0);
            try testArgs(u16, i2, -1 << 1);
            try testArgs(u16, i2, -1);
            try testArgs(u16, i2, 0);
            try testArgs(u32, i2, -1 << 1);
            try testArgs(u32, i2, -1);
            try testArgs(u32, i2, 0);
            try testArgs(u64, i2, -1 << 1);
            try testArgs(u64, i2, -1);
            try testArgs(u64, i2, 0);
            try testArgs(u128, i2, -1 << 1);
            try testArgs(u128, i2, -1);
            try testArgs(u128, i2, 0);
            try testArgs(u256, i2, -1 << 1);
            try testArgs(u256, i2, -1);
            try testArgs(u256, i2, 0);
            try testArgs(u512, i2, -1 << 1);
            try testArgs(u512, i2, -1);
            try testArgs(u512, i2, 0);
            try testArgs(u1024, i2, -1 << 1);
            try testArgs(u1024, i2, -1);
            try testArgs(u1024, i2, 0);
            try testArgs(i8, u2, 0);
            try testArgs(i8, u2, 1 << 0);
            try testArgs(i8, u2, 1 << 1);
            try testArgs(i16, u2, 0);
            try testArgs(i16, u2, 1 << 0);
            try testArgs(i16, u2, 1 << 1);
            try testArgs(i32, u2, 0);
            try testArgs(i32, u2, 1 << 0);
            try testArgs(i32, u2, 1 << 1);
            try testArgs(i64, u2, 0);
            try testArgs(i64, u2, 1 << 0);
            try testArgs(i64, u2, 1 << 1);
            try testArgs(i128, u2, 0);
            try testArgs(i128, u2, 1 << 0);
            try testArgs(i128, u2, 1 << 1);
            try testArgs(i256, u2, 0);
            try testArgs(i256, u2, 1 << 0);
            try testArgs(i256, u2, 1 << 1);
            try testArgs(i512, u2, 0);
            try testArgs(i512, u2, 1 << 0);
            try testArgs(i512, u2, 1 << 1);
            try testArgs(i1024, u2, 0);
            try testArgs(i1024, u2, 1 << 0);
            try testArgs(i1024, u2, 1 << 1);

            try testArgs(u8, i3, -1 << 2);
            try testArgs(u8, i3, -1);
            try testArgs(u8, i3, 0);
            try testArgs(u16, i3, -1 << 2);
            try testArgs(u16, i3, -1);
            try testArgs(u16, i3, 0);
            try testArgs(u32, i3, -1 << 2);
            try testArgs(u32, i3, -1);
            try testArgs(u32, i3, 0);
            try testArgs(u64, i3, -1 << 2);
            try testArgs(u64, i3, -1);
            try testArgs(u64, i3, 0);
            try testArgs(u128, i3, -1 << 2);
            try testArgs(u128, i3, -1);
            try testArgs(u128, i3, 0);
            try testArgs(u256, i3, -1 << 2);
            try testArgs(u256, i3, -1);
            try testArgs(u256, i3, 0);
            try testArgs(u512, i3, -1 << 2);
            try testArgs(u512, i3, -1);
            try testArgs(u512, i3, 0);
            try testArgs(u1024, i3, -1 << 2);
            try testArgs(u1024, i3, -1);
            try testArgs(u1024, i3, 0);
            try testArgs(i8, u3, 0);
            try testArgs(i8, u3, 1 << 0);
            try testArgs(i8, u3, 1 << 2);
            try testArgs(i16, u3, 0);
            try testArgs(i16, u3, 1 << 0);
            try testArgs(i16, u3, 1 << 2);
            try testArgs(i32, u3, 0);
            try testArgs(i32, u3, 1 << 0);
            try testArgs(i32, u3, 1 << 2);
            try testArgs(i64, u3, 0);
            try testArgs(i64, u3, 1 << 0);
            try testArgs(i64, u3, 1 << 2);
            try testArgs(i128, u3, 0);
            try testArgs(i128, u3, 1 << 0);
            try testArgs(i128, u3, 1 << 2);
            try testArgs(i256, u3, 0);
            try testArgs(i256, u3, 1 << 0);
            try testArgs(i256, u3, 1 << 2);
            try testArgs(i512, u3, 0);
            try testArgs(i512, u3, 1 << 0);
            try testArgs(i512, u3, 1 << 2);
            try testArgs(i1024, u3, 0);
            try testArgs(i1024, u3, 1 << 0);
            try testArgs(i1024, u3, 1 << 2);

            try testArgs(u8, i4, -1 << 3);
            try testArgs(u8, i4, -1);
            try testArgs(u8, i4, 0);
            try testArgs(u16, i4, -1 << 3);
            try testArgs(u16, i4, -1);
            try testArgs(u16, i4, 0);
            try testArgs(u32, i4, -1 << 3);
            try testArgs(u32, i4, -1);
            try testArgs(u32, i4, 0);
            try testArgs(u64, i4, -1 << 3);
            try testArgs(u64, i4, -1);
            try testArgs(u64, i4, 0);
            try testArgs(u128, i4, -1 << 3);
            try testArgs(u128, i4, -1);
            try testArgs(u128, i4, 0);
            try testArgs(u256, i4, -1 << 3);
            try testArgs(u256, i4, -1);
            try testArgs(u256, i4, 0);
            try testArgs(u512, i4, -1 << 3);
            try testArgs(u512, i4, -1);
            try testArgs(u512, i4, 0);
            try testArgs(u1024, i4, -1 << 3);
            try testArgs(u1024, i4, -1);
            try testArgs(u1024, i4, 0);
            try testArgs(i8, u4, 0);
            try testArgs(i8, u4, 1 << 0);
            try testArgs(i8, u4, 1 << 3);
            try testArgs(i16, u4, 0);
            try testArgs(i16, u4, 1 << 0);
            try testArgs(i16, u4, 1 << 3);
            try testArgs(i32, u4, 0);
            try testArgs(i32, u4, 1 << 0);
            try testArgs(i32, u4, 1 << 3);
            try testArgs(i64, u4, 0);
            try testArgs(i64, u4, 1 << 0);
            try testArgs(i64, u4, 1 << 3);
            try testArgs(i128, u4, 0);
            try testArgs(i128, u4, 1 << 0);
            try testArgs(i128, u4, 1 << 3);
            try testArgs(i256, u4, 0);
            try testArgs(i256, u4, 1 << 0);
            try testArgs(i256, u4, 1 << 3);
            try testArgs(i512, u4, 0);
            try testArgs(i512, u4, 1 << 0);
            try testArgs(i512, u4, 1 << 3);
            try testArgs(i1024, u4, 0);
            try testArgs(i1024, u4, 1 << 0);
            try testArgs(i1024, u4, 1 << 3);

            try testArgs(u8, i5, -1 << 4);
            try testArgs(u8, i5, -1);
            try testArgs(u8, i5, 0);
            try testArgs(u16, i5, -1 << 4);
            try testArgs(u16, i5, -1);
            try testArgs(u16, i5, 0);
            try testArgs(u32, i5, -1 << 4);
            try testArgs(u32, i5, -1);
            try testArgs(u32, i5, 0);
            try testArgs(u64, i5, -1 << 4);
            try testArgs(u64, i5, -1);
            try testArgs(u64, i5, 0);
            try testArgs(u128, i5, -1 << 4);
            try testArgs(u128, i5, -1);
            try testArgs(u128, i5, 0);
            try testArgs(u256, i5, -1 << 4);
            try testArgs(u256, i5, -1);
            try testArgs(u256, i5, 0);
            try testArgs(u512, i5, -1 << 4);
            try testArgs(u512, i5, -1);
            try testArgs(u512, i5, 0);
            try testArgs(u1024, i5, -1 << 4);
            try testArgs(u1024, i5, -1);
            try testArgs(u1024, i5, 0);
            try testArgs(i8, u5, 0);
            try testArgs(i8, u5, 1 << 0);
            try testArgs(i8, u5, 1 << 4);
            try testArgs(i16, u5, 0);
            try testArgs(i16, u5, 1 << 0);
            try testArgs(i16, u5, 1 << 4);
            try testArgs(i32, u5, 0);
            try testArgs(i32, u5, 1 << 0);
            try testArgs(i32, u5, 1 << 4);
            try testArgs(i64, u5, 0);
            try testArgs(i64, u5, 1 << 0);
            try testArgs(i64, u5, 1 << 4);
            try testArgs(i128, u5, 0);
            try testArgs(i128, u5, 1 << 0);
            try testArgs(i128, u5, 1 << 4);
            try testArgs(i256, u5, 0);
            try testArgs(i256, u5, 1 << 0);
            try testArgs(i256, u5, 1 << 4);
            try testArgs(i512, u5, 0);
            try testArgs(i512, u5, 1 << 0);
            try testArgs(i512, u5, 1 << 4);
            try testArgs(i1024, u5, 0);
            try testArgs(i1024, u5, 1 << 0);
            try testArgs(i1024, u5, 1 << 4);

            try testArgs(u8, i7, -1 << 6);
            try testArgs(u8, i7, -1);
            try testArgs(u8, i7, 0);
            try testArgs(u16, i7, -1 << 6);
            try testArgs(u16, i7, -1);
            try testArgs(u16, i7, 0);
            try testArgs(u32, i7, -1 << 6);
            try testArgs(u32, i7, -1);
            try testArgs(u32, i7, 0);
            try testArgs(u64, i7, -1 << 6);
            try testArgs(u64, i7, -1);
            try testArgs(u64, i7, 0);
            try testArgs(u128, i7, -1 << 6);
            try testArgs(u128, i7, -1);
            try testArgs(u128, i7, 0);
            try testArgs(u256, i7, -1 << 6);
            try testArgs(u256, i7, -1);
            try testArgs(u256, i7, 0);
            try testArgs(u512, i7, -1 << 6);
            try testArgs(u512, i7, -1);
            try testArgs(u512, i7, 0);
            try testArgs(u1024, i7, -1 << 6);
            try testArgs(u1024, i7, -1);
            try testArgs(u1024, i7, 0);
            try testArgs(i8, u7, 0);
            try testArgs(i8, u7, 1 << 0);
            try testArgs(i8, u7, 1 << 6);
            try testArgs(i16, u7, 0);
            try testArgs(i16, u7, 1 << 0);
            try testArgs(i16, u7, 1 << 6);
            try testArgs(i32, u7, 0);
            try testArgs(i32, u7, 1 << 0);
            try testArgs(i32, u7, 1 << 6);
            try testArgs(i64, u7, 0);
            try testArgs(i64, u7, 1 << 0);
            try testArgs(i64, u7, 1 << 6);
            try testArgs(i128, u7, 0);
            try testArgs(i128, u7, 1 << 0);
            try testArgs(i128, u7, 1 << 6);
            try testArgs(i256, u7, 0);
            try testArgs(i256, u7, 1 << 0);
            try testArgs(i256, u7, 1 << 6);
            try testArgs(i512, u7, 0);
            try testArgs(i512, u7, 1 << 0);
            try testArgs(i512, u7, 1 << 6);
            try testArgs(i1024, u7, 0);
            try testArgs(i1024, u7, 1 << 0);
            try testArgs(i1024, u7, 1 << 6);

            try testArgs(u8, i8, -1 << 7);
            try testArgs(u8, i8, -1);
            try testArgs(u8, i8, 0);
            try testArgs(u16, i8, -1 << 7);
            try testArgs(u16, i8, -1);
            try testArgs(u16, i8, 0);
            try testArgs(u32, i8, -1 << 7);
            try testArgs(u32, i8, -1);
            try testArgs(u32, i8, 0);
            try testArgs(u64, i8, -1 << 7);
            try testArgs(u64, i8, -1);
            try testArgs(u64, i8, 0);
            try testArgs(u128, i8, -1 << 7);
            try testArgs(u128, i8, -1);
            try testArgs(u128, i8, 0);
            try testArgs(u256, i8, -1 << 7);
            try testArgs(u256, i8, -1);
            try testArgs(u256, i8, 0);
            try testArgs(u512, i8, -1 << 7);
            try testArgs(u512, i8, -1);
            try testArgs(u512, i8, 0);
            try testArgs(u1024, i8, -1 << 7);
            try testArgs(u1024, i8, -1);
            try testArgs(u1024, i8, 0);
            try testArgs(i8, u8, 0);
            try testArgs(i8, u8, 1 << 0);
            try testArgs(i8, u8, 1 << 7);
            try testArgs(i16, u8, 0);
            try testArgs(i16, u8, 1 << 0);
            try testArgs(i16, u8, 1 << 7);
            try testArgs(i32, u8, 0);
            try testArgs(i32, u8, 1 << 0);
            try testArgs(i32, u8, 1 << 7);
            try testArgs(i64, u8, 0);
            try testArgs(i64, u8, 1 << 0);
            try testArgs(i64, u8, 1 << 7);
            try testArgs(i128, u8, 0);
            try testArgs(i128, u8, 1 << 0);
            try testArgs(i128, u8, 1 << 7);
            try testArgs(i256, u8, 0);
            try testArgs(i256, u8, 1 << 0);
            try testArgs(i256, u8, 1 << 7);
            try testArgs(i512, u8, 0);
            try testArgs(i512, u8, 1 << 0);
            try testArgs(i512, u8, 1 << 7);
            try testArgs(i1024, u8, 0);
            try testArgs(i1024, u8, 1 << 0);
            try testArgs(i1024, u8, 1 << 7);

            try testArgs(u8, i9, -1 << 8);
            try testArgs(u8, i9, -1);
            try testArgs(u8, i9, 0);
            try testArgs(u16, i9, -1 << 8);
            try testArgs(u16, i9, -1);
            try testArgs(u16, i9, 0);
            try testArgs(u32, i9, -1 << 8);
            try testArgs(u32, i9, -1);
            try testArgs(u32, i9, 0);
            try testArgs(u64, i9, -1 << 8);
            try testArgs(u64, i9, -1);
            try testArgs(u64, i9, 0);
            try testArgs(u128, i9, -1 << 8);
            try testArgs(u128, i9, -1);
            try testArgs(u128, i9, 0);
            try testArgs(u256, i9, -1 << 8);
            try testArgs(u256, i9, -1);
            try testArgs(u256, i9, 0);
            try testArgs(u512, i9, -1 << 8);
            try testArgs(u512, i9, -1);
            try testArgs(u512, i9, 0);
            try testArgs(u1024, i9, -1 << 8);
            try testArgs(u1024, i9, -1);
            try testArgs(u1024, i9, 0);
            try testArgs(i8, u9, 0);
            try testArgs(i8, u9, 1 << 0);
            try testArgs(i8, u9, 1 << 8);
            try testArgs(i16, u9, 0);
            try testArgs(i16, u9, 1 << 0);
            try testArgs(i16, u9, 1 << 8);
            try testArgs(i32, u9, 0);
            try testArgs(i32, u9, 1 << 0);
            try testArgs(i32, u9, 1 << 8);
            try testArgs(i64, u9, 0);
            try testArgs(i64, u9, 1 << 0);
            try testArgs(i64, u9, 1 << 8);
            try testArgs(i128, u9, 0);
            try testArgs(i128, u9, 1 << 0);
            try testArgs(i128, u9, 1 << 8);
            try testArgs(i256, u9, 0);
            try testArgs(i256, u9, 1 << 0);
            try testArgs(i256, u9, 1 << 8);
            try testArgs(i512, u9, 0);
            try testArgs(i512, u9, 1 << 0);
            try testArgs(i512, u9, 1 << 8);
            try testArgs(i1024, u9, 0);
            try testArgs(i1024, u9, 1 << 0);
            try testArgs(i1024, u9, 1 << 8);

            try testArgs(u8, i15, -1 << 14);
            try testArgs(u8, i15, -1);
            try testArgs(u8, i15, 0);
            try testArgs(u16, i15, -1 << 14);
            try testArgs(u16, i15, -1);
            try testArgs(u16, i15, 0);
            try testArgs(u32, i15, -1 << 14);
            try testArgs(u32, i15, -1);
            try testArgs(u32, i15, 0);
            try testArgs(u64, i15, -1 << 14);
            try testArgs(u64, i15, -1);
            try testArgs(u64, i15, 0);
            try testArgs(u128, i15, -1 << 14);
            try testArgs(u128, i15, -1);
            try testArgs(u128, i15, 0);
            try testArgs(u256, i15, -1 << 14);
            try testArgs(u256, i15, -1);
            try testArgs(u256, i15, 0);
            try testArgs(u512, i15, -1 << 14);
            try testArgs(u512, i15, -1);
            try testArgs(u512, i15, 0);
            try testArgs(u1024, i15, -1 << 14);
            try testArgs(u1024, i15, -1);
            try testArgs(u1024, i15, 0);
            try testArgs(i8, u15, 0);
            try testArgs(i8, u15, 1 << 0);
            try testArgs(i8, u15, 1 << 14);
            try testArgs(i16, u15, 0);
            try testArgs(i16, u15, 1 << 0);
            try testArgs(i16, u15, 1 << 14);
            try testArgs(i32, u15, 0);
            try testArgs(i32, u15, 1 << 0);
            try testArgs(i32, u15, 1 << 14);
            try testArgs(i64, u15, 0);
            try testArgs(i64, u15, 1 << 0);
            try testArgs(i64, u15, 1 << 14);
            try testArgs(i128, u15, 0);
            try testArgs(i128, u15, 1 << 0);
            try testArgs(i128, u15, 1 << 14);
            try testArgs(i256, u15, 0);
            try testArgs(i256, u15, 1 << 0);
            try testArgs(i256, u15, 1 << 14);
            try testArgs(i512, u15, 0);
            try testArgs(i512, u15, 1 << 0);
            try testArgs(i512, u15, 1 << 14);
            try testArgs(i1024, u15, 0);
            try testArgs(i1024, u15, 1 << 0);
            try testArgs(i1024, u15, 1 << 14);

            try testArgs(u8, i16, -1 << 15);
            try testArgs(u8, i16, -1);
            try testArgs(u8, i16, 0);
            try testArgs(u16, i16, -1 << 15);
            try testArgs(u16, i16, -1);
            try testArgs(u16, i16, 0);
            try testArgs(u32, i16, -1 << 15);
            try testArgs(u32, i16, -1);
            try testArgs(u32, i16, 0);
            try testArgs(u64, i16, -1 << 15);
            try testArgs(u64, i16, -1);
            try testArgs(u64, i16, 0);
            try testArgs(u128, i16, -1 << 15);
            try testArgs(u128, i16, -1);
            try testArgs(u128, i16, 0);
            try testArgs(u256, i16, -1 << 15);
            try testArgs(u256, i16, -1);
            try testArgs(u256, i16, 0);
            try testArgs(u512, i16, -1 << 15);
            try testArgs(u512, i16, -1);
            try testArgs(u512, i16, 0);
            try testArgs(u1024, i16, -1 << 15);
            try testArgs(u1024, i16, -1);
            try testArgs(u1024, i16, 0);
            try testArgs(i8, u16, 0);
            try testArgs(i8, u16, 1 << 0);
            try testArgs(i8, u16, 1 << 15);
            try testArgs(i16, u16, 0);
            try testArgs(i16, u16, 1 << 0);
            try testArgs(i16, u16, 1 << 15);
            try testArgs(i32, u16, 0);
            try testArgs(i32, u16, 1 << 0);
            try testArgs(i32, u16, 1 << 15);
            try testArgs(i64, u16, 0);
            try testArgs(i64, u16, 1 << 0);
            try testArgs(i64, u16, 1 << 15);
            try testArgs(i128, u16, 0);
            try testArgs(i128, u16, 1 << 0);
            try testArgs(i128, u16, 1 << 15);
            try testArgs(i256, u16, 0);
            try testArgs(i256, u16, 1 << 0);
            try testArgs(i256, u16, 1 << 15);
            try testArgs(i512, u16, 0);
            try testArgs(i512, u16, 1 << 0);
            try testArgs(i512, u16, 1 << 15);
            try testArgs(i1024, u16, 0);
            try testArgs(i1024, u16, 1 << 0);
            try testArgs(i1024, u16, 1 << 15);

            try testArgs(u8, i17, -1 << 16);
            try testArgs(u8, i17, -1);
            try testArgs(u8, i17, 0);
            try testArgs(u16, i17, -1 << 16);
            try testArgs(u16, i17, -1);
            try testArgs(u16, i17, 0);
            try testArgs(u32, i17, -1 << 16);
            try testArgs(u32, i17, -1);
            try testArgs(u32, i17, 0);
            try testArgs(u64, i17, -1 << 16);
            try testArgs(u64, i17, -1);
            try testArgs(u64, i17, 0);
            try testArgs(u128, i17, -1 << 16);
            try testArgs(u128, i17, -1);
            try testArgs(u128, i17, 0);
            try testArgs(u256, i17, -1 << 16);
            try testArgs(u256, i17, -1);
            try testArgs(u256, i17, 0);
            try testArgs(u512, i17, -1 << 16);
            try testArgs(u512, i17, -1);
            try testArgs(u512, i17, 0);
            try testArgs(u1024, i17, -1 << 16);
            try testArgs(u1024, i17, -1);
            try testArgs(u1024, i17, 0);
            try testArgs(i8, u17, 0);
            try testArgs(i8, u17, 1 << 0);
            try testArgs(i8, u17, 1 << 16);
            try testArgs(i16, u17, 0);
            try testArgs(i16, u17, 1 << 0);
            try testArgs(i16, u17, 1 << 16);
            try testArgs(i32, u17, 0);
            try testArgs(i32, u17, 1 << 0);
            try testArgs(i32, u17, 1 << 16);
            try testArgs(i64, u17, 0);
            try testArgs(i64, u17, 1 << 0);
            try testArgs(i64, u17, 1 << 16);
            try testArgs(i128, u17, 0);
            try testArgs(i128, u17, 1 << 0);
            try testArgs(i128, u17, 1 << 16);
            try testArgs(i256, u17, 0);
            try testArgs(i256, u17, 1 << 0);
            try testArgs(i256, u17, 1 << 16);
            try testArgs(i512, u17, 0);
            try testArgs(i512, u17, 1 << 0);
            try testArgs(i512, u17, 1 << 16);
            try testArgs(i1024, u17, 0);
            try testArgs(i1024, u17, 1 << 0);
            try testArgs(i1024, u17, 1 << 16);

            try testArgs(u8, i31, -1 << 30);
            try testArgs(u8, i31, -1);
            try testArgs(u8, i31, 0);
            try testArgs(u16, i31, -1 << 30);
            try testArgs(u16, i31, -1);
            try testArgs(u16, i31, 0);
            try testArgs(u32, i31, -1 << 30);
            try testArgs(u32, i31, -1);
            try testArgs(u32, i31, 0);
            try testArgs(u64, i31, -1 << 30);
            try testArgs(u64, i31, -1);
            try testArgs(u64, i31, 0);
            try testArgs(u128, i31, -1 << 30);
            try testArgs(u128, i31, -1);
            try testArgs(u128, i31, 0);
            try testArgs(u256, i31, -1 << 30);
            try testArgs(u256, i31, -1);
            try testArgs(u256, i31, 0);
            try testArgs(u512, i31, -1 << 30);
            try testArgs(u512, i31, -1);
            try testArgs(u512, i31, 0);
            try testArgs(u1024, i31, -1 << 30);
            try testArgs(u1024, i31, -1);
            try testArgs(u1024, i31, 0);
            try testArgs(i8, u31, 0);
            try testArgs(i8, u31, 1 << 0);
            try testArgs(i8, u31, 1 << 30);
            try testArgs(i16, u31, 0);
            try testArgs(i16, u31, 1 << 0);
            try testArgs(i16, u31, 1 << 30);
            try testArgs(i32, u31, 0);
            try testArgs(i32, u31, 1 << 0);
            try testArgs(i32, u31, 1 << 30);
            try testArgs(i64, u31, 0);
            try testArgs(i64, u31, 1 << 0);
            try testArgs(i64, u31, 1 << 30);
            try testArgs(i128, u31, 0);
            try testArgs(i128, u31, 1 << 0);
            try testArgs(i128, u31, 1 << 30);
            try testArgs(i256, u31, 0);
            try testArgs(i256, u31, 1 << 0);
            try testArgs(i256, u31, 1 << 30);
            try testArgs(i512, u31, 0);
            try testArgs(i512, u31, 1 << 0);
            try testArgs(i512, u31, 1 << 30);
            try testArgs(i1024, u31, 0);
            try testArgs(i1024, u31, 1 << 0);
            try testArgs(i1024, u31, 1 << 30);

            try testArgs(u8, i32, -1 << 31);
            try testArgs(u8, i32, -1);
            try testArgs(u8, i32, 0);
            try testArgs(u16, i32, -1 << 31);
            try testArgs(u16, i32, -1);
            try testArgs(u16, i32, 0);
            try testArgs(u32, i32, -1 << 31);
            try testArgs(u32, i32, -1);
            try testArgs(u32, i32, 0);
            try testArgs(u64, i32, -1 << 31);
            try testArgs(u64, i32, -1);
            try testArgs(u64, i32, 0);
            try testArgs(u128, i32, -1 << 31);
            try testArgs(u128, i32, -1);
            try testArgs(u128, i32, 0);
            try testArgs(u256, i32, -1 << 31);
            try testArgs(u256, i32, -1);
            try testArgs(u256, i32, 0);
            try testArgs(u512, i32, -1 << 31);
            try testArgs(u512, i32, -1);
            try testArgs(u512, i32, 0);
            try testArgs(u1024, i32, -1 << 31);
            try testArgs(u1024, i32, -1);
            try testArgs(u1024, i32, 0);
            try testArgs(i8, u32, 0);
            try testArgs(i8, u32, 1 << 0);
            try testArgs(i8, u32, 1 << 31);
            try testArgs(i16, u32, 0);
            try testArgs(i16, u32, 1 << 0);
            try testArgs(i16, u32, 1 << 31);
            try testArgs(i32, u32, 0);
            try testArgs(i32, u32, 1 << 0);
            try testArgs(i32, u32, 1 << 31);
            try testArgs(i64, u32, 0);
            try testArgs(i64, u32, 1 << 0);
            try testArgs(i64, u32, 1 << 31);
            try testArgs(i128, u32, 0);
            try testArgs(i128, u32, 1 << 0);
            try testArgs(i128, u32, 1 << 31);
            try testArgs(i256, u32, 0);
            try testArgs(i256, u32, 1 << 0);
            try testArgs(i256, u32, 1 << 31);
            try testArgs(i512, u32, 0);
            try testArgs(i512, u32, 1 << 0);
            try testArgs(i512, u32, 1 << 31);
            try testArgs(i1024, u32, 0);
            try testArgs(i1024, u32, 1 << 0);
            try testArgs(i1024, u32, 1 << 31);

            try testArgs(u8, i33, -1 << 32);
            try testArgs(u8, i33, -1);
            try testArgs(u8, i33, 0);
            try testArgs(u16, i33, -1 << 32);
            try testArgs(u16, i33, -1);
            try testArgs(u16, i33, 0);
            try testArgs(u32, i33, -1 << 32);
            try testArgs(u32, i33, -1);
            try testArgs(u32, i33, 0);
            try testArgs(u64, i33, -1 << 32);
            try testArgs(u64, i33, -1);
            try testArgs(u64, i33, 0);
            try testArgs(u128, i33, -1 << 32);
            try testArgs(u128, i33, -1);
            try testArgs(u128, i33, 0);
            try testArgs(u256, i33, -1 << 32);
            try testArgs(u256, i33, -1);
            try testArgs(u256, i33, 0);
            try testArgs(u512, i33, -1 << 32);
            try testArgs(u512, i33, -1);
            try testArgs(u512, i33, 0);
            try testArgs(u1024, i33, -1 << 32);
            try testArgs(u1024, i33, -1);
            try testArgs(u1024, i33, 0);
            try testArgs(i8, u33, 0);
            try testArgs(i8, u33, 1 << 0);
            try testArgs(i8, u33, 1 << 32);
            try testArgs(i16, u33, 0);
            try testArgs(i16, u33, 1 << 0);
            try testArgs(i16, u33, 1 << 32);
            try testArgs(i32, u33, 0);
            try testArgs(i32, u33, 1 << 0);
            try testArgs(i32, u33, 1 << 32);
            try testArgs(i64, u33, 0);
            try testArgs(i64, u33, 1 << 0);
            try testArgs(i64, u33, 1 << 32);
            try testArgs(i128, u33, 0);
            try testArgs(i128, u33, 1 << 0);
            try testArgs(i128, u33, 1 << 32);
            try testArgs(i256, u33, 0);
            try testArgs(i256, u33, 1 << 0);
            try testArgs(i256, u33, 1 << 32);
            try testArgs(i512, u33, 0);
            try testArgs(i512, u33, 1 << 0);
            try testArgs(i512, u33, 1 << 32);
            try testArgs(i1024, u33, 0);
            try testArgs(i1024, u33, 1 << 0);
            try testArgs(i1024, u33, 1 << 32);

            try testArgs(u8, i63, -1 << 62);
            try testArgs(u8, i63, -1);
            try testArgs(u8, i63, 0);
            try testArgs(u16, i63, -1 << 62);
            try testArgs(u16, i63, -1);
            try testArgs(u16, i63, 0);
            try testArgs(u32, i63, -1 << 62);
            try testArgs(u32, i63, -1);
            try testArgs(u32, i63, 0);
            try testArgs(u64, i63, -1 << 62);
            try testArgs(u64, i63, -1);
            try testArgs(u64, i63, 0);
            try testArgs(u128, i63, -1 << 62);
            try testArgs(u128, i63, -1);
            try testArgs(u128, i63, 0);
            try testArgs(u256, i63, -1 << 62);
            try testArgs(u256, i63, -1);
            try testArgs(u256, i63, 0);
            try testArgs(u512, i63, -1 << 62);
            try testArgs(u512, i63, -1);
            try testArgs(u512, i63, 0);
            try testArgs(u1024, i63, -1 << 62);
            try testArgs(u1024, i63, -1);
            try testArgs(u1024, i63, 0);
            try testArgs(i8, u63, 0);
            try testArgs(i8, u63, 1 << 0);
            try testArgs(i8, u63, 1 << 62);
            try testArgs(i16, u63, 0);
            try testArgs(i16, u63, 1 << 0);
            try testArgs(i16, u63, 1 << 62);
            try testArgs(i32, u63, 0);
            try testArgs(i32, u63, 1 << 0);
            try testArgs(i32, u63, 1 << 62);
            try testArgs(i64, u63, 0);
            try testArgs(i64, u63, 1 << 0);
            try testArgs(i64, u63, 1 << 62);
            try testArgs(i128, u63, 0);
            try testArgs(i128, u63, 1 << 0);
            try testArgs(i128, u63, 1 << 62);
            try testArgs(i256, u63, 0);
            try testArgs(i256, u63, 1 << 0);
            try testArgs(i256, u63, 1 << 62);
            try testArgs(i512, u63, 0);
            try testArgs(i512, u63, 1 << 0);
            try testArgs(i512, u63, 1 << 62);
            try testArgs(i1024, u63, 0);
            try testArgs(i1024, u63, 1 << 0);
            try testArgs(i1024, u63, 1 << 62);

            try testArgs(u8, i64, -1 << 63);
            try testArgs(u8, i64, -1);
            try testArgs(u8, i64, 0);
            try testArgs(u16, i64, -1 << 63);
            try testArgs(u16, i64, -1);
            try testArgs(u16, i64, 0);
            try testArgs(u32, i64, -1 << 63);
            try testArgs(u32, i64, -1);
            try testArgs(u32, i64, 0);
            try testArgs(u64, i64, -1 << 63);
            try testArgs(u64, i64, -1);
            try testArgs(u64, i64, 0);
            try testArgs(u128, i64, -1 << 63);
            try testArgs(u128, i64, -1);
            try testArgs(u128, i64, 0);
            try testArgs(u256, i64, -1 << 63);
            try testArgs(u256, i64, -1);
            try testArgs(u256, i64, 0);
            try testArgs(u512, i64, -1 << 63);
            try testArgs(u512, i64, -1);
            try testArgs(u512, i64, 0);
            try testArgs(u1024, i64, -1 << 63);
            try testArgs(u1024, i64, -1);
            try testArgs(u1024, i64, 0);
            try testArgs(i8, u64, 0);
            try testArgs(i8, u64, 1 << 0);
            try testArgs(i8, u64, 1 << 63);
            try testArgs(i16, u64, 0);
            try testArgs(i16, u64, 1 << 0);
            try testArgs(i16, u64, 1 << 63);
            try testArgs(i32, u64, 0);
            try testArgs(i32, u64, 1 << 0);
            try testArgs(i32, u64, 1 << 63);
            try testArgs(i64, u64, 0);
            try testArgs(i64, u64, 1 << 0);
            try testArgs(i64, u64, 1 << 63);
            try testArgs(i128, u64, 0);
            try testArgs(i128, u64, 1 << 0);
            try testArgs(i128, u64, 1 << 63);
            try testArgs(i256, u64, 0);
            try testArgs(i256, u64, 1 << 0);
            try testArgs(i256, u64, 1 << 63);
            try testArgs(i512, u64, 0);
            try testArgs(i512, u64, 1 << 0);
            try testArgs(i512, u64, 1 << 63);
            try testArgs(i1024, u64, 0);
            try testArgs(i1024, u64, 1 << 0);
            try testArgs(i1024, u64, 1 << 63);

            try testArgs(u8, i65, -1 << 64);
            try testArgs(u8, i65, -1);
            try testArgs(u8, i65, 0);
            try testArgs(u16, i65, -1 << 64);
            try testArgs(u16, i65, -1);
            try testArgs(u16, i65, 0);
            try testArgs(u32, i65, -1 << 64);
            try testArgs(u32, i65, -1);
            try testArgs(u32, i65, 0);
            try testArgs(u64, i65, -1 << 64);
            try testArgs(u64, i65, -1);
            try testArgs(u64, i65, 0);
            try testArgs(u128, i65, -1 << 64);
            try testArgs(u128, i65, -1);
            try testArgs(u128, i65, 0);
            try testArgs(u256, i65, -1 << 64);
            try testArgs(u256, i65, -1);
            try testArgs(u256, i65, 0);
            try testArgs(u512, i65, -1 << 64);
            try testArgs(u512, i65, -1);
            try testArgs(u512, i65, 0);
            try testArgs(u1024, i65, -1 << 64);
            try testArgs(u1024, i65, -1);
            try testArgs(u1024, i65, 0);
            try testArgs(i8, u65, 0);
            try testArgs(i8, u65, 1 << 0);
            try testArgs(i8, u65, 1 << 64);
            try testArgs(i16, u65, 0);
            try testArgs(i16, u65, 1 << 0);
            try testArgs(i16, u65, 1 << 64);
            try testArgs(i32, u65, 0);
            try testArgs(i32, u65, 1 << 0);
            try testArgs(i32, u65, 1 << 64);
            try testArgs(i64, u65, 0);
            try testArgs(i64, u65, 1 << 0);
            try testArgs(i64, u65, 1 << 64);
            try testArgs(i128, u65, 0);
            try testArgs(i128, u65, 1 << 0);
            try testArgs(i128, u65, 1 << 64);
            try testArgs(i256, u65, 0);
            try testArgs(i256, u65, 1 << 0);
            try testArgs(i256, u65, 1 << 64);
            try testArgs(i512, u65, 0);
            try testArgs(i512, u65, 1 << 0);
            try testArgs(i512, u65, 1 << 64);
            try testArgs(i1024, u65, 0);
            try testArgs(i1024, u65, 1 << 0);
            try testArgs(i1024, u65, 1 << 64);

            try testArgs(u8, i95, -1 << 94);
            try testArgs(u8, i95, -1);
            try testArgs(u8, i95, 0);
            try testArgs(u16, i95, -1 << 94);
            try testArgs(u16, i95, -1);
            try testArgs(u16, i95, 0);
            try testArgs(u32, i95, -1 << 94);
            try testArgs(u32, i95, -1);
            try testArgs(u32, i95, 0);
            try testArgs(u64, i95, -1 << 94);
            try testArgs(u64, i95, -1);
            try testArgs(u64, i95, 0);
            try testArgs(u128, i95, -1 << 94);
            try testArgs(u128, i95, -1);
            try testArgs(u128, i95, 0);
            try testArgs(u256, i95, -1 << 94);
            try testArgs(u256, i95, -1);
            try testArgs(u256, i95, 0);
            try testArgs(u512, i95, -1 << 94);
            try testArgs(u512, i95, -1);
            try testArgs(u512, i95, 0);
            try testArgs(u1024, i95, -1 << 94);
            try testArgs(u1024, i95, -1);
            try testArgs(u1024, i95, 0);
            try testArgs(i8, u95, 0);
            try testArgs(i8, u95, 1 << 0);
            try testArgs(i8, u95, 1 << 94);
            try testArgs(i16, u95, 0);
            try testArgs(i16, u95, 1 << 0);
            try testArgs(i16, u95, 1 << 94);
            try testArgs(i32, u95, 0);
            try testArgs(i32, u95, 1 << 0);
            try testArgs(i32, u95, 1 << 94);
            try testArgs(i64, u95, 0);
            try testArgs(i64, u95, 1 << 0);
            try testArgs(i64, u95, 1 << 94);
            try testArgs(i128, u95, 0);
            try testArgs(i128, u95, 1 << 0);
            try testArgs(i128, u95, 1 << 94);
            try testArgs(i256, u95, 0);
            try testArgs(i256, u95, 1 << 0);
            try testArgs(i256, u95, 1 << 94);
            try testArgs(i512, u95, 0);
            try testArgs(i512, u95, 1 << 0);
            try testArgs(i512, u95, 1 << 94);
            try testArgs(i1024, u95, 0);
            try testArgs(i1024, u95, 1 << 0);
            try testArgs(i1024, u95, 1 << 94);

            try testArgs(u8, i96, -1 << 95);
            try testArgs(u8, i96, -1);
            try testArgs(u8, i96, 0);
            try testArgs(u16, i96, -1 << 95);
            try testArgs(u16, i96, -1);
            try testArgs(u16, i96, 0);
            try testArgs(u32, i96, -1 << 95);
            try testArgs(u32, i96, -1);
            try testArgs(u32, i96, 0);
            try testArgs(u64, i96, -1 << 95);
            try testArgs(u64, i96, -1);
            try testArgs(u64, i96, 0);
            try testArgs(u128, i96, -1 << 95);
            try testArgs(u128, i96, -1);
            try testArgs(u128, i96, 0);
            try testArgs(u256, i96, -1 << 95);
            try testArgs(u256, i96, -1);
            try testArgs(u256, i96, 0);
            try testArgs(u512, i96, -1 << 95);
            try testArgs(u512, i96, -1);
            try testArgs(u512, i96, 0);
            try testArgs(u1024, i96, -1 << 95);
            try testArgs(u1024, i96, -1);
            try testArgs(u1024, i96, 0);
            try testArgs(i8, u96, 0);
            try testArgs(i8, u96, 1 << 0);
            try testArgs(i8, u96, 1 << 95);
            try testArgs(i16, u96, 0);
            try testArgs(i16, u96, 1 << 0);
            try testArgs(i16, u96, 1 << 95);
            try testArgs(i32, u96, 0);
            try testArgs(i32, u96, 1 << 0);
            try testArgs(i32, u96, 1 << 95);
            try testArgs(i64, u96, 0);
            try testArgs(i64, u96, 1 << 0);
            try testArgs(i64, u96, 1 << 95);
            try testArgs(i128, u96, 0);
            try testArgs(i128, u96, 1 << 0);
            try testArgs(i128, u96, 1 << 95);
            try testArgs(i256, u96, 0);
            try testArgs(i256, u96, 1 << 0);
            try testArgs(i256, u96, 1 << 95);
            try testArgs(i512, u96, 0);
            try testArgs(i512, u96, 1 << 0);
            try testArgs(i512, u96, 1 << 95);
            try testArgs(i1024, u96, 0);
            try testArgs(i1024, u96, 1 << 0);
            try testArgs(i1024, u96, 1 << 95);

            try testArgs(u8, i97, -1 << 96);
            try testArgs(u8, i97, -1);
            try testArgs(u8, i97, 0);
            try testArgs(u16, i97, -1 << 96);
            try testArgs(u16, i97, -1);
            try testArgs(u16, i97, 0);
            try testArgs(u32, i97, -1 << 96);
            try testArgs(u32, i97, -1);
            try testArgs(u32, i97, 0);
            try testArgs(u64, i97, -1 << 96);
            try testArgs(u64, i97, -1);
            try testArgs(u64, i97, 0);
            try testArgs(u128, i97, -1 << 96);
            try testArgs(u128, i97, -1);
            try testArgs(u128, i97, 0);
            try testArgs(u256, i97, -1 << 96);
            try testArgs(u256, i97, -1);
            try testArgs(u256, i97, 0);
            try testArgs(u512, i97, -1 << 96);
            try testArgs(u512, i97, -1);
            try testArgs(u512, i97, 0);
            try testArgs(u1024, i97, -1 << 96);
            try testArgs(u1024, i97, -1);
            try testArgs(u1024, i97, 0);
            try testArgs(i8, u97, 0);
            try testArgs(i8, u97, 1 << 0);
            try testArgs(i8, u97, 1 << 96);
            try testArgs(i16, u97, 0);
            try testArgs(i16, u97, 1 << 0);
            try testArgs(i16, u97, 1 << 96);
            try testArgs(i32, u97, 0);
            try testArgs(i32, u97, 1 << 0);
            try testArgs(i32, u97, 1 << 96);
            try testArgs(i64, u97, 0);
            try testArgs(i64, u97, 1 << 0);
            try testArgs(i64, u97, 1 << 96);
            try testArgs(i128, u97, 0);
            try testArgs(i128, u97, 1 << 0);
            try testArgs(i128, u97, 1 << 96);
            try testArgs(i256, u97, 0);
            try testArgs(i256, u97, 1 << 0);
            try testArgs(i256, u97, 1 << 96);
            try testArgs(i512, u97, 0);
            try testArgs(i512, u97, 1 << 0);
            try testArgs(i512, u97, 1 << 96);
            try testArgs(i1024, u97, 0);
            try testArgs(i1024, u97, 1 << 0);
            try testArgs(i1024, u97, 1 << 96);

            try testArgs(u8, i127, -1 << 126);
            try testArgs(u8, i127, -1);
            try testArgs(u8, i127, 0);
            try testArgs(u16, i127, -1 << 126);
            try testArgs(u16, i127, -1);
            try testArgs(u16, i127, 0);
            try testArgs(u32, i127, -1 << 126);
            try testArgs(u32, i127, -1);
            try testArgs(u32, i127, 0);
            try testArgs(u64, i127, -1 << 126);
            try testArgs(u64, i127, -1);
            try testArgs(u64, i127, 0);
            try testArgs(u128, i127, -1 << 126);
            try testArgs(u128, i127, -1);
            try testArgs(u128, i127, 0);
            try testArgs(u256, i127, -1 << 126);
            try testArgs(u256, i127, -1);
            try testArgs(u256, i127, 0);
            try testArgs(u512, i127, -1 << 126);
            try testArgs(u512, i127, -1);
            try testArgs(u512, i127, 0);
            try testArgs(u1024, i127, -1 << 126);
            try testArgs(u1024, i127, -1);
            try testArgs(u1024, i127, 0);
            try testArgs(i8, u127, 0);
            try testArgs(i8, u127, 1 << 0);
            try testArgs(i8, u127, 1 << 126);
            try testArgs(i16, u127, 0);
            try testArgs(i16, u127, 1 << 0);
            try testArgs(i16, u127, 1 << 126);
            try testArgs(i32, u127, 0);
            try testArgs(i32, u127, 1 << 0);
            try testArgs(i32, u127, 1 << 126);
            try testArgs(i64, u127, 0);
            try testArgs(i64, u127, 1 << 0);
            try testArgs(i64, u127, 1 << 126);
            try testArgs(i128, u127, 0);
            try testArgs(i128, u127, 1 << 0);
            try testArgs(i128, u127, 1 << 126);
            try testArgs(i256, u127, 0);
            try testArgs(i256, u127, 1 << 0);
            try testArgs(i256, u127, 1 << 126);
            try testArgs(i512, u127, 0);
            try testArgs(i512, u127, 1 << 0);
            try testArgs(i512, u127, 1 << 126);
            try testArgs(i1024, u127, 0);
            try testArgs(i1024, u127, 1 << 0);
            try testArgs(i1024, u127, 1 << 126);

            try testArgs(u8, i128, -1 << 127);
            try testArgs(u8, i128, -1);
            try testArgs(u8, i128, 0);
            try testArgs(u16, i128, -1 << 127);
            try testArgs(u16, i128, -1);
            try testArgs(u16, i128, 0);
            try testArgs(u32, i128, -1 << 127);
            try testArgs(u32, i128, -1);
            try testArgs(u32, i128, 0);
            try testArgs(u64, i128, -1 << 127);
            try testArgs(u64, i128, -1);
            try testArgs(u64, i128, 0);
            try testArgs(u128, i128, -1 << 127);
            try testArgs(u128, i128, -1);
            try testArgs(u128, i128, 0);
            try testArgs(u256, i128, -1 << 127);
            try testArgs(u256, i128, -1);
            try testArgs(u256, i128, 0);
            try testArgs(u512, i128, -1 << 127);
            try testArgs(u512, i128, -1);
            try testArgs(u512, i128, 0);
            try testArgs(u1024, i128, -1 << 127);
            try testArgs(u1024, i128, -1);
            try testArgs(u1024, i128, 0);
            try testArgs(i8, u128, 0);
            try testArgs(i8, u128, 1 << 0);
            try testArgs(i8, u128, 1 << 127);
            try testArgs(i16, u128, 0);
            try testArgs(i16, u128, 1 << 0);
            try testArgs(i16, u128, 1 << 127);
            try testArgs(i32, u128, 0);
            try testArgs(i32, u128, 1 << 0);
            try testArgs(i32, u128, 1 << 127);
            try testArgs(i64, u128, 0);
            try testArgs(i64, u128, 1 << 0);
            try testArgs(i64, u128, 1 << 127);
            try testArgs(i128, u128, 0);
            try testArgs(i128, u128, 1 << 0);
            try testArgs(i128, u128, 1 << 127);
            try testArgs(i256, u128, 0);
            try testArgs(i256, u128, 1 << 0);
            try testArgs(i256, u128, 1 << 127);
            try testArgs(i512, u128, 0);
            try testArgs(i512, u128, 1 << 0);
            try testArgs(i512, u128, 1 << 127);
            try testArgs(i1024, u128, 0);
            try testArgs(i1024, u128, 1 << 0);
            try testArgs(i1024, u128, 1 << 127);

            try testArgs(u8, i129, -1 << 128);
            try testArgs(u8, i129, -1);
            try testArgs(u8, i129, 0);
            try testArgs(u16, i129, -1 << 128);
            try testArgs(u16, i129, -1);
            try testArgs(u16, i129, 0);
            try testArgs(u32, i129, -1 << 128);
            try testArgs(u32, i129, -1);
            try testArgs(u32, i129, 0);
            try testArgs(u64, i129, -1 << 128);
            try testArgs(u64, i129, -1);
            try testArgs(u64, i129, 0);
            try testArgs(u128, i129, -1 << 128);
            try testArgs(u128, i129, -1);
            try testArgs(u128, i129, 0);
            try testArgs(u256, i129, -1 << 128);
            try testArgs(u256, i129, -1);
            try testArgs(u256, i129, 0);
            try testArgs(u512, i129, -1 << 128);
            try testArgs(u512, i129, -1);
            try testArgs(u512, i129, 0);
            try testArgs(u1024, i129, -1 << 128);
            try testArgs(u1024, i129, -1);
            try testArgs(u1024, i129, 0);
            try testArgs(i8, u129, 0);
            try testArgs(i8, u129, 1 << 0);
            try testArgs(i8, u129, 1 << 128);
            try testArgs(i16, u129, 0);
            try testArgs(i16, u129, 1 << 0);
            try testArgs(i16, u129, 1 << 128);
            try testArgs(i32, u129, 0);
            try testArgs(i32, u129, 1 << 0);
            try testArgs(i32, u129, 1 << 128);
            try testArgs(i64, u129, 0);
            try testArgs(i64, u129, 1 << 0);
            try testArgs(i64, u129, 1 << 128);
            try testArgs(i128, u129, 0);
            try testArgs(i128, u129, 1 << 0);
            try testArgs(i128, u129, 1 << 128);
            try testArgs(i256, u129, 0);
            try testArgs(i256, u129, 1 << 0);
            try testArgs(i256, u129, 1 << 128);
            try testArgs(i512, u129, 0);
            try testArgs(i512, u129, 1 << 0);
            try testArgs(i512, u129, 1 << 128);
            try testArgs(i1024, u129, 0);
            try testArgs(i1024, u129, 1 << 0);
            try testArgs(i1024, u129, 1 << 128);

            try testArgs(u8, i255, -1 << 254);
            try testArgs(u8, i255, -1);
            try testArgs(u8, i255, 0);
            try testArgs(u16, i255, -1 << 254);
            try testArgs(u16, i255, -1);
            try testArgs(u16, i255, 0);
            try testArgs(u32, i255, -1 << 254);
            try testArgs(u32, i255, -1);
            try testArgs(u32, i255, 0);
            try testArgs(u64, i255, -1 << 254);
            try testArgs(u64, i255, -1);
            try testArgs(u64, i255, 0);
            try testArgs(u128, i255, -1 << 254);
            try testArgs(u128, i255, -1);
            try testArgs(u128, i255, 0);
            try testArgs(u256, i255, -1 << 254);
            try testArgs(u256, i255, -1);
            try testArgs(u256, i255, 0);
            try testArgs(u512, i255, -1 << 254);
            try testArgs(u512, i255, -1);
            try testArgs(u512, i255, 0);
            try testArgs(u1024, i255, -1 << 254);
            try testArgs(u1024, i255, -1);
            try testArgs(u1024, i255, 0);
            try testArgs(i8, u255, 0);
            try testArgs(i8, u255, 1 << 0);
            try testArgs(i8, u255, 1 << 254);
            try testArgs(i16, u255, 0);
            try testArgs(i16, u255, 1 << 0);
            try testArgs(i16, u255, 1 << 254);
            try testArgs(i32, u255, 0);
            try testArgs(i32, u255, 1 << 0);
            try testArgs(i32, u255, 1 << 254);
            try testArgs(i64, u255, 0);
            try testArgs(i64, u255, 1 << 0);
            try testArgs(i64, u255, 1 << 254);
            try testArgs(i128, u255, 0);
            try testArgs(i128, u255, 1 << 0);
            try testArgs(i128, u255, 1 << 254);
            try testArgs(i256, u255, 0);
            try testArgs(i256, u255, 1 << 0);
            try testArgs(i256, u255, 1 << 254);
            try testArgs(i512, u255, 0);
            try testArgs(i512, u255, 1 << 0);
            try testArgs(i512, u255, 1 << 254);
            try testArgs(i1024, u255, 0);
            try testArgs(i1024, u255, 1 << 0);
            try testArgs(i1024, u255, 1 << 254);

            try testArgs(u8, i256, -1 << 255);
            try testArgs(u8, i256, -1);
            try testArgs(u8, i256, 0);
            try testArgs(u16, i256, -1 << 255);
            try testArgs(u16, i256, -1);
            try testArgs(u16, i256, 0);
            try testArgs(u32, i256, -1 << 255);
            try testArgs(u32, i256, -1);
            try testArgs(u32, i256, 0);
            try testArgs(u64, i256, -1 << 255);
            try testArgs(u64, i256, -1);
            try testArgs(u64, i256, 0);
            try testArgs(u128, i256, -1 << 255);
            try testArgs(u128, i256, -1);
            try testArgs(u128, i256, 0);
            try testArgs(u256, i256, -1 << 255);
            try testArgs(u256, i256, -1);
            try testArgs(u256, i256, 0);
            try testArgs(u512, i256, -1 << 255);
            try testArgs(u512, i256, -1);
            try testArgs(u512, i256, 0);
            try testArgs(u1024, i256, -1 << 255);
            try testArgs(u1024, i256, -1);
            try testArgs(u1024, i256, 0);
            try testArgs(i8, u256, 0);
            try testArgs(i8, u256, 1 << 0);
            try testArgs(i8, u256, 1 << 255);
            try testArgs(i16, u256, 0);
            try testArgs(i16, u256, 1 << 0);
            try testArgs(i16, u256, 1 << 255);
            try testArgs(i32, u256, 0);
            try testArgs(i32, u256, 1 << 0);
            try testArgs(i32, u256, 1 << 255);
            try testArgs(i64, u256, 0);
            try testArgs(i64, u256, 1 << 0);
            try testArgs(i64, u256, 1 << 255);
            try testArgs(i128, u256, 0);
            try testArgs(i128, u256, 1 << 0);
            try testArgs(i128, u256, 1 << 255);
            try testArgs(i256, u256, 0);
            try testArgs(i256, u256, 1 << 0);
            try testArgs(i256, u256, 1 << 255);
            try testArgs(i512, u256, 0);
            try testArgs(i512, u256, 1 << 0);
            try testArgs(i512, u256, 1 << 255);
            try testArgs(i1024, u256, 0);
            try testArgs(i1024, u256, 1 << 0);
            try testArgs(i1024, u256, 1 << 255);

            try testArgs(u8, i257, -1 << 256);
            try testArgs(u8, i257, -1);
            try testArgs(u8, i257, 0);
            try testArgs(u16, i257, -1 << 256);
            try testArgs(u16, i257, -1);
            try testArgs(u16, i257, 0);
            try testArgs(u32, i257, -1 << 256);
            try testArgs(u32, i257, -1);
            try testArgs(u32, i257, 0);
            try testArgs(u64, i257, -1 << 256);
            try testArgs(u64, i257, -1);
            try testArgs(u64, i257, 0);
            try testArgs(u128, i257, -1 << 256);
            try testArgs(u128, i257, -1);
            try testArgs(u128, i257, 0);
            try testArgs(u256, i257, -1 << 256);
            try testArgs(u256, i257, -1);
            try testArgs(u256, i257, 0);
            try testArgs(u512, i257, -1 << 256);
            try testArgs(u512, i257, -1);
            try testArgs(u512, i257, 0);
            try testArgs(u1024, i257, -1 << 256);
            try testArgs(u1024, i257, -1);
            try testArgs(u1024, i257, 0);
            try testArgs(i8, u257, 0);
            try testArgs(i8, u257, 1 << 0);
            try testArgs(i8, u257, 1 << 256);
            try testArgs(i16, u257, 0);
            try testArgs(i16, u257, 1 << 0);
            try testArgs(i16, u257, 1 << 256);
            try testArgs(i32, u257, 0);
            try testArgs(i32, u257, 1 << 0);
            try testArgs(i32, u257, 1 << 256);
            try testArgs(i64, u257, 0);
            try testArgs(i64, u257, 1 << 0);
            try testArgs(i64, u257, 1 << 256);
            try testArgs(i128, u257, 0);
            try testArgs(i128, u257, 1 << 0);
            try testArgs(i128, u257, 1 << 256);
            try testArgs(i256, u257, 0);
            try testArgs(i256, u257, 1 << 0);
            try testArgs(i256, u257, 1 << 256);
            try testArgs(i512, u257, 0);
            try testArgs(i512, u257, 1 << 0);
            try testArgs(i512, u257, 1 << 256);
            try testArgs(i1024, u257, 0);
            try testArgs(i1024, u257, 1 << 0);
            try testArgs(i1024, u257, 1 << 256);

            try testArgs(u8, i511, -1 << 510);
            try testArgs(u8, i511, -1);
            try testArgs(u8, i511, 0);
            try testArgs(u16, i511, -1 << 510);
            try testArgs(u16, i511, -1);
            try testArgs(u16, i511, 0);
            try testArgs(u32, i511, -1 << 510);
            try testArgs(u32, i511, -1);
            try testArgs(u32, i511, 0);
            try testArgs(u64, i511, -1 << 510);
            try testArgs(u64, i511, -1);
            try testArgs(u64, i511, 0);
            try testArgs(u128, i511, -1 << 510);
            try testArgs(u128, i511, -1);
            try testArgs(u128, i511, 0);
            try testArgs(u256, i511, -1 << 510);
            try testArgs(u256, i511, -1);
            try testArgs(u256, i511, 0);
            try testArgs(u512, i511, -1 << 510);
            try testArgs(u512, i511, -1);
            try testArgs(u512, i511, 0);
            try testArgs(u1024, i511, -1 << 510);
            try testArgs(u1024, i511, -1);
            try testArgs(u1024, i511, 0);
            try testArgs(i8, u511, 0);
            try testArgs(i8, u511, 1 << 0);
            try testArgs(i8, u511, 1 << 510);
            try testArgs(i16, u511, 0);
            try testArgs(i16, u511, 1 << 0);
            try testArgs(i16, u511, 1 << 510);
            try testArgs(i32, u511, 0);
            try testArgs(i32, u511, 1 << 0);
            try testArgs(i32, u511, 1 << 510);
            try testArgs(i64, u511, 0);
            try testArgs(i64, u511, 1 << 0);
            try testArgs(i64, u511, 1 << 510);
            try testArgs(i128, u511, 0);
            try testArgs(i128, u511, 1 << 0);
            try testArgs(i128, u511, 1 << 510);
            try testArgs(i256, u511, 0);
            try testArgs(i256, u511, 1 << 0);
            try testArgs(i256, u511, 1 << 510);
            try testArgs(i512, u511, 0);
            try testArgs(i512, u511, 1 << 0);
            try testArgs(i512, u511, 1 << 510);
            try testArgs(i1024, u511, 0);
            try testArgs(i1024, u511, 1 << 0);
            try testArgs(i1024, u511, 1 << 510);

            try testArgs(u8, i512, -1 << 511);
            try testArgs(u8, i512, -1);
            try testArgs(u8, i512, 0);
            try testArgs(u16, i512, -1 << 511);
            try testArgs(u16, i512, -1);
            try testArgs(u16, i512, 0);
            try testArgs(u32, i512, -1 << 511);
            try testArgs(u32, i512, -1);
            try testArgs(u32, i512, 0);
            try testArgs(u64, i512, -1 << 511);
            try testArgs(u64, i512, -1);
            try testArgs(u64, i512, 0);
            try testArgs(u128, i512, -1 << 511);
            try testArgs(u128, i512, -1);
            try testArgs(u128, i512, 0);
            try testArgs(u256, i512, -1 << 511);
            try testArgs(u256, i512, -1);
            try testArgs(u256, i512, 0);
            try testArgs(u512, i512, -1 << 511);
            try testArgs(u512, i512, -1);
            try testArgs(u512, i512, 0);
            try testArgs(u1024, i512, -1 << 511);
            try testArgs(u1024, i512, -1);
            try testArgs(u1024, i512, 0);
            try testArgs(i8, u512, 0);
            try testArgs(i8, u512, 1 << 0);
            try testArgs(i8, u512, 1 << 511);
            try testArgs(i16, u512, 0);
            try testArgs(i16, u512, 1 << 0);
            try testArgs(i16, u512, 1 << 511);
            try testArgs(i32, u512, 0);
            try testArgs(i32, u512, 1 << 0);
            try testArgs(i32, u512, 1 << 511);
            try testArgs(i64, u512, 0);
            try testArgs(i64, u512, 1 << 0);
            try testArgs(i64, u512, 1 << 511);
            try testArgs(i128, u512, 0);
            try testArgs(i128, u512, 1 << 0);
            try testArgs(i128, u512, 1 << 511);
            try testArgs(i256, u512, 0);
            try testArgs(i256, u512, 1 << 0);
            try testArgs(i256, u512, 1 << 511);
            try testArgs(i512, u512, 0);
            try testArgs(i512, u512, 1 << 0);
            try testArgs(i512, u512, 1 << 511);
            try testArgs(i1024, u512, 0);
            try testArgs(i1024, u512, 1 << 0);
            try testArgs(i1024, u512, 1 << 511);

            try testArgs(u8, i513, -1 << 512);
            try testArgs(u8, i513, -1);
            try testArgs(u8, i513, 0);
            try testArgs(u16, i513, -1 << 512);
            try testArgs(u16, i513, -1);
            try testArgs(u16, i513, 0);
            try testArgs(u32, i513, -1 << 512);
            try testArgs(u32, i513, -1);
            try testArgs(u32, i513, 0);
            try testArgs(u64, i513, -1 << 512);
            try testArgs(u64, i513, -1);
            try testArgs(u64, i513, 0);
            try testArgs(u128, i513, -1 << 512);
            try testArgs(u128, i513, -1);
            try testArgs(u128, i513, 0);
            try testArgs(u256, i513, -1 << 512);
            try testArgs(u256, i513, -1);
            try testArgs(u256, i513, 0);
            try testArgs(u512, i513, -1 << 512);
            try testArgs(u512, i513, -1);
            try testArgs(u512, i513, 0);
            try testArgs(u1024, i513, -1 << 512);
            try testArgs(u1024, i513, -1);
            try testArgs(u1024, i513, 0);
            try testArgs(i8, u513, 0);
            try testArgs(i8, u513, 1 << 0);
            try testArgs(i8, u513, 1 << 512);
            try testArgs(i16, u513, 0);
            try testArgs(i16, u513, 1 << 0);
            try testArgs(i16, u513, 1 << 512);
            try testArgs(i32, u513, 0);
            try testArgs(i32, u513, 1 << 0);
            try testArgs(i32, u513, 1 << 512);
            try testArgs(i64, u513, 0);
            try testArgs(i64, u513, 1 << 0);
            try testArgs(i64, u513, 1 << 512);
            try testArgs(i128, u513, 0);
            try testArgs(i128, u513, 1 << 0);
            try testArgs(i128, u513, 1 << 512);
            try testArgs(i256, u513, 0);
            try testArgs(i256, u513, 1 << 0);
            try testArgs(i256, u513, 1 << 512);
            try testArgs(i512, u513, 0);
            try testArgs(i512, u513, 1 << 0);
            try testArgs(i512, u513, 1 << 512);
            try testArgs(i1024, u513, 0);
            try testArgs(i1024, u513, 1 << 0);
            try testArgs(i1024, u513, 1 << 512);

            try testArgs(u8, i1023, -1 << 1022);
            try testArgs(u8, i1023, -1);
            try testArgs(u8, i1023, 0);
            try testArgs(u16, i1023, -1 << 1022);
            try testArgs(u16, i1023, -1);
            try testArgs(u16, i1023, 0);
            try testArgs(u32, i1023, -1 << 1022);
            try testArgs(u32, i1023, -1);
            try testArgs(u32, i1023, 0);
            try testArgs(u64, i1023, -1 << 1022);
            try testArgs(u64, i1023, -1);
            try testArgs(u64, i1023, 0);
            try testArgs(u128, i1023, -1 << 1022);
            try testArgs(u128, i1023, -1);
            try testArgs(u128, i1023, 0);
            try testArgs(u256, i1023, -1 << 1022);
            try testArgs(u256, i1023, -1);
            try testArgs(u256, i1023, 0);
            try testArgs(u512, i1023, -1 << 1022);
            try testArgs(u512, i1023, -1);
            try testArgs(u512, i1023, 0);
            try testArgs(u1024, i1023, -1 << 1022);
            try testArgs(u1024, i1023, -1);
            try testArgs(u1024, i1023, 0);
            try testArgs(i8, u1023, 0);
            try testArgs(i8, u1023, 1 << 0);
            try testArgs(i8, u1023, 1 << 1022);
            try testArgs(i16, u1023, 0);
            try testArgs(i16, u1023, 1 << 0);
            try testArgs(i16, u1023, 1 << 1022);
            try testArgs(i32, u1023, 0);
            try testArgs(i32, u1023, 1 << 0);
            try testArgs(i32, u1023, 1 << 1022);
            try testArgs(i64, u1023, 0);
            try testArgs(i64, u1023, 1 << 0);
            try testArgs(i64, u1023, 1 << 1022);
            try testArgs(i128, u1023, 0);
            try testArgs(i128, u1023, 1 << 0);
            try testArgs(i128, u1023, 1 << 1022);
            try testArgs(i256, u1023, 0);
            try testArgs(i256, u1023, 1 << 0);
            try testArgs(i256, u1023, 1 << 1022);
            try testArgs(i512, u1023, 0);
            try testArgs(i512, u1023, 1 << 0);
            try testArgs(i512, u1023, 1 << 1022);
            try testArgs(i1024, u1023, 0);
            try testArgs(i1024, u1023, 1 << 0);
            try testArgs(i1024, u1023, 1 << 1022);

            try testArgs(u8, i1024, -1 << 1023);
            try testArgs(u8, i1024, -1);
            try testArgs(u8, i1024, 0);
            try testArgs(u16, i1024, -1 << 1023);
            try testArgs(u16, i1024, -1);
            try testArgs(u16, i1024, 0);
            try testArgs(u32, i1024, -1 << 1023);
            try testArgs(u32, i1024, -1);
            try testArgs(u32, i1024, 0);
            try testArgs(u64, i1024, -1 << 1023);
            try testArgs(u64, i1024, -1);
            try testArgs(u64, i1024, 0);
            try testArgs(u128, i1024, -1 << 1023);
            try testArgs(u128, i1024, -1);
            try testArgs(u128, i1024, 0);
            try testArgs(u256, i1024, -1 << 1023);
            try testArgs(u256, i1024, -1);
            try testArgs(u256, i1024, 0);
            try testArgs(u512, i1024, -1 << 1023);
            try testArgs(u512, i1024, -1);
            try testArgs(u512, i1024, 0);
            try testArgs(u1024, i1024, -1 << 1023);
            try testArgs(u1024, i1024, -1);
            try testArgs(u1024, i1024, 0);
            try testArgs(i8, u1024, 0);
            try testArgs(i8, u1024, 1 << 0);
            try testArgs(i8, u1024, 1 << 1023);
            try testArgs(i16, u1024, 0);
            try testArgs(i16, u1024, 1 << 0);
            try testArgs(i16, u1024, 1 << 1023);
            try testArgs(i32, u1024, 0);
            try testArgs(i32, u1024, 1 << 0);
            try testArgs(i32, u1024, 1 << 1023);
            try testArgs(i64, u1024, 0);
            try testArgs(i64, u1024, 1 << 0);
            try testArgs(i64, u1024, 1 << 1023);
            try testArgs(i128, u1024, 0);
            try testArgs(i128, u1024, 1 << 0);
            try testArgs(i128, u1024, 1 << 1023);
            try testArgs(i256, u1024, 0);
            try testArgs(i256, u1024, 1 << 0);
            try testArgs(i256, u1024, 1 << 1023);
            try testArgs(i512, u1024, 0);
            try testArgs(i512, u1024, 1 << 0);
            try testArgs(i512, u1024, 1 << 1023);
            try testArgs(i1024, u1024, 0);
            try testArgs(i1024, u1024, 1 << 0);
            try testArgs(i1024, u1024, 1 << 1023);

            try testArgs(u8, i1025, -1 << 1024);
            try testArgs(u8, i1025, -1);
            try testArgs(u8, i1025, 0);
            try testArgs(u16, i1025, -1 << 1024);
            try testArgs(u16, i1025, -1);
            try testArgs(u16, i1025, 0);
            try testArgs(u32, i1025, -1 << 1024);
            try testArgs(u32, i1025, -1);
            try testArgs(u32, i1025, 0);
            try testArgs(u64, i1025, -1 << 1024);
            try testArgs(u64, i1025, -1);
            try testArgs(u64, i1025, 0);
            try testArgs(u128, i1025, -1 << 1024);
            try testArgs(u128, i1025, -1);
            try testArgs(u128, i1025, 0);
            try testArgs(u256, i1025, -1 << 1024);
            try testArgs(u256, i1025, -1);
            try testArgs(u256, i1025, 0);
            try testArgs(u512, i1025, -1 << 1024);
            try testArgs(u512, i1025, -1);
            try testArgs(u512, i1025, 0);
            try testArgs(u1024, i1025, -1 << 1024);
            try testArgs(u1024, i1025, -1);
            try testArgs(u1024, i1025, 0);
            try testArgs(i8, u1025, 0);
            try testArgs(i8, u1025, 1 << 0);
            try testArgs(i8, u1025, 1 << 1024);
            try testArgs(i16, u1025, 0);
            try testArgs(i16, u1025, 1 << 0);
            try testArgs(i16, u1025, 1 << 1024);
            try testArgs(i32, u1025, 0);
            try testArgs(i32, u1025, 1 << 0);
            try testArgs(i32, u1025, 1 << 1024);
            try testArgs(i64, u1025, 0);
            try testArgs(i64, u1025, 1 << 0);
            try testArgs(i64, u1025, 1 << 1024);
            try testArgs(i128, u1025, 0);
            try testArgs(i128, u1025, 1 << 0);
            try testArgs(i128, u1025, 1 << 1024);
            try testArgs(i256, u1025, 0);
            try testArgs(i256, u1025, 1 << 0);
            try testArgs(i256, u1025, 1 << 1024);
            try testArgs(i512, u1025, 0);
            try testArgs(i512, u1025, 1 << 0);
            try testArgs(i512, u1025, 1 << 1024);
            try testArgs(i1024, u1025, 0);
            try testArgs(i1024, u1025, 1 << 0);
            try testArgs(i1024, u1025, 1 << 1024);
        }
        fn testFloats() !void {
            @setEvalBranchQuota(3_100);

            try testArgs(f16, f16, -nan(f16));
            try testArgs(f16, f16, -inf(f16));
            try testArgs(f16, f16, -fmax(f16));
            try testArgs(f16, f16, -1e1);
            try testArgs(f16, f16, -1e0);
            try testArgs(f16, f16, -1e-1);
            try testArgs(f16, f16, -fmin(f16));
            try testArgs(f16, f16, -tmin(f16));
            try testArgs(f16, f16, -0.0);
            try testArgs(f16, f16, 0.0);
            try testArgs(f16, f16, tmin(f16));
            try testArgs(f16, f16, fmin(f16));
            try testArgs(f16, f16, 1e-1);
            try testArgs(f16, f16, 1e0);
            try testArgs(f16, f16, 1e1);
            try testArgs(f16, f16, fmax(f16));
            try testArgs(f16, f16, inf(f16));
            try testArgs(f16, f16, nan(f16));

            try testArgs(f32, f16, -nan(f16));
            try testArgs(f32, f16, -inf(f16));
            try testArgs(f32, f16, -fmax(f16));
            try testArgs(f32, f16, -1e1);
            try testArgs(f32, f16, -1e0);
            try testArgs(f32, f16, -1e-1);
            try testArgs(f32, f16, -fmin(f16));
            try testArgs(f32, f16, -tmin(f16));
            try testArgs(f32, f16, -0.0);
            try testArgs(f32, f16, 0.0);
            try testArgs(f32, f16, tmin(f16));
            try testArgs(f32, f16, fmin(f16));
            try testArgs(f32, f16, 1e-1);
            try testArgs(f32, f16, 1e0);
            try testArgs(f32, f16, 1e1);
            try testArgs(f32, f16, fmax(f16));
            try testArgs(f32, f16, inf(f16));
            try testArgs(f32, f16, nan(f16));

            try testArgs(f64, f16, -nan(f16));
            try testArgs(f64, f16, -inf(f16));
            try testArgs(f64, f16, -fmax(f16));
            try testArgs(f64, f16, -1e1);
            try testArgs(f64, f16, -1e0);
            try testArgs(f64, f16, -1e-1);
            try testArgs(f64, f16, -fmin(f16));
            try testArgs(f64, f16, -tmin(f16));
            try testArgs(f64, f16, -0.0);
            try testArgs(f64, f16, 0.0);
            try testArgs(f64, f16, tmin(f16));
            try testArgs(f64, f16, fmin(f16));
            try testArgs(f64, f16, 1e-1);
            try testArgs(f64, f16, 1e0);
            try testArgs(f64, f16, 1e1);
            try testArgs(f64, f16, fmax(f16));
            try testArgs(f64, f16, inf(f16));
            try testArgs(f64, f16, nan(f16));

            try testArgs(f80, f16, -nan(f16));
            try testArgs(f80, f16, -inf(f16));
            try testArgs(f80, f16, -fmax(f16));
            try testArgs(f80, f16, -1e1);
            try testArgs(f80, f16, -1e0);
            try testArgs(f80, f16, -1e-1);
            try testArgs(f80, f16, -fmin(f16));
            try testArgs(f80, f16, -tmin(f16));
            try testArgs(f80, f16, -0.0);
            try testArgs(f80, f16, 0.0);
            try testArgs(f80, f16, tmin(f16));
            try testArgs(f80, f16, fmin(f16));
            try testArgs(f80, f16, 1e-1);
            try testArgs(f80, f16, 1e0);
            try testArgs(f80, f16, 1e1);
            try testArgs(f80, f16, fmax(f16));
            try testArgs(f80, f16, inf(f16));
            try testArgs(f80, f16, nan(f16));

            try testArgs(f128, f16, -nan(f16));
            try testArgs(f128, f16, -inf(f16));
            try testArgs(f128, f16, -fmax(f16));
            try testArgs(f128, f16, -1e1);
            try testArgs(f128, f16, -1e0);
            try testArgs(f128, f16, -1e-1);
            try testArgs(f128, f16, -fmin(f16));
            try testArgs(f128, f16, -tmin(f16));
            try testArgs(f128, f16, -0.0);
            try testArgs(f128, f16, 0.0);
            try testArgs(f128, f16, tmin(f16));
            try testArgs(f128, f16, fmin(f16));
            try testArgs(f128, f16, 1e-1);
            try testArgs(f128, f16, 1e0);
            try testArgs(f128, f16, 1e1);
            try testArgs(f128, f16, fmax(f16));
            try testArgs(f128, f16, inf(f16));
            try testArgs(f128, f16, nan(f16));

            try testArgs(f16, f32, -nan(f32));
            try testArgs(f16, f32, -inf(f32));
            try testArgs(f16, f32, -fmax(f32));
            try testArgs(f16, f32, -1e1);
            try testArgs(f16, f32, -1e0);
            try testArgs(f16, f32, -1e-1);
            try testArgs(f16, f32, -fmin(f32));
            try testArgs(f16, f32, -tmin(f32));
            try testArgs(f16, f32, -0.0);
            try testArgs(f16, f32, 0.0);
            try testArgs(f16, f32, tmin(f32));
            try testArgs(f16, f32, fmin(f32));
            try testArgs(f16, f32, 1e-1);
            try testArgs(f16, f32, 1e0);
            try testArgs(f16, f32, 1e1);
            try testArgs(f16, f32, fmax(f32));
            try testArgs(f16, f32, inf(f32));
            try testArgs(f16, f32, nan(f32));

            try testArgs(f32, f32, -nan(f32));
            try testArgs(f32, f32, -inf(f32));
            try testArgs(f32, f32, -fmax(f32));
            try testArgs(f32, f32, -1e1);
            try testArgs(f32, f32, -1e0);
            try testArgs(f32, f32, -1e-1);
            try testArgs(f32, f32, -fmin(f32));
            try testArgs(f32, f32, -tmin(f32));
            try testArgs(f32, f32, -0.0);
            try testArgs(f32, f32, 0.0);
            try testArgs(f32, f32, tmin(f32));
            try testArgs(f32, f32, fmin(f32));
            try testArgs(f32, f32, 1e-1);
            try testArgs(f32, f32, 1e0);
            try testArgs(f32, f32, 1e1);
            try testArgs(f32, f32, fmax(f32));
            try testArgs(f32, f32, inf(f32));
            try testArgs(f32, f32, nan(f32));

            try testArgs(f64, f32, -nan(f32));
            try testArgs(f64, f32, -inf(f32));
            try testArgs(f64, f32, -fmax(f32));
            try testArgs(f64, f32, -1e1);
            try testArgs(f64, f32, -1e0);
            try testArgs(f64, f32, -1e-1);
            try testArgs(f64, f32, -fmin(f32));
            try testArgs(f64, f32, -tmin(f32));
            try testArgs(f64, f32, -0.0);
            try testArgs(f64, f32, 0.0);
            try testArgs(f64, f32, tmin(f32));
            try testArgs(f64, f32, fmin(f32));
            try testArgs(f64, f32, 1e-1);
            try testArgs(f64, f32, 1e0);
            try testArgs(f64, f32, 1e1);
            try testArgs(f64, f32, fmax(f32));
            try testArgs(f64, f32, inf(f32));
            try testArgs(f64, f32, nan(f32));

            try testArgs(f80, f32, -nan(f32));
            try testArgs(f80, f32, -inf(f32));
            try testArgs(f80, f32, -fmax(f32));
            try testArgs(f80, f32, -1e1);
            try testArgs(f80, f32, -1e0);
            try testArgs(f80, f32, -1e-1);
            try testArgs(f80, f32, -fmin(f32));
            try testArgs(f80, f32, -tmin(f32));
            try testArgs(f80, f32, -0.0);
            try testArgs(f80, f32, 0.0);
            try testArgs(f80, f32, tmin(f32));
            try testArgs(f80, f32, fmin(f32));
            try testArgs(f80, f32, 1e-1);
            try testArgs(f80, f32, 1e0);
            try testArgs(f80, f32, 1e1);
            try testArgs(f80, f32, fmax(f32));
            try testArgs(f80, f32, inf(f32));
            try testArgs(f80, f32, nan(f32));

            try testArgs(f128, f32, -nan(f32));
            try testArgs(f128, f32, -inf(f32));
            try testArgs(f128, f32, -fmax(f32));
            try testArgs(f128, f32, -1e1);
            try testArgs(f128, f32, -1e0);
            try testArgs(f128, f32, -1e-1);
            try testArgs(f128, f32, -fmin(f32));
            try testArgs(f128, f32, -tmin(f32));
            try testArgs(f128, f32, -0.0);
            try testArgs(f128, f32, 0.0);
            try testArgs(f128, f32, tmin(f32));
            try testArgs(f128, f32, fmin(f32));
            try testArgs(f128, f32, 1e-1);
            try testArgs(f128, f32, 1e0);
            try testArgs(f128, f32, 1e1);
            try testArgs(f128, f32, fmax(f32));
            try testArgs(f128, f32, inf(f32));
            try testArgs(f128, f32, nan(f32));

            try testArgs(f16, f64, -nan(f64));
            try testArgs(f16, f64, -inf(f64));
            try testArgs(f16, f64, -fmax(f64));
            try testArgs(f16, f64, -1e1);
            try testArgs(f16, f64, -1e0);
            try testArgs(f16, f64, -1e-1);
            try testArgs(f16, f64, -fmin(f64));
            try testArgs(f16, f64, -tmin(f64));
            try testArgs(f16, f64, -0.0);
            try testArgs(f16, f64, 0.0);
            try testArgs(f16, f64, tmin(f64));
            try testArgs(f16, f64, fmin(f64));
            try testArgs(f16, f64, 1e-1);
            try testArgs(f16, f64, 1e0);
            try testArgs(f16, f64, 1e1);
            try testArgs(f16, f64, fmax(f64));
            try testArgs(f16, f64, inf(f64));
            try testArgs(f16, f64, nan(f64));

            try testArgs(f32, f64, -nan(f64));
            try testArgs(f32, f64, -inf(f64));
            try testArgs(f32, f64, -fmax(f64));
            try testArgs(f32, f64, -1e1);
            try testArgs(f32, f64, -1e0);
            try testArgs(f32, f64, -1e-1);
            try testArgs(f32, f64, -fmin(f64));
            try testArgs(f32, f64, -tmin(f64));
            try testArgs(f32, f64, -0.0);
            try testArgs(f32, f64, 0.0);
            try testArgs(f32, f64, tmin(f64));
            try testArgs(f32, f64, fmin(f64));
            try testArgs(f32, f64, 1e-1);
            try testArgs(f32, f64, 1e0);
            try testArgs(f32, f64, 1e1);
            try testArgs(f32, f64, fmax(f64));
            try testArgs(f32, f64, inf(f64));
            try testArgs(f32, f64, nan(f64));

            try testArgs(f64, f64, -nan(f64));
            try testArgs(f64, f64, -inf(f64));
            try testArgs(f64, f64, -fmax(f64));
            try testArgs(f64, f64, -1e1);
            try testArgs(f64, f64, -1e0);
            try testArgs(f64, f64, -1e-1);
            try testArgs(f64, f64, -fmin(f64));
            try testArgs(f64, f64, -tmin(f64));
            try testArgs(f64, f64, -0.0);
            try testArgs(f64, f64, 0.0);
            try testArgs(f64, f64, tmin(f64));
            try testArgs(f64, f64, fmin(f64));
            try testArgs(f64, f64, 1e-1);
            try testArgs(f64, f64, 1e0);
            try testArgs(f64, f64, 1e1);
            try testArgs(f64, f64, fmax(f64));
            try testArgs(f64, f64, inf(f64));
            try testArgs(f64, f64, nan(f64));

            try testArgs(f80, f64, -nan(f64));
            try testArgs(f80, f64, -inf(f64));
            try testArgs(f80, f64, -fmax(f64));
            try testArgs(f80, f64, -1e1);
            try testArgs(f80, f64, -1e0);
            try testArgs(f80, f64, -1e-1);
            try testArgs(f80, f64, -fmin(f64));
            try testArgs(f80, f64, -tmin(f64));
            try testArgs(f80, f64, -0.0);
            try testArgs(f80, f64, 0.0);
            try testArgs(f80, f64, tmin(f64));
            try testArgs(f80, f64, fmin(f64));
            try testArgs(f80, f64, 1e-1);
            try testArgs(f80, f64, 1e0);
            try testArgs(f80, f64, 1e1);
            try testArgs(f80, f64, fmax(f64));
            try testArgs(f80, f64, inf(f64));
            try testArgs(f80, f64, nan(f64));

            try testArgs(f128, f64, -nan(f64));
            try testArgs(f128, f64, -inf(f64));
            try testArgs(f128, f64, -fmax(f64));
            try testArgs(f128, f64, -1e1);
            try testArgs(f128, f64, -1e0);
            try testArgs(f128, f64, -1e-1);
            try testArgs(f128, f64, -fmin(f64));
            try testArgs(f128, f64, -tmin(f64));
            try testArgs(f128, f64, -0.0);
            try testArgs(f128, f64, 0.0);
            try testArgs(f128, f64, tmin(f64));
            try testArgs(f128, f64, fmin(f64));
            try testArgs(f128, f64, 1e-1);
            try testArgs(f128, f64, 1e0);
            try testArgs(f128, f64, 1e1);
            try testArgs(f128, f64, fmax(f64));
            try testArgs(f128, f64, inf(f64));
            try testArgs(f128, f64, nan(f64));

            try testArgs(f16, f80, -nan(f80));
            try testArgs(f16, f80, -inf(f80));
            try testArgs(f16, f80, -fmax(f80));
            try testArgs(f16, f80, -1e1);
            try testArgs(f16, f80, -1e0);
            try testArgs(f16, f80, -1e-1);
            try testArgs(f16, f80, -fmin(f80));
            try testArgs(f16, f80, -tmin(f80));
            try testArgs(f16, f80, -0.0);
            try testArgs(f16, f80, 0.0);
            try testArgs(f16, f80, tmin(f80));
            try testArgs(f16, f80, fmin(f80));
            try testArgs(f16, f80, 1e-1);
            try testArgs(f16, f80, 1e0);
            try testArgs(f16, f80, 1e1);
            try testArgs(f16, f80, fmax(f80));
            try testArgs(f16, f80, inf(f80));
            try testArgs(f16, f80, nan(f80));

            try testArgs(f32, f80, -nan(f80));
            try testArgs(f32, f80, -inf(f80));
            try testArgs(f32, f80, -fmax(f80));
            try testArgs(f32, f80, -1e1);
            try testArgs(f32, f80, -1e0);
            try testArgs(f32, f80, -1e-1);
            try testArgs(f32, f80, -fmin(f80));
            try testArgs(f32, f80, -tmin(f80));
            try testArgs(f32, f80, -0.0);
            try testArgs(f32, f80, 0.0);
            try testArgs(f32, f80, tmin(f80));
            try testArgs(f32, f80, fmin(f80));
            try testArgs(f32, f80, 1e-1);
            try testArgs(f32, f80, 1e0);
            try testArgs(f32, f80, 1e1);
            try testArgs(f32, f80, fmax(f80));
            try testArgs(f32, f80, inf(f80));
            try testArgs(f32, f80, nan(f80));

            try testArgs(f64, f80, -nan(f80));
            try testArgs(f64, f80, -inf(f80));
            try testArgs(f64, f80, -fmax(f80));
            try testArgs(f64, f80, -1e1);
            try testArgs(f64, f80, -1e0);
            try testArgs(f64, f80, -1e-1);
            try testArgs(f64, f80, -fmin(f80));
            try testArgs(f64, f80, -tmin(f80));
            try testArgs(f64, f80, -0.0);
            try testArgs(f64, f80, 0.0);
            try testArgs(f64, f80, tmin(f80));
            try testArgs(f64, f80, fmin(f80));
            try testArgs(f64, f80, 1e-1);
            try testArgs(f64, f80, 1e0);
            try testArgs(f64, f80, 1e1);
            try testArgs(f64, f80, fmax(f80));
            try testArgs(f64, f80, inf(f80));
            try testArgs(f64, f80, nan(f80));

            try testArgs(f80, f80, -nan(f80));
            try testArgs(f80, f80, -inf(f80));
            try testArgs(f80, f80, -fmax(f80));
            try testArgs(f80, f80, -1e1);
            try testArgs(f80, f80, -1e0);
            try testArgs(f80, f80, -1e-1);
            try testArgs(f80, f80, -fmin(f80));
            try testArgs(f80, f80, -tmin(f80));
            try testArgs(f80, f80, -0.0);
            try testArgs(f80, f80, 0.0);
            try testArgs(f80, f80, tmin(f80));
            try testArgs(f80, f80, fmin(f80));
            try testArgs(f80, f80, 1e-1);
            try testArgs(f80, f80, 1e0);
            try testArgs(f80, f80, 1e1);
            try testArgs(f80, f80, fmax(f80));
            try testArgs(f80, f80, inf(f80));
            try testArgs(f80, f80, nan(f80));

            try testArgs(f128, f80, -nan(f80));
            try testArgs(f128, f80, -inf(f80));
            try testArgs(f128, f80, -fmax(f80));
            try testArgs(f128, f80, -1e1);
            try testArgs(f128, f80, -1e0);
            try testArgs(f128, f80, -1e-1);
            try testArgs(f128, f80, -fmin(f80));
            try testArgs(f128, f80, -tmin(f80));
            try testArgs(f128, f80, -0.0);
            try testArgs(f128, f80, 0.0);
            try testArgs(f128, f80, tmin(f80));
            try testArgs(f128, f80, fmin(f80));
            try testArgs(f128, f80, 1e-1);
            try testArgs(f128, f80, 1e0);
            try testArgs(f128, f80, 1e1);
            try testArgs(f128, f80, fmax(f80));
            try testArgs(f128, f80, inf(f80));
            try testArgs(f128, f80, nan(f80));

            try testArgs(f16, f128, -nan(f128));
            try testArgs(f16, f128, -inf(f128));
            try testArgs(f16, f128, -fmax(f128));
            try testArgs(f16, f128, -1e1);
            try testArgs(f16, f128, -1e0);
            try testArgs(f16, f128, -1e-1);
            try testArgs(f16, f128, -fmin(f128));
            try testArgs(f16, f128, -tmin(f128));
            try testArgs(f16, f128, -0.0);
            try testArgs(f16, f128, 0.0);
            try testArgs(f16, f128, tmin(f128));
            try testArgs(f16, f128, fmin(f128));
            try testArgs(f16, f128, 1e-1);
            try testArgs(f16, f128, 1e0);
            try testArgs(f16, f128, 1e1);
            try testArgs(f16, f128, fmax(f128));
            try testArgs(f16, f128, inf(f128));
            try testArgs(f16, f128, nan(f128));

            try testArgs(f32, f128, -nan(f128));
            try testArgs(f32, f128, -inf(f128));
            try testArgs(f32, f128, -fmax(f128));
            try testArgs(f32, f128, -1e1);
            try testArgs(f32, f128, -1e0);
            try testArgs(f32, f128, -1e-1);
            try testArgs(f32, f128, -fmin(f128));
            try testArgs(f32, f128, -tmin(f128));
            try testArgs(f32, f128, -0.0);
            try testArgs(f32, f128, 0.0);
            try testArgs(f32, f128, tmin(f128));
            try testArgs(f32, f128, fmin(f128));
            try testArgs(f32, f128, 1e-1);
            try testArgs(f32, f128, 1e0);
            try testArgs(f32, f128, 1e1);
            try testArgs(f32, f128, fmax(f128));
            try testArgs(f32, f128, inf(f128));
            try testArgs(f32, f128, nan(f128));

            try testArgs(f64, f128, -nan(f128));
            try testArgs(f64, f128, -inf(f128));
            try testArgs(f64, f128, -fmax(f128));
            try testArgs(f64, f128, -1e1);
            try testArgs(f64, f128, -1e0);
            try testArgs(f64, f128, -1e-1);
            try testArgs(f64, f128, -fmin(f128));
            try testArgs(f64, f128, -tmin(f128));
            try testArgs(f64, f128, -0.0);
            try testArgs(f64, f128, 0.0);
            try testArgs(f64, f128, tmin(f128));
            try testArgs(f64, f128, fmin(f128));
            try testArgs(f64, f128, 1e-1);
            try testArgs(f64, f128, 1e0);
            try testArgs(f64, f128, 1e1);
            try testArgs(f64, f128, fmax(f128));
            try testArgs(f64, f128, inf(f128));
            try testArgs(f64, f128, nan(f128));

            try testArgs(f80, f128, -nan(f128));
            try testArgs(f80, f128, -inf(f128));
            try testArgs(f80, f128, -fmax(f128));
            try testArgs(f80, f128, -1e1);
            try testArgs(f80, f128, -1e0);
            try testArgs(f80, f128, -1e-1);
            try testArgs(f80, f128, -fmin(f128));
            try testArgs(f80, f128, -tmin(f128));
            try testArgs(f80, f128, -0.0);
            try testArgs(f80, f128, 0.0);
            try testArgs(f80, f128, tmin(f128));
            try testArgs(f80, f128, fmin(f128));
            try testArgs(f80, f128, 1e-1);
            try testArgs(f80, f128, 1e0);
            try testArgs(f80, f128, 1e1);
            try testArgs(f80, f128, fmax(f128));
            try testArgs(f80, f128, inf(f128));
            try testArgs(f80, f128, nan(f128));

            try testArgs(f128, f128, -nan(f128));
            try testArgs(f128, f128, -inf(f128));
            try testArgs(f128, f128, -fmax(f128));
            try testArgs(f128, f128, -1e1);
            try testArgs(f128, f128, -1e0);
            try testArgs(f128, f128, -1e-1);
            try testArgs(f128, f128, -fmin(f128));
            try testArgs(f128, f128, -tmin(f128));
            try testArgs(f128, f128, -0.0);
            try testArgs(f128, f128, 0.0);
            try testArgs(f128, f128, tmin(f128));
            try testArgs(f128, f128, fmin(f128));
            try testArgs(f128, f128, 1e-1);
            try testArgs(f128, f128, 1e0);
            try testArgs(f128, f128, 1e1);
            try testArgs(f128, f128, fmax(f128));
            try testArgs(f128, f128, inf(f128));
            try testArgs(f128, f128, nan(f128));
        }
        fn testSameSignednessIntVectors() !void {
            try testArgs(@Vector(1, i7), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i8), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i9), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i15), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i16), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i17), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i31), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i32), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i33), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i63), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i64), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i65), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i127), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i128), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i129), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i255), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i256), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i257), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i511), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i512), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i513), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i1023), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i1024), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i1025), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u7), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u8), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u9), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u15), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u16), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u17), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u31), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u32), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u33), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u63), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u64), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u65), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u127), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u128), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u129), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u255), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u256), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u257), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u511), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u512), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u513), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u1023), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u1024), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, u1025), @Vector(1, u1), .{1});

            try testArgs(@Vector(2, i7), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i8), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i9), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i15), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i16), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i17), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i31), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i32), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i33), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i63), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i64), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i65), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i127), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i128), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i129), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i255), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i256), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i257), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i511), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i512), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i513), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i1023), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i1024), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i1025), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u7), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u8), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u9), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u15), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u16), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u17), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u31), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u32), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u33), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u63), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u64), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u65), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u127), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u128), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u129), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u255), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u256), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u257), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u511), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u512), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u513), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u1023), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u1024), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, u1025), @Vector(2, u1), .{ 0, 1 });

            try testArgs(@Vector(3, i7), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u8), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u9), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u15), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u16), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u17), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u31), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u32), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u33), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u63), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u64), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u65), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u127), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u128), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u129), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u255), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u256), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u257), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u511), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u512), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u513), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u1023), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u1024), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, u1025), @Vector(3, u2), .{ 0, 1, 1 << 1 });

            try testArgs(@Vector(3, i7), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u8), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u9), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u15), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u16), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u17), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u31), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u32), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u33), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u63), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u64), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u65), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u127), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u128), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u129), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u255), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u256), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u257), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u511), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u512), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u513), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u1023), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u1024), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, u1025), @Vector(3, u3), .{ 0, 1, 1 << 2 });

            try testArgs(@Vector(3, i7), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u8), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u9), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u15), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u16), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u17), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u31), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u32), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u33), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u63), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u64), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u65), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u127), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u128), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u129), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u255), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u256), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u257), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u511), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u512), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u513), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u1023), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u1024), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, u1025), @Vector(3, u4), .{ 0, 1, 1 << 3 });

            try testArgs(@Vector(3, i7), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u8), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u9), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u15), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u16), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u17), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u31), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u32), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u33), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u63), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u64), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u65), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u127), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u128), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u129), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u255), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u256), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u257), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u511), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u512), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u513), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u1023), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u1024), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, u1025), @Vector(3, u5), .{ 0, 1, 1 << 4 });

            try testArgs(@Vector(3, i7), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u8), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u9), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u15), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u16), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u17), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u31), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u32), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u33), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u63), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u64), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u65), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u127), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u128), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u129), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u255), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u256), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u257), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u511), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u512), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u513), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u1023), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u1024), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, u1025), @Vector(3, u7), .{ 0, 1, 1 << 6 });

            try testArgs(@Vector(3, i7), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u8), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u9), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u15), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u16), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u17), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u31), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u32), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u33), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u63), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u64), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u65), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u127), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u128), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u129), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u255), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u256), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u257), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u511), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u512), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u513), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u1023), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u1024), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, u1025), @Vector(3, u8), .{ 0, 1, 1 << 7 });

            try testArgs(@Vector(3, i7), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u8), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u9), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u15), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u16), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u17), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u31), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u32), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u33), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u63), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u64), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u65), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u127), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u128), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u129), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u255), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u256), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u257), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u511), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u512), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u513), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u1023), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u1024), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, u1025), @Vector(3, u9), .{ 0, 1, 1 << 8 });

            try testArgs(@Vector(3, i7), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u8), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u9), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u15), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u16), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u17), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u31), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u32), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u33), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u63), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u64), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u65), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u127), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u128), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u129), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u255), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u256), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u257), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u511), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u512), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u513), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u1023), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u1024), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, u1025), @Vector(3, u15), .{ 0, 1, 1 << 14 });

            try testArgs(@Vector(3, i7), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u8), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u9), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u15), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u16), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u17), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u31), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u32), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u33), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u63), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u64), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u65), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u127), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u128), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u129), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u255), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u256), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u257), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u511), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u512), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u513), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u1023), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u1024), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, u1025), @Vector(3, u16), .{ 0, 1, 1 << 15 });

            try testArgs(@Vector(3, i7), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u8), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u9), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u15), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u16), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u17), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u31), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u32), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u33), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u63), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u64), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u65), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u127), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u128), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u129), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u255), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u256), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u257), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u511), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u512), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u513), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u1023), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u1024), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, u1025), @Vector(3, u17), .{ 0, 1, 1 << 16 });

            try testArgs(@Vector(3, i7), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u8), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u9), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u15), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u16), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u17), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u31), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u32), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u33), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u63), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u64), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u65), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u127), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u128), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u129), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u255), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u256), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u257), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u511), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u512), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u513), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u1023), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u1024), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, u1025), @Vector(3, u31), .{ 0, 1, 1 << 30 });

            try testArgs(@Vector(3, i7), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u8), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u9), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u15), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u16), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u17), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u31), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u32), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u33), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u63), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u64), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u65), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u127), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u128), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u129), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u255), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u256), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u257), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u511), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u512), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u513), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u1023), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u1024), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, u1025), @Vector(3, u32), .{ 0, 1, 1 << 31 });

            try testArgs(@Vector(3, i7), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u8), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u9), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u15), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u16), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u17), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u31), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u32), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u33), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u63), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u64), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u65), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u127), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u128), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u129), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u255), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u256), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u257), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u511), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u512), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u513), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u1023), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u1024), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, u1025), @Vector(3, u33), .{ 0, 1, 1 << 32 });

            try testArgs(@Vector(3, i7), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u8), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u9), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u15), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u16), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u17), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u31), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u32), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u33), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u63), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u64), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u65), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u127), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u128), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u129), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u255), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u256), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u257), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u511), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u512), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u513), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u1023), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u1024), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, u1025), @Vector(3, u63), .{ 0, 1, 1 << 62 });

            try testArgs(@Vector(3, i7), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u8), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u9), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u15), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u16), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u17), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u31), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u32), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u33), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u63), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u64), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u65), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u127), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u128), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u129), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u255), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u256), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u257), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u511), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u512), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u513), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u1023), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u1024), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, u1025), @Vector(3, u64), .{ 0, 1, 1 << 63 });

            try testArgs(@Vector(3, i7), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u8), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u9), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u15), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u16), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u17), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u31), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u32), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u33), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u63), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u64), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u65), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u127), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u128), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u129), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u255), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u256), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u257), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u511), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u512), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u513), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u1023), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u1024), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, u1025), @Vector(3, u65), .{ 0, 1, 1 << 64 });

            try testArgs(@Vector(3, i7), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u8), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u9), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u15), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u16), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u17), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u31), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u32), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u33), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u63), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u64), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u65), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u127), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u128), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u129), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u255), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u256), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u257), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u511), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u512), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u513), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u1023), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u1024), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, u1025), @Vector(3, u95), .{ 0, 1, 1 << 94 });

            try testArgs(@Vector(3, i7), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u8), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u9), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u15), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u16), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u17), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u31), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u32), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u33), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u63), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u64), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u65), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u127), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u128), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u129), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u255), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u256), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u257), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u511), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u512), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u513), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u1023), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u1024), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, u1025), @Vector(3, u96), .{ 0, 1, 1 << 95 });

            try testArgs(@Vector(3, i7), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u8), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u9), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u15), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u16), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u17), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u31), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u32), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u33), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u63), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u64), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u65), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u127), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u128), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u129), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u255), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u256), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u257), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u511), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u512), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u513), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u1023), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u1024), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, u1025), @Vector(3, u97), .{ 0, 1, 1 << 96 });

            try testArgs(@Vector(3, i7), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u8), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u9), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u15), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u16), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u17), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u31), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u32), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u33), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u63), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u64), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u65), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u127), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u128), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u129), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u255), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u256), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u257), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u511), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u512), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u513), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u1023), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u1024), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, u1025), @Vector(3, u127), .{ 0, 1, 1 << 126 });

            try testArgs(@Vector(3, i7), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u8), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u9), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u15), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u16), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u17), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u31), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u32), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u33), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u63), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u64), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u65), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u127), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u128), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u129), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u255), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u256), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u257), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u511), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u512), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u513), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u1023), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u1024), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, u1025), @Vector(3, u128), .{ 0, 1, 1 << 127 });

            try testArgs(@Vector(3, i7), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u8), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u9), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u15), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u16), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u17), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u31), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u32), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u33), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u63), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u64), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u65), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u127), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u128), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u129), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u255), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u256), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u257), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u511), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u512), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u513), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u1023), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u1024), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, u1025), @Vector(3, u129), .{ 0, 1, 1 << 128 });

            try testArgs(@Vector(3, i7), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u8), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u9), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u15), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u16), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u17), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u31), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u32), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u33), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u63), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u64), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u65), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u127), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u128), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u129), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u255), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u256), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u257), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u511), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u512), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u513), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u1023), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u1024), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, u1025), @Vector(3, u159), .{ 0, 1, 1 << 158 });

            try testArgs(@Vector(3, i7), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u8), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u9), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u15), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u16), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u17), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u31), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u32), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u33), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u63), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u64), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u65), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u127), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u128), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u129), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u255), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u256), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u257), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u511), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u512), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u513), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u1023), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u1024), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, u1025), @Vector(3, u160), .{ 0, 1, 1 << 159 });

            try testArgs(@Vector(3, i7), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u8), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u9), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u15), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u16), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u17), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u31), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u32), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u33), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u63), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u64), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u65), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u127), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u128), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u129), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u255), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u256), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u257), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u511), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u512), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u513), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u1023), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u1024), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, u1025), @Vector(3, u161), .{ 0, 1, 1 << 160 });

            try testArgs(@Vector(3, i7), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u8), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u9), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u15), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u16), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u17), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u31), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u32), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u33), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u63), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u64), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u65), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u127), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u128), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u129), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u255), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u256), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u257), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u511), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u512), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u513), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u1023), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u1024), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, u1025), @Vector(3, u191), .{ 0, 1, 1 << 190 });

            try testArgs(@Vector(3, i7), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u8), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u9), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u15), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u16), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u17), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u31), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u32), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u33), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u63), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u64), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u65), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u127), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u128), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u129), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u255), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u256), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u257), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u511), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u512), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u513), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u1023), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u1024), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, u1025), @Vector(3, u192), .{ 0, 1, 1 << 191 });

            try testArgs(@Vector(3, i7), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u8), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u9), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u15), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u16), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u17), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u31), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u32), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u33), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u63), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u64), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u65), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u127), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u128), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u129), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u255), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u256), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u257), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u511), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u512), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u513), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u1023), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u1024), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, u1025), @Vector(3, u193), .{ 0, 1, 1 << 192 });

            try testArgs(@Vector(3, i7), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u8), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u9), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u15), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u16), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u17), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u31), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u32), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u33), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u63), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u64), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u65), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u127), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u128), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u129), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u255), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u256), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u257), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u511), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u512), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u513), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u1023), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u1024), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, u1025), @Vector(3, u223), .{ 0, 1, 1 << 222 });

            try testArgs(@Vector(3, i7), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u8), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u9), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u15), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u16), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u17), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u31), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u32), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u33), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u63), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u64), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u65), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u127), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u128), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u129), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u255), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u256), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u257), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u511), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u512), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u513), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u1023), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u1024), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, u1025), @Vector(3, u224), .{ 0, 1, 1 << 223 });

            try testArgs(@Vector(3, i7), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u8), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u9), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u15), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u16), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u17), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u31), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u32), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u33), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u63), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u64), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u65), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u127), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u128), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u129), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u255), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u256), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u257), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u511), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u512), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u513), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u1023), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u1024), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, u1025), @Vector(3, u225), .{ 0, 1, 1 << 224 });

            try testArgs(@Vector(3, i7), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u8), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u9), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u15), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u16), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u17), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u31), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u32), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u33), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u63), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u64), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u65), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u127), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u128), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u129), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u255), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u256), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u257), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u511), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u512), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u513), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u1023), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u1024), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, u1025), @Vector(3, u255), .{ 0, 1, 1 << 254 });

            try testArgs(@Vector(3, i7), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u8), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u9), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u15), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u16), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u17), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u31), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u32), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u33), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u63), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u64), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u65), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u127), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u128), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u129), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u255), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u256), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u257), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u511), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u512), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u513), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u1023), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u1024), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, u1025), @Vector(3, u256), .{ 0, 1, 1 << 255 });

            try testArgs(@Vector(3, i7), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u8), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u9), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u15), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u16), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u17), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u31), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u32), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u33), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u63), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u64), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u65), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u127), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u128), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u129), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u255), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u256), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u257), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u511), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u512), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u513), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u1023), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u1024), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, u1025), @Vector(3, u257), .{ 0, 1, 1 << 256 });

            try testArgs(@Vector(3, i7), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u8), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u9), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u15), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u16), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u17), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u31), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u32), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u33), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u63), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u64), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u65), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u127), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u128), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u129), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u255), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u256), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u257), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u511), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u512), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u513), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u1023), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u1024), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, u1025), @Vector(3, u511), .{ 0, 1, 1 << 510 });

            try testArgs(@Vector(3, i7), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u8), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u9), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u15), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u16), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u17), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u31), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u32), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u33), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u63), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u64), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u65), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u127), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u128), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u129), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u255), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u256), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u257), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u511), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u512), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u513), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u1023), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u1024), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, u1025), @Vector(3, u512), .{ 0, 1, 1 << 511 });

            try testArgs(@Vector(3, i7), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u8), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u9), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u15), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u16), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u17), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u31), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u32), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u33), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u63), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u64), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u65), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u127), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u128), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u129), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u255), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u256), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u257), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u511), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u512), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u513), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u1023), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u1024), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, u1025), @Vector(3, u513), .{ 0, 1, 1 << 512 });

            try testArgs(@Vector(3, i7), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u8), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u9), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u15), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u16), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u17), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u31), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u32), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u33), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u63), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u64), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u65), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u127), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u128), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u129), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u255), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u256), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u257), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u511), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u512), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u513), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u1023), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u1024), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, u1025), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });

            try testArgs(@Vector(3, i7), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u8), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u9), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u15), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u16), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u17), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u31), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u32), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u33), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u63), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u64), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u65), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u127), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u128), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u129), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u255), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u256), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u257), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u511), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u512), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u513), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u1023), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u1024), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, u1025), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });

            try testArgs(@Vector(3, i7), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i9), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i15), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i16), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i17), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i31), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i32), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i33), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i63), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i64), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i65), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i127), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i128), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i129), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i255), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i256), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i257), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i511), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i512), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i513), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i1023), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i1024), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i1025), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u7), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u8), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u9), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u15), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u16), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u17), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u31), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u32), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u33), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u63), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u64), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u65), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u127), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u128), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u129), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u255), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u256), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u257), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u511), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u512), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u513), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u1023), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u1024), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, u1025), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
        }
        fn testIntVectors() !void {
            try testSameSignednessIntVectors();

            try testArgs(@Vector(1, u8), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u16), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u32), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u64), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u128), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u256), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u512), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, u1024), @Vector(1, i1), .{-1});
            try testArgs(@Vector(1, i8), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i16), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i32), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i64), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i128), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i256), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i512), @Vector(1, u1), .{1});
            try testArgs(@Vector(1, i1024), @Vector(1, u1), .{1});

            try testArgs(@Vector(2, u8), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u16), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u32), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u64), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u128), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u256), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u512), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, u1024), @Vector(2, i1), .{ -1, 0 });
            try testArgs(@Vector(2, i8), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i16), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i32), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i64), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i128), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i256), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i512), @Vector(2, u1), .{ 0, 1 });
            try testArgs(@Vector(2, i1024), @Vector(2, u1), .{ 0, 1 });

            try testArgs(@Vector(3, u8), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i2), .{ -1 << 1, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i16), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i32), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i64), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i128), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i256), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i512), @Vector(3, u2), .{ 0, 1, 1 << 1 });
            try testArgs(@Vector(3, i1024), @Vector(3, u2), .{ 0, 1, 1 << 1 });

            try testArgs(@Vector(3, u8), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i3), .{ -1 << 2, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i16), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i32), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i64), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i128), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i256), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i512), @Vector(3, u3), .{ 0, 1, 1 << 2 });
            try testArgs(@Vector(3, i1024), @Vector(3, u3), .{ 0, 1, 1 << 2 });

            try testArgs(@Vector(3, u8), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i4), .{ -1 << 3, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i16), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i32), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i64), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i128), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i256), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i512), @Vector(3, u4), .{ 0, 1, 1 << 3 });
            try testArgs(@Vector(3, i1024), @Vector(3, u4), .{ 0, 1, 1 << 3 });

            try testArgs(@Vector(3, u8), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i5), .{ -1 << 4, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i16), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i32), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i64), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i128), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i256), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i512), @Vector(3, u5), .{ 0, 1, 1 << 4 });
            try testArgs(@Vector(3, i1024), @Vector(3, u5), .{ 0, 1, 1 << 4 });

            try testArgs(@Vector(3, u8), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i7), .{ -1 << 6, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i16), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i32), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i64), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i128), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i256), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i512), @Vector(3, u7), .{ 0, 1, 1 << 6 });
            try testArgs(@Vector(3, i1024), @Vector(3, u7), .{ 0, 1, 1 << 6 });

            try testArgs(@Vector(3, u8), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i8), .{ -1 << 7, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i16), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i32), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i64), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i128), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i256), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i512), @Vector(3, u8), .{ 0, 1, 1 << 7 });
            try testArgs(@Vector(3, i1024), @Vector(3, u8), .{ 0, 1, 1 << 7 });

            try testArgs(@Vector(3, u8), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i9), .{ -1 << 8, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i16), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i32), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i64), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i128), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i256), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i512), @Vector(3, u9), .{ 0, 1, 1 << 8 });
            try testArgs(@Vector(3, i1024), @Vector(3, u9), .{ 0, 1, 1 << 8 });

            try testArgs(@Vector(3, u8), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i15), .{ -1 << 14, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i16), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i32), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i64), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i128), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i256), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i512), @Vector(3, u15), .{ 0, 1, 1 << 14 });
            try testArgs(@Vector(3, i1024), @Vector(3, u15), .{ 0, 1, 1 << 14 });

            try testArgs(@Vector(3, u8), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i16), .{ -1 << 15, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i16), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i32), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i64), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i128), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i256), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i512), @Vector(3, u16), .{ 0, 1, 1 << 15 });
            try testArgs(@Vector(3, i1024), @Vector(3, u16), .{ 0, 1, 1 << 15 });

            try testArgs(@Vector(3, u8), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i17), .{ -1 << 16, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i16), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i32), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i64), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i128), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i256), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i512), @Vector(3, u17), .{ 0, 1, 1 << 16 });
            try testArgs(@Vector(3, i1024), @Vector(3, u17), .{ 0, 1, 1 << 16 });

            try testArgs(@Vector(3, u8), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i31), .{ -1 << 30, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i16), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i32), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i64), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i128), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i256), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i512), @Vector(3, u31), .{ 0, 1, 1 << 30 });
            try testArgs(@Vector(3, i1024), @Vector(3, u31), .{ 0, 1, 1 << 30 });

            try testArgs(@Vector(3, u8), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i32), .{ -1 << 31, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i16), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i32), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i64), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i128), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i256), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i512), @Vector(3, u32), .{ 0, 1, 1 << 31 });
            try testArgs(@Vector(3, i1024), @Vector(3, u32), .{ 0, 1, 1 << 31 });

            try testArgs(@Vector(3, u8), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i33), .{ -1 << 32, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i16), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i32), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i64), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i128), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i256), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i512), @Vector(3, u33), .{ 0, 1, 1 << 32 });
            try testArgs(@Vector(3, i1024), @Vector(3, u33), .{ 0, 1, 1 << 32 });

            try testArgs(@Vector(3, u8), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i63), .{ -1 << 62, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i16), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i32), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i64), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i128), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i256), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i512), @Vector(3, u63), .{ 0, 1, 1 << 62 });
            try testArgs(@Vector(3, i1024), @Vector(3, u63), .{ 0, 1, 1 << 62 });

            try testArgs(@Vector(3, u8), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i64), .{ -1 << 63, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i16), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i32), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i64), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i128), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i256), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i512), @Vector(3, u64), .{ 0, 1, 1 << 63 });
            try testArgs(@Vector(3, i1024), @Vector(3, u64), .{ 0, 1, 1 << 63 });

            try testArgs(@Vector(3, u8), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i65), .{ -1 << 64, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i16), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i32), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i64), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i128), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i256), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i512), @Vector(3, u65), .{ 0, 1, 1 << 64 });
            try testArgs(@Vector(3, i1024), @Vector(3, u65), .{ 0, 1, 1 << 64 });

            try testArgs(@Vector(3, u8), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i95), .{ -1 << 94, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i16), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i32), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i64), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i128), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i256), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i512), @Vector(3, u95), .{ 0, 1, 1 << 94 });
            try testArgs(@Vector(3, i1024), @Vector(3, u95), .{ 0, 1, 1 << 94 });

            try testArgs(@Vector(3, u8), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i96), .{ -1 << 95, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i16), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i32), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i64), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i128), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i256), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i512), @Vector(3, u96), .{ 0, 1, 1 << 95 });
            try testArgs(@Vector(3, i1024), @Vector(3, u96), .{ 0, 1, 1 << 95 });

            try testArgs(@Vector(3, u8), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i97), .{ -1 << 96, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i16), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i32), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i64), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i128), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i256), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i512), @Vector(3, u97), .{ 0, 1, 1 << 96 });
            try testArgs(@Vector(3, i1024), @Vector(3, u97), .{ 0, 1, 1 << 96 });

            try testArgs(@Vector(3, u8), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i127), .{ -1 << 126, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i16), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i32), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i64), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i128), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i256), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i512), @Vector(3, u127), .{ 0, 1, 1 << 126 });
            try testArgs(@Vector(3, i1024), @Vector(3, u127), .{ 0, 1, 1 << 126 });

            try testArgs(@Vector(3, u8), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i128), .{ -1 << 127, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i16), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i32), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i64), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i128), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i256), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i512), @Vector(3, u128), .{ 0, 1, 1 << 127 });
            try testArgs(@Vector(3, i1024), @Vector(3, u128), .{ 0, 1, 1 << 127 });

            try testArgs(@Vector(3, u8), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i129), .{ -1 << 128, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i16), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i32), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i64), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i128), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i256), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i512), @Vector(3, u129), .{ 0, 1, 1 << 128 });
            try testArgs(@Vector(3, i1024), @Vector(3, u129), .{ 0, 1, 1 << 128 });

            try testArgs(@Vector(3, u8), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i159), .{ -1 << 158, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i16), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i32), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i64), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i128), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i256), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i512), @Vector(3, u159), .{ 0, 1, 1 << 158 });
            try testArgs(@Vector(3, i1024), @Vector(3, u159), .{ 0, 1, 1 << 158 });

            try testArgs(@Vector(3, u8), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i160), .{ -1 << 159, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i16), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i32), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i64), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i128), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i256), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i512), @Vector(3, u160), .{ 0, 1, 1 << 159 });
            try testArgs(@Vector(3, i1024), @Vector(3, u160), .{ 0, 1, 1 << 159 });

            try testArgs(@Vector(3, u8), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i161), .{ -1 << 160, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i16), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i32), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i64), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i128), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i256), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i512), @Vector(3, u161), .{ 0, 1, 1 << 160 });
            try testArgs(@Vector(3, i1024), @Vector(3, u161), .{ 0, 1, 1 << 160 });

            try testArgs(@Vector(3, u8), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i191), .{ -1 << 190, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i16), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i32), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i64), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i128), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i256), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i512), @Vector(3, u191), .{ 0, 1, 1 << 190 });
            try testArgs(@Vector(3, i1024), @Vector(3, u191), .{ 0, 1, 1 << 190 });

            try testArgs(@Vector(3, u8), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i192), .{ -1 << 191, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i16), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i32), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i64), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i128), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i256), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i512), @Vector(3, u192), .{ 0, 1, 1 << 191 });
            try testArgs(@Vector(3, i1024), @Vector(3, u192), .{ 0, 1, 1 << 191 });

            try testArgs(@Vector(3, u8), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i193), .{ -1 << 192, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i16), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i32), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i64), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i128), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i256), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i512), @Vector(3, u193), .{ 0, 1, 1 << 192 });
            try testArgs(@Vector(3, i1024), @Vector(3, u193), .{ 0, 1, 1 << 192 });

            try testArgs(@Vector(3, u8), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i223), .{ -1 << 222, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i16), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i32), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i64), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i128), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i256), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i512), @Vector(3, u223), .{ 0, 1, 1 << 222 });
            try testArgs(@Vector(3, i1024), @Vector(3, u223), .{ 0, 1, 1 << 222 });

            try testArgs(@Vector(3, u8), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i224), .{ -1 << 223, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i16), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i32), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i64), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i128), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i256), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i512), @Vector(3, u224), .{ 0, 1, 1 << 223 });
            try testArgs(@Vector(3, i1024), @Vector(3, u224), .{ 0, 1, 1 << 223 });

            try testArgs(@Vector(3, u8), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i225), .{ -1 << 224, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i16), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i32), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i64), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i128), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i256), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i512), @Vector(3, u225), .{ 0, 1, 1 << 224 });
            try testArgs(@Vector(3, i1024), @Vector(3, u225), .{ 0, 1, 1 << 224 });

            try testArgs(@Vector(3, u8), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i255), .{ -1 << 254, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i16), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i32), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i64), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i128), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i256), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i512), @Vector(3, u255), .{ 0, 1, 1 << 254 });
            try testArgs(@Vector(3, i1024), @Vector(3, u255), .{ 0, 1, 1 << 254 });

            try testArgs(@Vector(3, u8), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i256), .{ -1 << 255, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i16), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i32), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i64), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i128), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i256), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i512), @Vector(3, u256), .{ 0, 1, 1 << 255 });
            try testArgs(@Vector(3, i1024), @Vector(3, u256), .{ 0, 1, 1 << 255 });

            try testArgs(@Vector(3, u8), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i257), .{ -1 << 256, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i16), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i32), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i64), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i128), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i256), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i512), @Vector(3, u257), .{ 0, 1, 1 << 256 });
            try testArgs(@Vector(3, i1024), @Vector(3, u257), .{ 0, 1, 1 << 256 });

            try testArgs(@Vector(3, u8), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i511), .{ -1 << 510, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i16), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i32), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i64), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i128), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i256), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i512), @Vector(3, u511), .{ 0, 1, 1 << 510 });
            try testArgs(@Vector(3, i1024), @Vector(3, u511), .{ 0, 1, 1 << 510 });

            try testArgs(@Vector(3, u8), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i512), .{ -1 << 511, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i16), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i32), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i64), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i128), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i256), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i512), @Vector(3, u512), .{ 0, 1, 1 << 511 });
            try testArgs(@Vector(3, i1024), @Vector(3, u512), .{ 0, 1, 1 << 511 });

            try testArgs(@Vector(3, u8), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i513), .{ -1 << 512, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i16), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i32), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i64), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i128), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i256), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i512), @Vector(3, u513), .{ 0, 1, 1 << 512 });
            try testArgs(@Vector(3, i1024), @Vector(3, u513), .{ 0, 1, 1 << 512 });

            try testArgs(@Vector(3, u8), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i1023), .{ -1 << 1022, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i16), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i32), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i64), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i128), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i256), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i512), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });
            try testArgs(@Vector(3, i1024), @Vector(3, u1023), .{ 0, 1, 1 << 1022 });

            try testArgs(@Vector(3, u8), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i1024), .{ -1 << 1023, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i16), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i32), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i64), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i128), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i256), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i512), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });
            try testArgs(@Vector(3, i1024), @Vector(3, u1024), .{ 0, 1, 1 << 1023 });

            try testArgs(@Vector(3, u8), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u16), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u32), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u64), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u128), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u256), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u512), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, u1024), @Vector(3, i1025), .{ -1 << 1024, -1, 0 });
            try testArgs(@Vector(3, i8), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i16), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i32), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i64), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i128), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i256), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i512), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
            try testArgs(@Vector(3, i1024), @Vector(3, u1025), .{ 0, 1, 1 << 1024 });
        }
        fn testFloatVectors() !void {
            @setEvalBranchQuota(6_700);

            try testArgs(@Vector(1, f16), @Vector(1, f16), .{
                1e0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, f16), .{
                -inf(f16), -1e-2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, f16), .{
                -1e2, 1e-1, fmax(f16), 1e-2,
            });
            try testArgs(@Vector(8, f16), @Vector(8, f16), .{
                -1e-1, tmin(f16), -1e3, fmin(f16), nan(f16), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, f16), .{
                -fmax(f16), -1e0, 1e-4, 1e2, -fmin(f16), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f16), -tmin(f16), -1e-4, inf(f16), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, f16), .{
                -1e3, -tmin(f16), inf(f16),   -1e4,      -0.0, fmax(f16), 1e2,       1e4, -nan(f16), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f16), -1e0,
                1e3,  -1e-3,      -fmin(f16), -inf(f16), 1e-3, tmin(f16), fmin(f16), 1e1, 1e-4,      -fmax(f16), -1e2,  1e-2, -1e-2, 1e3,  inf(f16), -fmin(f16),
            });

            try testArgs(@Vector(1, f32), @Vector(1, f16), .{
                1e0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, f16), .{
                -inf(f16), -1e-2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, f16), .{
                -1e2, 1e-1, fmax(f16), 1e-2,
            });
            try testArgs(@Vector(8, f32), @Vector(8, f16), .{
                -1e-1, tmin(f16), -1e3, fmin(f16), nan(f16), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, f16), .{
                -fmax(f16), -1e0, 1e-4, 1e2, -fmin(f16), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f16), -tmin(f16), -1e-4, inf(f16), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, f16), .{
                -1e3, -tmin(f16), inf(f16),   -1e4,      -0.0, fmax(f16), 1e2,       1e4, -nan(f16), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f16), -1e0,
                1e3,  -1e-3,      -fmin(f16), -inf(f16), 1e-3, tmin(f16), fmin(f16), 1e1, 1e-4,      -fmax(f16), -1e2,  1e-2, -1e-2, 1e3,  inf(f16), -fmin(f16),
            });

            try testArgs(@Vector(1, f64), @Vector(1, f16), .{
                1e0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, f16), .{
                -inf(f16), -1e-2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, f16), .{
                -1e2, 1e-1, fmax(f16), 1e-2,
            });
            try testArgs(@Vector(8, f64), @Vector(8, f16), .{
                -1e-1, tmin(f16), -1e3, fmin(f16), nan(f16), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, f16), .{
                -fmax(f16), -1e0, 1e-4, 1e2, -fmin(f16), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f16), -tmin(f16), -1e-4, inf(f16), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, f16), .{
                -1e3, -tmin(f16), inf(f16),   -1e4,      -0.0, fmax(f16), 1e2,       1e4, -nan(f16), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f16), -1e0,
                1e3,  -1e-3,      -fmin(f16), -inf(f16), 1e-3, tmin(f16), fmin(f16), 1e1, 1e-4,      -fmax(f16), -1e2,  1e-2, -1e-2, 1e3,  inf(f16), -fmin(f16),
            });

            try testArgs(@Vector(1, f80), @Vector(1, f16), .{
                1e0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, f16), .{
                -inf(f16), -1e-2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, f16), .{
                -1e2, 1e-1, fmax(f16), 1e-2,
            });
            try testArgs(@Vector(8, f80), @Vector(8, f16), .{
                -1e-1, tmin(f16), -1e3, fmin(f16), nan(f16), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, f16), .{
                -fmax(f16), -1e0, 1e-4, 1e2, -fmin(f16), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f16), -tmin(f16), -1e-4, inf(f16), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, f16), .{
                -1e3, -tmin(f16), inf(f16),   -1e4,      -0.0, fmax(f16), 1e2,       1e4, -nan(f16), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f16), -1e0,
                1e3,  -1e-3,      -fmin(f16), -inf(f16), 1e-3, tmin(f16), fmin(f16), 1e1, 1e-4,      -fmax(f16), -1e2,  1e-2, -1e-2, 1e3,  inf(f16), -fmin(f16),
            });

            try testArgs(@Vector(1, f128), @Vector(1, f16), .{
                1e0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, f16), .{
                -inf(f16), -1e-2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, f16), .{
                -1e2, 1e-1, fmax(f16), 1e-2,
            });
            try testArgs(@Vector(8, f128), @Vector(8, f16), .{
                -1e-1, tmin(f16), -1e3, fmin(f16), nan(f16), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, f16), .{
                -fmax(f16), -1e0, 1e-4, 1e2, -fmin(f16), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f16), -tmin(f16), -1e-4, inf(f16), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, f16), .{
                -1e3, -tmin(f16), inf(f16),   -1e4,      -0.0, fmax(f16), 1e2,       1e4, -nan(f16), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f16), -1e0,
                1e3,  -1e-3,      -fmin(f16), -inf(f16), 1e-3, tmin(f16), fmin(f16), 1e1, 1e-4,      -fmax(f16), -1e2,  1e-2, -1e-2, 1e3,  inf(f16), -fmin(f16),
            });

            try testArgs(@Vector(1, f16), @Vector(1, f32), .{
                1e0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, f32), .{
                -inf(f32), -1e-2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, f32), .{
                -1e2, 1e-1, fmax(f32), 1e-2,
            });
            try testArgs(@Vector(8, f16), @Vector(8, f32), .{
                -1e-1, tmin(f32), -1e3, fmin(f32), nan(f32), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, f32), .{
                -fmax(f32), -1e0, 1e-4, 1e2, -fmin(f32), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f32), -tmin(f32), -1e-4, inf(f32), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, f32), .{
                -1e3, -tmin(f32), inf(f32),   -1e4,      -0.0, fmax(f32), 1e2,       1e4, -nan(f32), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f32), -1e0,
                1e3,  -1e-3,      -fmin(f32), -inf(f32), 1e-3, tmin(f32), fmin(f32), 1e1, 1e-4,      -fmax(f32), -1e2,  1e-2, -1e-2, 1e3,  inf(f32), -fmin(f32),
            });

            try testArgs(@Vector(1, f32), @Vector(1, f32), .{
                1e0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, f32), .{
                -inf(f32), -1e-2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, f32), .{
                -1e2, 1e-1, fmax(f32), 1e-2,
            });
            try testArgs(@Vector(8, f32), @Vector(8, f32), .{
                -1e-1, tmin(f32), -1e3, fmin(f32), nan(f32), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, f32), .{
                -fmax(f32), -1e0, 1e-4, 1e2, -fmin(f32), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f32), -tmin(f32), -1e-4, inf(f32), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, f32), .{
                -1e3, -tmin(f32), inf(f32),   -1e4,      -0.0, fmax(f32), 1e2,       1e4, -nan(f32), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f32), -1e0,
                1e3,  -1e-3,      -fmin(f32), -inf(f32), 1e-3, tmin(f32), fmin(f32), 1e1, 1e-4,      -fmax(f32), -1e2,  1e-2, -1e-2, 1e3,  inf(f32), -fmin(f32),
            });

            try testArgs(@Vector(1, f64), @Vector(1, f32), .{
                1e0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, f32), .{
                -inf(f32), -1e-2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, f32), .{
                -1e2, 1e-1, fmax(f32), 1e-2,
            });
            try testArgs(@Vector(8, f64), @Vector(8, f32), .{
                -1e-1, tmin(f32), -1e3, fmin(f32), nan(f32), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, f32), .{
                -fmax(f32), -1e0, 1e-4, 1e2, -fmin(f32), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f32), -tmin(f32), -1e-4, inf(f32), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, f32), .{
                -1e3, -tmin(f32), inf(f32),   -1e4,      -0.0, fmax(f32), 1e2,       1e4, -nan(f32), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f32), -1e0,
                1e3,  -1e-3,      -fmin(f32), -inf(f32), 1e-3, tmin(f32), fmin(f32), 1e1, 1e-4,      -fmax(f32), -1e2,  1e-2, -1e-2, 1e3,  inf(f32), -fmin(f32),
            });

            try testArgs(@Vector(1, f80), @Vector(1, f32), .{
                1e0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, f32), .{
                -inf(f32), -1e-2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, f32), .{
                -1e2, 1e-1, fmax(f32), 1e-2,
            });
            try testArgs(@Vector(8, f80), @Vector(8, f32), .{
                -1e-1, tmin(f32), -1e3, fmin(f32), nan(f32), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, f32), .{
                -fmax(f32), -1e0, 1e-4, 1e2, -fmin(f32), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f32), -tmin(f32), -1e-4, inf(f32), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, f32), .{
                -1e3, -tmin(f32), inf(f32),   -1e4,      -0.0, fmax(f32), 1e2,       1e4, -nan(f32), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f32), -1e0,
                1e3,  -1e-3,      -fmin(f32), -inf(f32), 1e-3, tmin(f32), fmin(f32), 1e1, 1e-4,      -fmax(f32), -1e2,  1e-2, -1e-2, 1e3,  inf(f32), -fmin(f32),
            });

            try testArgs(@Vector(1, f128), @Vector(1, f32), .{
                1e0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, f32), .{
                -inf(f32), -1e-2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, f32), .{
                -1e2, 1e-1, fmax(f32), 1e-2,
            });
            try testArgs(@Vector(8, f128), @Vector(8, f32), .{
                -1e-1, tmin(f32), -1e3, fmin(f32), nan(f32), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, f32), .{
                -fmax(f32), -1e0, 1e-4, 1e2, -fmin(f32), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f32), -tmin(f32), -1e-4, inf(f32), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, f32), .{
                -1e3, -tmin(f32), inf(f32),   -1e4,      -0.0, fmax(f32), 1e2,       1e4, -nan(f32), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f32), -1e0,
                1e3,  -1e-3,      -fmin(f32), -inf(f32), 1e-3, tmin(f32), fmin(f32), 1e1, 1e-4,      -fmax(f32), -1e2,  1e-2, -1e-2, 1e3,  inf(f32), -fmin(f32),
            });

            try testArgs(@Vector(1, f16), @Vector(1, f64), .{
                1e0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, f64), .{
                -inf(f64), -1e-2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, f64), .{
                -1e2, 1e-1, fmax(f64), 1e-2,
            });
            try testArgs(@Vector(8, f16), @Vector(8, f64), .{
                -1e-1, tmin(f64), -1e3, fmin(f64), nan(f64), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, f64), .{
                -fmax(f64), -1e0, 1e-4, 1e2, -fmin(f64), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f64), -tmin(f64), -1e-4, inf(f64), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, f64), .{
                -1e3, -tmin(f64), inf(f64),   -1e4,      -0.0, fmax(f64), 1e2,       1e4, -nan(f64), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f64), -1e0,
                1e3,  -1e-3,      -fmin(f64), -inf(f64), 1e-3, tmin(f64), fmin(f64), 1e1, 1e-4,      -fmax(f64), -1e2,  1e-2, -1e-2, 1e3,  inf(f64), -fmin(f64),
            });

            try testArgs(@Vector(1, f32), @Vector(1, f64), .{
                1e0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, f64), .{
                -inf(f64), -1e-2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, f64), .{
                -1e2, 1e-1, fmax(f64), 1e-2,
            });
            try testArgs(@Vector(8, f32), @Vector(8, f64), .{
                -1e-1, tmin(f64), -1e3, fmin(f64), nan(f64), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, f64), .{
                -fmax(f64), -1e0, 1e-4, 1e2, -fmin(f64), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f64), -tmin(f64), -1e-4, inf(f64), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, f64), .{
                -1e3, -tmin(f64), inf(f64),   -1e4,      -0.0, fmax(f64), 1e2,       1e4, -nan(f64), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f64), -1e0,
                1e3,  -1e-3,      -fmin(f64), -inf(f64), 1e-3, tmin(f64), fmin(f64), 1e1, 1e-4,      -fmax(f64), -1e2,  1e-2, -1e-2, 1e3,  inf(f64), -fmin(f64),
            });

            try testArgs(@Vector(1, f64), @Vector(1, f64), .{
                1e0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, f64), .{
                -inf(f64), -1e-2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, f64), .{
                -1e2, 1e-1, fmax(f64), 1e-2,
            });
            try testArgs(@Vector(8, f64), @Vector(8, f64), .{
                -1e-1, tmin(f64), -1e3, fmin(f64), nan(f64), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, f64), .{
                -fmax(f64), -1e0, 1e-4, 1e2, -fmin(f64), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f64), -tmin(f64), -1e-4, inf(f64), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, f64), .{
                -1e3, -tmin(f64), inf(f64),   -1e4,      -0.0, fmax(f64), 1e2,       1e4, -nan(f64), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f64), -1e0,
                1e3,  -1e-3,      -fmin(f64), -inf(f64), 1e-3, tmin(f64), fmin(f64), 1e1, 1e-4,      -fmax(f64), -1e2,  1e-2, -1e-2, 1e3,  inf(f64), -fmin(f64),
            });

            try testArgs(@Vector(1, f80), @Vector(1, f64), .{
                1e0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, f64), .{
                -inf(f64), -1e-2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, f64), .{
                -1e2, 1e-1, fmax(f64), 1e-2,
            });
            try testArgs(@Vector(8, f80), @Vector(8, f64), .{
                -1e-1, tmin(f64), -1e3, fmin(f64), nan(f64), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, f64), .{
                -fmax(f64), -1e0, 1e-4, 1e2, -fmin(f64), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f64), -tmin(f64), -1e-4, inf(f64), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, f64), .{
                -1e3, -tmin(f64), inf(f64),   -1e4,      -0.0, fmax(f64), 1e2,       1e4, -nan(f64), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f64), -1e0,
                1e3,  -1e-3,      -fmin(f64), -inf(f64), 1e-3, tmin(f64), fmin(f64), 1e1, 1e-4,      -fmax(f64), -1e2,  1e-2, -1e-2, 1e3,  inf(f64), -fmin(f64),
            });

            try testArgs(@Vector(1, f128), @Vector(1, f64), .{
                1e0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, f64), .{
                -inf(f64), -1e-2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, f64), .{
                -1e2, 1e-1, fmax(f64), 1e-2,
            });
            try testArgs(@Vector(8, f128), @Vector(8, f64), .{
                -1e-1, tmin(f64), -1e3, fmin(f64), nan(f64), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, f64), .{
                -fmax(f64), -1e0, 1e-4, 1e2, -fmin(f64), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f64), -tmin(f64), -1e-4, inf(f64), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, f64), .{
                -1e3, -tmin(f64), inf(f64),   -1e4,      -0.0, fmax(f64), 1e2,       1e4, -nan(f64), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f64), -1e0,
                1e3,  -1e-3,      -fmin(f64), -inf(f64), 1e-3, tmin(f64), fmin(f64), 1e1, 1e-4,      -fmax(f64), -1e2,  1e-2, -1e-2, 1e3,  inf(f64), -fmin(f64),
            });

            try testArgs(@Vector(1, f16), @Vector(1, f80), .{
                1e0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, f80), .{
                -inf(f80), -1e-2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, f80), .{
                -1e2, 1e-1, fmax(f80), 1e-2,
            });
            try testArgs(@Vector(8, f16), @Vector(8, f80), .{
                -1e-1, tmin(f80), -1e3, fmin(f80), nan(f80), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, f80), .{
                -fmax(f80), -1e0, 1e-4, 1e2, -fmin(f80), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f80), -tmin(f80), -1e-4, inf(f80), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, f80), .{
                -1e3, -tmin(f80), inf(f80),   -1e4,      -0.0, fmax(f80), 1e2,       1e4, -nan(f80), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f80), -1e0,
                1e3,  -1e-3,      -fmin(f80), -inf(f80), 1e-3, tmin(f80), fmin(f80), 1e1, 1e-4,      -fmax(f80), -1e2,  1e-2, -1e-2, 1e3,  inf(f80), -fmin(f80),
            });

            try testArgs(@Vector(1, f32), @Vector(1, f80), .{
                1e0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, f80), .{
                -inf(f80), -1e-2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, f80), .{
                -1e2, 1e-1, fmax(f80), 1e-2,
            });
            try testArgs(@Vector(8, f32), @Vector(8, f80), .{
                -1e-1, tmin(f80), -1e3, fmin(f80), nan(f80), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, f80), .{
                -fmax(f80), -1e0, 1e-4, 1e2, -fmin(f80), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f80), -tmin(f80), -1e-4, inf(f80), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, f80), .{
                -1e3, -tmin(f80), inf(f80),   -1e4,      -0.0, fmax(f80), 1e2,       1e4, -nan(f80), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f80), -1e0,
                1e3,  -1e-3,      -fmin(f80), -inf(f80), 1e-3, tmin(f80), fmin(f80), 1e1, 1e-4,      -fmax(f80), -1e2,  1e-2, -1e-2, 1e3,  inf(f80), -fmin(f80),
            });

            try testArgs(@Vector(1, f64), @Vector(1, f80), .{
                1e0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, f80), .{
                -inf(f80), -1e-2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, f80), .{
                -1e2, 1e-1, fmax(f80), 1e-2,
            });
            try testArgs(@Vector(8, f64), @Vector(8, f80), .{
                -1e-1, tmin(f80), -1e3, fmin(f80), nan(f80), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, f80), .{
                -fmax(f80), -1e0, 1e-4, 1e2, -fmin(f80), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f80), -tmin(f80), -1e-4, inf(f80), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, f80), .{
                -1e3, -tmin(f80), inf(f80),   -1e4,      -0.0, fmax(f80), 1e2,       1e4, -nan(f80), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f80), -1e0,
                1e3,  -1e-3,      -fmin(f80), -inf(f80), 1e-3, tmin(f80), fmin(f80), 1e1, 1e-4,      -fmax(f80), -1e2,  1e-2, -1e-2, 1e3,  inf(f80), -fmin(f80),
            });

            try testArgs(@Vector(1, f80), @Vector(1, f80), .{
                1e0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, f80), .{
                -inf(f80), -1e-2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, f80), .{
                -1e2, 1e-1, fmax(f80), 1e-2,
            });
            try testArgs(@Vector(8, f80), @Vector(8, f80), .{
                -1e-1, tmin(f80), -1e3, fmin(f80), nan(f80), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, f80), .{
                -fmax(f80), -1e0, 1e-4, 1e2, -fmin(f80), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f80), -tmin(f80), -1e-4, inf(f80), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, f80), .{
                -1e3, -tmin(f80), inf(f80),   -1e4,      -0.0, fmax(f80), 1e2,       1e4, -nan(f80), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f80), -1e0,
                1e3,  -1e-3,      -fmin(f80), -inf(f80), 1e-3, tmin(f80), fmin(f80), 1e1, 1e-4,      -fmax(f80), -1e2,  1e-2, -1e-2, 1e3,  inf(f80), -fmin(f80),
            });

            try testArgs(@Vector(1, f128), @Vector(1, f80), .{
                1e0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, f80), .{
                -inf(f80), -1e-2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, f80), .{
                -1e2, 1e-1, fmax(f80), 1e-2,
            });
            try testArgs(@Vector(8, f128), @Vector(8, f80), .{
                -1e-1, tmin(f80), -1e3, fmin(f80), nan(f80), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, f80), .{
                -fmax(f80), -1e0, 1e-4, 1e2, -fmin(f80), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f80), -tmin(f80), -1e-4, inf(f80), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, f80), .{
                -1e3, -tmin(f80), inf(f80),   -1e4,      -0.0, fmax(f80), 1e2,       1e4, -nan(f80), 0.0,        -1e-4, -1e1, 1e0,   1e-1, nan(f80), -1e0,
                1e3,  -1e-3,      -fmin(f80), -inf(f80), 1e-3, tmin(f80), fmin(f80), 1e1, 1e-4,      -fmax(f80), -1e2,  1e-2, -1e-2, 1e3,  inf(f80), -fmin(f80),
            });

            try testArgs(@Vector(1, f16), @Vector(1, f128), .{
                1e0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, f128), .{
                -inf(f128), -1e-2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, f128), .{
                -1e2, 1e-1, fmax(f128), 1e-2,
            });
            try testArgs(@Vector(8, f16), @Vector(8, f128), .{
                -1e-1, tmin(f128), -1e3, fmin(f128), nan(f128), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, f128), .{
                -fmax(f128), -1e0, 1e-4, 1e2, -fmin(f128), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f128), -tmin(f128), -1e-4, inf(f128), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, f128), .{
                -1e3, -tmin(f128), inf(f128),   -1e4,       -0.0, fmax(f128), 1e2,        1e4, -nan(f128), 0.0,         -1e-4, -1e1, 1e0,   1e-1, nan(f128), -1e0,
                1e3,  -1e-3,       -fmin(f128), -inf(f128), 1e-3, tmin(f128), fmin(f128), 1e1, 1e-4,       -fmax(f128), -1e2,  1e-2, -1e-2, 1e3,  inf(f128), -fmin(f128),
            });

            try testArgs(@Vector(1, f32), @Vector(1, f128), .{
                1e0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, f128), .{
                -inf(f128), -1e-2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, f128), .{
                -1e2, 1e-1, fmax(f128), 1e-2,
            });
            try testArgs(@Vector(8, f32), @Vector(8, f128), .{
                -1e-1, tmin(f128), -1e3, fmin(f128), nan(f128), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, f128), .{
                -fmax(f128), -1e0, 1e-4, 1e2, -fmin(f128), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f128), -tmin(f128), -1e-4, inf(f128), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, f128), .{
                -1e3, -tmin(f128), inf(f128),   -1e4,       -0.0, fmax(f128), 1e2,        1e4, -nan(f128), 0.0,         -1e-4, -1e1, 1e0,   1e-1, nan(f128), -1e0,
                1e3,  -1e-3,       -fmin(f128), -inf(f128), 1e-3, tmin(f128), fmin(f128), 1e1, 1e-4,       -fmax(f128), -1e2,  1e-2, -1e-2, 1e3,  inf(f128), -fmin(f128),
            });

            try testArgs(@Vector(1, f64), @Vector(1, f128), .{
                1e0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, f128), .{
                -inf(f128), -1e-2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, f128), .{
                -1e2, 1e-1, fmax(f128), 1e-2,
            });
            try testArgs(@Vector(8, f64), @Vector(8, f128), .{
                -1e-1, tmin(f128), -1e3, fmin(f128), nan(f128), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, f128), .{
                -fmax(f128), -1e0, 1e-4, 1e2, -fmin(f128), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f128), -tmin(f128), -1e-4, inf(f128), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, f128), .{
                -1e3, -tmin(f128), inf(f128),   -1e4,       -0.0, fmax(f128), 1e2,        1e4, -nan(f128), 0.0,         -1e-4, -1e1, 1e0,   1e-1, nan(f128), -1e0,
                1e3,  -1e-3,       -fmin(f128), -inf(f128), 1e-3, tmin(f128), fmin(f128), 1e1, 1e-4,       -fmax(f128), -1e2,  1e-2, -1e-2, 1e3,  inf(f128), -fmin(f128),
            });

            try testArgs(@Vector(1, f80), @Vector(1, f128), .{
                1e0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, f128), .{
                -inf(f128), -1e-2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, f128), .{
                -1e2, 1e-1, fmax(f128), 1e-2,
            });
            try testArgs(@Vector(8, f80), @Vector(8, f128), .{
                -1e-1, tmin(f128), -1e3, fmin(f128), nan(f128), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, f128), .{
                -fmax(f128), -1e0, 1e-4, 1e2, -fmin(f128), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f128), -tmin(f128), -1e-4, inf(f128), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, f128), .{
                -1e3, -tmin(f128), inf(f128),   -1e4,       -0.0, fmax(f128), 1e2,        1e4, -nan(f128), 0.0,         -1e-4, -1e1, 1e0,   1e-1, nan(f128), -1e0,
                1e3,  -1e-3,       -fmin(f128), -inf(f128), 1e-3, tmin(f128), fmin(f128), 1e1, 1e-4,       -fmax(f128), -1e2,  1e-2, -1e-2, 1e3,  inf(f128), -fmin(f128),
            });

            try testArgs(@Vector(1, f128), @Vector(1, f128), .{
                1e0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, f128), .{
                -inf(f128), -1e-2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, f128), .{
                -1e2, 1e-1, fmax(f128), 1e-2,
            });
            try testArgs(@Vector(8, f128), @Vector(8, f128), .{
                -1e-1, tmin(f128), -1e3, fmin(f128), nan(f128), -1e-3, 1e1, 1e4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, f128), .{
                -fmax(f128), -1e0, 1e-4, 1e2, -fmin(f128), -1e1, 0.0, -1e4, -0.0, 1e3, -nan(f128), -tmin(f128), -1e-4, inf(f128), 1e-3, -1e-1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, f128), .{
                -1e3, -tmin(f128), inf(f128),   -1e4,       -0.0, fmax(f128), 1e2,        1e4, -nan(f128), 0.0,         -1e-4, -1e1, 1e0,   1e-1, nan(f128), -1e0,
                1e3,  -1e-3,       -fmin(f128), -inf(f128), 1e-3, tmin(f128), fmin(f128), 1e1, 1e-4,       -fmax(f128), -1e2,  1e-2, -1e-2, 1e3,  inf(f128), -fmin(f128),
            });
        }
        fn testIntsFromFloats() !void {
            @setEvalBranchQuota(2_600);

            try testArgs(i8, f16, -0x0.8p8);
            try testArgs(i8, f16, next(f16, -0x0.8p8, -0.0));
            try testArgs(i8, f16, next(f16, next(f16, -0x0.8p8, -0.0), -0.0));
            try testArgs(i8, f16, -1e2);
            try testArgs(i8, f16, -1e1);
            try testArgs(i8, f16, -1e0);
            try testArgs(i8, f16, -1e-1);
            try testArgs(i8, f16, -0.0);
            try testArgs(i8, f16, 0.0);
            try testArgs(i8, f16, 1e-1);
            try testArgs(i8, f16, 1e0);
            try testArgs(i8, f16, 1e1);
            try testArgs(i8, f16, 1e2);
            try testArgs(i8, f16, next(f16, next(f16, 0x0.8p8, 0.0), 0.0));
            try testArgs(i8, f16, next(f16, 0x0.8p8, 0.0));

            try testArgs(u8, f16, -0.0);
            try testArgs(u8, f16, 0.0);
            try testArgs(u8, f16, 1e-1);
            try testArgs(u8, f16, 1e0);
            try testArgs(u8, f16, 1e1);
            try testArgs(u8, f16, 1e2);
            try testArgs(u8, f16, next(f16, next(f16, 0x1p8, 0.0), 0.0));
            try testArgs(u8, f16, next(f16, 0x1p8, 0.0));

            try testArgs(i16, f16, -1e4);
            try testArgs(i16, f16, -1e3);
            try testArgs(i16, f16, -1e2);
            try testArgs(i16, f16, -1e1);
            try testArgs(i16, f16, -1e0);
            try testArgs(i16, f16, -1e-1);
            try testArgs(i16, f16, -0.0);
            try testArgs(i16, f16, 0.0);
            try testArgs(i16, f16, 1e-1);
            try testArgs(i16, f16, 1e0);
            try testArgs(i16, f16, 1e1);
            try testArgs(i16, f16, 1e2);
            try testArgs(i16, f16, 1e3);
            try testArgs(i16, f16, 1e4);
            try testArgs(i16, f16, next(f16, next(f16, 0x0.8p16, 0.0), 0.0));
            try testArgs(i16, f16, next(f16, 0x0.8p16, 0.0));

            try testArgs(u16, f16, -0.0);
            try testArgs(u16, f16, 0.0);
            try testArgs(u16, f16, 1e-1);
            try testArgs(u16, f16, 1e0);
            try testArgs(u16, f16, 1e1);
            try testArgs(u16, f16, 1e2);
            try testArgs(u16, f16, 1e3);
            try testArgs(u16, f16, 1e4);
            try testArgs(u16, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(u16, f16, next(f16, fmax(f16), 0.0));
            try testArgs(u16, f16, fmax(f16));

            try testArgs(i32, f16, -fmax(f16));
            try testArgs(i32, f16, next(f16, -fmax(f16), -0.0));
            try testArgs(i32, f16, next(f16, next(f16, -fmax(f16), -0.0), -0.0));
            try testArgs(i32, f16, -1e4);
            try testArgs(i32, f16, -1e3);
            try testArgs(i32, f16, -1e2);
            try testArgs(i32, f16, -1e1);
            try testArgs(i32, f16, -1e0);
            try testArgs(i32, f16, -1e-1);
            try testArgs(i32, f16, -0.0);
            try testArgs(i32, f16, 0.0);
            try testArgs(i32, f16, 1e-1);
            try testArgs(i32, f16, 1e0);
            try testArgs(i32, f16, 1e1);
            try testArgs(i32, f16, 1e2);
            try testArgs(i32, f16, 1e3);
            try testArgs(i32, f16, 1e4);
            try testArgs(i32, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(i32, f16, next(f16, fmax(f16), 0.0));
            try testArgs(i32, f16, fmax(f16));

            try testArgs(u32, f16, -0.0);
            try testArgs(u32, f16, 0.0);
            try testArgs(u32, f16, 1e-1);
            try testArgs(u32, f16, 1e0);
            try testArgs(u32, f16, 1e1);
            try testArgs(u32, f16, 1e2);
            try testArgs(u32, f16, 1e3);
            try testArgs(u32, f16, 1e4);
            try testArgs(u32, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(u32, f16, next(f16, fmax(f16), 0.0));
            try testArgs(u32, f16, fmax(f16));

            try testArgs(i64, f16, -fmax(f16));
            try testArgs(i64, f16, next(f16, -fmax(f16), -0.0));
            try testArgs(i64, f16, next(f16, next(f16, -fmax(f16), -0.0), -0.0));
            try testArgs(i64, f16, -1e4);
            try testArgs(i64, f16, -1e3);
            try testArgs(i64, f16, -1e2);
            try testArgs(i64, f16, -1e1);
            try testArgs(i64, f16, -1e0);
            try testArgs(i64, f16, -1e-1);
            try testArgs(i64, f16, -0.0);
            try testArgs(i64, f16, 0.0);
            try testArgs(i64, f16, 1e-1);
            try testArgs(i64, f16, 1e0);
            try testArgs(i64, f16, 1e1);
            try testArgs(i64, f16, 1e2);
            try testArgs(i64, f16, 1e3);
            try testArgs(i64, f16, 1e4);
            try testArgs(i64, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(i64, f16, next(f16, fmax(f16), 0.0));
            try testArgs(i64, f16, fmax(f16));

            try testArgs(u64, f16, -0.0);
            try testArgs(u64, f16, 0.0);
            try testArgs(u64, f16, 1e-1);
            try testArgs(u64, f16, 1e0);
            try testArgs(u64, f16, 1e1);
            try testArgs(u64, f16, 1e2);
            try testArgs(u64, f16, 1e3);
            try testArgs(u64, f16, 1e4);
            try testArgs(u64, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(u64, f16, next(f16, fmax(f16), 0.0));
            try testArgs(u64, f16, fmax(f16));

            try testArgs(i128, f16, -fmax(f16));
            try testArgs(i128, f16, next(f16, -fmax(f16), -0.0));
            try testArgs(i128, f16, next(f16, next(f16, -fmax(f16), -0.0), -0.0));
            try testArgs(i128, f16, -1e4);
            try testArgs(i128, f16, -1e3);
            try testArgs(i128, f16, -1e2);
            try testArgs(i128, f16, -1e1);
            try testArgs(i128, f16, -1e0);
            try testArgs(i128, f16, -1e-1);
            try testArgs(i128, f16, -0.0);
            try testArgs(i128, f16, 0.0);
            try testArgs(i128, f16, 1e-1);
            try testArgs(i128, f16, 1e0);
            try testArgs(i128, f16, 1e1);
            try testArgs(i128, f16, 1e2);
            try testArgs(i128, f16, 1e3);
            try testArgs(i128, f16, 1e4);
            try testArgs(i128, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(i128, f16, next(f16, fmax(f16), 0.0));
            try testArgs(i128, f16, fmax(f16));

            try testArgs(u128, f16, -0.0);
            try testArgs(u128, f16, 0.0);
            try testArgs(u128, f16, 1e-1);
            try testArgs(u128, f16, 1e0);
            try testArgs(u128, f16, 1e1);
            try testArgs(u128, f16, 1e2);
            try testArgs(u128, f16, 1e3);
            try testArgs(u128, f16, 1e4);
            try testArgs(u128, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(u128, f16, next(f16, fmax(f16), 0.0));
            try testArgs(u128, f16, fmax(f16));

            try testArgs(i256, f16, -fmax(f16));
            try testArgs(i256, f16, next(f16, -fmax(f16), -0.0));
            try testArgs(i256, f16, next(f16, next(f16, -fmax(f16), -0.0), -0.0));
            try testArgs(i256, f16, -1e4);
            try testArgs(i256, f16, -1e3);
            try testArgs(i256, f16, -1e2);
            try testArgs(i256, f16, -1e1);
            try testArgs(i256, f16, -1e0);
            try testArgs(i256, f16, -1e-1);
            try testArgs(i256, f16, -0.0);
            try testArgs(i256, f16, 0.0);
            try testArgs(i256, f16, 1e-1);
            try testArgs(i256, f16, 1e0);
            try testArgs(i256, f16, 1e1);
            try testArgs(i256, f16, 1e2);
            try testArgs(i256, f16, 1e3);
            try testArgs(i256, f16, 1e4);
            try testArgs(i256, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(i256, f16, next(f16, fmax(f16), 0.0));
            try testArgs(i256, f16, fmax(f16));

            try testArgs(u256, f16, -0.0);
            try testArgs(u256, f16, 0.0);
            try testArgs(u256, f16, 1e-1);
            try testArgs(u256, f16, 1e0);
            try testArgs(u256, f16, 1e1);
            try testArgs(u256, f16, 1e2);
            try testArgs(u256, f16, 1e3);
            try testArgs(u256, f16, 1e4);
            try testArgs(u256, f16, next(f16, next(f16, fmax(f16), 0.0), 0.0));
            try testArgs(u256, f16, next(f16, fmax(f16), 0.0));
            try testArgs(u256, f16, fmax(f16));

            try testArgs(i8, f32, -0x0.8p8);
            try testArgs(i8, f32, next(f32, -0x0.8p8, -0.0));
            try testArgs(i8, f32, next(f32, next(f32, -0x0.8p8, -0.0), -0.0));
            try testArgs(i8, f32, -1e2);
            try testArgs(i8, f32, -1e1);
            try testArgs(i8, f32, -1e0);
            try testArgs(i8, f32, -1e-1);
            try testArgs(i8, f32, -0.0);
            try testArgs(i8, f32, 0.0);
            try testArgs(i8, f32, 1e-1);
            try testArgs(i8, f32, 1e0);
            try testArgs(i8, f32, 1e1);
            try testArgs(i8, f32, 1e2);
            try testArgs(i8, f32, next(f32, next(f32, 0x0.8p8, 0.0), 0.0));
            try testArgs(i8, f32, next(f32, 0x0.8p8, 0.0));

            try testArgs(u8, f32, -0.0);
            try testArgs(u8, f32, 0.0);
            try testArgs(u8, f32, 1e-1);
            try testArgs(u8, f32, 1e0);
            try testArgs(u8, f32, 1e1);
            try testArgs(u8, f32, 1e2);
            try testArgs(u8, f32, next(f32, next(f32, 0x1p8, 0.0), 0.0));
            try testArgs(u8, f32, next(f32, 0x1p8, 0.0));

            try testArgs(i16, f32, -0x0.8p16);
            try testArgs(i16, f32, next(f32, -0x0.8p16, -0.0));
            try testArgs(i16, f32, next(f32, next(f32, -0x0.8p16, -0.0), -0.0));
            try testArgs(i16, f32, -1e4);
            try testArgs(i16, f32, -1e3);
            try testArgs(i16, f32, -1e2);
            try testArgs(i16, f32, -1e1);
            try testArgs(i16, f32, -1e0);
            try testArgs(i16, f32, -1e-1);
            try testArgs(i16, f32, -0.0);
            try testArgs(i16, f32, 0.0);
            try testArgs(i16, f32, 1e-1);
            try testArgs(i16, f32, 1e0);
            try testArgs(i16, f32, 1e1);
            try testArgs(i16, f32, 1e2);
            try testArgs(i16, f32, 1e3);
            try testArgs(i16, f32, 1e4);
            try testArgs(i16, f32, next(f32, next(f32, 0x0.8p16, 0.0), 0.0));
            try testArgs(i16, f32, next(f32, 0x0.8p16, 0.0));

            try testArgs(u16, f32, -0.0);
            try testArgs(u16, f32, 0.0);
            try testArgs(u16, f32, 1e-1);
            try testArgs(u16, f32, 1e0);
            try testArgs(u16, f32, 1e1);
            try testArgs(u16, f32, 1e2);
            try testArgs(u16, f32, 1e3);
            try testArgs(u16, f32, 1e4);
            try testArgs(u16, f32, next(f32, next(f32, 0x1p16, 0.0), 0.0));
            try testArgs(u16, f32, next(f32, 0x1p16, 0.0));

            try testArgs(i32, f32, -0x0.8p32);
            try testArgs(i32, f32, next(f32, -0x0.8p32, -0.0));
            try testArgs(i32, f32, next(f32, next(f32, -0x0.8p32, -0.0), -0.0));
            try testArgs(i32, f32, -1e9);
            try testArgs(i32, f32, -1e8);
            try testArgs(i32, f32, -1e7);
            try testArgs(i32, f32, -1e6);
            try testArgs(i32, f32, -1e5);
            try testArgs(i32, f32, -1e4);
            try testArgs(i32, f32, -1e3);
            try testArgs(i32, f32, -1e2);
            try testArgs(i32, f32, -1e1);
            try testArgs(i32, f32, -1e0);
            try testArgs(i32, f32, -1e-1);
            try testArgs(i32, f32, -0.0);
            try testArgs(i32, f32, 0.0);
            try testArgs(i32, f32, 1e-1);
            try testArgs(i32, f32, 1e0);
            try testArgs(i32, f32, 1e1);
            try testArgs(i32, f32, 1e2);
            try testArgs(i32, f32, 1e3);
            try testArgs(i32, f32, 1e4);
            try testArgs(i32, f32, 1e5);
            try testArgs(i32, f32, 1e6);
            try testArgs(i32, f32, 1e7);
            try testArgs(i32, f32, 1e8);
            try testArgs(i32, f32, 1e9);
            try testArgs(i32, f32, next(f32, next(f32, 0x0.8p32, 0.0), 0.0));
            try testArgs(i32, f32, next(f32, 0x0.8p32, 0.0));

            try testArgs(u32, f32, -0.0);
            try testArgs(u32, f32, 0.0);
            try testArgs(u32, f32, 1e-1);
            try testArgs(u32, f32, 1e0);
            try testArgs(u32, f32, 1e1);
            try testArgs(u32, f32, 1e2);
            try testArgs(u32, f32, 1e3);
            try testArgs(u32, f32, 1e4);
            try testArgs(u32, f32, 1e5);
            try testArgs(u32, f32, 1e6);
            try testArgs(u32, f32, 1e7);
            try testArgs(u32, f32, 1e8);
            try testArgs(u32, f32, 1e9);
            try testArgs(u32, f32, next(f32, next(f32, 0x1p32, 0.0), 0.0));
            try testArgs(u32, f32, next(f32, 0x1p32, 0.0));

            try testArgs(i64, f32, -0x0.8p64);
            try testArgs(i64, f32, next(f32, -0x0.8p64, -0.0));
            try testArgs(i64, f32, next(f32, next(f32, -0x0.8p64, -0.0), -0.0));
            try testArgs(i64, f32, -1e18);
            try testArgs(i64, f32, -1e16);
            try testArgs(i64, f32, -1e14);
            try testArgs(i64, f32, -1e12);
            try testArgs(i64, f32, -1e10);
            try testArgs(i64, f32, -1e8);
            try testArgs(i64, f32, -1e6);
            try testArgs(i64, f32, -1e4);
            try testArgs(i64, f32, -1e2);
            try testArgs(i64, f32, -1e0);
            try testArgs(i64, f32, -1e-1);
            try testArgs(i64, f32, -0.0);
            try testArgs(i64, f32, 0.0);
            try testArgs(i64, f32, 1e-1);
            try testArgs(i64, f32, 1e0);
            try testArgs(i64, f32, 1e2);
            try testArgs(i64, f32, 1e4);
            try testArgs(i64, f32, 1e6);
            try testArgs(i64, f32, 1e8);
            try testArgs(i64, f32, 1e10);
            try testArgs(i64, f32, 1e12);
            try testArgs(i64, f32, 1e14);
            try testArgs(i64, f32, 1e16);
            try testArgs(i64, f32, 1e18);
            try testArgs(i64, f32, next(f32, next(f32, 0x0.8p64, 0.0), 0.0));
            try testArgs(i64, f32, next(f32, 0x0.8p64, 0.0));

            try testArgs(u64, f32, -0.0);
            try testArgs(u64, f32, 0.0);
            try testArgs(u64, f32, 1e-1);
            try testArgs(u64, f32, 1e0);
            try testArgs(u64, f32, 1e2);
            try testArgs(u64, f32, 1e4);
            try testArgs(u64, f32, 1e6);
            try testArgs(u64, f32, 1e8);
            try testArgs(u64, f32, 1e10);
            try testArgs(u64, f32, 1e12);
            try testArgs(u64, f32, 1e14);
            try testArgs(u64, f32, 1e16);
            try testArgs(u64, f32, 1e18);
            try testArgs(u64, f32, next(f32, next(f32, 0x1p64, 0.0), 0.0));
            try testArgs(u64, f32, next(f32, 0x1p64, 0.0));

            try testArgs(i128, f32, -0x0.8p128);
            try testArgs(i128, f32, next(f32, -0x0.8p128, -0.0));
            try testArgs(i128, f32, next(f32, next(f32, -0x0.8p128, -0.0), -0.0));
            try testArgs(i128, f32, -1e38);
            try testArgs(i128, f32, -1e34);
            try testArgs(i128, f32, -1e30);
            try testArgs(i128, f32, -1e26);
            try testArgs(i128, f32, -1e22);
            try testArgs(i128, f32, -1e18);
            try testArgs(i128, f32, -1e14);
            try testArgs(i128, f32, -1e10);
            try testArgs(i128, f32, -1e6);
            try testArgs(i128, f32, -1e2);
            try testArgs(i128, f32, -1e0);
            try testArgs(i128, f32, -1e-1);
            try testArgs(i128, f32, -0.0);
            try testArgs(i128, f32, 0.0);
            try testArgs(i128, f32, 1e-1);
            try testArgs(i128, f32, 1e0);
            try testArgs(i128, f32, 1e2);
            try testArgs(i128, f32, 1e6);
            try testArgs(i128, f32, 1e10);
            try testArgs(i128, f32, 1e14);
            try testArgs(i128, f32, 1e18);
            try testArgs(i128, f32, 1e22);
            try testArgs(i128, f32, 1e26);
            try testArgs(i128, f32, 1e30);
            try testArgs(i128, f32, 1e34);
            try testArgs(i128, f32, 1e38);
            try testArgs(i128, f32, next(f32, next(f32, 0x0.8p128, 0.0), 0.0));
            try testArgs(i128, f32, next(f32, 0x0.8p128, 0.0));

            try testArgs(u128, f32, -0.0);
            try testArgs(u128, f32, 0.0);
            try testArgs(u128, f32, 1e-1);
            try testArgs(u128, f32, 1e0);
            try testArgs(u128, f32, 1e2);
            try testArgs(u128, f32, 1e6);
            try testArgs(u128, f32, 1e10);
            try testArgs(u128, f32, 1e14);
            try testArgs(u128, f32, 1e18);
            try testArgs(u128, f32, 1e22);
            try testArgs(u128, f32, 1e26);
            try testArgs(u128, f32, 1e30);
            try testArgs(u128, f32, 1e34);
            try testArgs(u128, f32, 1e38);
            try testArgs(u128, f32, next(f32, next(f32, fmax(f32), 0.0), 0.0));
            try testArgs(u128, f32, next(f32, fmax(f32), 0.0));

            try testArgs(i256, f32, -fmax(f32));
            try testArgs(i256, f32, next(f32, -fmax(f32), -0.0));
            try testArgs(i256, f32, next(f32, next(f32, -fmax(f32), -0.0), -0.0));
            try testArgs(i256, f32, -1e38);
            try testArgs(i256, f32, -1e34);
            try testArgs(i256, f32, -1e30);
            try testArgs(i256, f32, -1e26);
            try testArgs(i256, f32, -1e22);
            try testArgs(i256, f32, -1e18);
            try testArgs(i256, f32, -1e14);
            try testArgs(i256, f32, -1e10);
            try testArgs(i256, f32, -1e6);
            try testArgs(i256, f32, -1e2);
            try testArgs(i256, f32, -1e0);
            try testArgs(i256, f32, -1e-1);
            try testArgs(i256, f32, -0.0);
            try testArgs(i256, f32, 0.0);
            try testArgs(i256, f32, 1e-1);
            try testArgs(i256, f32, 1e0);
            try testArgs(i256, f32, 1e2);
            try testArgs(i256, f32, 1e6);
            try testArgs(i256, f32, 1e10);
            try testArgs(i256, f32, 1e14);
            try testArgs(i256, f32, 1e18);
            try testArgs(i256, f32, 1e22);
            try testArgs(i256, f32, 1e26);
            try testArgs(i256, f32, 1e30);
            try testArgs(i256, f32, 1e34);
            try testArgs(i256, f32, 1e38);
            try testArgs(i256, f32, next(f32, next(f32, fmax(f32), 0.0), 0.0));
            try testArgs(i256, f32, next(f32, fmax(f32), 0.0));

            try testArgs(u256, f32, -0.0);
            try testArgs(u256, f32, 0.0);
            try testArgs(u256, f32, 1e-1);
            try testArgs(u256, f32, 1e0);
            try testArgs(u256, f32, 1e2);
            try testArgs(u256, f32, 1e6);
            try testArgs(u256, f32, 1e10);
            try testArgs(u256, f32, 1e14);
            try testArgs(u256, f32, 1e18);
            try testArgs(u256, f32, 1e22);
            try testArgs(u256, f32, 1e26);
            try testArgs(u256, f32, 1e30);
            try testArgs(u256, f32, 1e34);
            try testArgs(u256, f32, 1e38);
            try testArgs(u256, f32, next(f32, next(f32, fmax(f32), 0.0), 0.0));
            try testArgs(u256, f32, next(f32, fmax(f32), 0.0));

            try testArgs(i8, f64, -0x0.8p8);
            try testArgs(i8, f64, next(f64, -0x0.8p8, -0.0));
            try testArgs(i8, f64, next(f64, next(f64, -0x0.8p8, -0.0), -0.0));
            try testArgs(i8, f64, -1e2);
            try testArgs(i8, f64, -1e1);
            try testArgs(i8, f64, -1e0);
            try testArgs(i8, f64, -1e-1);
            try testArgs(i8, f64, -0.0);
            try testArgs(i8, f64, 0.0);
            try testArgs(i8, f64, 1e-1);
            try testArgs(i8, f64, 1e0);
            try testArgs(i8, f64, 1e1);
            try testArgs(i8, f64, 1e2);
            try testArgs(i8, f64, next(f64, next(f64, 0x0.8p8, 0.0), 0.0));
            try testArgs(i8, f64, next(f64, 0x0.8p8, 0.0));

            try testArgs(u8, f64, -0.0);
            try testArgs(u8, f64, 0.0);
            try testArgs(u8, f64, 1e-1);
            try testArgs(u8, f64, 1e0);
            try testArgs(u8, f64, 1e1);
            try testArgs(u8, f64, 1e2);
            try testArgs(u8, f64, next(f64, next(f64, 0x1p8, 0.0), 0.0));
            try testArgs(u8, f64, next(f64, 0x1p8, 0.0));

            try testArgs(i16, f64, -0x0.8p16);
            try testArgs(i16, f64, next(f64, -0x0.8p16, -0.0));
            try testArgs(i16, f64, next(f64, next(f64, -0x0.8p16, -0.0), -0.0));
            try testArgs(i16, f64, -1e4);
            try testArgs(i16, f64, -1e3);
            try testArgs(i16, f64, -1e2);
            try testArgs(i16, f64, -1e1);
            try testArgs(i16, f64, -1e0);
            try testArgs(i16, f64, -1e-1);
            try testArgs(i16, f64, -0.0);
            try testArgs(i16, f64, 0.0);
            try testArgs(i16, f64, 1e-1);
            try testArgs(i16, f64, 1e0);
            try testArgs(i16, f64, 1e1);
            try testArgs(i16, f64, 1e2);
            try testArgs(i16, f64, 1e3);
            try testArgs(i16, f64, 1e4);
            try testArgs(i16, f64, next(f64, next(f64, 0x0.8p16, 0.0), 0.0));
            try testArgs(i16, f64, next(f64, 0x0.8p16, 0.0));

            try testArgs(u16, f64, -0.0);
            try testArgs(u16, f64, 0.0);
            try testArgs(u16, f64, 1e-1);
            try testArgs(u16, f64, 1e0);
            try testArgs(u16, f64, 1e1);
            try testArgs(u16, f64, 1e2);
            try testArgs(u16, f64, 1e3);
            try testArgs(u16, f64, 1e4);
            try testArgs(u16, f64, next(f64, next(f64, 0x1p16, 0.0), 0.0));
            try testArgs(u16, f64, next(f64, 0x1p16, 0.0));

            try testArgs(i32, f64, -0x0.8p32);
            try testArgs(i32, f64, next(f64, -0x0.8p32, -0.0));
            try testArgs(i32, f64, next(f64, next(f64, -0x0.8p32, -0.0), -0.0));
            try testArgs(i32, f64, -1e9);
            try testArgs(i32, f64, -1e8);
            try testArgs(i32, f64, -1e7);
            try testArgs(i32, f64, -1e6);
            try testArgs(i32, f64, -1e5);
            try testArgs(i32, f64, -1e4);
            try testArgs(i32, f64, -1e3);
            try testArgs(i32, f64, -1e2);
            try testArgs(i32, f64, -1e1);
            try testArgs(i32, f64, -1e0);
            try testArgs(i32, f64, -1e-1);
            try testArgs(i32, f64, -0.0);
            try testArgs(i32, f64, 0.0);
            try testArgs(i32, f64, 1e-1);
            try testArgs(i32, f64, 1e0);
            try testArgs(i32, f64, 1e1);
            try testArgs(i32, f64, 1e2);
            try testArgs(i32, f64, 1e3);
            try testArgs(i32, f64, 1e4);
            try testArgs(i32, f64, 1e5);
            try testArgs(i32, f64, 1e6);
            try testArgs(i32, f64, 1e7);
            try testArgs(i32, f64, 1e8);
            try testArgs(i32, f64, 1e9);
            try testArgs(i32, f64, next(f64, next(f64, 0x0.8p32, 0.0), 0.0));
            try testArgs(i32, f64, next(f64, 0x0.8p32, 0.0));

            try testArgs(u32, f64, -0.0);
            try testArgs(u32, f64, 0.0);
            try testArgs(u32, f64, 1e-1);
            try testArgs(u32, f64, 1e0);
            try testArgs(u32, f64, 1e1);
            try testArgs(u32, f64, 1e2);
            try testArgs(u32, f64, 1e3);
            try testArgs(u32, f64, 1e4);
            try testArgs(u32, f64, 1e5);
            try testArgs(u32, f64, 1e6);
            try testArgs(u32, f64, 1e7);
            try testArgs(u32, f64, 1e8);
            try testArgs(u32, f64, 1e9);
            try testArgs(u32, f64, next(f64, next(f64, 0x1p32, 0.0), 0.0));
            try testArgs(u32, f64, next(f64, 0x1p32, 0.0));

            try testArgs(i64, f64, -0x0.8p64);
            try testArgs(i64, f64, next(f64, -0x0.8p64, -0.0));
            try testArgs(i64, f64, next(f64, next(f64, -0x0.8p64, -0.0), -0.0));
            try testArgs(i64, f64, -1e18);
            try testArgs(i64, f64, -1e16);
            try testArgs(i64, f64, -1e14);
            try testArgs(i64, f64, -1e12);
            try testArgs(i64, f64, -1e10);
            try testArgs(i64, f64, -1e8);
            try testArgs(i64, f64, -1e6);
            try testArgs(i64, f64, -1e4);
            try testArgs(i64, f64, -1e2);
            try testArgs(i64, f64, -1e0);
            try testArgs(i64, f64, -1e-1);
            try testArgs(i64, f64, -0.0);
            try testArgs(i64, f64, 0.0);
            try testArgs(i64, f64, 1e-1);
            try testArgs(i64, f64, 1e0);
            try testArgs(i64, f64, 1e2);
            try testArgs(i64, f64, 1e4);
            try testArgs(i64, f64, 1e6);
            try testArgs(i64, f64, 1e8);
            try testArgs(i64, f64, 1e10);
            try testArgs(i64, f64, 1e12);
            try testArgs(i64, f64, 1e14);
            try testArgs(i64, f64, 1e16);
            try testArgs(i64, f64, 1e18);
            try testArgs(i64, f64, next(f64, next(f64, 0x0.8p64, 0.0), 0.0));
            try testArgs(i64, f64, next(f64, 0x0.8p64, 0.0));

            try testArgs(u64, f64, -0.0);
            try testArgs(u64, f64, 0.0);
            try testArgs(u64, f64, 1e-1);
            try testArgs(u64, f64, 1e0);
            try testArgs(u64, f64, 1e2);
            try testArgs(u64, f64, 1e4);
            try testArgs(u64, f64, 1e6);
            try testArgs(u64, f64, 1e8);
            try testArgs(u64, f64, 1e10);
            try testArgs(u64, f64, 1e12);
            try testArgs(u64, f64, 1e14);
            try testArgs(u64, f64, 1e16);
            try testArgs(u64, f64, 1e18);
            try testArgs(u64, f64, next(f64, next(f64, 0x1p64, 0.0), 0.0));
            try testArgs(u64, f64, next(f64, 0x1p64, 0.0));

            try testArgs(i128, f64, -0x0.8p128);
            try testArgs(i128, f64, next(f64, -0x0.8p128, -0.0));
            try testArgs(i128, f64, next(f64, next(f64, -0x0.8p128, -0.0), -0.0));
            try testArgs(i128, f64, -1e38);
            try testArgs(i128, f64, -1e34);
            try testArgs(i128, f64, -1e30);
            try testArgs(i128, f64, -1e26);
            try testArgs(i128, f64, -1e22);
            try testArgs(i128, f64, -1e18);
            try testArgs(i128, f64, -1e14);
            try testArgs(i128, f64, -1e10);
            try testArgs(i128, f64, -1e6);
            try testArgs(i128, f64, -1e2);
            try testArgs(i128, f64, -1e0);
            try testArgs(i128, f64, -1e-1);
            try testArgs(i128, f64, -0.0);
            try testArgs(i128, f64, 0.0);
            try testArgs(i128, f64, 1e-1);
            try testArgs(i128, f64, 1e0);
            try testArgs(i128, f64, 1e2);
            try testArgs(i128, f64, 1e6);
            try testArgs(i128, f64, 1e10);
            try testArgs(i128, f64, 1e14);
            try testArgs(i128, f64, 1e18);
            try testArgs(i128, f64, 1e22);
            try testArgs(i128, f64, 1e26);
            try testArgs(i128, f64, 1e30);
            try testArgs(i128, f64, 1e34);
            try testArgs(i128, f64, 1e38);
            try testArgs(i128, f64, next(f64, next(f64, 0x0.8p128, 0.0), 0.0));
            try testArgs(i128, f64, next(f64, 0x0.8p128, 0.0));

            try testArgs(u128, f64, -0.0);
            try testArgs(u128, f64, 0.0);
            try testArgs(u128, f64, 1e-1);
            try testArgs(u128, f64, 1e0);
            try testArgs(u128, f64, 1e2);
            try testArgs(u128, f64, 1e6);
            try testArgs(u128, f64, 1e10);
            try testArgs(u128, f64, 1e14);
            try testArgs(u128, f64, 1e18);
            try testArgs(u128, f64, 1e22);
            try testArgs(u128, f64, 1e26);
            try testArgs(u128, f64, 1e30);
            try testArgs(u128, f64, 1e34);
            try testArgs(u128, f64, 1e38);
            try testArgs(u128, f64, next(f64, next(f64, 0x1p128, 0.0), 0.0));
            try testArgs(u128, f64, next(f64, 0x1p128, 0.0));

            try testArgs(i256, f64, -0x0.8p256);
            try testArgs(i256, f64, next(f64, -0x0.8p256, -0.0));
            try testArgs(i256, f64, next(f64, next(f64, -0x0.8p256, -0.0), -0.0));
            try testArgs(i256, f64, -1e76);
            try testArgs(i256, f64, -1e69);
            try testArgs(i256, f64, -1e62);
            try testArgs(i256, f64, -1e55);
            try testArgs(i256, f64, -1e48);
            try testArgs(i256, f64, -1e41);
            try testArgs(i256, f64, -1e34);
            try testArgs(i256, f64, -1e27);
            try testArgs(i256, f64, -1e20);
            try testArgs(i256, f64, -1e13);
            try testArgs(i256, f64, -1e6);
            try testArgs(i256, f64, -1e0);
            try testArgs(i256, f64, -1e-1);
            try testArgs(i256, f64, -0.0);
            try testArgs(i256, f64, 0.0);
            try testArgs(i256, f64, 1e-1);
            try testArgs(i256, f64, 1e0);
            try testArgs(i256, f64, 1e6);
            try testArgs(i256, f64, 1e13);
            try testArgs(i256, f64, 1e20);
            try testArgs(i256, f64, 1e27);
            try testArgs(i256, f64, 1e34);
            try testArgs(i256, f64, 1e41);
            try testArgs(i256, f64, 1e48);
            try testArgs(i256, f64, 1e55);
            try testArgs(i256, f64, 1e62);
            try testArgs(i256, f64, 1e69);
            try testArgs(i256, f64, 1e76);
            try testArgs(i256, f64, next(f64, next(f64, 0x0.8p256, 0.0), 0.0));
            try testArgs(i256, f64, next(f64, 0x0.8p256, 0.0));

            try testArgs(u256, f64, -0.0);
            try testArgs(u256, f64, 0.0);
            try testArgs(u256, f64, 1e-1);
            try testArgs(u256, f64, 1e0);
            try testArgs(u256, f64, 1e7);
            try testArgs(u256, f64, 1e14);
            try testArgs(u256, f64, 1e21);
            try testArgs(u256, f64, 1e28);
            try testArgs(u256, f64, 1e35);
            try testArgs(u256, f64, 1e42);
            try testArgs(u256, f64, 1e49);
            try testArgs(u256, f64, 1e56);
            try testArgs(u256, f64, 1e63);
            try testArgs(u256, f64, 1e70);
            try testArgs(u256, f64, 1e77);
            try testArgs(u256, f64, next(f64, next(f64, 0x1p256, 0.0), 0.0));
            try testArgs(u256, f64, next(f64, 0x1p256, 0.0));

            try testArgs(i8, f80, -0x0.8p8);
            try testArgs(i8, f80, next(f80, -0x0.8p8, -0.0));
            try testArgs(i8, f80, next(f80, next(f80, -0x0.8p8, -0.0), -0.0));
            try testArgs(i8, f80, -1e2);
            try testArgs(i8, f80, -1e1);
            try testArgs(i8, f80, -1e0);
            try testArgs(i8, f80, -1e-1);
            try testArgs(i8, f80, -0.0);
            try testArgs(i8, f80, 0.0);
            try testArgs(i8, f80, 1e-1);
            try testArgs(i8, f80, 1e0);
            try testArgs(i8, f80, 1e1);
            try testArgs(i8, f80, 1e2);
            try testArgs(i8, f80, next(f80, next(f80, 0x0.8p8, 0.0), 0.0));
            try testArgs(i8, f80, next(f80, 0x0.8p8, 0.0));

            try testArgs(u8, f80, -0.0);
            try testArgs(u8, f80, 0.0);
            try testArgs(u8, f80, 1e-1);
            try testArgs(u8, f80, 1e0);
            try testArgs(u8, f80, 1e1);
            try testArgs(u8, f80, 1e2);
            try testArgs(u8, f80, next(f80, next(f80, 0x1p8, 0.0), 0.0));
            try testArgs(u8, f80, next(f80, 0x1p8, 0.0));

            try testArgs(i16, f80, -0x0.8p16);
            try testArgs(i16, f80, next(f80, -0x0.8p16, -0.0));
            try testArgs(i16, f80, next(f80, next(f80, -0x0.8p16, -0.0), -0.0));
            try testArgs(i16, f80, -1e4);
            try testArgs(i16, f80, -1e3);
            try testArgs(i16, f80, -1e2);
            try testArgs(i16, f80, -1e1);
            try testArgs(i16, f80, -1e0);
            try testArgs(i16, f80, -1e-1);
            try testArgs(i16, f80, -0.0);
            try testArgs(i16, f80, 0.0);
            try testArgs(i16, f80, 1e-1);
            try testArgs(i16, f80, 1e0);
            try testArgs(i16, f80, 1e1);
            try testArgs(i16, f80, 1e2);
            try testArgs(i16, f80, 1e3);
            try testArgs(i16, f80, 1e4);
            try testArgs(i16, f80, next(f80, next(f80, 0x0.8p16, 0.0), 0.0));
            try testArgs(i16, f80, next(f80, 0x0.8p16, 0.0));

            try testArgs(u16, f80, -0.0);
            try testArgs(u16, f80, 0.0);
            try testArgs(u16, f80, 1e-1);
            try testArgs(u16, f80, 1e0);
            try testArgs(u16, f80, 1e1);
            try testArgs(u16, f80, 1e2);
            try testArgs(u16, f80, 1e3);
            try testArgs(u16, f80, 1e4);
            try testArgs(u16, f80, next(f80, next(f80, 0x1p16, 0.0), 0.0));
            try testArgs(u16, f80, next(f80, 0x1p16, 0.0));

            try testArgs(i32, f80, -0x0.8p32);
            try testArgs(i32, f80, next(f80, -0x0.8p32, -0.0));
            try testArgs(i32, f80, next(f80, next(f80, -0x0.8p32, -0.0), -0.0));
            try testArgs(i32, f80, -1e9);
            try testArgs(i32, f80, -1e8);
            try testArgs(i32, f80, -1e7);
            try testArgs(i32, f80, -1e6);
            try testArgs(i32, f80, -1e5);
            try testArgs(i32, f80, -1e4);
            try testArgs(i32, f80, -1e3);
            try testArgs(i32, f80, -1e2);
            try testArgs(i32, f80, -1e1);
            try testArgs(i32, f80, -1e0);
            try testArgs(i32, f80, -1e-1);
            try testArgs(i32, f80, -0.0);
            try testArgs(i32, f80, 0.0);
            try testArgs(i32, f80, 1e-1);
            try testArgs(i32, f80, 1e0);
            try testArgs(i32, f80, 1e1);
            try testArgs(i32, f80, 1e2);
            try testArgs(i32, f80, 1e3);
            try testArgs(i32, f80, 1e4);
            try testArgs(i32, f80, 1e5);
            try testArgs(i32, f80, 1e6);
            try testArgs(i32, f80, 1e7);
            try testArgs(i32, f80, 1e8);
            try testArgs(i32, f80, 1e9);
            try testArgs(i32, f80, next(f80, next(f80, 0x0.8p32, 0.0), 0.0));
            try testArgs(i32, f80, next(f80, 0x0.8p32, 0.0));

            try testArgs(u32, f80, -0.0);
            try testArgs(u32, f80, 0.0);
            try testArgs(u32, f80, 1e-1);
            try testArgs(u32, f80, 1e0);
            try testArgs(u32, f80, 1e1);
            try testArgs(u32, f80, 1e2);
            try testArgs(u32, f80, 1e3);
            try testArgs(u32, f80, 1e4);
            try testArgs(u32, f80, 1e5);
            try testArgs(u32, f80, 1e6);
            try testArgs(u32, f80, 1e7);
            try testArgs(u32, f80, 1e8);
            try testArgs(u32, f80, 1e9);
            try testArgs(u32, f80, next(f80, next(f80, 0x1p32, 0.0), 0.0));
            try testArgs(u32, f80, next(f80, 0x1p32, 0.0));

            try testArgs(i64, f80, -0x0.8p64);
            try testArgs(i64, f80, next(f80, -0x0.8p64, -0.0));
            try testArgs(i64, f80, next(f80, next(f80, -0x0.8p64, -0.0), -0.0));
            try testArgs(i64, f80, -1e18);
            try testArgs(i64, f80, -1e16);
            try testArgs(i64, f80, -1e14);
            try testArgs(i64, f80, -1e12);
            try testArgs(i64, f80, -1e10);
            try testArgs(i64, f80, -1e8);
            try testArgs(i64, f80, -1e6);
            try testArgs(i64, f80, -1e4);
            try testArgs(i64, f80, -1e2);
            try testArgs(i64, f80, -1e0);
            try testArgs(i64, f80, -1e-1);
            try testArgs(i64, f80, -0.0);
            try testArgs(i64, f80, 0.0);
            try testArgs(i64, f80, 1e-1);
            try testArgs(i64, f80, 1e0);
            try testArgs(i64, f80, 1e2);
            try testArgs(i64, f80, 1e4);
            try testArgs(i64, f80, 1e6);
            try testArgs(i64, f80, 1e8);
            try testArgs(i64, f80, 1e10);
            try testArgs(i64, f80, 1e12);
            try testArgs(i64, f80, 1e14);
            try testArgs(i64, f80, 1e16);
            try testArgs(i64, f80, 1e18);
            try testArgs(i64, f80, next(f80, next(f80, 0x0.8p64, 0.0), 0.0));
            try testArgs(i64, f80, next(f80, 0x0.8p64, 0.0));

            try testArgs(u64, f80, -0.0);
            try testArgs(u64, f80, 0.0);
            try testArgs(u64, f80, 1e-1);
            try testArgs(u64, f80, 1e0);
            try testArgs(u64, f80, 1e2);
            try testArgs(u64, f80, 1e4);
            try testArgs(u64, f80, 1e6);
            try testArgs(u64, f80, 1e8);
            try testArgs(u64, f80, 1e10);
            try testArgs(u64, f80, 1e12);
            try testArgs(u64, f80, 1e14);
            try testArgs(u64, f80, 1e16);
            try testArgs(u64, f80, 1e18);
            try testArgs(u64, f80, next(f80, next(f80, 0x1p64, 0.0), 0.0));
            try testArgs(u64, f80, next(f80, 0x1p64, 0.0));

            try testArgs(i128, f80, -0x0.8p128);
            try testArgs(i128, f80, next(f80, -0x0.8p128, -0.0));
            try testArgs(i128, f80, next(f80, next(f80, -0x0.8p128, -0.0), -0.0));
            try testArgs(i128, f80, -1e38);
            try testArgs(i128, f80, -1e34);
            try testArgs(i128, f80, -1e30);
            try testArgs(i128, f80, -1e26);
            try testArgs(i128, f80, -1e22);
            try testArgs(i128, f80, -1e18);
            try testArgs(i128, f80, -1e14);
            try testArgs(i128, f80, -1e10);
            try testArgs(i128, f80, -1e6);
            try testArgs(i128, f80, -1e2);
            try testArgs(i128, f80, -1e0);
            try testArgs(i128, f80, -1e-1);
            try testArgs(i128, f80, -0.0);
            try testArgs(i128, f80, 0.0);
            try testArgs(i128, f80, 1e-1);
            try testArgs(i128, f80, 1e0);
            try testArgs(i128, f80, 1e2);
            try testArgs(i128, f80, 1e6);
            try testArgs(i128, f80, 1e10);
            try testArgs(i128, f80, 1e14);
            try testArgs(i128, f80, 1e18);
            try testArgs(i128, f80, 1e22);
            try testArgs(i128, f80, 1e26);
            try testArgs(i128, f80, 1e30);
            try testArgs(i128, f80, 1e34);
            try testArgs(i128, f80, 1e38);
            try testArgs(i128, f80, next(f80, next(f80, 0x0.8p128, 0.0), 0.0));
            try testArgs(i128, f80, next(f80, 0x0.8p128, 0.0));

            try testArgs(u128, f80, -0.0);
            try testArgs(u128, f80, 0.0);
            try testArgs(u128, f80, 1e-1);
            try testArgs(u128, f80, 1e0);
            try testArgs(u128, f80, 1e2);
            try testArgs(u128, f80, 1e6);
            try testArgs(u128, f80, 1e10);
            try testArgs(u128, f80, 1e14);
            try testArgs(u128, f80, 1e18);
            try testArgs(u128, f80, 1e22);
            try testArgs(u128, f80, 1e26);
            try testArgs(u128, f80, 1e30);
            try testArgs(u128, f80, 1e34);
            try testArgs(u128, f80, 1e38);
            try testArgs(u128, f80, next(f80, next(f80, 0x1p128, 0.0), 0.0));
            try testArgs(u128, f80, next(f80, 0x1p128, 0.0));

            try testArgs(i256, f80, -0x0.8p256);
            try testArgs(i256, f80, next(f80, -0x0.8p256, -0.0));
            try testArgs(i256, f80, next(f80, next(f80, -0x0.8p256, -0.0), -0.0));
            try testArgs(i256, f80, -1e76);
            try testArgs(i256, f80, -1e69);
            try testArgs(i256, f80, -1e62);
            try testArgs(i256, f80, -1e55);
            try testArgs(i256, f80, -1e48);
            try testArgs(i256, f80, -1e41);
            try testArgs(i256, f80, -1e34);
            try testArgs(i256, f80, -1e27);
            try testArgs(i256, f80, -1e20);
            try testArgs(i256, f80, -1e13);
            try testArgs(i256, f80, -1e6);
            try testArgs(i256, f80, -1e0);
            try testArgs(i256, f80, -1e-1);
            try testArgs(i256, f80, -0.0);
            try testArgs(i256, f80, 0.0);
            try testArgs(i256, f80, 1e-1);
            try testArgs(i256, f80, 1e0);
            try testArgs(i256, f80, 1e6);
            try testArgs(i256, f80, 1e13);
            try testArgs(i256, f80, 1e20);
            try testArgs(i256, f80, 1e27);
            try testArgs(i256, f80, 1e34);
            try testArgs(i256, f80, 1e41);
            try testArgs(i256, f80, 1e48);
            try testArgs(i256, f80, 1e55);
            try testArgs(i256, f80, 1e62);
            try testArgs(i256, f80, 1e69);
            try testArgs(i256, f80, 1e76);
            try testArgs(i256, f80, next(f80, next(f80, 0x0.8p256, 0.0), 0.0));
            try testArgs(i256, f80, next(f80, 0x0.8p256, 0.0));

            try testArgs(u256, f80, -0.0);
            try testArgs(u256, f80, 0.0);
            try testArgs(u256, f80, 1e-1);
            try testArgs(u256, f80, 1e0);
            try testArgs(u256, f80, 1e7);
            try testArgs(u256, f80, 1e14);
            try testArgs(u256, f80, 1e21);
            try testArgs(u256, f80, 1e28);
            try testArgs(u256, f80, 1e35);
            try testArgs(u256, f80, 1e42);
            try testArgs(u256, f80, 1e49);
            try testArgs(u256, f80, 1e56);
            try testArgs(u256, f80, 1e63);
            try testArgs(u256, f80, 1e70);
            try testArgs(u256, f80, 1e77);
            try testArgs(u256, f80, next(f80, next(f80, 0x1p256, 0.0), 0.0));
            try testArgs(u256, f80, next(f80, 0x1p256, 0.0));

            try testArgs(i8, f128, -0x0.8p8);
            try testArgs(i8, f128, next(f128, -0x0.8p8, -0.0));
            try testArgs(i8, f128, next(f128, next(f128, -0x0.8p8, -0.0), -0.0));
            try testArgs(i8, f128, -1e2);
            try testArgs(i8, f128, -1e1);
            try testArgs(i8, f128, -1e0);
            try testArgs(i8, f128, -1e-1);
            try testArgs(i8, f128, -0.0);
            try testArgs(i8, f128, 0.0);
            try testArgs(i8, f128, 1e-1);
            try testArgs(i8, f128, 1e0);
            try testArgs(i8, f128, 1e1);
            try testArgs(i8, f128, 1e2);
            try testArgs(i8, f128, next(f128, next(f128, 0x0.8p8, 0.0), 0.0));
            try testArgs(i8, f128, next(f128, 0x0.8p8, 0.0));

            try testArgs(u8, f128, -0.0);
            try testArgs(u8, f128, 0.0);
            try testArgs(u8, f128, 1e-1);
            try testArgs(u8, f128, 1e0);
            try testArgs(u8, f128, 1e1);
            try testArgs(u8, f128, 1e2);
            try testArgs(u8, f128, next(f128, next(f128, 0x1p8, 0.0), 0.0));
            try testArgs(u8, f128, next(f128, 0x1p8, 0.0));

            try testArgs(i16, f128, -0x0.8p16);
            try testArgs(i16, f128, next(f128, -0x0.8p16, -0.0));
            try testArgs(i16, f128, next(f128, next(f128, -0x0.8p16, -0.0), -0.0));
            try testArgs(i16, f128, -1e4);
            try testArgs(i16, f128, -1e3);
            try testArgs(i16, f128, -1e2);
            try testArgs(i16, f128, -1e1);
            try testArgs(i16, f128, -1e0);
            try testArgs(i16, f128, -1e-1);
            try testArgs(i16, f128, -0.0);
            try testArgs(i16, f128, 0.0);
            try testArgs(i16, f128, 1e-1);
            try testArgs(i16, f128, 1e0);
            try testArgs(i16, f128, 1e1);
            try testArgs(i16, f128, 1e2);
            try testArgs(i16, f128, 1e3);
            try testArgs(i16, f128, 1e4);
            try testArgs(i16, f128, next(f128, next(f128, 0x0.8p16, 0.0), 0.0));
            try testArgs(i16, f128, next(f128, 0x0.8p16, 0.0));

            try testArgs(u16, f128, -0.0);
            try testArgs(u16, f128, 0.0);
            try testArgs(u16, f128, 1e-1);
            try testArgs(u16, f128, 1e0);
            try testArgs(u16, f128, 1e1);
            try testArgs(u16, f128, 1e2);
            try testArgs(u16, f128, 1e3);
            try testArgs(u16, f128, 1e4);
            try testArgs(u16, f128, next(f128, next(f128, 0x1p16, 0.0), 0.0));
            try testArgs(u16, f128, next(f128, 0x1p16, 0.0));

            try testArgs(i32, f128, -0x0.8p32);
            try testArgs(i32, f128, next(f128, -0x0.8p32, -0.0));
            try testArgs(i32, f128, next(f128, next(f128, -0x0.8p32, -0.0), -0.0));
            try testArgs(i32, f128, -1e9);
            try testArgs(i32, f128, -1e8);
            try testArgs(i32, f128, -1e7);
            try testArgs(i32, f128, -1e6);
            try testArgs(i32, f128, -1e5);
            try testArgs(i32, f128, -1e4);
            try testArgs(i32, f128, -1e3);
            try testArgs(i32, f128, -1e2);
            try testArgs(i32, f128, -1e1);
            try testArgs(i32, f128, -1e0);
            try testArgs(i32, f128, -1e-1);
            try testArgs(i32, f128, -0.0);
            try testArgs(i32, f128, 0.0);
            try testArgs(i32, f128, 1e-1);
            try testArgs(i32, f128, 1e0);
            try testArgs(i32, f128, 1e1);
            try testArgs(i32, f128, 1e2);
            try testArgs(i32, f128, 1e3);
            try testArgs(i32, f128, 1e4);
            try testArgs(i32, f128, 1e5);
            try testArgs(i32, f128, 1e6);
            try testArgs(i32, f128, 1e7);
            try testArgs(i32, f128, 1e8);
            try testArgs(i32, f128, 1e9);
            try testArgs(i32, f128, next(f128, next(f128, 0x0.8p32, 0.0), 0.0));
            try testArgs(i32, f128, next(f128, 0x0.8p32, 0.0));

            try testArgs(u32, f128, -0.0);
            try testArgs(u32, f128, 0.0);
            try testArgs(u32, f128, 1e-1);
            try testArgs(u32, f128, 1e0);
            try testArgs(u32, f128, 1e1);
            try testArgs(u32, f128, 1e2);
            try testArgs(u32, f128, 1e3);
            try testArgs(u32, f128, 1e4);
            try testArgs(u32, f128, 1e5);
            try testArgs(u32, f128, 1e6);
            try testArgs(u32, f128, 1e7);
            try testArgs(u32, f128, 1e8);
            try testArgs(u32, f128, 1e9);
            try testArgs(u32, f128, next(f128, next(f128, 0x1p32, 0.0), 0.0));
            try testArgs(u32, f128, next(f128, 0x1p32, 0.0));

            try testArgs(i64, f128, -0x0.8p64);
            try testArgs(i64, f128, next(f128, -0x0.8p64, -0.0));
            try testArgs(i64, f128, next(f128, next(f128, -0x0.8p64, -0.0), -0.0));
            try testArgs(i64, f128, -1e18);
            try testArgs(i64, f128, -1e16);
            try testArgs(i64, f128, -1e14);
            try testArgs(i64, f128, -1e12);
            try testArgs(i64, f128, -1e10);
            try testArgs(i64, f128, -1e8);
            try testArgs(i64, f128, -1e6);
            try testArgs(i64, f128, -1e4);
            try testArgs(i64, f128, -1e2);
            try testArgs(i64, f128, -1e0);
            try testArgs(i64, f128, -1e-1);
            try testArgs(i64, f128, -0.0);
            try testArgs(i64, f128, 0.0);
            try testArgs(i64, f128, 1e-1);
            try testArgs(i64, f128, 1e0);
            try testArgs(i64, f128, 1e2);
            try testArgs(i64, f128, 1e4);
            try testArgs(i64, f128, 1e6);
            try testArgs(i64, f128, 1e8);
            try testArgs(i64, f128, 1e10);
            try testArgs(i64, f128, 1e11);
            try testArgs(i64, f128, 1e12);
            try testArgs(i64, f128, 1e13);
            try testArgs(i64, f128, 1e14);
            try testArgs(i64, f128, 1e15);
            try testArgs(i64, f128, 1e16);
            try testArgs(i64, f128, 1e17);
            try testArgs(i64, f128, 1e18);
            try testArgs(i64, f128, next(f128, next(f128, 0x0.8p64, 0.0), 0.0));
            try testArgs(i64, f128, next(f128, 0x0.8p64, 0.0));

            try testArgs(u64, f128, -0.0);
            try testArgs(u64, f128, 0.0);
            try testArgs(u64, f128, 1e-1);
            try testArgs(u64, f128, 1e0);
            try testArgs(u64, f128, 1e2);
            try testArgs(u64, f128, 1e4);
            try testArgs(u64, f128, 1e6);
            try testArgs(u64, f128, 1e8);
            try testArgs(u64, f128, 1e10);
            try testArgs(u64, f128, 1e12);
            try testArgs(u64, f128, 1e14);
            try testArgs(u64, f128, 1e16);
            try testArgs(u64, f128, 1e18);
            try testArgs(u64, f128, next(f128, next(f128, 0x1p64, 0.0), 0.0));
            try testArgs(u64, f128, next(f128, 0x1p64, 0.0));

            try testArgs(i128, f128, -0x0.8p128);
            try testArgs(i128, f128, next(f128, -0x0.8p128, -0.0));
            try testArgs(i128, f128, next(f128, next(f128, -0x0.8p128, -0.0), -0.0));
            try testArgs(i128, f128, -1e38);
            try testArgs(i128, f128, -1e34);
            try testArgs(i128, f128, -1e30);
            try testArgs(i128, f128, -1e26);
            try testArgs(i128, f128, -1e22);
            try testArgs(i128, f128, -1e18);
            try testArgs(i128, f128, -1e14);
            try testArgs(i128, f128, -1e10);
            try testArgs(i128, f128, -1e6);
            try testArgs(i128, f128, -1e2);
            try testArgs(i128, f128, -1e0);
            try testArgs(i128, f128, -1e-1);
            try testArgs(i128, f128, -0.0);
            try testArgs(i128, f128, 0.0);
            try testArgs(i128, f128, 1e-1);
            try testArgs(i128, f128, 1e0);
            try testArgs(i128, f128, 1e2);
            try testArgs(i128, f128, 1e6);
            try testArgs(i128, f128, 1e10);
            try testArgs(i128, f128, 1e14);
            try testArgs(i128, f128, 1e18);
            try testArgs(i128, f128, 1e22);
            try testArgs(i128, f128, 1e26);
            try testArgs(i128, f128, 1e30);
            try testArgs(i128, f128, 1e34);
            try testArgs(i128, f128, 1e38);
            try testArgs(i128, f128, next(f128, next(f128, 0x0.8p128, 0.0), 0.0));
            try testArgs(i128, f128, next(f128, 0x0.8p128, 0.0));

            try testArgs(u128, f128, -0.0);
            try testArgs(u128, f128, 0.0);
            try testArgs(u128, f128, 1e-1);
            try testArgs(u128, f128, 1e0);
            try testArgs(u128, f128, 1e2);
            try testArgs(u128, f128, 1e6);
            try testArgs(u128, f128, 1e10);
            try testArgs(u128, f128, 1e14);
            try testArgs(u128, f128, 1e18);
            try testArgs(u128, f128, 1e22);
            try testArgs(u128, f128, 1e26);
            try testArgs(u128, f128, 1e30);
            try testArgs(u128, f128, 1e34);
            try testArgs(u128, f128, 1e38);
            try testArgs(u128, f128, next(f128, next(f128, 0x1p128, 0.0), 0.0));
            try testArgs(u128, f128, next(f128, 0x1p128, 0.0));

            try testArgs(i256, f128, -0x0.8p256);
            try testArgs(i256, f128, next(f128, -0x0.8p256, -0.0));
            try testArgs(i256, f128, next(f128, next(f128, -0x0.8p256, -0.0), -0.0));
            try testArgs(i256, f128, -1e76);
            try testArgs(i256, f128, -1e69);
            try testArgs(i256, f128, -1e62);
            try testArgs(i256, f128, -1e55);
            try testArgs(i256, f128, -1e48);
            try testArgs(i256, f128, -1e41);
            try testArgs(i256, f128, -1e34);
            try testArgs(i256, f128, -1e27);
            try testArgs(i256, f128, -1e20);
            try testArgs(i256, f128, -1e13);
            try testArgs(i256, f128, -1e6);
            try testArgs(i256, f128, -1e0);
            try testArgs(i256, f128, -1e-1);
            try testArgs(i256, f128, -0.0);
            try testArgs(i256, f128, 0.0);
            try testArgs(i256, f128, 1e-1);
            try testArgs(i256, f128, 1e0);
            try testArgs(i256, f128, 1e6);
            try testArgs(i256, f128, 1e13);
            try testArgs(i256, f128, 1e20);
            try testArgs(i256, f128, 1e27);
            try testArgs(i256, f128, 1e34);
            try testArgs(i256, f128, 1e41);
            try testArgs(i256, f128, 1e48);
            try testArgs(i256, f128, 1e55);
            try testArgs(i256, f128, 1e62);
            try testArgs(i256, f128, 1e69);
            try testArgs(i256, f128, 1e76);
            try testArgs(i256, f128, next(f128, next(f128, 0x0.8p256, 0.0), 0.0));
            try testArgs(i256, f128, next(f128, 0x0.8p256, 0.0));

            try testArgs(u256, f128, -0.0);
            try testArgs(u256, f128, 0.0);
            try testArgs(u256, f128, 1e-1);
            try testArgs(u256, f128, 1e0);
            try testArgs(u256, f128, 1e7);
            try testArgs(u256, f128, 1e14);
            try testArgs(u256, f128, 1e21);
            try testArgs(u256, f128, 1e28);
            try testArgs(u256, f128, 1e35);
            try testArgs(u256, f128, 1e42);
            try testArgs(u256, f128, 1e49);
            try testArgs(u256, f128, 1e56);
            try testArgs(u256, f128, 1e63);
            try testArgs(u256, f128, 1e70);
            try testArgs(u256, f128, 1e77);
            try testArgs(u256, f128, next(f128, next(f128, 0x1p256, 0.0), 0.0));
            try testArgs(u256, f128, next(f128, 0x1p256, 0.0));
        }
        fn testFloatsFromInts() !void {
            try testArgs(f16, i8, imin(i8));
            try testArgs(f16, i8, imin(i8) + 1);
            try testArgs(f16, i8, -1e2);
            try testArgs(f16, i8, -1e1);
            try testArgs(f16, i8, -1e0);
            try testArgs(f16, i8, 0);
            try testArgs(f16, i8, 1e0);
            try testArgs(f16, i8, 1e1);
            try testArgs(f16, i8, 1e2);
            try testArgs(f16, i8, imax(i8) - 1);
            try testArgs(f16, i8, imax(i8));

            try testArgs(f16, u8, 0);
            try testArgs(f16, u8, 1e0);
            try testArgs(f16, u8, 1e1);
            try testArgs(f16, u8, 1e2);
            try testArgs(f16, u8, imax(u8) - 1);
            try testArgs(f16, u8, imax(u8));

            try testArgs(f16, i16, imin(i16));
            try testArgs(f16, i16, imin(i16) + 1);
            try testArgs(f16, i16, -1e4);
            try testArgs(f16, i16, -1e3);
            try testArgs(f16, i16, -1e2);
            try testArgs(f16, i16, -1e1);
            try testArgs(f16, i16, -1e0);
            try testArgs(f16, i16, 0);
            try testArgs(f16, i16, 1e0);
            try testArgs(f16, i16, 1e1);
            try testArgs(f16, i16, 1e2);
            try testArgs(f16, i16, 1e3);
            try testArgs(f16, i16, 1e4);
            try testArgs(f16, i16, imax(i16) - 1);
            try testArgs(f16, i16, imax(i16));

            try testArgs(f16, u16, 0);
            try testArgs(f16, u16, 1e0);
            try testArgs(f16, u16, 1e1);
            try testArgs(f16, u16, 1e2);
            try testArgs(f16, u16, 1e3);
            try testArgs(f16, u16, 1e4);
            try testArgs(f16, u16, imax(u16) - 1);
            try testArgs(f16, u16, imax(u16));

            try testArgs(f16, i32, imin(i32));
            try testArgs(f16, i32, imin(i32) + 1);
            try testArgs(f16, i32, -1e9);
            try testArgs(f16, i32, -1e8);
            try testArgs(f16, i32, -1e7);
            try testArgs(f16, i32, -1e6);
            try testArgs(f16, i32, -1e5);
            try testArgs(f16, i32, -1e4);
            try testArgs(f16, i32, -1e3);
            try testArgs(f16, i32, -1e2);
            try testArgs(f16, i32, -1e1);
            try testArgs(f16, i32, -1e0);
            try testArgs(f16, i32, 0);
            try testArgs(f16, i32, 1e0);
            try testArgs(f16, i32, 1e1);
            try testArgs(f16, i32, 1e2);
            try testArgs(f16, i32, 1e3);
            try testArgs(f16, i32, 1e4);
            try testArgs(f16, i32, 1e5);
            try testArgs(f16, i32, 1e6);
            try testArgs(f16, i32, 1e7);
            try testArgs(f16, i32, 1e8);
            try testArgs(f16, i32, 1e9);
            try testArgs(f16, i32, imax(i32) - 1);
            try testArgs(f16, i32, imax(i32));

            try testArgs(f16, u32, 0);
            try testArgs(f16, u32, 1e0);
            try testArgs(f16, u32, 1e1);
            try testArgs(f16, u32, 1e2);
            try testArgs(f16, u32, 1e3);
            try testArgs(f16, u32, 1e4);
            try testArgs(f16, u32, 1e5);
            try testArgs(f16, u32, 1e6);
            try testArgs(f16, u32, 1e7);
            try testArgs(f16, u32, 1e8);
            try testArgs(f16, u32, 1e9);
            try testArgs(f16, u32, imax(u32) - 1);
            try testArgs(f16, u32, imax(u32));

            try testArgs(f16, i64, imin(i64));
            try testArgs(f16, i64, imin(i64) + 1);
            try testArgs(f16, i64, -1e18);
            try testArgs(f16, i64, -1e16);
            try testArgs(f16, i64, -1e14);
            try testArgs(f16, i64, -1e12);
            try testArgs(f16, i64, -1e10);
            try testArgs(f16, i64, -1e8);
            try testArgs(f16, i64, -1e6);
            try testArgs(f16, i64, -1e4);
            try testArgs(f16, i64, -1e2);
            try testArgs(f16, i64, -1e0);
            try testArgs(f16, i64, 0);
            try testArgs(f16, i64, 1e0);
            try testArgs(f16, i64, 1e2);
            try testArgs(f16, i64, 1e4);
            try testArgs(f16, i64, 1e6);
            try testArgs(f16, i64, 1e8);
            try testArgs(f16, i64, 1e10);
            try testArgs(f16, i64, 1e12);
            try testArgs(f16, i64, 1e14);
            try testArgs(f16, i64, 1e16);
            try testArgs(f16, i64, 1e18);
            try testArgs(f16, i64, imax(i64) - 1);
            try testArgs(f16, i64, imax(i64));

            try testArgs(f16, u64, 0);
            try testArgs(f16, u64, 1e0);
            try testArgs(f16, u64, 1e2);
            try testArgs(f16, u64, 1e4);
            try testArgs(f16, u64, 1e6);
            try testArgs(f16, u64, 1e8);
            try testArgs(f16, u64, 1e10);
            try testArgs(f16, u64, 1e12);
            try testArgs(f16, u64, 1e14);
            try testArgs(f16, u64, 1e16);
            try testArgs(f16, u64, 1e18);
            try testArgs(f16, u64, imax(u64) - 1);
            try testArgs(f16, u64, imax(u64));

            try testArgs(f16, i128, imin(i128));
            try testArgs(f16, i128, imin(i128) + 1);
            try testArgs(f16, i128, -1e38);
            try testArgs(f16, i128, -1e34);
            try testArgs(f16, i128, -1e30);
            try testArgs(f16, i128, -1e26);
            try testArgs(f16, i128, -1e22);
            try testArgs(f16, i128, -1e18);
            try testArgs(f16, i128, -1e14);
            try testArgs(f16, i128, -1e10);
            try testArgs(f16, i128, -1e6);
            try testArgs(f16, i128, -1e2);
            try testArgs(f16, i128, -1e0);
            try testArgs(f16, i128, 0);
            try testArgs(f16, i128, 1e0);
            try testArgs(f16, i128, 1e2);
            try testArgs(f16, i128, 1e6);
            try testArgs(f16, i128, 1e10);
            try testArgs(f16, i128, 1e14);
            try testArgs(f16, i128, 1e18);
            try testArgs(f16, i128, 1e22);
            try testArgs(f16, i128, 1e26);
            try testArgs(f16, i128, 1e30);
            try testArgs(f16, i128, 1e34);
            try testArgs(f16, i128, 1e38);
            try testArgs(f16, i128, imax(i128) - 1);
            try testArgs(f16, i128, imax(i128));

            try testArgs(f16, u128, 0);
            try testArgs(f16, u128, 1e0);
            try testArgs(f16, u128, 1e2);
            try testArgs(f16, u128, 1e6);
            try testArgs(f16, u128, 1e10);
            try testArgs(f16, u128, 1e14);
            try testArgs(f16, u128, 1e18);
            try testArgs(f16, u128, 1e22);
            try testArgs(f16, u128, 1e26);
            try testArgs(f16, u128, 1e30);
            try testArgs(f16, u128, 1e34);
            try testArgs(f16, u128, 1e38);
            try testArgs(f16, u128, imax(u128) - 1);
            try testArgs(f16, u128, imax(u128));

            try testArgs(f16, i256, imin(i256));
            try testArgs(f16, i256, imin(i256) + 1);
            try testArgs(f16, i256, -1e76);
            try testArgs(f16, i256, -1e69);
            try testArgs(f16, i256, -1e62);
            try testArgs(f16, i256, -1e55);
            try testArgs(f16, i256, -1e48);
            try testArgs(f16, i256, -1e41);
            try testArgs(f16, i256, -1e34);
            try testArgs(f16, i256, -1e27);
            try testArgs(f16, i256, -1e20);
            try testArgs(f16, i256, -1e13);
            try testArgs(f16, i256, -1e6);
            try testArgs(f16, i256, -1e0);
            try testArgs(f16, i256, 0);
            try testArgs(f16, i256, 1e0);
            try testArgs(f16, i256, 1e6);
            try testArgs(f16, i256, 1e13);
            try testArgs(f16, i256, 1e20);
            try testArgs(f16, i256, 1e27);
            try testArgs(f16, i256, 1e34);
            try testArgs(f16, i256, 1e41);
            try testArgs(f16, i256, 1e48);
            try testArgs(f16, i256, 1e55);
            try testArgs(f16, i256, 1e62);
            try testArgs(f16, i256, 1e69);
            try testArgs(f16, i256, 1e76);
            try testArgs(f16, i256, imax(i256) - 1);
            try testArgs(f16, i256, imax(i256));

            try testArgs(f16, u256, 0);
            try testArgs(f16, u256, 1e0);
            try testArgs(f16, u256, 1e7);
            try testArgs(f16, u256, 1e14);
            try testArgs(f16, u256, 1e21);
            try testArgs(f16, u256, 1e28);
            try testArgs(f16, u256, 1e35);
            try testArgs(f16, u256, 1e42);
            try testArgs(f16, u256, 1e49);
            try testArgs(f16, u256, 1e56);
            try testArgs(f16, u256, 1e63);
            try testArgs(f16, u256, 1e70);
            try testArgs(f16, u256, 1e77);
            try testArgs(f16, u256, imax(u256) - 1);
            try testArgs(f16, u256, imax(u256));

            try testArgs(f32, i8, imin(i8));
            try testArgs(f32, i8, imin(i8) + 1);
            try testArgs(f32, i8, -1e2);
            try testArgs(f32, i8, -1e1);
            try testArgs(f32, i8, -1e0);
            try testArgs(f32, i8, 0);
            try testArgs(f32, i8, 1e0);
            try testArgs(f32, i8, 1e1);
            try testArgs(f32, i8, 1e2);
            try testArgs(f32, i8, imax(i8) - 1);
            try testArgs(f32, i8, imax(i8));

            try testArgs(f32, u8, 0);
            try testArgs(f32, u8, 1e0);
            try testArgs(f32, u8, 1e1);
            try testArgs(f32, u8, 1e2);
            try testArgs(f32, u8, imax(u8) - 1);
            try testArgs(f32, u8, imax(u8));

            try testArgs(f32, i16, imin(i16));
            try testArgs(f32, i16, imin(i16) + 1);
            try testArgs(f32, i16, -1e4);
            try testArgs(f32, i16, -1e3);
            try testArgs(f32, i16, -1e2);
            try testArgs(f32, i16, -1e1);
            try testArgs(f32, i16, -1e0);
            try testArgs(f32, i16, 0);
            try testArgs(f32, i16, 1e0);
            try testArgs(f32, i16, 1e1);
            try testArgs(f32, i16, 1e2);
            try testArgs(f32, i16, 1e3);
            try testArgs(f32, i16, 1e4);
            try testArgs(f32, i16, imax(i16) - 1);
            try testArgs(f32, i16, imax(i16));

            try testArgs(f32, u16, 0);
            try testArgs(f32, u16, 1e0);
            try testArgs(f32, u16, 1e1);
            try testArgs(f32, u16, 1e2);
            try testArgs(f32, u16, 1e3);
            try testArgs(f32, u16, 1e4);
            try testArgs(f32, u16, imax(u16) - 1);
            try testArgs(f32, u16, imax(u16));

            try testArgs(f32, i32, imin(i32));
            try testArgs(f32, i32, imin(i32) + 1);
            try testArgs(f32, i32, -1e9);
            try testArgs(f32, i32, -1e8);
            try testArgs(f32, i32, -1e7);
            try testArgs(f32, i32, -1e6);
            try testArgs(f32, i32, -1e5);
            try testArgs(f32, i32, -1e4);
            try testArgs(f32, i32, -1e3);
            try testArgs(f32, i32, -1e2);
            try testArgs(f32, i32, -1e1);
            try testArgs(f32, i32, -1e0);
            try testArgs(f32, i32, 0);
            try testArgs(f32, i32, 1e0);
            try testArgs(f32, i32, 1e1);
            try testArgs(f32, i32, 1e2);
            try testArgs(f32, i32, 1e3);
            try testArgs(f32, i32, 1e4);
            try testArgs(f32, i32, 1e5);
            try testArgs(f32, i32, 1e6);
            try testArgs(f32, i32, 1e7);
            try testArgs(f32, i32, 1e8);
            try testArgs(f32, i32, 1e9);
            try testArgs(f32, i32, imax(i32) - 1);
            try testArgs(f32, i32, imax(i32));

            try testArgs(f32, u32, 0);
            try testArgs(f32, u32, 1e0);
            try testArgs(f32, u32, 1e1);
            try testArgs(f32, u32, 1e2);
            try testArgs(f32, u32, 1e3);
            try testArgs(f32, u32, 1e4);
            try testArgs(f32, u32, 1e5);
            try testArgs(f32, u32, 1e6);
            try testArgs(f32, u32, 1e7);
            try testArgs(f32, u32, 1e8);
            try testArgs(f32, u32, 1e9);
            try testArgs(f32, u32, imax(u32) - 1);
            try testArgs(f32, u32, imax(u32));

            try testArgs(f32, i64, imin(i64));
            try testArgs(f32, i64, imin(i64) + 1);
            try testArgs(f32, i64, -1e18);
            try testArgs(f32, i64, -1e16);
            try testArgs(f32, i64, -1e14);
            try testArgs(f32, i64, -1e12);
            try testArgs(f32, i64, -1e10);
            try testArgs(f32, i64, -1e8);
            try testArgs(f32, i64, -1e6);
            try testArgs(f32, i64, -1e4);
            try testArgs(f32, i64, -1e2);
            try testArgs(f32, i64, -1e0);
            try testArgs(f32, i64, 0);
            try testArgs(f32, i64, 1e0);
            try testArgs(f32, i64, 1e2);
            try testArgs(f32, i64, 1e4);
            try testArgs(f32, i64, 1e6);
            try testArgs(f32, i64, 1e8);
            try testArgs(f32, i64, 1e10);
            try testArgs(f32, i64, 1e12);
            try testArgs(f32, i64, 1e14);
            try testArgs(f32, i64, 1e16);
            try testArgs(f32, i64, 1e18);
            try testArgs(f32, i64, imax(i64) - 1);
            try testArgs(f32, i64, imax(i64));

            try testArgs(f32, u64, 0);
            try testArgs(f32, u64, 1e0);
            try testArgs(f32, u64, 1e2);
            try testArgs(f32, u64, 1e4);
            try testArgs(f32, u64, 1e6);
            try testArgs(f32, u64, 1e8);
            try testArgs(f32, u64, 1e10);
            try testArgs(f32, u64, 1e12);
            try testArgs(f32, u64, 1e14);
            try testArgs(f32, u64, 1e16);
            try testArgs(f32, u64, 1e18);
            try testArgs(f32, u64, imax(u64) - 1);
            try testArgs(f32, u64, imax(u64));

            try testArgs(f32, i128, imin(i128));
            try testArgs(f32, i128, imin(i128) + 1);
            try testArgs(f32, i128, -1e38);
            try testArgs(f32, i128, -1e34);
            try testArgs(f32, i128, -1e30);
            try testArgs(f32, i128, -1e26);
            try testArgs(f32, i128, -1e22);
            try testArgs(f32, i128, -1e18);
            try testArgs(f32, i128, -1e14);
            try testArgs(f32, i128, -1e10);
            try testArgs(f32, i128, -1e6);
            try testArgs(f32, i128, -1e2);
            try testArgs(f32, i128, -1e0);
            try testArgs(f32, i128, 0);
            try testArgs(f32, i128, 1e0);
            try testArgs(f32, i128, 1e2);
            try testArgs(f32, i128, 1e6);
            try testArgs(f32, i128, 1e10);
            try testArgs(f32, i128, 1e14);
            try testArgs(f32, i128, 1e18);
            try testArgs(f32, i128, 1e22);
            try testArgs(f32, i128, 1e26);
            try testArgs(f32, i128, 1e30);
            try testArgs(f32, i128, 1e34);
            try testArgs(f32, i128, 1e38);
            try testArgs(f32, i128, imax(i128) - 1);
            try testArgs(f32, i128, imax(i128));

            try testArgs(f32, u128, 0);
            try testArgs(f32, u128, 1e0);
            try testArgs(f32, u128, 1e2);
            try testArgs(f32, u128, 1e6);
            try testArgs(f32, u128, 1e10);
            try testArgs(f32, u128, 1e14);
            try testArgs(f32, u128, 1e18);
            try testArgs(f32, u128, 1e22);
            try testArgs(f32, u128, 1e26);
            try testArgs(f32, u128, 1e30);
            try testArgs(f32, u128, 1e34);
            try testArgs(f32, u128, 1e38);
            try testArgs(f32, u128, imax(u128) - 1);
            try testArgs(f32, u128, imax(u128));

            try testArgs(f32, i256, imin(i256));
            try testArgs(f32, i256, imin(i256) + 1);
            try testArgs(f32, i256, -1e76);
            try testArgs(f32, i256, -1e69);
            try testArgs(f32, i256, -1e62);
            try testArgs(f32, i256, -1e55);
            try testArgs(f32, i256, -1e48);
            try testArgs(f32, i256, -1e41);
            try testArgs(f32, i256, -1e34);
            try testArgs(f32, i256, -1e27);
            try testArgs(f32, i256, -1e20);
            try testArgs(f32, i256, -1e13);
            try testArgs(f32, i256, -1e6);
            try testArgs(f32, i256, -1e0);
            try testArgs(f32, i256, 0);
            try testArgs(f32, i256, 1e0);
            try testArgs(f32, i256, 1e6);
            try testArgs(f32, i256, 1e13);
            try testArgs(f32, i256, 1e20);
            try testArgs(f32, i256, 1e27);
            try testArgs(f32, i256, 1e34);
            try testArgs(f32, i256, 1e41);
            try testArgs(f32, i256, 1e48);
            try testArgs(f32, i256, 1e55);
            try testArgs(f32, i256, 1e62);
            try testArgs(f32, i256, 1e69);
            try testArgs(f32, i256, 1e76);
            try testArgs(f32, i256, imax(i256) - 1);
            try testArgs(f32, i256, imax(i256));

            try testArgs(f32, u256, 0);
            try testArgs(f32, u256, 1e0);
            try testArgs(f32, u256, 1e7);
            try testArgs(f32, u256, 1e14);
            try testArgs(f32, u256, 1e21);
            try testArgs(f32, u256, 1e28);
            try testArgs(f32, u256, 1e35);
            try testArgs(f32, u256, 1e42);
            try testArgs(f32, u256, 1e49);
            try testArgs(f32, u256, 1e56);
            try testArgs(f32, u256, 1e63);
            try testArgs(f32, u256, 1e70);
            try testArgs(f32, u256, 1e77);
            try testArgs(f32, u256, imax(u256) - 1);
            try testArgs(f32, u256, imax(u256));

            try testArgs(f64, i8, imin(i8));
            try testArgs(f64, i8, imin(i8) + 1);
            try testArgs(f64, i8, -1e2);
            try testArgs(f64, i8, -1e1);
            try testArgs(f64, i8, -1e0);
            try testArgs(f64, i8, 0);
            try testArgs(f64, i8, 1e0);
            try testArgs(f64, i8, 1e1);
            try testArgs(f64, i8, 1e2);
            try testArgs(f64, i8, imax(i8) - 1);
            try testArgs(f64, i8, imax(i8));

            try testArgs(f64, u8, 0);
            try testArgs(f64, u8, 1e0);
            try testArgs(f64, u8, 1e1);
            try testArgs(f64, u8, 1e2);
            try testArgs(f64, u8, imax(u8) - 1);
            try testArgs(f64, u8, imax(u8));

            try testArgs(f64, i16, imin(i16));
            try testArgs(f64, i16, imin(i16) + 1);
            try testArgs(f64, i16, -1e4);
            try testArgs(f64, i16, -1e3);
            try testArgs(f64, i16, -1e2);
            try testArgs(f64, i16, -1e1);
            try testArgs(f64, i16, -1e0);
            try testArgs(f64, i16, 0);
            try testArgs(f64, i16, 1e0);
            try testArgs(f64, i16, 1e1);
            try testArgs(f64, i16, 1e2);
            try testArgs(f64, i16, 1e3);
            try testArgs(f64, i16, 1e4);
            try testArgs(f64, i16, imax(i16) - 1);
            try testArgs(f64, i16, imax(i16));

            try testArgs(f64, u16, 0);
            try testArgs(f64, u16, 1e0);
            try testArgs(f64, u16, 1e1);
            try testArgs(f64, u16, 1e2);
            try testArgs(f64, u16, 1e3);
            try testArgs(f64, u16, 1e4);
            try testArgs(f64, u16, imax(u16) - 1);
            try testArgs(f64, u16, imax(u16));

            try testArgs(f64, i32, imin(i32));
            try testArgs(f64, i32, imin(i32) + 1);
            try testArgs(f64, i32, -1e9);
            try testArgs(f64, i32, -1e8);
            try testArgs(f64, i32, -1e7);
            try testArgs(f64, i32, -1e6);
            try testArgs(f64, i32, -1e5);
            try testArgs(f64, i32, -1e4);
            try testArgs(f64, i32, -1e3);
            try testArgs(f64, i32, -1e2);
            try testArgs(f64, i32, -1e1);
            try testArgs(f64, i32, -1e0);
            try testArgs(f64, i32, 0);
            try testArgs(f64, i32, 1e0);
            try testArgs(f64, i32, 1e1);
            try testArgs(f64, i32, 1e2);
            try testArgs(f64, i32, 1e3);
            try testArgs(f64, i32, 1e4);
            try testArgs(f64, i32, 1e5);
            try testArgs(f64, i32, 1e6);
            try testArgs(f64, i32, 1e7);
            try testArgs(f64, i32, 1e8);
            try testArgs(f64, i32, 1e9);
            try testArgs(f64, i32, imax(i32) - 1);
            try testArgs(f64, i32, imax(i32));

            try testArgs(f64, u32, 0);
            try testArgs(f64, u32, 1e0);
            try testArgs(f64, u32, 1e1);
            try testArgs(f64, u32, 1e2);
            try testArgs(f64, u32, 1e3);
            try testArgs(f64, u32, 1e4);
            try testArgs(f64, u32, 1e5);
            try testArgs(f64, u32, 1e6);
            try testArgs(f64, u32, 1e7);
            try testArgs(f64, u32, 1e8);
            try testArgs(f64, u32, 1e9);
            try testArgs(f64, u32, imax(u32) - 1);
            try testArgs(f64, u32, imax(u32));

            try testArgs(f64, i64, imin(i64));
            try testArgs(f64, i64, imin(i64) + 1);
            try testArgs(f64, i64, -1e18);
            try testArgs(f64, i64, -1e16);
            try testArgs(f64, i64, -1e14);
            try testArgs(f64, i64, -1e12);
            try testArgs(f64, i64, -1e10);
            try testArgs(f64, i64, -1e8);
            try testArgs(f64, i64, -1e6);
            try testArgs(f64, i64, -1e4);
            try testArgs(f64, i64, -1e2);
            try testArgs(f64, i64, -1e0);
            try testArgs(f64, i64, 0);
            try testArgs(f64, i64, 1e0);
            try testArgs(f64, i64, 1e2);
            try testArgs(f64, i64, 1e4);
            try testArgs(f64, i64, 1e6);
            try testArgs(f64, i64, 1e8);
            try testArgs(f64, i64, 1e10);
            try testArgs(f64, i64, 1e12);
            try testArgs(f64, i64, 1e14);
            try testArgs(f64, i64, 1e16);
            try testArgs(f64, i64, 1e18);
            try testArgs(f64, i64, imax(i64) - 1);
            try testArgs(f64, i64, imax(i64));

            try testArgs(f64, u64, 0);
            try testArgs(f64, u64, 1e0);
            try testArgs(f64, u64, 1e2);
            try testArgs(f64, u64, 1e4);
            try testArgs(f64, u64, 1e6);
            try testArgs(f64, u64, 1e8);
            try testArgs(f64, u64, 1e10);
            try testArgs(f64, u64, 1e12);
            try testArgs(f64, u64, 1e14);
            try testArgs(f64, u64, 1e16);
            try testArgs(f64, u64, 1e18);
            try testArgs(f64, u64, imax(u64) - 1);
            try testArgs(f64, u64, imax(u64));

            try testArgs(f64, i128, imin(i128));
            try testArgs(f64, i128, imin(i128) + 1);
            try testArgs(f64, i128, -1e38);
            try testArgs(f64, i128, -1e34);
            try testArgs(f64, i128, -1e30);
            try testArgs(f64, i128, -1e26);
            try testArgs(f64, i128, -1e22);
            try testArgs(f64, i128, -1e18);
            try testArgs(f64, i128, -1e14);
            try testArgs(f64, i128, -1e10);
            try testArgs(f64, i128, -1e6);
            try testArgs(f64, i128, -1e2);
            try testArgs(f64, i128, -1e0);
            try testArgs(f64, i128, 0);
            try testArgs(f64, i128, 1e0);
            try testArgs(f64, i128, 1e2);
            try testArgs(f64, i128, 1e6);
            try testArgs(f64, i128, 1e10);
            try testArgs(f64, i128, 1e14);
            try testArgs(f64, i128, 1e18);
            try testArgs(f64, i128, 1e22);
            try testArgs(f64, i128, 1e26);
            try testArgs(f64, i128, 1e30);
            try testArgs(f64, i128, 1e34);
            try testArgs(f64, i128, 1e38);
            try testArgs(f64, i128, imax(i128) - 1);
            try testArgs(f64, i128, imax(i128));

            try testArgs(f64, u128, 0);
            try testArgs(f64, u128, 1e0);
            try testArgs(f64, u128, 1e2);
            try testArgs(f64, u128, 1e6);
            try testArgs(f64, u128, 1e10);
            try testArgs(f64, u128, 1e14);
            try testArgs(f64, u128, 1e18);
            try testArgs(f64, u128, 1e22);
            try testArgs(f64, u128, 1e26);
            try testArgs(f64, u128, 1e30);
            try testArgs(f64, u128, 1e34);
            try testArgs(f64, u128, 1e38);
            try testArgs(f64, u128, imax(u128) - 1);
            try testArgs(f64, u128, imax(u128));

            try testArgs(f64, i256, imin(i256));
            try testArgs(f64, i256, imin(i256) + 1);
            try testArgs(f64, i256, -1e76);
            try testArgs(f64, i256, -1e69);
            try testArgs(f64, i256, -1e62);
            try testArgs(f64, i256, -1e55);
            try testArgs(f64, i256, -1e48);
            try testArgs(f64, i256, -1e41);
            try testArgs(f64, i256, -1e34);
            try testArgs(f64, i256, -1e27);
            try testArgs(f64, i256, -1e20);
            try testArgs(f64, i256, -1e13);
            try testArgs(f64, i256, -1e6);
            try testArgs(f64, i256, -1e0);
            try testArgs(f64, i256, 0);
            try testArgs(f64, i256, 1e0);
            try testArgs(f64, i256, 1e6);
            try testArgs(f64, i256, 1e13);
            try testArgs(f64, i256, 1e20);
            try testArgs(f64, i256, 1e27);
            try testArgs(f64, i256, 1e34);
            try testArgs(f64, i256, 1e41);
            try testArgs(f64, i256, 1e48);
            try testArgs(f64, i256, 1e55);
            try testArgs(f64, i256, 1e62);
            try testArgs(f64, i256, 1e69);
            try testArgs(f64, i256, 1e76);
            try testArgs(f64, i256, imax(i256) - 1);
            try testArgs(f64, i256, imax(i256));

            try testArgs(f64, u256, 0);
            try testArgs(f64, u256, 1e0);
            try testArgs(f64, u256, 1e7);
            try testArgs(f64, u256, 1e14);
            try testArgs(f64, u256, 1e21);
            try testArgs(f64, u256, 1e28);
            try testArgs(f64, u256, 1e35);
            try testArgs(f64, u256, 1e42);
            try testArgs(f64, u256, 1e49);
            try testArgs(f64, u256, 1e56);
            try testArgs(f64, u256, 1e63);
            try testArgs(f64, u256, 1e70);
            try testArgs(f64, u256, 1e77);
            try testArgs(f64, u256, imax(u256) - 1);
            try testArgs(f64, u256, imax(u256));

            try testArgs(f80, i8, imin(i8));
            try testArgs(f80, i8, imin(i8) + 1);
            try testArgs(f80, i8, -1e2);
            try testArgs(f80, i8, -1e1);
            try testArgs(f80, i8, -1e0);
            try testArgs(f80, i8, 0);
            try testArgs(f80, i8, 1e0);
            try testArgs(f80, i8, 1e1);
            try testArgs(f80, i8, 1e2);
            try testArgs(f80, i8, imax(i8) - 1);
            try testArgs(f80, i8, imax(i8));

            try testArgs(f80, u8, 0);
            try testArgs(f80, u8, 1e0);
            try testArgs(f80, u8, 1e1);
            try testArgs(f80, u8, 1e2);
            try testArgs(f80, u8, imax(u8) - 1);
            try testArgs(f80, u8, imax(u8));

            try testArgs(f80, i16, imin(i16));
            try testArgs(f80, i16, imin(i16) + 1);
            try testArgs(f80, i16, -1e4);
            try testArgs(f80, i16, -1e3);
            try testArgs(f80, i16, -1e2);
            try testArgs(f80, i16, -1e1);
            try testArgs(f80, i16, -1e0);
            try testArgs(f80, i16, 0);
            try testArgs(f80, i16, 1e0);
            try testArgs(f80, i16, 1e1);
            try testArgs(f80, i16, 1e2);
            try testArgs(f80, i16, 1e3);
            try testArgs(f80, i16, 1e4);
            try testArgs(f80, i16, imax(i16) - 1);
            try testArgs(f80, i16, imax(i16));

            try testArgs(f80, u16, 0);
            try testArgs(f80, u16, 1e0);
            try testArgs(f80, u16, 1e1);
            try testArgs(f80, u16, 1e2);
            try testArgs(f80, u16, 1e3);
            try testArgs(f80, u16, 1e4);
            try testArgs(f80, u16, imax(u16) - 1);
            try testArgs(f80, u16, imax(u16));

            try testArgs(f80, i32, imin(i32));
            try testArgs(f80, i32, imin(i32) + 1);
            try testArgs(f80, i32, -1e9);
            try testArgs(f80, i32, -1e8);
            try testArgs(f80, i32, -1e7);
            try testArgs(f80, i32, -1e6);
            try testArgs(f80, i32, -1e5);
            try testArgs(f80, i32, -1e4);
            try testArgs(f80, i32, -1e3);
            try testArgs(f80, i32, -1e2);
            try testArgs(f80, i32, -1e1);
            try testArgs(f80, i32, -1e0);
            try testArgs(f80, i32, 0);
            try testArgs(f80, i32, 1e0);
            try testArgs(f80, i32, 1e1);
            try testArgs(f80, i32, 1e2);
            try testArgs(f80, i32, 1e3);
            try testArgs(f80, i32, 1e4);
            try testArgs(f80, i32, 1e5);
            try testArgs(f80, i32, 1e6);
            try testArgs(f80, i32, 1e7);
            try testArgs(f80, i32, 1e8);
            try testArgs(f80, i32, 1e9);
            try testArgs(f80, i32, imax(i32) - 1);
            try testArgs(f80, i32, imax(i32));

            try testArgs(f80, u32, 0);
            try testArgs(f80, u32, 1e0);
            try testArgs(f80, u32, 1e1);
            try testArgs(f80, u32, 1e2);
            try testArgs(f80, u32, 1e3);
            try testArgs(f80, u32, 1e4);
            try testArgs(f80, u32, 1e5);
            try testArgs(f80, u32, 1e6);
            try testArgs(f80, u32, 1e7);
            try testArgs(f80, u32, 1e8);
            try testArgs(f80, u32, 1e9);
            try testArgs(f80, u32, imax(u32) - 1);
            try testArgs(f80, u32, imax(u32));

            try testArgs(f80, i64, imin(i64));
            try testArgs(f80, i64, imin(i64) + 1);
            try testArgs(f80, i64, -1e18);
            try testArgs(f80, i64, -1e16);
            try testArgs(f80, i64, -1e14);
            try testArgs(f80, i64, -1e12);
            try testArgs(f80, i64, -1e10);
            try testArgs(f80, i64, -1e8);
            try testArgs(f80, i64, -1e6);
            try testArgs(f80, i64, -1e4);
            try testArgs(f80, i64, -1e2);
            try testArgs(f80, i64, -1e0);
            try testArgs(f80, i64, 0);
            try testArgs(f80, i64, 1e0);
            try testArgs(f80, i64, 1e2);
            try testArgs(f80, i64, 1e4);
            try testArgs(f80, i64, 1e6);
            try testArgs(f80, i64, 1e8);
            try testArgs(f80, i64, 1e10);
            try testArgs(f80, i64, 1e12);
            try testArgs(f80, i64, 1e14);
            try testArgs(f80, i64, 1e16);
            try testArgs(f80, i64, 1e18);
            try testArgs(f80, i64, imax(i64) - 1);
            try testArgs(f80, i64, imax(i64));

            try testArgs(f80, u64, 0);
            try testArgs(f80, u64, 1e0);
            try testArgs(f80, u64, 1e2);
            try testArgs(f80, u64, 1e4);
            try testArgs(f80, u64, 1e6);
            try testArgs(f80, u64, 1e8);
            try testArgs(f80, u64, 1e10);
            try testArgs(f80, u64, 1e12);
            try testArgs(f80, u64, 1e14);
            try testArgs(f80, u64, 1e16);
            try testArgs(f80, u64, 1e18);
            try testArgs(f80, u64, imax(u64) - 1);
            try testArgs(f80, u64, imax(u64));

            try testArgs(f80, i128, imin(i128));
            try testArgs(f80, i128, imin(i128) + 1);
            try testArgs(f80, i128, -1e38);
            try testArgs(f80, i128, -1e34);
            try testArgs(f80, i128, -1e30);
            try testArgs(f80, i128, -1e26);
            try testArgs(f80, i128, -1e22);
            try testArgs(f80, i128, -1e18);
            try testArgs(f80, i128, -1e14);
            try testArgs(f80, i128, -1e10);
            try testArgs(f80, i128, -1e6);
            try testArgs(f80, i128, -1e2);
            try testArgs(f80, i128, -1e0);
            try testArgs(f80, i128, 0);
            try testArgs(f80, i128, 1e0);
            try testArgs(f80, i128, 1e2);
            try testArgs(f80, i128, 1e6);
            try testArgs(f80, i128, 1e10);
            try testArgs(f80, i128, 1e14);
            try testArgs(f80, i128, 1e18);
            try testArgs(f80, i128, 1e22);
            try testArgs(f80, i128, 1e26);
            try testArgs(f80, i128, 1e30);
            try testArgs(f80, i128, 1e34);
            try testArgs(f80, i128, 1e38);
            try testArgs(f80, i128, imax(i128) - 1);
            try testArgs(f80, i128, imax(i128));

            try testArgs(f80, u128, 0);
            try testArgs(f80, u128, 1e0);
            try testArgs(f80, u128, 1e2);
            try testArgs(f80, u128, 1e6);
            try testArgs(f80, u128, 1e10);
            try testArgs(f80, u128, 1e14);
            try testArgs(f80, u128, 1e18);
            try testArgs(f80, u128, 1e22);
            try testArgs(f80, u128, 1e26);
            try testArgs(f80, u128, 1e30);
            try testArgs(f80, u128, 1e34);
            try testArgs(f80, u128, 1e38);
            try testArgs(f80, u128, imax(u128) - 1);
            try testArgs(f80, u128, imax(u128));

            try testArgs(f80, i256, imin(i256));
            try testArgs(f80, i256, imin(i256) + 1);
            try testArgs(f80, i256, -1e76);
            try testArgs(f80, i256, -1e69);
            try testArgs(f80, i256, -1e62);
            try testArgs(f80, i256, -1e55);
            try testArgs(f80, i256, -1e48);
            try testArgs(f80, i256, -1e41);
            try testArgs(f80, i256, -1e34);
            try testArgs(f80, i256, -1e27);
            try testArgs(f80, i256, -1e20);
            try testArgs(f80, i256, -1e13);
            try testArgs(f80, i256, -1e6);
            try testArgs(f80, i256, -1e0);
            try testArgs(f80, i256, 0);
            try testArgs(f80, i256, 1e0);
            try testArgs(f80, i256, 1e6);
            try testArgs(f80, i256, 1e13);
            try testArgs(f80, i256, 1e20);
            try testArgs(f80, i256, 1e27);
            try testArgs(f80, i256, 1e34);
            try testArgs(f80, i256, 1e41);
            try testArgs(f80, i256, 1e48);
            try testArgs(f80, i256, 1e55);
            try testArgs(f80, i256, 1e62);
            try testArgs(f80, i256, 1e69);
            try testArgs(f80, i256, 1e76);
            try testArgs(f80, i256, imax(i256) - 1);
            try testArgs(f80, i256, imax(i256));

            try testArgs(f80, u256, 0);
            try testArgs(f80, u256, 1e0);
            try testArgs(f80, u256, 1e7);
            try testArgs(f80, u256, 1e14);
            try testArgs(f80, u256, 1e21);
            try testArgs(f80, u256, 1e28);
            try testArgs(f80, u256, 1e35);
            try testArgs(f80, u256, 1e42);
            try testArgs(f80, u256, 1e49);
            try testArgs(f80, u256, 1e56);
            try testArgs(f80, u256, 1e63);
            try testArgs(f80, u256, 1e70);
            try testArgs(f80, u256, 1e77);
            try testArgs(f80, u256, imax(u256) - 1);
            try testArgs(f80, u256, imax(u256));

            try testArgs(f128, i8, imin(i8));
            try testArgs(f128, i8, imin(i8) + 1);
            try testArgs(f128, i8, -1e2);
            try testArgs(f128, i8, -1e1);
            try testArgs(f128, i8, -1e0);
            try testArgs(f128, i8, 0);
            try testArgs(f128, i8, 1e0);
            try testArgs(f128, i8, 1e1);
            try testArgs(f128, i8, 1e2);
            try testArgs(f128, i8, imax(i8) - 1);
            try testArgs(f128, i8, imax(i8));

            try testArgs(f128, u8, 0);
            try testArgs(f128, u8, 1e0);
            try testArgs(f128, u8, 1e1);
            try testArgs(f128, u8, 1e2);
            try testArgs(f128, u8, imax(u8) - 1);
            try testArgs(f128, u8, imax(u8));

            try testArgs(f128, i16, imin(i16));
            try testArgs(f128, i16, imin(i16) + 1);
            try testArgs(f128, i16, -1e4);
            try testArgs(f128, i16, -1e3);
            try testArgs(f128, i16, -1e2);
            try testArgs(f128, i16, -1e1);
            try testArgs(f128, i16, -1e0);
            try testArgs(f128, i16, 0);
            try testArgs(f128, i16, 1e0);
            try testArgs(f128, i16, 1e1);
            try testArgs(f128, i16, 1e2);
            try testArgs(f128, i16, 1e3);
            try testArgs(f128, i16, 1e4);
            try testArgs(f128, i16, imax(i16) - 1);
            try testArgs(f128, i16, imax(i16));

            try testArgs(f128, u16, 0);
            try testArgs(f128, u16, 1e0);
            try testArgs(f128, u16, 1e1);
            try testArgs(f128, u16, 1e2);
            try testArgs(f128, u16, 1e3);
            try testArgs(f128, u16, 1e4);
            try testArgs(f128, u16, imax(u16) - 1);
            try testArgs(f128, u16, imax(u16));

            try testArgs(f128, i32, imin(i32));
            try testArgs(f128, i32, imin(i32) + 1);
            try testArgs(f128, i32, -1e9);
            try testArgs(f128, i32, -1e8);
            try testArgs(f128, i32, -1e7);
            try testArgs(f128, i32, -1e6);
            try testArgs(f128, i32, -1e5);
            try testArgs(f128, i32, -1e4);
            try testArgs(f128, i32, -1e3);
            try testArgs(f128, i32, -1e2);
            try testArgs(f128, i32, -1e1);
            try testArgs(f128, i32, -1e0);
            try testArgs(f128, i32, 0);
            try testArgs(f128, i32, 1e0);
            try testArgs(f128, i32, 1e1);
            try testArgs(f128, i32, 1e2);
            try testArgs(f128, i32, 1e3);
            try testArgs(f128, i32, 1e4);
            try testArgs(f128, i32, 1e5);
            try testArgs(f128, i32, 1e6);
            try testArgs(f128, i32, 1e7);
            try testArgs(f128, i32, 1e8);
            try testArgs(f128, i32, 1e9);
            try testArgs(f128, i32, imax(i32) - 1);
            try testArgs(f128, i32, imax(i32));

            try testArgs(f128, u32, 0);
            try testArgs(f128, u32, 1e0);
            try testArgs(f128, u32, 1e1);
            try testArgs(f128, u32, 1e2);
            try testArgs(f128, u32, 1e3);
            try testArgs(f128, u32, 1e4);
            try testArgs(f128, u32, 1e5);
            try testArgs(f128, u32, 1e6);
            try testArgs(f128, u32, 1e7);
            try testArgs(f128, u32, 1e8);
            try testArgs(f128, u32, 1e9);
            try testArgs(f128, u32, imax(u32) - 1);
            try testArgs(f128, u32, imax(u32));

            try testArgs(f128, i64, imin(i64));
            try testArgs(f128, i64, imin(i64) + 1);
            try testArgs(f128, i64, -1e18);
            try testArgs(f128, i64, -1e16);
            try testArgs(f128, i64, -1e14);
            try testArgs(f128, i64, -1e12);
            try testArgs(f128, i64, -1e10);
            try testArgs(f128, i64, -1e8);
            try testArgs(f128, i64, -1e6);
            try testArgs(f128, i64, -1e4);
            try testArgs(f128, i64, -1e2);
            try testArgs(f128, i64, -1e0);
            try testArgs(f128, i64, 0);
            try testArgs(f128, i64, 1e0);
            try testArgs(f128, i64, 1e2);
            try testArgs(f128, i64, 1e4);
            try testArgs(f128, i64, 1e6);
            try testArgs(f128, i64, 1e8);
            try testArgs(f128, i64, 1e10);
            try testArgs(f128, i64, 1e12);
            try testArgs(f128, i64, 1e14);
            try testArgs(f128, i64, 1e16);
            try testArgs(f128, i64, 1e18);
            try testArgs(f128, i64, imax(i64) - 1);
            try testArgs(f128, i64, imax(i64));

            try testArgs(f128, u64, 0);
            try testArgs(f128, u64, 1e0);
            try testArgs(f128, u64, 1e2);
            try testArgs(f128, u64, 1e4);
            try testArgs(f128, u64, 1e6);
            try testArgs(f128, u64, 1e8);
            try testArgs(f128, u64, 1e10);
            try testArgs(f128, u64, 1e12);
            try testArgs(f128, u64, 1e14);
            try testArgs(f128, u64, 1e16);
            try testArgs(f128, u64, 1e18);
            try testArgs(f128, u64, imax(u64) - 1);
            try testArgs(f128, u64, imax(u64));

            try testArgs(f128, i128, imin(i128));
            try testArgs(f128, i128, imin(i128) + 1);
            try testArgs(f128, i128, -1e38);
            try testArgs(f128, i128, -1e34);
            try testArgs(f128, i128, -1e30);
            try testArgs(f128, i128, -1e26);
            try testArgs(f128, i128, -1e22);
            try testArgs(f128, i128, -1e18);
            try testArgs(f128, i128, -1e14);
            try testArgs(f128, i128, -1e10);
            try testArgs(f128, i128, -1e6);
            try testArgs(f128, i128, -1e2);
            try testArgs(f128, i128, -1e0);
            try testArgs(f128, i128, 0);
            try testArgs(f128, i128, 1e0);
            try testArgs(f128, i128, 1e2);
            try testArgs(f128, i128, 1e6);
            try testArgs(f128, i128, 1e10);
            try testArgs(f128, i128, 1e14);
            try testArgs(f128, i128, 1e18);
            try testArgs(f128, i128, 1e22);
            try testArgs(f128, i128, 1e26);
            try testArgs(f128, i128, 1e30);
            try testArgs(f128, i128, 1e34);
            try testArgs(f128, i128, 1e38);
            try testArgs(f128, i128, imax(i128) - 1);
            try testArgs(f128, i128, imax(i128));

            try testArgs(f128, u128, 0);
            try testArgs(f128, u128, 1e0);
            try testArgs(f128, u128, 1e2);
            try testArgs(f128, u128, 1e6);
            try testArgs(f128, u128, 1e10);
            try testArgs(f128, u128, 1e14);
            try testArgs(f128, u128, 1e18);
            try testArgs(f128, u128, 1e22);
            try testArgs(f128, u128, 1e26);
            try testArgs(f128, u128, 1e30);
            try testArgs(f128, u128, 1e34);
            try testArgs(f128, u128, 1e38);
            try testArgs(f128, u128, imax(u128) - 1);
            try testArgs(f128, u128, imax(u128));

            try testArgs(f128, i256, imin(i256));
            try testArgs(f128, i256, imin(i256) + 1);
            try testArgs(f128, i256, -1e76);
            try testArgs(f128, i256, -1e69);
            try testArgs(f128, i256, -1e62);
            try testArgs(f128, i256, -1e55);
            try testArgs(f128, i256, -1e48);
            try testArgs(f128, i256, -1e41);
            try testArgs(f128, i256, -1e34);
            try testArgs(f128, i256, -1e27);
            try testArgs(f128, i256, -1e20);
            try testArgs(f128, i256, -1e13);
            try testArgs(f128, i256, -1e6);
            try testArgs(f128, i256, -1e0);
            try testArgs(f128, i256, 0);
            try testArgs(f128, i256, 1e0);
            try testArgs(f128, i256, 1e6);
            try testArgs(f128, i256, 1e13);
            try testArgs(f128, i256, 1e20);
            try testArgs(f128, i256, 1e27);
            try testArgs(f128, i256, 1e34);
            try testArgs(f128, i256, 1e41);
            try testArgs(f128, i256, 1e48);
            try testArgs(f128, i256, 1e55);
            try testArgs(f128, i256, 1e62);
            try testArgs(f128, i256, 1e69);
            try testArgs(f128, i256, 1e76);
            try testArgs(f128, i256, imax(i256) - 1);
            try testArgs(f128, i256, imax(i256));

            try testArgs(f128, u256, 0);
            try testArgs(f128, u256, 1e0);
            try testArgs(f128, u256, 1e7);
            try testArgs(f128, u256, 1e14);
            try testArgs(f128, u256, 1e21);
            try testArgs(f128, u256, 1e28);
            try testArgs(f128, u256, 1e35);
            try testArgs(f128, u256, 1e42);
            try testArgs(f128, u256, 1e49);
            try testArgs(f128, u256, 1e56);
            try testArgs(f128, u256, 1e63);
            try testArgs(f128, u256, 1e70);
            try testArgs(f128, u256, 1e77);
            try testArgs(f128, u256, imax(u256) - 1);
            try testArgs(f128, u256, imax(u256));
        }
        fn testIntVectorsFromFloatVectors() !void {
            @setEvalBranchQuota(2_500);

            try testArgs(@Vector(1, i8), @Vector(1, f16), .{
                -0x0.8p8,
            });
            try testArgs(@Vector(2, i8), @Vector(2, f16), .{
                next(f16, -0x0.8p8, -0.0), next(f16, next(f16, -0x0.8p8, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i8), @Vector(4, f16), .{
                -1e2, -1e1, -1e0, -1e-1,
            });
            try testArgs(@Vector(8, i8), @Vector(8, f16), .{
                -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0, 1e-5, 1e-4,
            });
            try testArgs(@Vector(16, i8), @Vector(16, f16), .{
                1e-3, 1e-2, 1e-1,  1e0,   1e1,   1e2,   next(f16, next(f16, 0x0.8p8, 0.0), 0.0), next(f16, 0x0.8p8, 0.0),
                -2e1, -2e0, -2e-1, -2e-2, -2e-3, -2e-4, -2e-5,                                   2e-5,
            });
            try testArgs(@Vector(32, i8), @Vector(32, f16), .{
                2e-4,  2e-3,  2e-2,  2e-1,  2e0,   2e1,  -3e1,  -3e0,
                -3e-1, -3e-2, -3e-3, -3e-4, -3e-5, 3e-5, 3e-4,  3e-3,
                3e-2,  3e-1,  3e0,   3e1,   -4e1,  -4e0, -4e-1, -4e-2,
                -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3, 4e-2,  4e-1,
            });
            try testArgs(@Vector(64, i8), @Vector(64, f16), .{
                4e0,   4e1,   -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4,
                -5e-5, 5e-5,  5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,
                -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4, -6e-5, 6e-5,
                6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,   -7e1,  -7e0,
                -7e-1, -7e-2, -7e-3, -7e-4, -7e-5, 7e-5,  7e-4,  7e-3,
                7e-2,  7e-1,  7e0,   7e1,   -8e1,  -8e0,  -8e-1, -8e-2,
                -8e-3, -8e-4, -8e-5, 8e-5,  8e-4,  8e-3,  8e-2,  8e-1,
                8e0,   8e1,   -9e1,  -9e0,  -9e-1, -9e-2, -9e-3, -9e-4,
            });
            try testArgs(@Vector(128, i8), @Vector(128, f16), .{
                -9e-5,  9e-5,   9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,
                -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4, -11e-5, 11e-5,
                11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,   -12e1,  -12e0,
                -12e-1, -12e-2, -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,
                12e-2,  12e-1,  12e0,   12e1,   -13e0,  -13e-1, -13e-2, -13e-3,
                -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,  13e-2,  13e-1,  13e0,
                -14e0,  -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,
                14e-3,  14e-2,  14e-1,  14e0,   -15e0,  -15e-1, -15e-2, -15e-3,
                -15e-4, -15e-5, 15e-5,  15e-4,  15e-3,  15e-2,  15e-1,  15e0,
                -16e0,  -16e-1, -16e-2, -16e-3, -16e-4, -16e-5, 16e-5,  16e-4,
                16e-3,  16e-2,  16e-1,  16e0,   -17e0,  -17e-1, -17e-2, -17e-3,
                -17e-4, -17e-5, 17e-5,  17e-4,  17e-3,  17e-2,  17e-1,  17e0,
                -18e0,  -18e-1, -18e-2, -18e-3, -18e-4, -18e-5, 18e-5,  18e-4,
                18e-3,  18e-2,  18e-1,  18e0,   -19e0,  -19e-1, -19e-2, -19e-3,
                -19e-4, -19e-5, 19e-5,  19e-4,  19e-3,  19e-2,  19e-1,  19e0,
                -21e0,  -21e-1, -21e-2, -21e-3, -21e-4, -21e-5, 21e-5,  21e-4,
            });

            try testArgs(@Vector(1, u8), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u8), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u8), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u8), @Vector(8, f16), .{
                1e0, 1e1, 1e2, next(f16, next(f16, 0x1p8, 0.0), 0.0), next(f16, 0x1p8, 0.0), 2e-5, 2e-4, 2e-3,
            });
            try testArgs(@Vector(16, u8), @Vector(16, f16), .{
                2e-2, 2e-1, 2e0, 2e1, 2e2,  3e-5, 3e-4, 3e-3,
                3e-2, 3e-1, 3e0, 3e1, 4e-5, 4e-4, 4e-3, 4e-2,
            });
            try testArgs(@Vector(32, u8), @Vector(32, f16), .{
                4e-1, 4e0,  4e1,  5e-5, 5e-4, 5e-3, 5e-2, 5e-1,
                5e0,  5e1,  6e-5, 6e-4, 6e-3, 6e-2, 6e-1, 6e0,
                6e1,  7e-5, 7e-4, 7e-3, 7e-2, 7e-1, 7e0,  7e1,
                8e-5, 8e-4, 8e-3, 8e-2, 8e-1, 8e0,  8e1,  9e-5,
            });
            try testArgs(@Vector(64, u8), @Vector(64, f16), .{
                9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   11e-5, 11e-4,
                11e-3, 11e-2, 11e-1, 11e0,  11e1,  13e-5, 13e-4, 13e-3,
                13e-2, 13e-1, 13e0,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  15e-5, 15e-4, 15e-3, 15e-2, 15e-1, 15e0,
                15e1,  16e-5, 16e-4, 16e-3, 16e-2, 16e-1, 16e0,  16e1,
                17e-5, 17e-4, 17e-3, 17e-2, 17e-1, 17e0,  17e1,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  19e-5, 19e-4,
                19e-3, 19e-2, 19e-1, 19e0,  19e1,  21e-5, 21e-4, 21e-3,
            });
            try testArgs(@Vector(128, u8), @Vector(128, f16), .{
                21e-2, 21e-1, 21e0,  21e1,  22e-5, 22e-4, 22e-3, 22e-2,
                22e-1, 22e0,  22e1,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  24e-5, 24e-4, 24e-3, 24e-2, 24e-1, 24e0,
                24e1,  25e-5, 25e-4, 25e-3, 25e-2, 25e-1, 25e0,  25e1,
                26e-5, 26e-4, 26e-3, 26e-2, 26e-1, 26e0,  27e-5, 27e-4,
                27e-3, 27e-2, 27e-1, 27e0,  28e-5, 28e-4, 28e-3, 28e-2,
                28e-1, 28e0,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,
                31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,  32e-5, 32e-4,
                32e-3, 32e-2, 32e-1, 32e0,  33e-5, 33e-4, 33e-3, 33e-2,
                33e-1, 33e0,  34e-5, 34e-4, 34e-3, 34e-2, 34e-1, 34e0,
                35e-5, 35e-4, 35e-3, 35e-2, 35e-1, 35e0,  36e-5, 36e-4,
                36e-3, 36e-2, 36e-1, 36e0,  37e-5, 37e-4, 37e-3, 37e-2,
                37e-1, 37e0,  38e-5, 38e-4, 38e-3, 38e-2, 38e-1, 38e0,
                39e-5, 39e-4, 39e-3, 39e-2, 39e-1, 39e0,  41e-5, 41e-4,
                41e-3, 41e-2, 41e-1, 41e0,  42e-5, 42e-4, 42e-3, 42e-2,
                42e-1, 42e0,  43e-5, 43e-4, 43e-3, 43e-2, 43e-1, 43e0,
            });

            try testArgs(@Vector(1, i16), @Vector(1, f16), .{
                -0x0.8p16,
            });
            try testArgs(@Vector(2, i16), @Vector(2, f16), .{
                next(f16, -0x0.8p16, -0.0), next(f16, next(f16, -0x0.8p16, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i16), @Vector(4, f16), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i16), @Vector(8, f16), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i16), @Vector(16, f16), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f16, next(f16, 0x0.8p16, 0.0), 0.0), next(f16, 0x0.8p16, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i16), @Vector(32, f16), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i16), @Vector(64, f16), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i16), @Vector(128, f16), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u16), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u16), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u16), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u16), @Vector(8, f16), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f16, next(f16, 0x1p16, 0.0), 0.0), next(f16, 0x1p16, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u16), @Vector(16, f16), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u16), @Vector(32, f16), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u16), @Vector(64, f16), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u16), @Vector(128, f16), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i32), @Vector(1, f16), .{
                -fmax(f16),
            });
            try testArgs(@Vector(2, i32), @Vector(2, f16), .{
                next(f16, -fmax(f16), -0.0), next(f16, next(f16, -fmax(f16), -0.0), -0.0),
            });
            try testArgs(@Vector(4, i32), @Vector(4, f16), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i32), @Vector(8, f16), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i32), @Vector(16, f16), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i32), @Vector(32, f16), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i32), @Vector(64, f16), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i32), @Vector(128, f16), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u32), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u32), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u32), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u32), @Vector(8, f16), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u32), @Vector(16, f16), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u32), @Vector(32, f16), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u32), @Vector(64, f16), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u32), @Vector(128, f16), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i64), @Vector(1, f16), .{
                -fmax(f16),
            });
            try testArgs(@Vector(2, i64), @Vector(2, f16), .{
                next(f16, -fmax(f16), -0.0), next(f16, next(f16, -fmax(f16), -0.0), -0.0),
            });
            try testArgs(@Vector(4, i64), @Vector(4, f16), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i64), @Vector(8, f16), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i64), @Vector(16, f16), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i64), @Vector(32, f16), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i64), @Vector(64, f16), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i64), @Vector(128, f16), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u64), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u64), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u64), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u64), @Vector(8, f16), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u64), @Vector(16, f16), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u64), @Vector(32, f16), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u64), @Vector(64, f16), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u64), @Vector(128, f16), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i128), @Vector(1, f16), .{
                -fmax(f16),
            });
            try testArgs(@Vector(2, i128), @Vector(2, f16), .{
                next(f16, -fmax(f16), -0.0), next(f16, next(f16, -fmax(f16), -0.0), -0.0),
            });
            try testArgs(@Vector(4, i128), @Vector(4, f16), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i128), @Vector(8, f16), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i128), @Vector(16, f16), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i128), @Vector(32, f16), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i128), @Vector(64, f16), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u128), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u128), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u128), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u128), @Vector(8, f16), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u128), @Vector(16, f16), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u128), @Vector(32, f16), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u128), @Vector(64, f16), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i256), @Vector(1, f16), .{
                -fmax(f16),
            });
            try testArgs(@Vector(2, i256), @Vector(2, f16), .{
                next(f16, -fmax(f16), -0.0), next(f16, next(f16, -fmax(f16), -0.0), -0.0),
            });
            try testArgs(@Vector(4, i256), @Vector(4, f16), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i256), @Vector(8, f16), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i256), @Vector(16, f16), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i256), @Vector(32, f16), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i256), @Vector(64, f16), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u256), @Vector(1, f16), .{
                -0.0,
            });
            try testArgs(@Vector(2, u256), @Vector(2, f16), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u256), @Vector(4, f16), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u256), @Vector(8, f16), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f16, next(f16, fmax(f16), 0.0), 0.0), next(f16, fmax(f16), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u256), @Vector(16, f16), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u256), @Vector(32, f16), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u256), @Vector(64, f16), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i8), @Vector(1, f32), .{
                -0x0.8p8,
            });
            try testArgs(@Vector(2, i8), @Vector(2, f32), .{
                next(f32, -0x0.8p8, -0.0), next(f32, next(f32, -0x0.8p8, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i8), @Vector(4, f32), .{
                -1e2, -1e1, -1e0, -1e-1,
            });
            try testArgs(@Vector(8, i8), @Vector(8, f32), .{
                -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0, 1e-5, 1e-4,
            });
            try testArgs(@Vector(16, i8), @Vector(16, f32), .{
                1e-3, 1e-2, 1e-1,  1e0,   1e1,   1e2,   next(f32, next(f32, 0x0.8p8, 0.0), 0.0), next(f32, 0x0.8p8, 0.0),
                -2e1, -2e0, -2e-1, -2e-2, -2e-3, -2e-4, -2e-5,                                   2e-5,
            });
            try testArgs(@Vector(32, i8), @Vector(32, f32), .{
                2e-4,  2e-3,  2e-2,  2e-1,  2e0,   2e1,  -3e1,  -3e0,
                -3e-1, -3e-2, -3e-3, -3e-4, -3e-5, 3e-5, 3e-4,  3e-3,
                3e-2,  3e-1,  3e0,   3e1,   -4e1,  -4e0, -4e-1, -4e-2,
                -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3, 4e-2,  4e-1,
            });
            try testArgs(@Vector(64, i8), @Vector(64, f32), .{
                4e0,   4e1,   -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4,
                -5e-5, 5e-5,  5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,
                -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4, -6e-5, 6e-5,
                6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,   -7e1,  -7e0,
                -7e-1, -7e-2, -7e-3, -7e-4, -7e-5, 7e-5,  7e-4,  7e-3,
                7e-2,  7e-1,  7e0,   7e1,   -8e1,  -8e0,  -8e-1, -8e-2,
                -8e-3, -8e-4, -8e-5, 8e-5,  8e-4,  8e-3,  8e-2,  8e-1,
                8e0,   8e1,   -9e1,  -9e0,  -9e-1, -9e-2, -9e-3, -9e-4,
            });
            try testArgs(@Vector(128, i8), @Vector(128, f32), .{
                -9e-5,  9e-5,   9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,
                -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4, -11e-5, 11e-5,
                11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,   -12e1,  -12e0,
                -12e-1, -12e-2, -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,
                12e-2,  12e-1,  12e0,   12e1,   -13e0,  -13e-1, -13e-2, -13e-3,
                -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,  13e-2,  13e-1,  13e0,
                -14e0,  -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,
                14e-3,  14e-2,  14e-1,  14e0,   -15e0,  -15e-1, -15e-2, -15e-3,
                -15e-4, -15e-5, 15e-5,  15e-4,  15e-3,  15e-2,  15e-1,  15e0,
                -16e0,  -16e-1, -16e-2, -16e-3, -16e-4, -16e-5, 16e-5,  16e-4,
                16e-3,  16e-2,  16e-1,  16e0,   -17e0,  -17e-1, -17e-2, -17e-3,
                -17e-4, -17e-5, 17e-5,  17e-4,  17e-3,  17e-2,  17e-1,  17e0,
                -18e0,  -18e-1, -18e-2, -18e-3, -18e-4, -18e-5, 18e-5,  18e-4,
                18e-3,  18e-2,  18e-1,  18e0,   -19e0,  -19e-1, -19e-2, -19e-3,
                -19e-4, -19e-5, 19e-5,  19e-4,  19e-3,  19e-2,  19e-1,  19e0,
                -21e0,  -21e-1, -21e-2, -21e-3, -21e-4, -21e-5, 21e-5,  21e-4,
            });

            try testArgs(@Vector(1, u8), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u8), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u8), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u8), @Vector(8, f32), .{
                1e0, 1e1, 1e2, next(f32, next(f32, 0x1p8, 0.0), 0.0), next(f32, 0x1p8, 0.0), 2e-5, 2e-4, 2e-3,
            });
            try testArgs(@Vector(16, u8), @Vector(16, f32), .{
                2e-2, 2e-1, 2e0, 2e1, 2e2,  3e-5, 3e-4, 3e-3,
                3e-2, 3e-1, 3e0, 3e1, 4e-5, 4e-4, 4e-3, 4e-2,
            });
            try testArgs(@Vector(32, u8), @Vector(32, f32), .{
                4e-1, 4e0,  4e1,  5e-5, 5e-4, 5e-3, 5e-2, 5e-1,
                5e0,  5e1,  6e-5, 6e-4, 6e-3, 6e-2, 6e-1, 6e0,
                6e1,  7e-5, 7e-4, 7e-3, 7e-2, 7e-1, 7e0,  7e1,
                8e-5, 8e-4, 8e-3, 8e-2, 8e-1, 8e0,  8e1,  9e-5,
            });
            try testArgs(@Vector(64, u8), @Vector(64, f32), .{
                9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   11e-5, 11e-4,
                11e-3, 11e-2, 11e-1, 11e0,  11e1,  13e-5, 13e-4, 13e-3,
                13e-2, 13e-1, 13e0,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  15e-5, 15e-4, 15e-3, 15e-2, 15e-1, 15e0,
                15e1,  16e-5, 16e-4, 16e-3, 16e-2, 16e-1, 16e0,  16e1,
                17e-5, 17e-4, 17e-3, 17e-2, 17e-1, 17e0,  17e1,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  19e-5, 19e-4,
                19e-3, 19e-2, 19e-1, 19e0,  19e1,  21e-5, 21e-4, 21e-3,
            });
            try testArgs(@Vector(128, u8), @Vector(128, f32), .{
                21e-2, 21e-1, 21e0,  21e1,  22e-5, 22e-4, 22e-3, 22e-2,
                22e-1, 22e0,  22e1,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  24e-5, 24e-4, 24e-3, 24e-2, 24e-1, 24e0,
                24e1,  25e-5, 25e-4, 25e-3, 25e-2, 25e-1, 25e0,  25e1,
                26e-5, 26e-4, 26e-3, 26e-2, 26e-1, 26e0,  27e-5, 27e-4,
                27e-3, 27e-2, 27e-1, 27e0,  28e-5, 28e-4, 28e-3, 28e-2,
                28e-1, 28e0,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,
                31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,  32e-5, 32e-4,
                32e-3, 32e-2, 32e-1, 32e0,  33e-5, 33e-4, 33e-3, 33e-2,
                33e-1, 33e0,  34e-5, 34e-4, 34e-3, 34e-2, 34e-1, 34e0,
                35e-5, 35e-4, 35e-3, 35e-2, 35e-1, 35e0,  36e-5, 36e-4,
                36e-3, 36e-2, 36e-1, 36e0,  37e-5, 37e-4, 37e-3, 37e-2,
                37e-1, 37e0,  38e-5, 38e-4, 38e-3, 38e-2, 38e-1, 38e0,
                39e-5, 39e-4, 39e-3, 39e-2, 39e-1, 39e0,  41e-5, 41e-4,
                41e-3, 41e-2, 41e-1, 41e0,  42e-5, 42e-4, 42e-3, 42e-2,
                42e-1, 42e0,  43e-5, 43e-4, 43e-3, 43e-2, 43e-1, 43e0,
            });

            try testArgs(@Vector(1, i16), @Vector(1, f32), .{
                -0x0.8p16,
            });
            try testArgs(@Vector(2, i16), @Vector(2, f32), .{
                next(f32, -0x0.8p16, -0.0), next(f32, next(f32, -0x0.8p16, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i16), @Vector(4, f32), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i16), @Vector(8, f32), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i16), @Vector(16, f32), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f32, next(f32, 0x0.8p16, 0.0), 0.0), next(f32, 0x0.8p16, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i16), @Vector(32, f32), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i16), @Vector(64, f32), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i16), @Vector(128, f32), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u16), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u16), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u16), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u16), @Vector(8, f32), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f32, next(f32, 0x1p16, 0.0), 0.0), next(f32, 0x1p16, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u16), @Vector(16, f32), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u16), @Vector(32, f32), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u16), @Vector(64, f32), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u16), @Vector(128, f32), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i32), @Vector(1, f32), .{
                -0x0.8p32,
            });
            try testArgs(@Vector(2, i32), @Vector(2, f32), .{
                next(f32, -0x0.8p32, -0.0), next(f32, next(f32, -0x0.8p32, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i32), @Vector(4, f32), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i32), @Vector(8, f32), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i32), @Vector(16, f32), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f32, next(f32, 0x0.8p32, 0.0), 0.0), next(f32, 0x0.8p32, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i32), @Vector(32, f32), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i32), @Vector(64, f32), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i32), @Vector(128, f32), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u32), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u32), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u32), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u32), @Vector(8, f32), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f32, next(f32, 0x1p32, 0.0), 0.0), next(f32, 0x1p32, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u32), @Vector(16, f32), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u32), @Vector(32, f32), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u32), @Vector(64, f32), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u32), @Vector(128, f32), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i64), @Vector(1, f32), .{
                -0x0.8p64,
            });
            try testArgs(@Vector(2, i64), @Vector(2, f32), .{
                next(f32, -0x0.8p64, -0.0), next(f32, next(f32, -0x0.8p64, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i64), @Vector(4, f32), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i64), @Vector(8, f32), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i64), @Vector(16, f32), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f32, next(f32, 0x0.8p64, 0.0), 0.0), next(f32, 0x0.8p64, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i64), @Vector(32, f32), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i64), @Vector(64, f32), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i64), @Vector(128, f32), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u64), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u64), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u64), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u64), @Vector(8, f32), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f32, next(f32, 0x1p64, 0.0), 0.0), next(f32, 0x1p64, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u64), @Vector(16, f32), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u64), @Vector(32, f32), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u64), @Vector(64, f32), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u64), @Vector(128, f32), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i128), @Vector(1, f32), .{
                -0x0.8p128,
            });
            try testArgs(@Vector(2, i128), @Vector(2, f32), .{
                next(f32, -0x0.8p128, -0.0), next(f32, next(f32, -0x0.8p128, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i128), @Vector(4, f32), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i128), @Vector(8, f32), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i128), @Vector(16, f32), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f32, next(f32, 0x0.8p128, 0.0), 0.0), next(f32, 0x0.8p128, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i128), @Vector(32, f32), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i128), @Vector(64, f32), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u128), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u128), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u128), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u128), @Vector(8, f32), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f32, next(f32, fmax(f32), 0.0), 0.0), next(f32, fmax(f32), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u128), @Vector(16, f32), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u128), @Vector(32, f32), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u128), @Vector(64, f32), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i256), @Vector(1, f32), .{
                -fmax(f32),
            });
            try testArgs(@Vector(2, i256), @Vector(2, f32), .{
                next(f32, -fmax(f32), -0.0), next(f32, next(f32, -fmax(f32), -0.0), -0.0),
            });
            try testArgs(@Vector(4, i256), @Vector(4, f32), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i256), @Vector(8, f32), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i256), @Vector(16, f32), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f32, next(f32, fmax(f32), 0.0), 0.0), next(f32, fmax(f32), 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i256), @Vector(32, f32), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i256), @Vector(64, f32), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u256), @Vector(1, f32), .{
                -0.0,
            });
            try testArgs(@Vector(2, u256), @Vector(2, f32), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u256), @Vector(4, f32), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u256), @Vector(8, f32), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f32, next(f32, fmax(f32), 0.0), 0.0), next(f32, fmax(f32), 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u256), @Vector(16, f32), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u256), @Vector(32, f32), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u256), @Vector(64, f32), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i8), @Vector(1, f64), .{
                -0x0.8p8,
            });
            try testArgs(@Vector(2, i8), @Vector(2, f64), .{
                next(f64, -0x0.8p8, -0.0), next(f64, next(f64, -0x0.8p8, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i8), @Vector(4, f64), .{
                -1e2, -1e1, -1e0, -1e-1,
            });
            try testArgs(@Vector(8, i8), @Vector(8, f64), .{
                -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0, 1e-5, 1e-4,
            });
            try testArgs(@Vector(16, i8), @Vector(16, f64), .{
                1e-3, 1e-2, 1e-1,  1e0,   1e1,   1e2,   next(f64, next(f64, 0x0.8p8, 0.0), 0.0), next(f64, 0x0.8p8, 0.0),
                -2e1, -2e0, -2e-1, -2e-2, -2e-3, -2e-4, -2e-5,                                   2e-5,
            });
            try testArgs(@Vector(32, i8), @Vector(32, f64), .{
                2e-4,  2e-3,  2e-2,  2e-1,  2e0,   2e1,  -3e1,  -3e0,
                -3e-1, -3e-2, -3e-3, -3e-4, -3e-5, 3e-5, 3e-4,  3e-3,
                3e-2,  3e-1,  3e0,   3e1,   -4e1,  -4e0, -4e-1, -4e-2,
                -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3, 4e-2,  4e-1,
            });
            try testArgs(@Vector(64, i8), @Vector(64, f64), .{
                4e0,   4e1,   -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4,
                -5e-5, 5e-5,  5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,
                -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4, -6e-5, 6e-5,
                6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,   -7e1,  -7e0,
                -7e-1, -7e-2, -7e-3, -7e-4, -7e-5, 7e-5,  7e-4,  7e-3,
                7e-2,  7e-1,  7e0,   7e1,   -8e1,  -8e0,  -8e-1, -8e-2,
                -8e-3, -8e-4, -8e-5, 8e-5,  8e-4,  8e-3,  8e-2,  8e-1,
                8e0,   8e1,   -9e1,  -9e0,  -9e-1, -9e-2, -9e-3, -9e-4,
            });
            try testArgs(@Vector(128, i8), @Vector(128, f64), .{
                -9e-5,  9e-5,   9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,
                -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4, -11e-5, 11e-5,
                11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,   -12e1,  -12e0,
                -12e-1, -12e-2, -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,
                12e-2,  12e-1,  12e0,   12e1,   -13e0,  -13e-1, -13e-2, -13e-3,
                -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,  13e-2,  13e-1,  13e0,
                -14e0,  -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,
                14e-3,  14e-2,  14e-1,  14e0,   -15e0,  -15e-1, -15e-2, -15e-3,
                -15e-4, -15e-5, 15e-5,  15e-4,  15e-3,  15e-2,  15e-1,  15e0,
                -16e0,  -16e-1, -16e-2, -16e-3, -16e-4, -16e-5, 16e-5,  16e-4,
                16e-3,  16e-2,  16e-1,  16e0,   -17e0,  -17e-1, -17e-2, -17e-3,
                -17e-4, -17e-5, 17e-5,  17e-4,  17e-3,  17e-2,  17e-1,  17e0,
                -18e0,  -18e-1, -18e-2, -18e-3, -18e-4, -18e-5, 18e-5,  18e-4,
                18e-3,  18e-2,  18e-1,  18e0,   -19e0,  -19e-1, -19e-2, -19e-3,
                -19e-4, -19e-5, 19e-5,  19e-4,  19e-3,  19e-2,  19e-1,  19e0,
                -21e0,  -21e-1, -21e-2, -21e-3, -21e-4, -21e-5, 21e-5,  21e-4,
            });

            try testArgs(@Vector(1, u8), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u8), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u8), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u8), @Vector(8, f64), .{
                1e0, 1e1, 1e2, next(f64, next(f64, 0x1p8, 0.0), 0.0), next(f64, 0x1p8, 0.0), 2e-5, 2e-4, 2e-3,
            });
            try testArgs(@Vector(16, u8), @Vector(16, f64), .{
                2e-2, 2e-1, 2e0, 2e1, 2e2,  3e-5, 3e-4, 3e-3,
                3e-2, 3e-1, 3e0, 3e1, 4e-5, 4e-4, 4e-3, 4e-2,
            });
            try testArgs(@Vector(32, u8), @Vector(32, f64), .{
                4e-1, 4e0,  4e1,  5e-5, 5e-4, 5e-3, 5e-2, 5e-1,
                5e0,  5e1,  6e-5, 6e-4, 6e-3, 6e-2, 6e-1, 6e0,
                6e1,  7e-5, 7e-4, 7e-3, 7e-2, 7e-1, 7e0,  7e1,
                8e-5, 8e-4, 8e-3, 8e-2, 8e-1, 8e0,  8e1,  9e-5,
            });
            try testArgs(@Vector(64, u8), @Vector(64, f64), .{
                9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   11e-5, 11e-4,
                11e-3, 11e-2, 11e-1, 11e0,  11e1,  13e-5, 13e-4, 13e-3,
                13e-2, 13e-1, 13e0,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  15e-5, 15e-4, 15e-3, 15e-2, 15e-1, 15e0,
                15e1,  16e-5, 16e-4, 16e-3, 16e-2, 16e-1, 16e0,  16e1,
                17e-5, 17e-4, 17e-3, 17e-2, 17e-1, 17e0,  17e1,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  19e-5, 19e-4,
                19e-3, 19e-2, 19e-1, 19e0,  19e1,  21e-5, 21e-4, 21e-3,
            });
            try testArgs(@Vector(128, u8), @Vector(128, f64), .{
                21e-2, 21e-1, 21e0,  21e1,  22e-5, 22e-4, 22e-3, 22e-2,
                22e-1, 22e0,  22e1,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  24e-5, 24e-4, 24e-3, 24e-2, 24e-1, 24e0,
                24e1,  25e-5, 25e-4, 25e-3, 25e-2, 25e-1, 25e0,  25e1,
                26e-5, 26e-4, 26e-3, 26e-2, 26e-1, 26e0,  27e-5, 27e-4,
                27e-3, 27e-2, 27e-1, 27e0,  28e-5, 28e-4, 28e-3, 28e-2,
                28e-1, 28e0,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,
                31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,  32e-5, 32e-4,
                32e-3, 32e-2, 32e-1, 32e0,  33e-5, 33e-4, 33e-3, 33e-2,
                33e-1, 33e0,  34e-5, 34e-4, 34e-3, 34e-2, 34e-1, 34e0,
                35e-5, 35e-4, 35e-3, 35e-2, 35e-1, 35e0,  36e-5, 36e-4,
                36e-3, 36e-2, 36e-1, 36e0,  37e-5, 37e-4, 37e-3, 37e-2,
                37e-1, 37e0,  38e-5, 38e-4, 38e-3, 38e-2, 38e-1, 38e0,
                39e-5, 39e-4, 39e-3, 39e-2, 39e-1, 39e0,  41e-5, 41e-4,
                41e-3, 41e-2, 41e-1, 41e0,  42e-5, 42e-4, 42e-3, 42e-2,
                42e-1, 42e0,  43e-5, 43e-4, 43e-3, 43e-2, 43e-1, 43e0,
            });

            try testArgs(@Vector(1, i16), @Vector(1, f64), .{
                -0x0.8p16,
            });
            try testArgs(@Vector(2, i16), @Vector(2, f64), .{
                next(f64, -0x0.8p16, -0.0), next(f64, next(f64, -0x0.8p16, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i16), @Vector(4, f64), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i16), @Vector(8, f64), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i16), @Vector(16, f64), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f64, next(f64, 0x0.8p16, 0.0), 0.0), next(f64, 0x0.8p16, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i16), @Vector(32, f64), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i16), @Vector(64, f64), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i16), @Vector(128, f64), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u16), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u16), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u16), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u16), @Vector(8, f64), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f64, next(f64, 0x1p16, 0.0), 0.0), next(f64, 0x1p16, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u16), @Vector(16, f64), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u16), @Vector(32, f64), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u16), @Vector(64, f64), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u16), @Vector(128, f64), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i32), @Vector(1, f64), .{
                -0x0.8p32,
            });
            try testArgs(@Vector(2, i32), @Vector(2, f64), .{
                next(f64, -0x0.8p32, -0.0), next(f64, next(f64, -0x0.8p32, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i32), @Vector(4, f64), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i32), @Vector(8, f64), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i32), @Vector(16, f64), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f64, next(f64, 0x0.8p32, 0.0), 0.0), next(f64, 0x0.8p32, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i32), @Vector(32, f64), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i32), @Vector(64, f64), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i32), @Vector(128, f64), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u32), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u32), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u32), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u32), @Vector(8, f64), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f64, next(f64, 0x1p32, 0.0), 0.0), next(f64, 0x1p32, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u32), @Vector(16, f64), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u32), @Vector(32, f64), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u32), @Vector(64, f64), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u32), @Vector(128, f64), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i64), @Vector(1, f64), .{
                -0x0.8p64,
            });
            try testArgs(@Vector(2, i64), @Vector(2, f64), .{
                next(f64, -0x0.8p64, -0.0), next(f64, next(f64, -0x0.8p64, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i64), @Vector(4, f64), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i64), @Vector(8, f64), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i64), @Vector(16, f64), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f64, next(f64, 0x0.8p64, 0.0), 0.0), next(f64, 0x0.8p64, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i64), @Vector(32, f64), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i64), @Vector(64, f64), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i64), @Vector(128, f64), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u64), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u64), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u64), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u64), @Vector(8, f64), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f64, next(f64, 0x1p64, 0.0), 0.0), next(f64, 0x1p64, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u64), @Vector(16, f64), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u64), @Vector(32, f64), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u64), @Vector(64, f64), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u64), @Vector(128, f64), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i128), @Vector(1, f64), .{
                -0x0.8p128,
            });
            try testArgs(@Vector(2, i128), @Vector(2, f64), .{
                next(f64, -0x0.8p128, -0.0), next(f64, next(f64, -0x0.8p128, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i128), @Vector(4, f64), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i128), @Vector(8, f64), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i128), @Vector(16, f64), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f64, next(f64, 0x0.8p128, 0.0), 0.0), next(f64, 0x0.8p128, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i128), @Vector(32, f64), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i128), @Vector(64, f64), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u128), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u128), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u128), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u128), @Vector(8, f64), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f64, next(f64, 0x1p128, 0.0), 0.0), next(f64, 0x1p128, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u128), @Vector(16, f64), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u128), @Vector(32, f64), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u128), @Vector(64, f64), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i256), @Vector(1, f64), .{
                -0x0.8p256,
            });
            try testArgs(@Vector(2, i256), @Vector(2, f64), .{
                next(f64, -0x0.8p256, -0.0), next(f64, next(f64, -0x0.8p256, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i256), @Vector(4, f64), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i256), @Vector(8, f64), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i256), @Vector(16, f64), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f64, next(f64, 0x0.8p256, 0.0), 0.0), next(f64, 0x0.8p256, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i256), @Vector(32, f64), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i256), @Vector(64, f64), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u256), @Vector(1, f64), .{
                -0.0,
            });
            try testArgs(@Vector(2, u256), @Vector(2, f64), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u256), @Vector(4, f64), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u256), @Vector(8, f64), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f64, next(f64, 0x1p256, 0.0), 0.0), next(f64, 0x1p256, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u256), @Vector(16, f64), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u256), @Vector(32, f64), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u256), @Vector(64, f64), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i8), @Vector(1, f80), .{
                -0x0.8p8,
            });
            try testArgs(@Vector(2, i8), @Vector(2, f80), .{
                next(f80, -0x0.8p8, -0.0), next(f80, next(f80, -0x0.8p8, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i8), @Vector(4, f80), .{
                -1e2, -1e1, -1e0, -1e-1,
            });
            try testArgs(@Vector(8, i8), @Vector(8, f80), .{
                -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0, 1e-5, 1e-4,
            });
            try testArgs(@Vector(16, i8), @Vector(16, f80), .{
                1e-3, 1e-2, 1e-1,  1e0,   1e1,   1e2,   next(f80, next(f80, 0x0.8p8, 0.0), 0.0), next(f80, 0x0.8p8, 0.0),
                -2e1, -2e0, -2e-1, -2e-2, -2e-3, -2e-4, -2e-5,                                   2e-5,
            });
            try testArgs(@Vector(32, i8), @Vector(32, f80), .{
                2e-4,  2e-3,  2e-2,  2e-1,  2e0,   2e1,  -3e1,  -3e0,
                -3e-1, -3e-2, -3e-3, -3e-4, -3e-5, 3e-5, 3e-4,  3e-3,
                3e-2,  3e-1,  3e0,   3e1,   -4e1,  -4e0, -4e-1, -4e-2,
                -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3, 4e-2,  4e-1,
            });
            try testArgs(@Vector(64, i8), @Vector(64, f80), .{
                4e0,   4e1,   -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4,
                -5e-5, 5e-5,  5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,
                -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4, -6e-5, 6e-5,
                6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,   -7e1,  -7e0,
                -7e-1, -7e-2, -7e-3, -7e-4, -7e-5, 7e-5,  7e-4,  7e-3,
                7e-2,  7e-1,  7e0,   7e1,   -8e1,  -8e0,  -8e-1, -8e-2,
                -8e-3, -8e-4, -8e-5, 8e-5,  8e-4,  8e-3,  8e-2,  8e-1,
                8e0,   8e1,   -9e1,  -9e0,  -9e-1, -9e-2, -9e-3, -9e-4,
            });
            try testArgs(@Vector(128, i8), @Vector(128, f80), .{
                -9e-5,  9e-5,   9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,
                -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4, -11e-5, 11e-5,
                11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,   -12e1,  -12e0,
                -12e-1, -12e-2, -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,
                12e-2,  12e-1,  12e0,   12e1,   -13e0,  -13e-1, -13e-2, -13e-3,
                -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,  13e-2,  13e-1,  13e0,
                -14e0,  -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,
                14e-3,  14e-2,  14e-1,  14e0,   -15e0,  -15e-1, -15e-2, -15e-3,
                -15e-4, -15e-5, 15e-5,  15e-4,  15e-3,  15e-2,  15e-1,  15e0,
                -16e0,  -16e-1, -16e-2, -16e-3, -16e-4, -16e-5, 16e-5,  16e-4,
                16e-3,  16e-2,  16e-1,  16e0,   -17e0,  -17e-1, -17e-2, -17e-3,
                -17e-4, -17e-5, 17e-5,  17e-4,  17e-3,  17e-2,  17e-1,  17e0,
                -18e0,  -18e-1, -18e-2, -18e-3, -18e-4, -18e-5, 18e-5,  18e-4,
                18e-3,  18e-2,  18e-1,  18e0,   -19e0,  -19e-1, -19e-2, -19e-3,
                -19e-4, -19e-5, 19e-5,  19e-4,  19e-3,  19e-2,  19e-1,  19e0,
                -21e0,  -21e-1, -21e-2, -21e-3, -21e-4, -21e-5, 21e-5,  21e-4,
            });

            try testArgs(@Vector(1, u8), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u8), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u8), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u8), @Vector(8, f80), .{
                1e0, 1e1, 1e2, next(f80, next(f80, 0x1p8, 0.0), 0.0), next(f80, 0x1p8, 0.0), 2e-5, 2e-4, 2e-3,
            });
            try testArgs(@Vector(16, u8), @Vector(16, f80), .{
                2e-2, 2e-1, 2e0, 2e1, 2e2,  3e-5, 3e-4, 3e-3,
                3e-2, 3e-1, 3e0, 3e1, 4e-5, 4e-4, 4e-3, 4e-2,
            });
            try testArgs(@Vector(32, u8), @Vector(32, f80), .{
                4e-1, 4e0,  4e1,  5e-5, 5e-4, 5e-3, 5e-2, 5e-1,
                5e0,  5e1,  6e-5, 6e-4, 6e-3, 6e-2, 6e-1, 6e0,
                6e1,  7e-5, 7e-4, 7e-3, 7e-2, 7e-1, 7e0,  7e1,
                8e-5, 8e-4, 8e-3, 8e-2, 8e-1, 8e0,  8e1,  9e-5,
            });
            try testArgs(@Vector(64, u8), @Vector(64, f80), .{
                9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   11e-5, 11e-4,
                11e-3, 11e-2, 11e-1, 11e0,  11e1,  13e-5, 13e-4, 13e-3,
                13e-2, 13e-1, 13e0,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  15e-5, 15e-4, 15e-3, 15e-2, 15e-1, 15e0,
                15e1,  16e-5, 16e-4, 16e-3, 16e-2, 16e-1, 16e0,  16e1,
                17e-5, 17e-4, 17e-3, 17e-2, 17e-1, 17e0,  17e1,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  19e-5, 19e-4,
                19e-3, 19e-2, 19e-1, 19e0,  19e1,  21e-5, 21e-4, 21e-3,
            });
            try testArgs(@Vector(128, u8), @Vector(128, f80), .{
                21e-2, 21e-1, 21e0,  21e1,  22e-5, 22e-4, 22e-3, 22e-2,
                22e-1, 22e0,  22e1,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  24e-5, 24e-4, 24e-3, 24e-2, 24e-1, 24e0,
                24e1,  25e-5, 25e-4, 25e-3, 25e-2, 25e-1, 25e0,  25e1,
                26e-5, 26e-4, 26e-3, 26e-2, 26e-1, 26e0,  27e-5, 27e-4,
                27e-3, 27e-2, 27e-1, 27e0,  28e-5, 28e-4, 28e-3, 28e-2,
                28e-1, 28e0,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,
                31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,  32e-5, 32e-4,
                32e-3, 32e-2, 32e-1, 32e0,  33e-5, 33e-4, 33e-3, 33e-2,
                33e-1, 33e0,  34e-5, 34e-4, 34e-3, 34e-2, 34e-1, 34e0,
                35e-5, 35e-4, 35e-3, 35e-2, 35e-1, 35e0,  36e-5, 36e-4,
                36e-3, 36e-2, 36e-1, 36e0,  37e-5, 37e-4, 37e-3, 37e-2,
                37e-1, 37e0,  38e-5, 38e-4, 38e-3, 38e-2, 38e-1, 38e0,
                39e-5, 39e-4, 39e-3, 39e-2, 39e-1, 39e0,  41e-5, 41e-4,
                41e-3, 41e-2, 41e-1, 41e0,  42e-5, 42e-4, 42e-3, 42e-2,
                42e-1, 42e0,  43e-5, 43e-4, 43e-3, 43e-2, 43e-1, 43e0,
            });

            try testArgs(@Vector(1, i16), @Vector(1, f80), .{
                -0x0.8p16,
            });
            try testArgs(@Vector(2, i16), @Vector(2, f80), .{
                next(f80, -0x0.8p16, -0.0), next(f80, next(f80, -0x0.8p16, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i16), @Vector(4, f80), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i16), @Vector(8, f80), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i16), @Vector(16, f80), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f80, next(f80, 0x0.8p16, 0.0), 0.0), next(f80, 0x0.8p16, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i16), @Vector(32, f80), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i16), @Vector(64, f80), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i16), @Vector(128, f80), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u16), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u16), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u16), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u16), @Vector(8, f80), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f80, next(f80, 0x1p16, 0.0), 0.0), next(f80, 0x1p16, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u16), @Vector(16, f80), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u16), @Vector(32, f80), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u16), @Vector(64, f80), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u16), @Vector(128, f80), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i32), @Vector(1, f80), .{
                -0x0.8p32,
            });
            try testArgs(@Vector(2, i32), @Vector(2, f80), .{
                next(f80, -0x0.8p32, -0.0), next(f80, next(f80, -0x0.8p32, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i32), @Vector(4, f80), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i32), @Vector(8, f80), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i32), @Vector(16, f80), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f80, next(f80, 0x0.8p32, 0.0), 0.0), next(f80, 0x0.8p32, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i32), @Vector(32, f80), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i32), @Vector(64, f80), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i32), @Vector(128, f80), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u32), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u32), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u32), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u32), @Vector(8, f80), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f80, next(f80, 0x1p32, 0.0), 0.0), next(f80, 0x1p32, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u32), @Vector(16, f80), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u32), @Vector(32, f80), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u32), @Vector(64, f80), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u32), @Vector(128, f80), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i64), @Vector(1, f80), .{
                -0x0.8p64,
            });
            try testArgs(@Vector(2, i64), @Vector(2, f80), .{
                next(f80, -0x0.8p64, -0.0), next(f80, next(f80, -0x0.8p64, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i64), @Vector(4, f80), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i64), @Vector(8, f80), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i64), @Vector(16, f80), .{
                1e-5, 1e-4, 1e-3,                                     1e-2,                     1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f80, next(f80, 0x0.8p64, 0.0), 0.0), next(f80, 0x0.8p64, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i64), @Vector(32, f80), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i64), @Vector(64, f80), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i64), @Vector(128, f80), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u64), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u64), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u64), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u64), @Vector(8, f80), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f80, next(f80, 0x1p64, 0.0), 0.0), next(f80, 0x1p64, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u64), @Vector(16, f80), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u64), @Vector(32, f80), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u64), @Vector(64, f80), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u64), @Vector(128, f80), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i128), @Vector(1, f80), .{
                -0x0.8p128,
            });
            try testArgs(@Vector(2, i128), @Vector(2, f80), .{
                next(f80, -0x0.8p128, -0.0), next(f80, next(f80, -0x0.8p128, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i128), @Vector(4, f80), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i128), @Vector(8, f80), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i128), @Vector(16, f80), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f80, next(f80, 0x0.8p128, 0.0), 0.0), next(f80, 0x0.8p128, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i128), @Vector(32, f80), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i128), @Vector(64, f80), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u128), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u128), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u128), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u128), @Vector(8, f80), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f80, next(f80, 0x1p128, 0.0), 0.0), next(f80, 0x1p128, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u128), @Vector(16, f80), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u128), @Vector(32, f80), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u128), @Vector(64, f80), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i256), @Vector(1, f80), .{
                -0x0.8p256,
            });
            try testArgs(@Vector(2, i256), @Vector(2, f80), .{
                next(f80, -0x0.8p256, -0.0), next(f80, next(f80, -0x0.8p256, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i256), @Vector(4, f80), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i256), @Vector(8, f80), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i256), @Vector(16, f80), .{
                1e-5, 1e-4, 1e-3,                                      1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f80, next(f80, 0x0.8p256, 0.0), 0.0), next(f80, 0x0.8p256, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i256), @Vector(32, f80), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i256), @Vector(64, f80), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u256), @Vector(1, f80), .{
                -0.0,
            });
            try testArgs(@Vector(2, u256), @Vector(2, f80), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u256), @Vector(4, f80), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u256), @Vector(8, f80), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f80, next(f80, 0x1p256, 0.0), 0.0), next(f80, 0x1p256, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u256), @Vector(16, f80), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u256), @Vector(32, f80), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u256), @Vector(64, f80), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i8), @Vector(1, f128), .{
                -0x0.8p8,
            });
            try testArgs(@Vector(2, i8), @Vector(2, f128), .{
                next(f128, -0x0.8p8, -0.0), next(f128, next(f128, -0x0.8p8, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i8), @Vector(4, f128), .{
                -1e2, -1e1, -1e0, -1e-1,
            });
            try testArgs(@Vector(8, i8), @Vector(8, f128), .{
                -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0, 1e-5, 1e-4,
            });
            try testArgs(@Vector(16, i8), @Vector(16, f128), .{
                1e-3, 1e-2, 1e-1,  1e0,   1e1,   1e2,   next(f128, next(f128, 0x0.8p8, 0.0), 0.0), next(f128, 0x0.8p8, 0.0),
                -2e1, -2e0, -2e-1, -2e-2, -2e-3, -2e-4, -2e-5,                                     2e-5,
            });
            try testArgs(@Vector(32, i8), @Vector(32, f128), .{
                2e-4,  2e-3,  2e-2,  2e-1,  2e0,   2e1,  -3e1,  -3e0,
                -3e-1, -3e-2, -3e-3, -3e-4, -3e-5, 3e-5, 3e-4,  3e-3,
                3e-2,  3e-1,  3e0,   3e1,   -4e1,  -4e0, -4e-1, -4e-2,
                -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3, 4e-2,  4e-1,
            });
            try testArgs(@Vector(64, i8), @Vector(64, f128), .{
                4e0,   4e1,   -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4,
                -5e-5, 5e-5,  5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,
                -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4, -6e-5, 6e-5,
                6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,   -7e1,  -7e0,
                -7e-1, -7e-2, -7e-3, -7e-4, -7e-5, 7e-5,  7e-4,  7e-3,
                7e-2,  7e-1,  7e0,   7e1,   -8e1,  -8e0,  -8e-1, -8e-2,
                -8e-3, -8e-4, -8e-5, 8e-5,  8e-4,  8e-3,  8e-2,  8e-1,
                8e0,   8e1,   -9e1,  -9e0,  -9e-1, -9e-2, -9e-3, -9e-4,
            });
            try testArgs(@Vector(128, i8), @Vector(128, f128), .{
                -9e-5,  9e-5,   9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,
                -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4, -11e-5, 11e-5,
                11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,   -12e1,  -12e0,
                -12e-1, -12e-2, -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,
                12e-2,  12e-1,  12e0,   12e1,   -13e0,  -13e-1, -13e-2, -13e-3,
                -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,  13e-2,  13e-1,  13e0,
                -14e0,  -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,
                14e-3,  14e-2,  14e-1,  14e0,   -15e0,  -15e-1, -15e-2, -15e-3,
                -15e-4, -15e-5, 15e-5,  15e-4,  15e-3,  15e-2,  15e-1,  15e0,
                -16e0,  -16e-1, -16e-2, -16e-3, -16e-4, -16e-5, 16e-5,  16e-4,
                16e-3,  16e-2,  16e-1,  16e0,   -17e0,  -17e-1, -17e-2, -17e-3,
                -17e-4, -17e-5, 17e-5,  17e-4,  17e-3,  17e-2,  17e-1,  17e0,
                -18e0,  -18e-1, -18e-2, -18e-3, -18e-4, -18e-5, 18e-5,  18e-4,
                18e-3,  18e-2,  18e-1,  18e0,   -19e0,  -19e-1, -19e-2, -19e-3,
                -19e-4, -19e-5, 19e-5,  19e-4,  19e-3,  19e-2,  19e-1,  19e0,
                -21e0,  -21e-1, -21e-2, -21e-3, -21e-4, -21e-5, 21e-5,  21e-4,
            });

            try testArgs(@Vector(1, u8), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u8), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u8), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u8), @Vector(8, f128), .{
                1e0, 1e1, 1e2, next(f128, next(f128, 0x1p8, 0.0), 0.0), next(f128, 0x1p8, 0.0), 2e-5, 2e-4, 2e-3,
            });
            try testArgs(@Vector(16, u8), @Vector(16, f128), .{
                2e-2, 2e-1, 2e0, 2e1, 2e2,  3e-5, 3e-4, 3e-3,
                3e-2, 3e-1, 3e0, 3e1, 4e-5, 4e-4, 4e-3, 4e-2,
            });
            try testArgs(@Vector(32, u8), @Vector(32, f128), .{
                4e-1, 4e0,  4e1,  5e-5, 5e-4, 5e-3, 5e-2, 5e-1,
                5e0,  5e1,  6e-5, 6e-4, 6e-3, 6e-2, 6e-1, 6e0,
                6e1,  7e-5, 7e-4, 7e-3, 7e-2, 7e-1, 7e0,  7e1,
                8e-5, 8e-4, 8e-3, 8e-2, 8e-1, 8e0,  8e1,  9e-5,
            });
            try testArgs(@Vector(64, u8), @Vector(64, f128), .{
                9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   11e-5, 11e-4,
                11e-3, 11e-2, 11e-1, 11e0,  11e1,  13e-5, 13e-4, 13e-3,
                13e-2, 13e-1, 13e0,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  15e-5, 15e-4, 15e-3, 15e-2, 15e-1, 15e0,
                15e1,  16e-5, 16e-4, 16e-3, 16e-2, 16e-1, 16e0,  16e1,
                17e-5, 17e-4, 17e-3, 17e-2, 17e-1, 17e0,  17e1,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  19e-5, 19e-4,
                19e-3, 19e-2, 19e-1, 19e0,  19e1,  21e-5, 21e-4, 21e-3,
            });
            try testArgs(@Vector(128, u8), @Vector(128, f128), .{
                21e-2, 21e-1, 21e0,  21e1,  22e-5, 22e-4, 22e-3, 22e-2,
                22e-1, 22e0,  22e1,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  24e-5, 24e-4, 24e-3, 24e-2, 24e-1, 24e0,
                24e1,  25e-5, 25e-4, 25e-3, 25e-2, 25e-1, 25e0,  25e1,
                26e-5, 26e-4, 26e-3, 26e-2, 26e-1, 26e0,  27e-5, 27e-4,
                27e-3, 27e-2, 27e-1, 27e0,  28e-5, 28e-4, 28e-3, 28e-2,
                28e-1, 28e0,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,
                31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,  32e-5, 32e-4,
                32e-3, 32e-2, 32e-1, 32e0,  33e-5, 33e-4, 33e-3, 33e-2,
                33e-1, 33e0,  34e-5, 34e-4, 34e-3, 34e-2, 34e-1, 34e0,
                35e-5, 35e-4, 35e-3, 35e-2, 35e-1, 35e0,  36e-5, 36e-4,
                36e-3, 36e-2, 36e-1, 36e0,  37e-5, 37e-4, 37e-3, 37e-2,
                37e-1, 37e0,  38e-5, 38e-4, 38e-3, 38e-2, 38e-1, 38e0,
                39e-5, 39e-4, 39e-3, 39e-2, 39e-1, 39e0,  41e-5, 41e-4,
                41e-3, 41e-2, 41e-1, 41e0,  42e-5, 42e-4, 42e-3, 42e-2,
                42e-1, 42e0,  43e-5, 43e-4, 43e-3, 43e-2, 43e-1, 43e0,
            });

            try testArgs(@Vector(1, i16), @Vector(1, f128), .{
                -0x0.8p16,
            });
            try testArgs(@Vector(2, i16), @Vector(2, f128), .{
                next(f128, -0x0.8p16, -0.0), next(f128, next(f128, -0x0.8p16, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i16), @Vector(4, f128), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i16), @Vector(8, f128), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i16), @Vector(16, f128), .{
                1e-5, 1e-4, 1e-3,                                       1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f128, next(f128, 0x0.8p16, 0.0), 0.0), next(f128, 0x0.8p16, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i16), @Vector(32, f128), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i16), @Vector(64, f128), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i16), @Vector(128, f128), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u16), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u16), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u16), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u16), @Vector(8, f128), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f128, next(f128, 0x1p16, 0.0), 0.0), next(f128, 0x1p16, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u16), @Vector(16, f128), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u16), @Vector(32, f128), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u16), @Vector(64, f128), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u16), @Vector(128, f128), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i32), @Vector(1, f128), .{
                -0x0.8p32,
            });
            try testArgs(@Vector(2, i32), @Vector(2, f128), .{
                next(f128, -0x0.8p32, -0.0), next(f128, next(f128, -0x0.8p32, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i32), @Vector(4, f128), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i32), @Vector(8, f128), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i32), @Vector(16, f128), .{
                1e-5, 1e-4, 1e-3,                                       1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f128, next(f128, 0x0.8p32, 0.0), 0.0), next(f128, 0x0.8p32, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i32), @Vector(32, f128), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i32), @Vector(64, f128), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i32), @Vector(128, f128), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u32), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u32), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u32), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u32), @Vector(8, f128), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f128, next(f128, 0x1p32, 0.0), 0.0), next(f128, 0x1p32, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u32), @Vector(16, f128), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u32), @Vector(32, f128), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u32), @Vector(64, f128), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u32), @Vector(128, f128), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i64), @Vector(1, f128), .{
                -0x0.8p64,
            });
            try testArgs(@Vector(2, i64), @Vector(2, f128), .{
                next(f128, -0x0.8p64, -0.0), next(f128, next(f128, -0x0.8p64, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i64), @Vector(4, f128), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i64), @Vector(8, f128), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i64), @Vector(16, f128), .{
                1e-5, 1e-4, 1e-3,                                       1e-2,                      1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f128, next(f128, 0x0.8p64, 0.0), 0.0), next(f128, 0x0.8p64, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i64), @Vector(32, f128), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i64), @Vector(64, f128), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });
            try testArgs(@Vector(128, i64), @Vector(128, f128), .{
                -7e-3,  -7e-4,  -7e-5,  7e-5,   7e-4,   7e-3,   7e-2,   7e-1,
                7e0,    7e1,    7e2,    7e3,    -8e3,   -8e2,   -8e1,   -8e0,
                -8e-1,  -8e-2,  -8e-3,  -8e-4,  -8e-5,  8e-5,   8e-4,   8e-3,
                8e-2,   8e-1,   8e0,    8e1,    8e2,    8e3,    -9e3,   -9e2,
                -9e1,   -9e0,   -9e-1,  -9e-2,  -9e-3,  -9e-4,  -9e-5,  9e-5,
                9e-4,   9e-3,   9e-2,   9e-1,   9e0,    9e1,    9e2,    9e3,
                -11e3,  -11e2,  -11e1,  -11e0,  -11e-1, -11e-2, -11e-3, -11e-4,
                -11e-5, 11e-5,  11e-4,  11e-3,  11e-2,  11e-1,  11e0,   11e1,
                11e2,   11e3,   -12e3,  -12e2,  -12e1,  -12e0,  -12e-1, -12e-2,
                -12e-3, -12e-4, -12e-5, 12e-5,  12e-4,  12e-3,  12e-2,  12e-1,
                12e0,   12e1,   12e2,   12e3,   -13e3,  -13e2,  -13e1,  -13e0,
                -13e-1, -13e-2, -13e-3, -13e-4, -13e-5, 13e-5,  13e-4,  13e-3,
                13e-2,  13e-1,  13e0,   13e1,   13e2,   -14e2,  -14e1,  -14e0,
                -14e-1, -14e-2, -14e-3, -14e-4, -14e-5, 14e-5,  14e-4,  14e-3,
                14e-2,  14e-1,  14e0,   14e1,   14e2,   14e3,   -15e3,  -15e2,
                -15e1,  -15e0,  -15e-1, -15e-2, -15e-3, -15e-4, -15e-5, 15e-5,
            });

            try testArgs(@Vector(1, u64), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u64), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u64), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u64), @Vector(8, f128), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f128, next(f128, 0x1p64, 0.0), 0.0), next(f128, 0x1p64, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u64), @Vector(16, f128), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u64), @Vector(32, f128), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u64), @Vector(64, f128), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
            try testArgs(@Vector(128, u64), @Vector(128, f128), .{
                16e-2, 16e-1, 16e0,  16e1,  16e2,  16e3,  17e-5, 17e-4,
                17e-3, 17e-2, 17e-1, 17e0,  17e1,  17e2,  17e3,  18e-5,
                18e-4, 18e-3, 18e-2, 18e-1, 18e0,  18e1,  18e2,  18e3,
                19e-5, 19e-4, 19e-3, 19e-2, 19e-1, 19e0,  19e1,  19e2,
                19e3,  21e-5, 21e-4, 21e-3, 21e-2, 21e-1, 21e0,  21e1,
                21e2,  21e3,  22e-5, 22e-4, 22e-3, 22e-2, 22e-1, 22e0,
                22e1,  22e2,  22e3,  23e-5, 23e-4, 23e-3, 23e-2, 23e-1,
                23e0,  23e1,  23e2,  23e3,  24e-5, 24e-4, 24e-3, 24e-2,
                24e-1, 24e0,  24e1,  24e2,  24e3,  25e-5, 25e-4, 25e-3,
                25e-2, 25e-1, 25e0,  25e1,  25e2,  25e3,  26e-5, 26e-4,
                26e-3, 26e-2, 26e-1, 26e0,  26e1,  26e2,  26e3,  27e-5,
                27e-4, 27e-3, 27e-2, 27e-1, 27e0,  27e1,  27e2,  27e3,
                28e-5, 28e-4, 28e-3, 28e-2, 28e-1, 28e0,  28e1,  28e2,
                28e3,  29e-5, 29e-4, 29e-3, 29e-2, 29e-1, 29e0,  29e1,
                29e2,  29e3,  31e-5, 31e-4, 31e-3, 31e-2, 31e-1, 31e0,
                31e1,  31e2,  31e3,  32e-5, 32e-4, 32e-3, 32e-2, 32e-1,
            });

            try testArgs(@Vector(1, i128), @Vector(1, f128), .{
                -0x0.8p128,
            });
            try testArgs(@Vector(2, i128), @Vector(2, f128), .{
                next(f128, -0x0.8p128, -0.0), next(f128, next(f128, -0x0.8p128, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i128), @Vector(4, f128), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i128), @Vector(8, f128), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i128), @Vector(16, f128), .{
                1e-5, 1e-4, 1e-3,                                        1e-2,                       1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f128, next(f128, 0x0.8p128, 0.0), 0.0), next(f128, 0x0.8p128, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i128), @Vector(32, f128), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i128), @Vector(64, f128), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u128), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u128), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u128), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u128), @Vector(8, f128), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f128, next(f128, 0x1p128, 0.0), 0.0), next(f128, 0x1p128, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u128), @Vector(16, f128), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u128), @Vector(32, f128), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u128), @Vector(64, f128), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });

            try testArgs(@Vector(1, i256), @Vector(1, f128), .{
                -0x0.8p256,
            });
            try testArgs(@Vector(2, i256), @Vector(2, f128), .{
                next(f128, -0x0.8p256, -0.0), next(f128, next(f128, -0x0.8p256, -0.0), -0.0),
            });
            try testArgs(@Vector(4, i256), @Vector(4, f128), .{
                -1e4, -1e3, -1e2, -1e1,
            });
            try testArgs(@Vector(8, i256), @Vector(8, f128), .{
                -1e0, -1e-1, -1e-2, -1e-3, -1e-4, -1e-5, -0.0, 0.0,
            });
            try testArgs(@Vector(16, i256), @Vector(16, f128), .{
                1e-5, 1e-4, 1e-3,                                        1e-2,                       1e-1, 1e0,  1e1,  1e2,
                1e3,  1e4,  next(f128, next(f128, 0x0.8p256, 0.0), 0.0), next(f128, 0x0.8p256, 0.0), -2e4, -2e3, -2e2, -2e1,
            });
            try testArgs(@Vector(32, i256), @Vector(32, f128), .{
                -2e0,  -2e-1, -2e-2, -2e-3, -2e-4, -2e-5, 2e-5,  2e-4,
                2e-3,  2e-2,  2e-1,  2e0,   2e1,   2e2,   2e3,   2e4,
                -3e4,  -3e3,  -3e2,  -3e1,  -3e0,  -3e-1, -3e-2, -3e-3,
                -3e-4, -3e-5, 3e-5,  3e-4,  3e-3,  3e-2,  3e-1,  3e0,
            });
            try testArgs(@Vector(64, i256), @Vector(64, f128), .{
                3e1,   3e2,   3e3,   3e4,   -4e3,  -4e2,  -4e1,  -4e0,
                -4e-1, -4e-2, -4e-3, -4e-4, -4e-5, 4e-5,  4e-4,  4e-3,
                4e-2,  4e-1,  4e0,   4e1,   4e2,   4e3,   -5e3,  -5e2,
                -5e1,  -5e0,  -5e-1, -5e-2, -5e-3, -5e-4, -5e-5, 5e-5,
                5e-4,  5e-3,  5e-2,  5e-1,  5e0,   5e1,   5e2,   5e3,
                -6e3,  -6e2,  -6e1,  -6e0,  -6e-1, -6e-2, -6e-3, -6e-4,
                -6e-5, 6e-5,  6e-4,  6e-3,  6e-2,  6e-1,  6e0,   6e1,
                6e2,   6e3,   -7e3,  -7e2,  -7e1,  -7e0,  -7e-1, -7e-2,
            });

            try testArgs(@Vector(1, u256), @Vector(1, f128), .{
                -0.0,
            });
            try testArgs(@Vector(2, u256), @Vector(2, f128), .{
                0.0, 1e-5,
            });
            try testArgs(@Vector(4, u256), @Vector(4, f128), .{
                1e-4, 1e-3, 1e-2, 1e-1,
            });
            try testArgs(@Vector(8, u256), @Vector(8, f128), .{
                1e0, 1e1, 1e2, 1e3, 1e4, next(f128, next(f128, 0x1p256, 0.0), 0.0), next(f128, 0x1p256, 0.0), 2e-5,
            });
            try testArgs(@Vector(16, u256), @Vector(16, f128), .{
                2e-4, 2e-3, 2e-2, 2e-1, 2e0,  2e1,  2e2, 2e3,
                2e4,  3e-5, 3e-4, 3e-3, 3e-2, 3e-1, 3e0, 3e1,
            });
            try testArgs(@Vector(32, u256), @Vector(32, f128), .{
                3e2,  3e3,  3e4, 4e-5, 4e-4, 4e-3, 4e-2, 4e-1,
                4e0,  4e1,  4e2, 4e3,  5e-5, 5e-4, 5e-3, 5e-2,
                5e-1, 5e0,  5e1, 5e2,  5e3,  6e-5, 6e-4, 6e-3,
                6e-2, 6e-1, 6e0, 6e1,  6e2,  6e3,  7e-5, 7e-4,
            });
            try testArgs(@Vector(64, u256), @Vector(64, f128), .{
                7e-3,  7e-2,  7e-1,  7e0,   7e1,   7e2,   7e3,   8e-5,
                8e-4,  8e-3,  8e-2,  8e-1,  8e0,   8e1,   8e2,   8e3,
                9e-5,  9e-4,  9e-3,  9e-2,  9e-1,  9e0,   9e1,   9e2,
                9e3,   11e-5, 11e-4, 11e-3, 11e-2, 11e-1, 11e0,  11e1,
                11e2,  11e3,  13e-5, 13e-4, 13e-3, 13e-2, 13e-1, 13e0,
                13e1,  13e2,  13e3,  14e-5, 14e-4, 14e-3, 14e-2, 14e-1,
                14e0,  14e1,  14e2,  14e3,  15e-5, 15e-4, 15e-3, 15e-2,
                15e-1, 15e0,  15e1,  15e2,  15e3,  16e-5, 16e-4, 16e-3,
            });
        }
        fn testFloatVectorsFromIntVectors() !void {
            @setEvalBranchQuota(2_700);

            try testArgs(@Vector(1, f16), @Vector(1, i8), .{
                imin(i8),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i8), .{
                imin(i8) + 1, -1e2,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i8), .{
                -1e1, -1e0, 0, 1e0,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i8), .{
                1e1, 1e2, imax(i8) - 1, imax(i8), imin(i8) + 2, imin(i8) + 3, -2e1, -2e0,
            });
            try testArgs(@Vector(16, f16), @Vector(16, i8), .{
                2e0, 2e1, imax(i8) - 3, imax(i8) - 2, imin(i8) + 4, imin(i8) + 5, -3e1, -3e0,
                3e0, 3e1, imax(i8) - 5, imax(i8) - 4, imin(i8) + 6, imin(i8) + 7, -4e1, -4e0,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i8), .{
                4e0, 4e1, imax(i8) - 7,  imax(i8) - 6,  imin(i8) + 8,  imin(i8) + 9,  -5e1, -5e0,
                5e0, 5e1, imax(i8) - 9,  imax(i8) - 8,  imin(i8) + 10, imin(i8) + 11, -6e1, -6e0,
                6e0, 6e1, imax(i8) - 11, imax(i8) - 10, imin(i8) + 12, imin(i8) + 13, -7e1, -7e0,
                7e0, 7e1, imax(i8) - 13, imax(i8) - 12, imin(i8) + 14, imin(i8) + 15, -8e1, -8e0,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i8), .{
                8e0,           8e1,           imax(i8) - 15, imax(i8) - 14, imin(i8) + 16, imin(i8) + 17, -9e1,          -9e0,
                9e0,           9e1,           imax(i8) - 17, imax(i8) - 16, imin(i8) + 18, imin(i8) + 19, -11e1,         -11e0,
                11e0,          11e1,          imax(i8) - 19, imax(i8) - 18, imin(i8) + 20, imin(i8) + 21, -12e1,         -12e0,
                12e0,          12e1,          imax(i8) - 21, imax(i8) - 20, imin(i8) + 22, imin(i8) + 23, -13e0,         13e0,
                imax(i8) - 23, imax(i8) - 22, imin(i8) + 24, imin(i8) + 25, -14e0,         14e0,          imax(i8) - 25, imax(i8) - 24,
                imin(i8) + 26, imin(i8) + 27, -15e0,         15e0,          imax(i8) - 27, imax(i8) - 26, imin(i8) + 28, imin(i8) + 29,
                -16e0,         16e0,          imax(i8) - 29, imax(i8) - 28, imin(i8) + 30, imin(i8) + 31, -17e0,         17e0,
                imax(i8) - 31, imax(i8) - 30, imin(i8) + 32, imin(i8) + 33, -18e0,         18e0,          imax(i8) - 33, imax(i8) - 32,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i8), .{
                imin(i8) + 34, imin(i8) + 35, -19e0,         19e0,          imax(i8) - 35, imax(i8) - 34, imin(i8) + 36, imin(i8) + 37,
                -21e0,         21e0,          imax(i8) - 37, imax(i8) - 36, imin(i8) + 38, imin(i8) + 39, -22e0,         22e0,
                imax(i8) - 39, imax(i8) - 38, imin(i8) + 40, imin(i8) + 41, -23e0,         23e0,          imax(i8) - 41, imax(i8) - 40,
                imin(i8) + 42, imin(i8) + 43, -24e0,         24e0,          imax(i8) - 43, imax(i8) - 42, imin(i8) + 44, imin(i8) + 45,
                -25e0,         25e0,          imax(i8) - 45, imax(i8) - 44, imin(i8) + 46, imin(i8) + 47, -26e0,         26e0,
                imax(i8) - 47, imax(i8) - 46, imin(i8) + 48, imin(i8) + 49, -27e0,         27e0,          imax(i8) - 49, imax(i8) - 48,
                imin(i8) + 50, imin(i8) + 51, -28e0,         28e0,          imax(i8) - 51, imax(i8) - 50, imin(i8) + 52, imin(i8) + 53,
                -29e0,         29e0,          imax(i8) - 53, imax(i8) - 52, imin(i8) + 54, imin(i8) + 55, -31e0,         31e0,
                imax(i8) - 55, imax(i8) - 54, imin(i8) + 56, imin(i8) + 57, -32e0,         32e0,          imax(i8) - 57, imax(i8) - 56,
                imin(i8) + 58, imin(i8) + 59, -33e0,         33e0,          imax(i8) - 59, imax(i8) - 58, imin(i8) + 60, imin(i8) + 61,
                -34e0,         34e0,          imax(i8) - 61, imax(i8) - 60, imin(i8) + 62, imin(i8) + 63, -35e0,         35e0,
                imax(i8) - 63, imax(i8) - 62, imin(i8) + 64, imin(i8) + 65, -36e0,         36e0,          imax(i8) - 65, imax(i8) - 64,
                imin(i8) + 66, imin(i8) + 67, -37e0,         37e0,          imax(i8) - 67, imax(i8) - 66, imin(i8) + 68, imin(i8) + 69,
                -38e0,         38e0,          imax(i8) - 69, imax(i8) - 68, imin(i8) + 70, imin(i8) + 71, -39e0,         39e0,
                imax(i8) - 71, imax(i8) - 70, imin(i8) + 72, imin(i8) + 73, -41e0,         41e0,          imax(i8) - 73, imax(i8) - 72,
                imin(i8) + 74, imin(i8) + 75, -42e0,         42e0,          imax(i8) - 75, imax(i8) - 74, imin(i8) + 76, imin(i8) + 77,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u8), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u8), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u8), .{
                1e2, imax(u8) - 1, imax(u8), 2e0,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u8), .{
                2e1, 2e2, imax(u8) - 3, imax(u8) - 2, 3e0, 3e1, imax(u8) - 5, imax(u8) - 4,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u8), .{
                imax(u8) - 7,  imax(u8) - 6,  5e0, 5e1, imax(u8) - 9,  imax(u8) - 8,  6e0, 6e1,
                imax(u8) - 11, imax(u8) - 10, 7e0, 7e1, imax(u8) - 13, imax(u8) - 12, 8e0, 8e1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u8), .{
                imax(u8) - 15, imax(u8) - 14, 9e0,  9e1,  imax(u8) - 17, imax(u8) - 16, 11e0, 11e1,
                imax(u8) - 19, imax(u8) - 18, 12e0, 12e1, imax(u8) - 21, imax(u8) - 20, 13e0, 13e1,
                imax(u8) - 23, imax(u8) - 22, 14e0, 14e1, imax(u8) - 25, imax(u8) - 24, 15e0, 15e1,
                imax(u8) - 27, imax(u8) - 26, 16e0, 16e1, imax(u8) - 29, imax(u8) - 28, 17e0, 17e1,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u8), .{
                imax(u8) - 31, imax(u8) - 30, 18e0,          18e1,          imax(u8) - 33, imax(u8) - 32, 19e0,          19e1,
                imax(u8) - 35, imax(u8) - 34, 21e0,          21e1,          imax(u8) - 37, imax(u8) - 36, 22e0,          22e1,
                imax(u8) - 39, imax(u8) - 38, 23e0,          23e1,          imax(u8) - 41, imax(u8) - 40, 24e0,          24e1,
                imax(u8) - 43, imax(u8) - 42, 25e0,          25e1,          imax(u8) - 45, imax(u8) - 44, 26e0,          imax(u8) - 47,
                imax(u8) - 46, 27e0,          imax(u8) - 49, imax(u8) - 48, 28e0,          imax(u8) - 51, imax(u8) - 50, 29e0,
                imax(u8) - 53, imax(u8) - 52, 31e0,          imax(u8) - 55, imax(u8) - 54, 32e0,          imax(u8) - 57, imax(u8) - 56,
                33e0,          imax(u8) - 59, imax(u8) - 58, 34e0,          imax(u8) - 61, imax(u8) - 60, 35e0,          imax(u8) - 63,
                imax(u8) - 62, 36e0,          imax(u8) - 65, imax(u8) - 64, 37e0,          imax(u8) - 67, imax(u8) - 66, 38e0,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u8), .{
                imax(u8) - 69,  imax(u8) - 68,  39e0,           imax(u8) - 71,  imax(u8) - 70,  41e0,           imax(u8) - 73,  imax(u8) - 72,
                42e0,           imax(u8) - 75,  imax(u8) - 74,  43e0,           imax(u8) - 77,  imax(u8) - 76,  44e0,           imax(u8) - 79,
                imax(u8) - 78,  45e0,           imax(u8) - 81,  imax(u8) - 80,  46e0,           imax(u8) - 83,  imax(u8) - 82,  47e0,
                imax(u8) - 85,  imax(u8) - 84,  48e0,           imax(u8) - 87,  imax(u8) - 86,  49e0,           imax(u8) - 89,  imax(u8) - 88,
                51e0,           imax(u8) - 91,  imax(u8) - 90,  52e0,           imax(u8) - 93,  imax(u8) - 92,  53e0,           imax(u8) - 95,
                imax(u8) - 94,  54e0,           imax(u8) - 97,  imax(u8) - 96,  55e0,           imax(u8) - 99,  imax(u8) - 98,  56e0,
                imax(u8) - 101, imax(u8) - 100, 57e0,           imax(u8) - 103, imax(u8) - 102, 58e0,           imax(u8) - 105, imax(u8) - 104,
                59e0,           imax(u8) - 107, imax(u8) - 106, 61e0,           imax(u8) - 109, imax(u8) - 108, 62e0,           imax(u8) - 111,
                imax(u8) - 110, 63e0,           imax(u8) - 113, imax(u8) - 112, 64e0,           imax(u8) - 115, imax(u8) - 114, 65e0,
                imax(u8) - 117, imax(u8) - 116, 66e0,           imax(u8) - 119, imax(u8) - 118, 67e0,           imax(u8) - 121, imax(u8) - 120,
                68e0,           imax(u8) - 123, imax(u8) - 122, 69e0,           imax(u8) - 125, imax(u8) - 124, 71e0,           imax(u8) - 127,
                imax(u8) - 126, 72e0,           imax(u8) - 129, imax(u8) - 128, 73e0,           imax(u8) - 131, imax(u8) - 130, 74e0,
                imax(u8) - 133, imax(u8) - 132, 75e0,           imax(u8) - 135, imax(u8) - 134, 76e0,           imax(u8) - 137, imax(u8) - 136,
                77e0,           imax(u8) - 139, imax(u8) - 138, 78e0,           imax(u8) - 141, imax(u8) - 140, 79e0,           imax(u8) - 143,
                imax(u8) - 142, 81e0,           imax(u8) - 145, imax(u8) - 144, 82e0,           imax(u8) - 147, imax(u8) - 146, 83e0,
                imax(u8) - 149, imax(u8) - 148, 84e0,           imax(u8) - 151, imax(u8) - 150, 85e0,           imax(u8) - 153, imax(u8) - 152,
            });

            try testArgs(@Vector(1, f16), @Vector(1, i16), .{
                imin(i16),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i16), .{
                imin(i16) + 1, -1e4,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i16), .{
                -1e3, -1e2, -1e1, -1e0,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i16), .{
                0, 1e0, 1e1, 1e2, 1e3, 1e4, imax(i16) - 1, imax(i16),
            });
            try testArgs(@Vector(16, f16), @Vector(16, i16), .{
                imin(i16) + 2, imin(i16) + 3, -2e4, -2e3, -2e2,          -2e1,          -2e0,          2e0,
                2e1,           2e2,           2e3,  2e4,  imax(i16) - 3, imax(i16) - 2, imin(i16) + 4, imin(i16) + 5,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i16), .{
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0,          3e0,           3e1,           3e2,
                3e3,           3e4,           imax(i16) - 5, imax(i16) - 4, imin(i16) + 6, imin(i16) + 7, -4e3,          -4e2,
                -4e1,          -4e0,          4e0,           4e1,           4e2,           4e3,           imax(i16) - 7, imax(i16) - 6,
                imin(i16) + 8, imin(i16) + 9, -5e3,          -5e2,          -5e1,          -5e0,          5e0,           5e1,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i16), .{
                5e2,            5e3,            imax(i16) - 9,  imax(i16) - 8,  imin(i16) + 10, imin(i16) + 11, -6e3,           -6e2,
                -6e1,           -6e0,           6e0,            6e1,            6e2,            6e3,            imax(i16) - 11, imax(i16) - 10,
                imin(i16) + 12, imin(i16) + 13, -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,
                7e2,            7e3,            imax(i16) - 13, imax(i16) - 12, imin(i16) + 14, imin(i16) + 15, -8e3,           -8e2,
                -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            imax(i16) - 15, imax(i16) - 14,
                imin(i16) + 16, imin(i16) + 17, -9e3,           -9e2,           -9e1,           -9e0,           9e0,            9e1,
                9e2,            9e3,            imax(i16) - 17, imax(i16) - 16, imin(i16) + 18, imin(i16) + 19, -11e3,          -11e2,
                -11e1,          -11e0,          11e0,           11e1,           11e2,           11e3,           imax(i16) - 19, imax(i16) - 18,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i16), .{
                imin(i16) + 20, imin(i16) + 21, -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,
                12e2,           12e3,           imax(i16) - 21, imax(i16) - 20, imin(i16) + 22, imin(i16) + 23, -13e3,          -13e2,
                -13e1,          -13e0,          13e0,           13e1,           13e2,           13e3,           imax(i16) - 23, imax(i16) - 22,
                imin(i16) + 24, imin(i16) + 25, -14e3,          -14e2,          -14e1,          -14e0,          14e0,           14e1,
                14e2,           14e3,           imax(i16) - 25, imax(i16) - 24, imin(i16) + 26, imin(i16) + 27, -15e3,          -15e2,
                -15e1,          -15e0,          15e0,           15e1,           15e2,           15e3,           imax(i16) - 27, imax(i16) - 26,
                imin(i16) + 28, imin(i16) + 29, -16e3,          -16e2,          -16e1,          -16e0,          16e0,           16e1,
                16e2,           16e3,           imax(i16) - 29, imax(i16) - 28, imin(i16) + 30, imin(i16) + 31, -17e3,          -17e2,
                -17e1,          -17e0,          17e0,           17e1,           17e2,           17e3,           imax(i16) - 31, imax(i16) - 30,
                imin(i16) + 32, imin(i16) + 33, -18e3,          -18e2,          -18e1,          -18e0,          18e0,           18e1,
                18e2,           18e3,           imax(i16) - 33, imax(i16) - 32, imin(i16) + 34, imin(i16) + 35, -19e3,          -19e2,
                -19e1,          -19e0,          19e0,           19e1,           19e2,           19e3,           imax(i16) - 35, imax(i16) - 34,
                imin(i16) + 36, imin(i16) + 37, -12e3,          -21e2,          -21e1,          -21e0,          21e0,           21e1,
                21e2,           21e3,           imax(i16) - 37, imax(i16) - 36, imin(i16) + 38, imin(i16) + 39, -22e3,          -22e2,
                -22e1,          -22e0,          22e0,           22e1,           22e2,           22e3,           imax(i16) - 39, imax(i16) - 38,
                imin(i16) + 40, imin(i16) + 41, -23e3,          -23e2,          -23e1,          -23e0,          23e0,           23e1,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u16), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u16), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u16), .{
                1e2, 1e3, 1e4, imax(u16) - 1,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u16), .{
                imax(u16), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u16) - 3, imax(u16) - 2,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u16), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u16) - 5, imax(u16) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u16) - 7, imax(u16) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u16), .{
                5e2,            5e3,            5e4,            imax(u16) - 9,  imax(u16) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u16) - 11, imax(u16) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u16) - 13, imax(u16) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u16) - 15, imax(u16) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u16) - 17, imax(u16) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u16), .{
                11e2,           11e3,           imax(u16) - 19, imax(u16) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u16) - 21, imax(u16) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u16) - 23, imax(u16) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u16) - 25, imax(u16) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u16) - 27, imax(u16) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u16) - 29, imax(u16) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u16) - 31, imax(u16) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u16) - 33, imax(u16) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u16) - 35, imax(u16) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u16) - 37, imax(u16) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u16) - 39, imax(u16) - 38,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u16), .{
                23e0,           23e1,           23e2,           23e3,           imax(u16) - 41, imax(u16) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u16) - 43, imax(u16) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u16) - 45, imax(u16) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u16) - 47, imax(u16) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u16) - 49, imax(u16) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u16) - 51, imax(u16) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u16) - 53, imax(u16) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u16) - 55, imax(u16) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u16) - 57, imax(u16) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u16) - 59, imax(u16) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u16) - 61, imax(u16) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u16) - 63, imax(u16) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u16) - 65, imax(u16) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u16) - 67, imax(u16) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u16) - 69, imax(u16) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u16) - 71, imax(u16) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u16) - 73, imax(u16) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u16) - 75, imax(u16) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u16) - 77, imax(u16) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u16) - 79, imax(u16) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u16) - 81, imax(u16) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f16), @Vector(1, i32), .{
                imin(i32),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i32), .{
                imin(i32) + 1, -1e9,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i32), .{
                -1e8, -1e7, -1e6, -1e5,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i32), .{
                -1e4, -1e3, -1e2, -1e1, -1e0, 0, 1e0, 1e1,
            });
            try testArgs(@Vector(16, f16), @Vector(16, i32), .{
                1e2,           1e3,       1e4,           1e5,           1e6,  1e7,  1e8,  1e9,
                imax(i32) - 1, imax(i32), imin(i32) + 2, imin(i32) + 3, -2e9, -2e8, -2e7, -2e6,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i32), .{
                -2e5,          -2e4,          -2e3,          -2e2,          -2e1, -2e0, 2e0,  2e1,
                2e2,           2e3,           2e4,           2e5,           2e6,  2e7,  2e8,  2e9,
                imax(i32) - 3, imax(i32) - 2, imin(i32) + 4, imin(i32) + 5, -3e8, -3e7, -3e6, -3e5,
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0, 3e0,  3e1,  3e2,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i32), .{
                3e3,           3e4,           3e5,           3e6,           3e7,            3e8,            imax(i32) - 5, imax(i32) - 4,
                imin(i32) + 6, imin(i32) + 7, -4e8,          -4e7,          -4e6,           -4e5,           -4e4,          -4e3,
                -4e2,          -4e1,          -4e0,          4e0,           4e1,            4e2,            4e3,           4e4,
                4e5,           4e6,           4e7,           4e8,           imax(i32) - 7,  imax(i32) - 6,  imin(i32) + 8, imin(i32) + 9,
                -5e8,          -5e7,          -5e6,          -5e5,          -5e4,           -5e3,           -5e2,          -5e1,
                -5e0,          5e0,           5e1,           5e2,           5e3,            5e4,            5e5,           5e6,
                5e7,           5e8,           imax(i32) - 9, imax(i32) - 8, imin(i32) + 10, imin(i32) + 11, -6e8,          -6e7,
                -6e6,          -6e5,          -6e4,          -6e3,          -6e2,           -6e1,           -6e0,          6e0,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i32), .{
                6e1,            6e2,            6e3,            6e4,            6e5,            6e6,            6e7,            6e8,
                imax(i32) - 11, imax(i32) - 10, imin(i32) + 12, imin(i32) + 13, -7e8,           -7e7,           -7e6,           -7e5,
                -7e4,           -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,            7e2,
                7e3,            7e4,            7e5,            7e6,            7e7,            7e8,            imax(i32) - 13, imax(i32) - 12,
                imin(i32) + 14, imin(i32) + 15, -8e8,           -8e7,           -8e6,           -8e5,           -8e4,           -8e3,
                -8e2,           -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            8e4,
                8e5,            8e6,            8e7,            8e8,            imax(i32) - 15, imax(i32) - 14, imin(i32) + 16, imin(i32) + 17,
                -9e8,           -9e7,           -9e6,           -9e5,           -9e4,           -9e3,           -9e2,           -9e1,
                -9e0,           9e0,            9e1,            9e2,            9e3,            9e4,            9e5,            9e6,
                9e7,            9e8,            imax(i32) - 17, imax(i32) - 16, imin(i32) + 18, imin(i32) + 19, -11e8,          -11e7,
                -11e6,          -11e5,          -11e4,          -11e3,          -11e2,          -11e1,          -11e0,          11e0,
                11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,           11e8,
                imax(i32) - 19, imax(i32) - 18, imin(i32) + 20, imin(i32) + 21, -12e8,          -12e7,          -12e6,          -12e5,
                -12e4,          -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           imax(i32) - 21, imax(i32) - 20,
                imin(i32) + 22, imin(i32) + 23, -13e8,          -13e7,          -13e6,          -13e5,          -13e4,          -13e3,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u32), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u32), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u32), .{
                1e2, 1e3, 1e4, imax(u32) - 1,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u32), .{
                imax(u32), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u32) - 3, imax(u32) - 2,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u32), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u32) - 5, imax(u32) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u32) - 7, imax(u32) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u32), .{
                5e2,            5e3,            5e4,            imax(u32) - 9,  imax(u32) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u32) - 11, imax(u32) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u32) - 13, imax(u32) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u32) - 15, imax(u32) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u32) - 17, imax(u32) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u32), .{
                11e2,           11e3,           imax(u32) - 19, imax(u32) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u32) - 21, imax(u32) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u32) - 23, imax(u32) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u32) - 25, imax(u32) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u32) - 27, imax(u32) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u32) - 29, imax(u32) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u32) - 31, imax(u32) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u32) - 33, imax(u32) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u32) - 35, imax(u32) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u32) - 37, imax(u32) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u32) - 39, imax(u32) - 38,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u32), .{
                23e0,           23e1,           23e2,           23e3,           imax(u32) - 41, imax(u32) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u32) - 43, imax(u32) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u32) - 45, imax(u32) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u32) - 47, imax(u32) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u32) - 49, imax(u32) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u32) - 51, imax(u32) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u32) - 53, imax(u32) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u32) - 55, imax(u32) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u32) - 57, imax(u32) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u32) - 59, imax(u32) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u32) - 61, imax(u32) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u32) - 63, imax(u32) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u32) - 65, imax(u32) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u32) - 67, imax(u32) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u32) - 69, imax(u32) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u32) - 71, imax(u32) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u32) - 73, imax(u32) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u32) - 75, imax(u32) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u32) - 77, imax(u32) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u32) - 79, imax(u32) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u32) - 81, imax(u32) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f16), @Vector(1, i64), .{
                imin(i64),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i64), .{
                imin(i64) + 1, -1e18,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i64), .{
                -1e17, -1e16, -1e15, -1e14,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i64), .{
                -1e13, -1e12, -1e11, -1e10, -1e9, -1e8, -1e7, -1e6,
            });
            try testArgs(@Vector(16, f16), @Vector(16, i64), .{
                -1e5, -1e4, -1e3, -1e2, -1e1, -1e0, 0,   1e0,
                1e1,  1e2,  1e3,  1e4,  1e5,  1e6,  1e7, 1e8,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i64), .{
                1e9,   1e10,  1e11,          1e12,      1e13,          1e14,          1e15,  1e16,
                1e17,  1e18,  imax(i64) - 1, imax(i64), imin(i64) + 2, imin(i64) + 3, -2e18, -2e17,
                -2e16, -2e15, -2e14,         -2e13,     -2e12,         -2e11,         -2e10, -2e9,
                -2e8,  -2e7,  -2e6,          -2e5,      -2e4,          -2e3,          -2e2,  -2e1,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i64), .{
                -2e0,  2e0,   2e1,   2e2,   2e3,           2e4,           2e5,           2e6,
                2e7,   2e8,   2e9,   2e10,  2e11,          2e12,          2e13,          2e14,
                2e15,  2e16,  2e17,  2e18,  imax(i64) - 3, imax(i64) - 2, imin(i64) + 4, imin(i64) + 5,
                -3e18, -3e17, -3e16, -3e15, -3e14,         -3e13,         -3e12,         -3e11,
                -3e10, -3e9,  -3e8,  -3e7,  -3e6,          -3e5,          -3e4,          -3e3,
                -3e2,  -3e1,  -3e0,  3e0,   3e1,           3e2,           3e3,           3e4,
                3e5,   3e6,   3e7,   3e8,   3e9,           3e10,          3e11,          3e12,
                3e13,  3e14,  3e15,  3e16,  3e17,          3e18,          imax(i64) - 5, imax(i64) - 4,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i64), .{
                imin(i64) + 6, imin(i64) + 7, -4e18,         -4e17,         -4e16,          -4e15,          -4e14,          -4e13,
                -4e12,         -4e11,         -4e10,         -4e9,          -4e8,           -4e7,           -4e6,           -4e5,
                -4e4,          -4e3,          -4e2,          -4e1,          -4e0,           4e0,            4e1,            4e2,
                4e3,           4e4,           4e5,           4e6,           4e7,            4e8,            4e9,            4e10,
                4e11,          4e12,          4e13,          4e14,          4e15,           4e16,           4e17,           4e18,
                imax(i64) - 7, imax(i64) - 6, imin(i64) + 8, imin(i64) + 9, -5e18,          -5e17,          -5e16,          -5e15,
                -5e14,         -5e13,         -5e12,         -5e11,         -5e10,          -5e9,           -5e8,           -5e7,
                -5e6,          -5e5,          -5e4,          -5e3,          -5e2,           -5e1,           -5e0,           5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,            5e6,            5e7,            5e8,
                5e9,           5e10,          5e11,          5e12,          5e13,           5e14,           5e15,           5e16,
                5e17,          5e18,          imax(i64) - 9, imax(i64) - 8, imin(i64) + 10, imin(i64) + 11, -6e18,          -6e17,
                -6e16,         -6e15,         -6e14,         -6e13,         -6e12,          -6e11,          -6e10,          -6e9,
                -6e8,          -6e7,          -6e6,          -6e5,          -6e4,           -6e3,           -6e2,           -6e1,
                -6e0,          6e0,           6e1,           6e2,           6e3,            6e4,            6e5,            6e6,
                6e7,           6e8,           6e9,           6e10,          6e11,           6e12,           6e13,           6e14,
                6e15,          6e16,          6e17,          6e18,          imax(i64) - 11, imax(i64) - 10, imin(i64) + 12, imin(i64) + 13,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u64), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u64), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u64), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u64), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u64), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, imax(u64) - 1, imax(u64),
                2e0,  2e1,  2e2,  2e3,  2e4,  2e5,  2e6,           2e7,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u64), .{
                2e8,  2e9,  2e10, 2e11,          2e12,          2e13, 2e14, 2e15,
                2e16, 2e17, 2e18, imax(u64) - 3, imax(u64) - 2, 3e0,  3e1,  3e2,
                3e3,  3e4,  3e5,  3e6,           3e7,           3e8,  3e9,  3e10,
                3e11, 3e12, 3e13, 3e14,          3e15,          3e16, 3e17, 3e18,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u64), .{
                imax(u64) - 5, imax(u64) - 4, 4e0,           4e1,           4e2,  4e3,           4e4,           4e5,
                4e6,           4e7,           4e8,           4e9,           4e10, 4e11,          4e12,          4e13,
                4e14,          4e15,          4e16,          4e17,          4e18, imax(u64) - 7, imax(u64) - 6, 5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,  5e6,           5e7,           5e8,
                5e9,           5e10,          5e11,          5e12,          5e13, 5e14,          5e15,          5e16,
                5e17,          5e18,          imax(u64) - 9, imax(u64) - 8, 6e0,  6e1,           6e2,           6e3,
                6e4,           6e5,           6e6,           6e7,           6e8,  6e9,           6e10,          6e11,
                6e12,          6e13,          6e14,          6e15,          6e16, 6e17,          6e18,          imax(u64) - 11,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u64), .{
                imax(u64) - 10, 7e0,            7e1,            7e2,            7e3,            7e4,            7e5,            7e6,
                7e7,            7e8,            7e9,            7e10,           7e11,           7e12,           7e13,           7e14,
                7e15,           7e16,           7e17,           7e18,           imax(u64) - 13, imax(u64) - 12, 8e0,            8e1,
                8e2,            8e3,            8e4,            8e5,            8e6,            8e7,            8e8,            8e9,
                8e10,           8e11,           8e12,           8e13,           8e14,           8e15,           8e16,           8e17,
                8e18,           imax(u64) - 15, imax(u64) - 14, 9e0,            9e1,            9e2,            9e3,            9e4,
                9e5,            9e6,            9e7,            9e8,            9e9,            9e10,           9e11,           9e12,
                9e13,           9e14,           9e15,           9e16,           9e17,           9e18,           imax(u64) - 17, imax(u64) - 16,
                11e0,           11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,
                11e8,           11e9,           11e10,          11e11,          11e12,          11e13,          11e14,          11e15,
                11e16,          11e17,          11e18,          imax(u64) - 19, imax(u64) - 18, 12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           12e9,           12e10,
                12e11,          12e12,          12e13,          12e14,          12e15,          12e16,          12e17,          12e18,
                imax(u64) - 21, imax(u64) - 20, 13e0,           13e1,           13e2,           13e3,           13e4,           13e5,
                13e6,           13e7,           13e8,           13e9,           13e10,          13e11,          13e12,          13e13,
                13e14,          13e15,          13e16,          13e17,          13e18,          imax(u64) - 23, imax(u64) - 22, 14e0,
            });

            try testArgs(@Vector(1, f16), @Vector(1, i128), .{
                imin(i128),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i128), .{
                imin(i128) + 1, -1e38,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i128), .{
                -1e37, -1e36, -1e35, -1e34,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i128), .{
                -1e33, -1e32, -1e31, -1e30, -1e29, -1e28, -1e27, -1e26,
            });
            try testArgs(@Vector(16, f16), @Vector(16, i128), .{
                -1e25, -1e24, -1e23, -1e22, -1e21, -1e20, -1e19, -1e18,
                -1e17, -1e16, -1e15, -1e14, -1e13, -1e12, -1e11, -1e10,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i128), .{
                -1e9, -1e8, -1e7, -1e6, -1e5, -1e4, -1e3, -1e2,
                -1e1, -1e0, 0,    1e0,  1e1,  1e2,  1e3,  1e4,
                1e5,  1e6,  1e7,  1e8,  1e9,  1e10, 1e11, 1e12,
                1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i128), .{
                1e21,  1e22,  1e23,           1e24,       1e25,           1e26,           1e27,  1e28,
                1e29,  1e30,  1e31,           1e32,       1e33,           1e34,           1e35,  1e36,
                1e37,  1e38,  imax(i128) - 1, imax(i128), imin(i128) + 2, imin(i128) + 3, -2e37, -2e36,
                -2e35, -2e34, -2e33,          -2e32,      -2e31,          -2e30,          -2e29, -2e28,
                -2e27, -2e26, -2e25,          -2e24,      -2e23,          -2e22,          -2e21, -2e20,
                -2e19, -2e18, -2e17,          -2e16,      -2e15,          -2e14,          -2e13, -2e12,
                -2e11, -2e10, -2e9,           -2e8,       -2e7,           -2e6,           -2e5,  -2e4,
                -2e3,  -2e2,  -2e1,           -2e0,       2e0,            2e1,            2e2,   2e3,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i128), .{
                2e4,   2e5,   2e6,            2e7,            2e8,            2e9,            2e10,  2e11,
                2e12,  2e13,  2e14,           2e15,           2e16,           2e17,           2e18,  2e19,
                2e20,  2e21,  2e22,           2e23,           2e24,           2e25,           2e26,  2e27,
                2e28,  2e29,  2e30,           2e31,           2e32,           2e33,           2e34,  2e35,
                2e36,  2e37,  imax(i128) - 3, imax(i128) - 2, imin(i128) + 4, imin(i128) + 5, -3e37, -3e36,
                -3e35, -3e34, -3e33,          -3e32,          -3e31,          -3e30,          -3e29, -3e28,
                -3e27, -3e26, -3e25,          -3e24,          -3e23,          -3e22,          -3e21, -3e20,
                -3e19, -3e18, -3e17,          -3e16,          -3e15,          -3e14,          -3e13, -3e12,
                -3e11, -3e10, -3e9,           -3e8,           -3e7,           -3e6,           -3e5,  -3e4,
                -3e3,  -3e2,  -3e1,           -3e0,           3e0,            3e1,            3e2,   3e3,
                3e4,   3e5,   3e6,            3e7,            3e8,            3e9,            3e10,  3e11,
                3e12,  3e13,  3e14,           3e15,           3e16,           3e17,           3e18,  3e19,
                3e20,  3e21,  3e22,           3e23,           3e24,           3e25,           3e26,  3e27,
                3e28,  3e29,  3e30,           3e31,           3e32,           3e33,           3e34,  3e35,
                3e36,  3e37,  imax(i128) - 5, imax(i128) - 4, imin(i128) + 6, imin(i128) + 7, -4e37, -4e36,
                -4e35, -4e34, -4e33,          -4e32,          -4e31,          -4e30,          -4e29, -4e28,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u128), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u128), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u128), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u128), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u128), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u128), .{
                1e30, 1e31,           1e32,       1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, imax(u128) - 1, imax(u128), 2e0,  2e1,  2e2,  2e3,  2e4,
                2e5,  2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12,
                2e13, 2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u128), .{
                2e21, 2e22, 2e23,           2e24,           2e25,           2e26, 2e27, 2e28,
                2e29, 2e30, 2e31,           2e32,           2e33,           2e34, 2e35, 2e36,
                2e37, 2e38, imax(u128) - 3, imax(u128) - 2, 3e0,            3e1,  3e2,  3e3,
                3e4,  3e5,  3e6,            3e7,            3e8,            3e9,  3e10, 3e11,
                3e12, 3e13, 3e14,           3e15,           3e16,           3e17, 3e18, 3e19,
                3e20, 3e21, 3e22,           3e23,           3e24,           3e25, 3e26, 3e27,
                3e28, 3e29, 3e30,           3e31,           3e32,           3e33, 3e34, 3e35,
                3e36, 3e37, 3e38,           imax(u128) - 5, imax(u128) - 4, 4e0,  4e1,  4e2,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u128), .{
                4e3,  4e4,  4e5,  4e6,             4e7,             4e8,  4e9,  4e10,
                4e11, 4e12, 4e13, 4e14,            4e15,            4e16, 4e17, 4e18,
                4e19, 4e20, 4e21, 4e22,            4e23,            4e24, 4e25, 4e26,
                4e27, 4e28, 4e29, 4e30,            4e31,            4e32, 4e33, 4e34,
                4e35, 4e36, 4e37, imax(u128) - 7,  imax(u128) - 6,  5e0,  5e1,  5e2,
                5e3,  5e4,  5e5,  5e6,             5e7,             5e8,  5e9,  5e10,
                5e11, 5e12, 5e13, 5e14,            5e15,            5e16, 5e17, 5e18,
                5e19, 5e20, 5e21, 5e22,            5e23,            5e24, 5e25, 5e26,
                5e27, 5e28, 5e29, 5e30,            5e31,            5e32, 5e33, 5e34,
                5e35, 5e36, 5e37, imax(u128) - 9,  imax(u128) - 8,  6e0,  6e1,  6e2,
                6e3,  6e4,  6e5,  6e6,             6e7,             6e8,  6e9,  6e10,
                6e11, 6e12, 6e13, 6e14,            6e15,            6e16, 6e17, 6e18,
                6e19, 6e20, 6e21, 6e22,            6e23,            6e24, 6e25, 6e26,
                6e27, 6e28, 6e29, 6e30,            6e31,            6e32, 6e33, 6e34,
                6e35, 6e36, 6e37, imax(u128) - 11, imax(u128) - 10, 7e0,  7e1,  7e2,
                7e3,  7e4,  7e5,  7e6,             7e7,             7e8,  7e9,  7e10,
            });

            try testArgs(@Vector(1, f16), @Vector(1, i256), .{
                imin(i256),
            });
            try testArgs(@Vector(2, f16), @Vector(2, i256), .{
                imin(i256) + 1, -1e76,
            });
            try testArgs(@Vector(4, f16), @Vector(4, i256), .{
                -1e75, -1e74, -1e73, -1e72,
            });
            try testArgs(@Vector(8, f16), @Vector(8, i256), .{
                -1e71, -1e70, -1e69, -1e68, -1e67, -1e66, -1e65, -1e64,
            });
            try testArgs(@Vector(16, f16), @Vector(16, i256), .{
                -1e63, -1e62, -1e61, -1e60, -1e59, -1e58, -1e57, -1e56,
                -1e55, -1e54, -1e53, -1e52, -1e51, -1e50, -1e49, -1e48,
            });
            try testArgs(@Vector(32, f16), @Vector(32, i256), .{
                -1e47, -1e46, -1e45, -1e44, -1e43, -1e42, -1e41, -1e40,
                -1e39, -1e38, -1e37, -1e36, -1e35, -1e34, -1e33, -1e32,
                -1e31, -1e30, -1e29, -1e28, -1e27, -1e26, -1e25, -1e24,
                -1e23, -1e22, -1e21, -1e20, -1e19, -1e18, -1e17, -1e16,
            });
            try testArgs(@Vector(64, f16), @Vector(64, i256), .{
                -1e15, -1e14, -1e13, -1e12, -1e11, -1e10, -1e9, -1e8,
                -1e7,  -1e6,  -1e5,  -1e4,  -1e3,  -1e2,  -1e1, -1e0,
                0,     1e0,   1e1,   1e2,   1e3,   1e4,   1e5,  1e6,
                1e7,   1e8,   1e9,   1e10,  1e11,  1e12,  1e13, 1e14,
                1e15,  1e16,  1e17,  1e18,  1e19,  1e20,  1e21, 1e22,
                1e23,  1e24,  1e25,  1e26,  1e27,  1e28,  1e29, 1e30,
                1e31,  1e32,  1e33,  1e34,  1e35,  1e36,  1e37, 1e38,
                1e39,  1e40,  1e41,  1e42,  1e43,  1e44,  1e45, 1e46,
            });
            try testArgs(@Vector(128, f16), @Vector(128, i256), .{
                1e47,           1e48,           1e49,  1e50,  1e51,  1e52,  1e53,           1e54,
                1e55,           1e56,           1e57,  1e58,  1e59,  1e60,  1e61,           1e62,
                1e63,           1e64,           1e65,  1e66,  1e67,  1e68,  1e69,           1e70,
                1e71,           1e72,           1e73,  1e74,  1e75,  1e76,  imax(i256) - 1, imax(i256),
                imin(i256) + 2, imin(i256) + 3, -2e76, -2e75, -2e74, -2e73, -2e72,          -2e71,
                -2e70,          -2e69,          -2e68, -2e67, -2e66, -2e65, -2e64,          -2e63,
                -2e62,          -2e61,          -2e60, -2e59, -2e58, -2e57, -2e56,          -2e55,
                -2e54,          -2e53,          -2e52, -2e51, -2e50, -2e49, -2e48,          -2e47,
                -2e46,          -2e45,          -2e44, -2e43, -2e42, -2e41, -2e40,          -2e39,
                -2e38,          -2e37,          -2e36, -2e35, -2e34, -2e33, -2e32,          -2e31,
                -2e30,          -2e29,          -2e28, -2e27, -2e26, -2e25, -2e24,          -2e23,
                -2e22,          -2e21,          -2e20, -2e19, -2e18, -2e17, -2e16,          -2e15,
                -2e14,          -2e13,          -2e12, -2e11, -2e10, -2e9,  -2e8,           -2e7,
                -2e6,           -2e5,           -2e4,  -2e3,  -2e2,  -2e1,  -2e0,           2e0,
                2e1,            2e2,            2e3,   2e4,   2e5,   2e6,   2e7,            2e8,
                2e9,            2e10,           2e11,  2e12,  2e13,  2e14,  2e15,           2e16,
            });

            try testArgs(@Vector(1, f16), @Vector(1, u256), .{
                0,
            });
            try testArgs(@Vector(2, f16), @Vector(2, u256), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f16), @Vector(4, u256), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f16), @Vector(8, u256), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f16), @Vector(16, u256), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f16), @Vector(32, u256), .{
                1e30, 1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, 1e39, 1e40, 1e41, 1e42, 1e43, 1e44, 1e45,
                1e46, 1e47, 1e48, 1e49, 1e50, 1e51, 1e52, 1e53,
                1e54, 1e55, 1e56, 1e57, 1e58, 1e59, 1e60, 1e61,
            });
            try testArgs(@Vector(64, f16), @Vector(64, u256), .{
                1e62,           1e63,       1e64, 1e65, 1e66, 1e67, 1e68, 1e69,
                1e70,           1e71,       1e72, 1e73, 1e74, 1e75, 1e76, 1e77,
                imax(u256) - 1, imax(u256), 2e0,  2e1,  2e2,  2e3,  2e4,  2e5,
                2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12, 2e13,
                2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20, 2e21,
                2e22,           2e23,       2e24, 2e25, 2e26, 2e27, 2e28, 2e29,
                2e30,           2e31,       2e32, 2e33, 2e34, 2e35, 2e36, 2e37,
                2e38,           2e39,       2e40, 2e41, 2e42, 2e43, 2e44, 2e45,
            });
            try testArgs(@Vector(128, f16), @Vector(128, u256), .{
                2e46,           2e47, 2e48, 2e49, 2e50, 2e51, 2e52,           2e53,
                2e54,           2e55, 2e56, 2e57, 2e58, 2e59, 2e60,           2e61,
                2e62,           2e63, 2e64, 2e65, 2e66, 2e67, 2e68,           2e69,
                2e70,           2e71, 2e72, 2e73, 2e74, 2e75, 2e76,           imax(u256) - 3,
                imax(u256) - 2, 3e0,  3e1,  3e2,  3e3,  3e4,  3e5,            3e6,
                3e7,            3e8,  3e9,  3e10, 3e11, 3e12, 3e13,           3e14,
                3e15,           3e16, 3e17, 3e18, 3e19, 3e20, 3e21,           3e22,
                3e23,           3e24, 3e25, 3e26, 3e27, 3e28, 3e29,           3e30,
                3e31,           3e32, 3e33, 3e34, 3e35, 3e36, 3e37,           3e38,
                3e39,           3e40, 3e41, 3e42, 3e43, 3e44, 3e45,           3e46,
                3e47,           3e48, 3e49, 3e50, 3e51, 3e52, 3e53,           3e54,
                3e55,           3e56, 3e57, 3e58, 3e59, 3e60, 3e61,           3e62,
                3e63,           3e64, 3e65, 3e66, 3e67, 3e68, 3e69,           3e70,
                3e71,           3e72, 3e73, 3e74, 3e75, 3e76, imax(u256) - 5, imax(u256) - 4,
                4e0,            4e1,  4e2,  4e3,  4e4,  4e5,  4e6,            4e7,
                4e8,            4e9,  4e10, 4e11, 4e12, 4e13, 4e14,           4e15,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i8), .{
                imin(i8),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i8), .{
                imin(i8) + 1, -1e2,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i8), .{
                -1e1, -1e0, 0, 1e0,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i8), .{
                1e1, 1e2, imax(i8) - 1, imax(i8), imin(i8) + 2, imin(i8) + 3, -2e1, -2e0,
            });
            try testArgs(@Vector(16, f32), @Vector(16, i8), .{
                2e0, 2e1, imax(i8) - 3, imax(i8) - 2, imin(i8) + 4, imin(i8) + 5, -3e1, -3e0,
                3e0, 3e1, imax(i8) - 5, imax(i8) - 4, imin(i8) + 6, imin(i8) + 7, -4e1, -4e0,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i8), .{
                4e0, 4e1, imax(i8) - 7,  imax(i8) - 6,  imin(i8) + 8,  imin(i8) + 9,  -5e1, -5e0,
                5e0, 5e1, imax(i8) - 9,  imax(i8) - 8,  imin(i8) + 10, imin(i8) + 11, -6e1, -6e0,
                6e0, 6e1, imax(i8) - 11, imax(i8) - 10, imin(i8) + 12, imin(i8) + 13, -7e1, -7e0,
                7e0, 7e1, imax(i8) - 13, imax(i8) - 12, imin(i8) + 14, imin(i8) + 15, -8e1, -8e0,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i8), .{
                8e0,           8e1,           imax(i8) - 15, imax(i8) - 14, imin(i8) + 16, imin(i8) + 17, -9e1,          -9e0,
                9e0,           9e1,           imax(i8) - 17, imax(i8) - 16, imin(i8) + 18, imin(i8) + 19, -11e1,         -11e0,
                11e0,          11e1,          imax(i8) - 19, imax(i8) - 18, imin(i8) + 20, imin(i8) + 21, -12e1,         -12e0,
                12e0,          12e1,          imax(i8) - 21, imax(i8) - 20, imin(i8) + 22, imin(i8) + 23, -13e0,         13e0,
                imax(i8) - 23, imax(i8) - 22, imin(i8) + 24, imin(i8) + 25, -14e0,         14e0,          imax(i8) - 25, imax(i8) - 24,
                imin(i8) + 26, imin(i8) + 27, -15e0,         15e0,          imax(i8) - 27, imax(i8) - 26, imin(i8) + 28, imin(i8) + 29,
                -16e0,         16e0,          imax(i8) - 29, imax(i8) - 28, imin(i8) + 30, imin(i8) + 31, -17e0,         17e0,
                imax(i8) - 31, imax(i8) - 30, imin(i8) + 32, imin(i8) + 33, -18e0,         18e0,          imax(i8) - 33, imax(i8) - 32,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i8), .{
                imin(i8) + 34, imin(i8) + 35, -19e0,         19e0,          imax(i8) - 35, imax(i8) - 34, imin(i8) + 36, imin(i8) + 37,
                -21e0,         21e0,          imax(i8) - 37, imax(i8) - 36, imin(i8) + 38, imin(i8) + 39, -22e0,         22e0,
                imax(i8) - 39, imax(i8) - 38, imin(i8) + 40, imin(i8) + 41, -23e0,         23e0,          imax(i8) - 41, imax(i8) - 40,
                imin(i8) + 42, imin(i8) + 43, -24e0,         24e0,          imax(i8) - 43, imax(i8) - 42, imin(i8) + 44, imin(i8) + 45,
                -25e0,         25e0,          imax(i8) - 45, imax(i8) - 44, imin(i8) + 46, imin(i8) + 47, -26e0,         26e0,
                imax(i8) - 47, imax(i8) - 46, imin(i8) + 48, imin(i8) + 49, -27e0,         27e0,          imax(i8) - 49, imax(i8) - 48,
                imin(i8) + 50, imin(i8) + 51, -28e0,         28e0,          imax(i8) - 51, imax(i8) - 50, imin(i8) + 52, imin(i8) + 53,
                -29e0,         29e0,          imax(i8) - 53, imax(i8) - 52, imin(i8) + 54, imin(i8) + 55, -31e0,         31e0,
                imax(i8) - 55, imax(i8) - 54, imin(i8) + 56, imin(i8) + 57, -32e0,         32e0,          imax(i8) - 57, imax(i8) - 56,
                imin(i8) + 58, imin(i8) + 59, -33e0,         33e0,          imax(i8) - 59, imax(i8) - 58, imin(i8) + 60, imin(i8) + 61,
                -34e0,         34e0,          imax(i8) - 61, imax(i8) - 60, imin(i8) + 62, imin(i8) + 63, -35e0,         35e0,
                imax(i8) - 63, imax(i8) - 62, imin(i8) + 64, imin(i8) + 65, -36e0,         36e0,          imax(i8) - 65, imax(i8) - 64,
                imin(i8) + 66, imin(i8) + 67, -37e0,         37e0,          imax(i8) - 67, imax(i8) - 66, imin(i8) + 68, imin(i8) + 69,
                -38e0,         38e0,          imax(i8) - 69, imax(i8) - 68, imin(i8) + 70, imin(i8) + 71, -39e0,         39e0,
                imax(i8) - 71, imax(i8) - 70, imin(i8) + 72, imin(i8) + 73, -41e0,         41e0,          imax(i8) - 73, imax(i8) - 72,
                imin(i8) + 74, imin(i8) + 75, -42e0,         42e0,          imax(i8) - 75, imax(i8) - 74, imin(i8) + 76, imin(i8) + 77,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u8), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u8), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u8), .{
                1e2, imax(u8) - 1, imax(u8), 2e0,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u8), .{
                2e1, 2e2, imax(u8) - 3, imax(u8) - 2, 3e0, 3e1, imax(u8) - 5, imax(u8) - 4,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u8), .{
                imax(u8) - 7,  imax(u8) - 6,  5e0, 5e1, imax(u8) - 9,  imax(u8) - 8,  6e0, 6e1,
                imax(u8) - 11, imax(u8) - 10, 7e0, 7e1, imax(u8) - 13, imax(u8) - 12, 8e0, 8e1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u8), .{
                imax(u8) - 15, imax(u8) - 14, 9e0,  9e1,  imax(u8) - 17, imax(u8) - 16, 11e0, 11e1,
                imax(u8) - 19, imax(u8) - 18, 12e0, 12e1, imax(u8) - 21, imax(u8) - 20, 13e0, 13e1,
                imax(u8) - 23, imax(u8) - 22, 14e0, 14e1, imax(u8) - 25, imax(u8) - 24, 15e0, 15e1,
                imax(u8) - 27, imax(u8) - 26, 16e0, 16e1, imax(u8) - 29, imax(u8) - 28, 17e0, 17e1,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u8), .{
                imax(u8) - 31, imax(u8) - 30, 18e0,          18e1,          imax(u8) - 33, imax(u8) - 32, 19e0,          19e1,
                imax(u8) - 35, imax(u8) - 34, 21e0,          21e1,          imax(u8) - 37, imax(u8) - 36, 22e0,          22e1,
                imax(u8) - 39, imax(u8) - 38, 23e0,          23e1,          imax(u8) - 41, imax(u8) - 40, 24e0,          24e1,
                imax(u8) - 43, imax(u8) - 42, 25e0,          25e1,          imax(u8) - 45, imax(u8) - 44, 26e0,          imax(u8) - 47,
                imax(u8) - 46, 27e0,          imax(u8) - 49, imax(u8) - 48, 28e0,          imax(u8) - 51, imax(u8) - 50, 29e0,
                imax(u8) - 53, imax(u8) - 52, 31e0,          imax(u8) - 55, imax(u8) - 54, 32e0,          imax(u8) - 57, imax(u8) - 56,
                33e0,          imax(u8) - 59, imax(u8) - 58, 34e0,          imax(u8) - 61, imax(u8) - 60, 35e0,          imax(u8) - 63,
                imax(u8) - 62, 36e0,          imax(u8) - 65, imax(u8) - 64, 37e0,          imax(u8) - 67, imax(u8) - 66, 38e0,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u8), .{
                imax(u8) - 69,  imax(u8) - 68,  39e0,           imax(u8) - 71,  imax(u8) - 70,  41e0,           imax(u8) - 73,  imax(u8) - 72,
                42e0,           imax(u8) - 75,  imax(u8) - 74,  43e0,           imax(u8) - 77,  imax(u8) - 76,  44e0,           imax(u8) - 79,
                imax(u8) - 78,  45e0,           imax(u8) - 81,  imax(u8) - 80,  46e0,           imax(u8) - 83,  imax(u8) - 82,  47e0,
                imax(u8) - 85,  imax(u8) - 84,  48e0,           imax(u8) - 87,  imax(u8) - 86,  49e0,           imax(u8) - 89,  imax(u8) - 88,
                51e0,           imax(u8) - 91,  imax(u8) - 90,  52e0,           imax(u8) - 93,  imax(u8) - 92,  53e0,           imax(u8) - 95,
                imax(u8) - 94,  54e0,           imax(u8) - 97,  imax(u8) - 96,  55e0,           imax(u8) - 99,  imax(u8) - 98,  56e0,
                imax(u8) - 101, imax(u8) - 100, 57e0,           imax(u8) - 103, imax(u8) - 102, 58e0,           imax(u8) - 105, imax(u8) - 104,
                59e0,           imax(u8) - 107, imax(u8) - 106, 61e0,           imax(u8) - 109, imax(u8) - 108, 62e0,           imax(u8) - 111,
                imax(u8) - 110, 63e0,           imax(u8) - 113, imax(u8) - 112, 64e0,           imax(u8) - 115, imax(u8) - 114, 65e0,
                imax(u8) - 117, imax(u8) - 116, 66e0,           imax(u8) - 119, imax(u8) - 118, 67e0,           imax(u8) - 121, imax(u8) - 120,
                68e0,           imax(u8) - 123, imax(u8) - 122, 69e0,           imax(u8) - 125, imax(u8) - 124, 71e0,           imax(u8) - 127,
                imax(u8) - 126, 72e0,           imax(u8) - 129, imax(u8) - 128, 73e0,           imax(u8) - 131, imax(u8) - 130, 74e0,
                imax(u8) - 133, imax(u8) - 132, 75e0,           imax(u8) - 135, imax(u8) - 134, 76e0,           imax(u8) - 137, imax(u8) - 136,
                77e0,           imax(u8) - 139, imax(u8) - 138, 78e0,           imax(u8) - 141, imax(u8) - 140, 79e0,           imax(u8) - 143,
                imax(u8) - 142, 81e0,           imax(u8) - 145, imax(u8) - 144, 82e0,           imax(u8) - 147, imax(u8) - 146, 83e0,
                imax(u8) - 149, imax(u8) - 148, 84e0,           imax(u8) - 151, imax(u8) - 150, 85e0,           imax(u8) - 153, imax(u8) - 152,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i16), .{
                imin(i16),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i16), .{
                imin(i16) + 1, -1e4,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i16), .{
                -1e3, -1e2, -1e1, -1e0,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i16), .{
                0, 1e0, 1e1, 1e2, 1e3, 1e4, imax(i16) - 1, imax(i16),
            });
            try testArgs(@Vector(16, f32), @Vector(16, i16), .{
                imin(i16) + 2, imin(i16) + 3, -2e4, -2e3, -2e2,          -2e1,          -2e0,          2e0,
                2e1,           2e2,           2e3,  2e4,  imax(i16) - 3, imax(i16) - 2, imin(i16) + 4, imin(i16) + 5,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i16), .{
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0,          3e0,           3e1,           3e2,
                3e3,           3e4,           imax(i16) - 5, imax(i16) - 4, imin(i16) + 6, imin(i16) + 7, -4e3,          -4e2,
                -4e1,          -4e0,          4e0,           4e1,           4e2,           4e3,           imax(i16) - 7, imax(i16) - 6,
                imin(i16) + 8, imin(i16) + 9, -5e3,          -5e2,          -5e1,          -5e0,          5e0,           5e1,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i16), .{
                5e2,            5e3,            imax(i16) - 9,  imax(i16) - 8,  imin(i16) + 10, imin(i16) + 11, -6e3,           -6e2,
                -6e1,           -6e0,           6e0,            6e1,            6e2,            6e3,            imax(i16) - 11, imax(i16) - 10,
                imin(i16) + 12, imin(i16) + 13, -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,
                7e2,            7e3,            imax(i16) - 13, imax(i16) - 12, imin(i16) + 14, imin(i16) + 15, -8e3,           -8e2,
                -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            imax(i16) - 15, imax(i16) - 14,
                imin(i16) + 16, imin(i16) + 17, -9e3,           -9e2,           -9e1,           -9e0,           9e0,            9e1,
                9e2,            9e3,            imax(i16) - 17, imax(i16) - 16, imin(i16) + 18, imin(i16) + 19, -11e3,          -11e2,
                -11e1,          -11e0,          11e0,           11e1,           11e2,           11e3,           imax(i16) - 19, imax(i16) - 18,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i16), .{
                imin(i16) + 20, imin(i16) + 21, -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,
                12e2,           12e3,           imax(i16) - 21, imax(i16) - 20, imin(i16) + 22, imin(i16) + 23, -13e3,          -13e2,
                -13e1,          -13e0,          13e0,           13e1,           13e2,           13e3,           imax(i16) - 23, imax(i16) - 22,
                imin(i16) + 24, imin(i16) + 25, -14e3,          -14e2,          -14e1,          -14e0,          14e0,           14e1,
                14e2,           14e3,           imax(i16) - 25, imax(i16) - 24, imin(i16) + 26, imin(i16) + 27, -15e3,          -15e2,
                -15e1,          -15e0,          15e0,           15e1,           15e2,           15e3,           imax(i16) - 27, imax(i16) - 26,
                imin(i16) + 28, imin(i16) + 29, -16e3,          -16e2,          -16e1,          -16e0,          16e0,           16e1,
                16e2,           16e3,           imax(i16) - 29, imax(i16) - 28, imin(i16) + 30, imin(i16) + 31, -17e3,          -17e2,
                -17e1,          -17e0,          17e0,           17e1,           17e2,           17e3,           imax(i16) - 31, imax(i16) - 30,
                imin(i16) + 32, imin(i16) + 33, -18e3,          -18e2,          -18e1,          -18e0,          18e0,           18e1,
                18e2,           18e3,           imax(i16) - 33, imax(i16) - 32, imin(i16) + 34, imin(i16) + 35, -19e3,          -19e2,
                -19e1,          -19e0,          19e0,           19e1,           19e2,           19e3,           imax(i16) - 35, imax(i16) - 34,
                imin(i16) + 36, imin(i16) + 37, -12e3,          -21e2,          -21e1,          -21e0,          21e0,           21e1,
                21e2,           21e3,           imax(i16) - 37, imax(i16) - 36, imin(i16) + 38, imin(i16) + 39, -22e3,          -22e2,
                -22e1,          -22e0,          22e0,           22e1,           22e2,           22e3,           imax(i16) - 39, imax(i16) - 38,
                imin(i16) + 40, imin(i16) + 41, -23e3,          -23e2,          -23e1,          -23e0,          23e0,           23e1,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u16), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u16), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u16), .{
                1e2, 1e3, 1e4, imax(u16) - 1,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u16), .{
                imax(u16), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u16) - 3, imax(u16) - 2,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u16), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u16) - 5, imax(u16) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u16) - 7, imax(u16) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u16), .{
                5e2,            5e3,            5e4,            imax(u16) - 9,  imax(u16) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u16) - 11, imax(u16) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u16) - 13, imax(u16) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u16) - 15, imax(u16) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u16) - 17, imax(u16) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u16), .{
                11e2,           11e3,           imax(u16) - 19, imax(u16) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u16) - 21, imax(u16) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u16) - 23, imax(u16) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u16) - 25, imax(u16) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u16) - 27, imax(u16) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u16) - 29, imax(u16) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u16) - 31, imax(u16) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u16) - 33, imax(u16) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u16) - 35, imax(u16) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u16) - 37, imax(u16) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u16) - 39, imax(u16) - 38,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u16), .{
                23e0,           23e1,           23e2,           23e3,           imax(u16) - 41, imax(u16) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u16) - 43, imax(u16) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u16) - 45, imax(u16) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u16) - 47, imax(u16) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u16) - 49, imax(u16) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u16) - 51, imax(u16) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u16) - 53, imax(u16) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u16) - 55, imax(u16) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u16) - 57, imax(u16) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u16) - 59, imax(u16) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u16) - 61, imax(u16) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u16) - 63, imax(u16) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u16) - 65, imax(u16) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u16) - 67, imax(u16) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u16) - 69, imax(u16) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u16) - 71, imax(u16) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u16) - 73, imax(u16) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u16) - 75, imax(u16) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u16) - 77, imax(u16) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u16) - 79, imax(u16) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u16) - 81, imax(u16) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i32), .{
                imin(i32),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i32), .{
                imin(i32) + 1, -1e9,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i32), .{
                -1e8, -1e7, -1e6, -1e5,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i32), .{
                -1e4, -1e3, -1e2, -1e1, -1e0, 0, 1e0, 1e1,
            });
            try testArgs(@Vector(16, f32), @Vector(16, i32), .{
                1e2,           1e3,       1e4,           1e5,           1e6,  1e7,  1e8,  1e9,
                imax(i32) - 1, imax(i32), imin(i32) + 2, imin(i32) + 3, -2e9, -2e8, -2e7, -2e6,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i32), .{
                -2e5,          -2e4,          -2e3,          -2e2,          -2e1, -2e0, 2e0,  2e1,
                2e2,           2e3,           2e4,           2e5,           2e6,  2e7,  2e8,  2e9,
                imax(i32) - 3, imax(i32) - 2, imin(i32) + 4, imin(i32) + 5, -3e8, -3e7, -3e6, -3e5,
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0, 3e0,  3e1,  3e2,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i32), .{
                3e3,           3e4,           3e5,           3e6,           3e7,            3e8,            imax(i32) - 5, imax(i32) - 4,
                imin(i32) + 6, imin(i32) + 7, -4e8,          -4e7,          -4e6,           -4e5,           -4e4,          -4e3,
                -4e2,          -4e1,          -4e0,          4e0,           4e1,            4e2,            4e3,           4e4,
                4e5,           4e6,           4e7,           4e8,           imax(i32) - 7,  imax(i32) - 6,  imin(i32) + 8, imin(i32) + 9,
                -5e8,          -5e7,          -5e6,          -5e5,          -5e4,           -5e3,           -5e2,          -5e1,
                -5e0,          5e0,           5e1,           5e2,           5e3,            5e4,            5e5,           5e6,
                5e7,           5e8,           imax(i32) - 9, imax(i32) - 8, imin(i32) + 10, imin(i32) + 11, -6e8,          -6e7,
                -6e6,          -6e5,          -6e4,          -6e3,          -6e2,           -6e1,           -6e0,          6e0,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i32), .{
                6e1,            6e2,            6e3,            6e4,            6e5,            6e6,            6e7,            6e8,
                imax(i32) - 11, imax(i32) - 10, imin(i32) + 12, imin(i32) + 13, -7e8,           -7e7,           -7e6,           -7e5,
                -7e4,           -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,            7e2,
                7e3,            7e4,            7e5,            7e6,            7e7,            7e8,            imax(i32) - 13, imax(i32) - 12,
                imin(i32) + 14, imin(i32) + 15, -8e8,           -8e7,           -8e6,           -8e5,           -8e4,           -8e3,
                -8e2,           -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            8e4,
                8e5,            8e6,            8e7,            8e8,            imax(i32) - 15, imax(i32) - 14, imin(i32) + 16, imin(i32) + 17,
                -9e8,           -9e7,           -9e6,           -9e5,           -9e4,           -9e3,           -9e2,           -9e1,
                -9e0,           9e0,            9e1,            9e2,            9e3,            9e4,            9e5,            9e6,
                9e7,            9e8,            imax(i32) - 17, imax(i32) - 16, imin(i32) + 18, imin(i32) + 19, -11e8,          -11e7,
                -11e6,          -11e5,          -11e4,          -11e3,          -11e2,          -11e1,          -11e0,          11e0,
                11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,           11e8,
                imax(i32) - 19, imax(i32) - 18, imin(i32) + 20, imin(i32) + 21, -12e8,          -12e7,          -12e6,          -12e5,
                -12e4,          -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           imax(i32) - 21, imax(i32) - 20,
                imin(i32) + 22, imin(i32) + 23, -13e8,          -13e7,          -13e6,          -13e5,          -13e4,          -13e3,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u32), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u32), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u32), .{
                1e2, 1e3, 1e4, imax(u32) - 1,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u32), .{
                imax(u32), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u32) - 3, imax(u32) - 2,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u32), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u32) - 5, imax(u32) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u32) - 7, imax(u32) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u32), .{
                5e2,            5e3,            5e4,            imax(u32) - 9,  imax(u32) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u32) - 11, imax(u32) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u32) - 13, imax(u32) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u32) - 15, imax(u32) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u32) - 17, imax(u32) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u32), .{
                11e2,           11e3,           imax(u32) - 19, imax(u32) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u32) - 21, imax(u32) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u32) - 23, imax(u32) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u32) - 25, imax(u32) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u32) - 27, imax(u32) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u32) - 29, imax(u32) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u32) - 31, imax(u32) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u32) - 33, imax(u32) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u32) - 35, imax(u32) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u32) - 37, imax(u32) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u32) - 39, imax(u32) - 38,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u32), .{
                23e0,           23e1,           23e2,           23e3,           imax(u32) - 41, imax(u32) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u32) - 43, imax(u32) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u32) - 45, imax(u32) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u32) - 47, imax(u32) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u32) - 49, imax(u32) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u32) - 51, imax(u32) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u32) - 53, imax(u32) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u32) - 55, imax(u32) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u32) - 57, imax(u32) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u32) - 59, imax(u32) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u32) - 61, imax(u32) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u32) - 63, imax(u32) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u32) - 65, imax(u32) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u32) - 67, imax(u32) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u32) - 69, imax(u32) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u32) - 71, imax(u32) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u32) - 73, imax(u32) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u32) - 75, imax(u32) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u32) - 77, imax(u32) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u32) - 79, imax(u32) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u32) - 81, imax(u32) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i64), .{
                imin(i64),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i64), .{
                imin(i64) + 1, -1e18,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i64), .{
                -1e17, -1e16, -1e15, -1e14,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i64), .{
                -1e13, -1e12, -1e11, -1e10, -1e9, -1e8, -1e7, -1e6,
            });
            try testArgs(@Vector(16, f32), @Vector(16, i64), .{
                -1e5, -1e4, -1e3, -1e2, -1e1, -1e0, 0,   1e0,
                1e1,  1e2,  1e3,  1e4,  1e5,  1e6,  1e7, 1e8,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i64), .{
                1e9,   1e10,  1e11,          1e12,      1e13,          1e14,          1e15,  1e16,
                1e17,  1e18,  imax(i64) - 1, imax(i64), imin(i64) + 2, imin(i64) + 3, -2e18, -2e17,
                -2e16, -2e15, -2e14,         -2e13,     -2e12,         -2e11,         -2e10, -2e9,
                -2e8,  -2e7,  -2e6,          -2e5,      -2e4,          -2e3,          -2e2,  -2e1,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i64), .{
                -2e0,  2e0,   2e1,   2e2,   2e3,           2e4,           2e5,           2e6,
                2e7,   2e8,   2e9,   2e10,  2e11,          2e12,          2e13,          2e14,
                2e15,  2e16,  2e17,  2e18,  imax(i64) - 3, imax(i64) - 2, imin(i64) + 4, imin(i64) + 5,
                -3e18, -3e17, -3e16, -3e15, -3e14,         -3e13,         -3e12,         -3e11,
                -3e10, -3e9,  -3e8,  -3e7,  -3e6,          -3e5,          -3e4,          -3e3,
                -3e2,  -3e1,  -3e0,  3e0,   3e1,           3e2,           3e3,           3e4,
                3e5,   3e6,   3e7,   3e8,   3e9,           3e10,          3e11,          3e12,
                3e13,  3e14,  3e15,  3e16,  3e17,          3e18,          imax(i64) - 5, imax(i64) - 4,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i64), .{
                imin(i64) + 6, imin(i64) + 7, -4e18,         -4e17,         -4e16,          -4e15,          -4e14,          -4e13,
                -4e12,         -4e11,         -4e10,         -4e9,          -4e8,           -4e7,           -4e6,           -4e5,
                -4e4,          -4e3,          -4e2,          -4e1,          -4e0,           4e0,            4e1,            4e2,
                4e3,           4e4,           4e5,           4e6,           4e7,            4e8,            4e9,            4e10,
                4e11,          4e12,          4e13,          4e14,          4e15,           4e16,           4e17,           4e18,
                imax(i64) - 7, imax(i64) - 6, imin(i64) + 8, imin(i64) + 9, -5e18,          -5e17,          -5e16,          -5e15,
                -5e14,         -5e13,         -5e12,         -5e11,         -5e10,          -5e9,           -5e8,           -5e7,
                -5e6,          -5e5,          -5e4,          -5e3,          -5e2,           -5e1,           -5e0,           5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,            5e6,            5e7,            5e8,
                5e9,           5e10,          5e11,          5e12,          5e13,           5e14,           5e15,           5e16,
                5e17,          5e18,          imax(i64) - 9, imax(i64) - 8, imin(i64) + 10, imin(i64) + 11, -6e18,          -6e17,
                -6e16,         -6e15,         -6e14,         -6e13,         -6e12,          -6e11,          -6e10,          -6e9,
                -6e8,          -6e7,          -6e6,          -6e5,          -6e4,           -6e3,           -6e2,           -6e1,
                -6e0,          6e0,           6e1,           6e2,           6e3,            6e4,            6e5,            6e6,
                6e7,           6e8,           6e9,           6e10,          6e11,           6e12,           6e13,           6e14,
                6e15,          6e16,          6e17,          6e18,          imax(i64) - 11, imax(i64) - 10, imin(i64) + 12, imin(i64) + 13,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u64), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u64), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u64), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u64), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u64), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, imax(u64) - 1, imax(u64),
                2e0,  2e1,  2e2,  2e3,  2e4,  2e5,  2e6,           2e7,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u64), .{
                2e8,  2e9,  2e10, 2e11,          2e12,          2e13, 2e14, 2e15,
                2e16, 2e17, 2e18, imax(u64) - 3, imax(u64) - 2, 3e0,  3e1,  3e2,
                3e3,  3e4,  3e5,  3e6,           3e7,           3e8,  3e9,  3e10,
                3e11, 3e12, 3e13, 3e14,          3e15,          3e16, 3e17, 3e18,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u64), .{
                imax(u64) - 5, imax(u64) - 4, 4e0,           4e1,           4e2,  4e3,           4e4,           4e5,
                4e6,           4e7,           4e8,           4e9,           4e10, 4e11,          4e12,          4e13,
                4e14,          4e15,          4e16,          4e17,          4e18, imax(u64) - 7, imax(u64) - 6, 5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,  5e6,           5e7,           5e8,
                5e9,           5e10,          5e11,          5e12,          5e13, 5e14,          5e15,          5e16,
                5e17,          5e18,          imax(u64) - 9, imax(u64) - 8, 6e0,  6e1,           6e2,           6e3,
                6e4,           6e5,           6e6,           6e7,           6e8,  6e9,           6e10,          6e11,
                6e12,          6e13,          6e14,          6e15,          6e16, 6e17,          6e18,          imax(u64) - 11,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u64), .{
                imax(u64) - 10, 7e0,            7e1,            7e2,            7e3,            7e4,            7e5,            7e6,
                7e7,            7e8,            7e9,            7e10,           7e11,           7e12,           7e13,           7e14,
                7e15,           7e16,           7e17,           7e18,           imax(u64) - 13, imax(u64) - 12, 8e0,            8e1,
                8e2,            8e3,            8e4,            8e5,            8e6,            8e7,            8e8,            8e9,
                8e10,           8e11,           8e12,           8e13,           8e14,           8e15,           8e16,           8e17,
                8e18,           imax(u64) - 15, imax(u64) - 14, 9e0,            9e1,            9e2,            9e3,            9e4,
                9e5,            9e6,            9e7,            9e8,            9e9,            9e10,           9e11,           9e12,
                9e13,           9e14,           9e15,           9e16,           9e17,           9e18,           imax(u64) - 17, imax(u64) - 16,
                11e0,           11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,
                11e8,           11e9,           11e10,          11e11,          11e12,          11e13,          11e14,          11e15,
                11e16,          11e17,          11e18,          imax(u64) - 19, imax(u64) - 18, 12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           12e9,           12e10,
                12e11,          12e12,          12e13,          12e14,          12e15,          12e16,          12e17,          12e18,
                imax(u64) - 21, imax(u64) - 20, 13e0,           13e1,           13e2,           13e3,           13e4,           13e5,
                13e6,           13e7,           13e8,           13e9,           13e10,          13e11,          13e12,          13e13,
                13e14,          13e15,          13e16,          13e17,          13e18,          imax(u64) - 23, imax(u64) - 22, 14e0,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i128), .{
                imin(i128),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i128), .{
                imin(i128) + 1, -1e38,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i128), .{
                -1e37, -1e36, -1e35, -1e34,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i128), .{
                -1e33, -1e32, -1e31, -1e30, -1e29, -1e28, -1e27, -1e26,
            });
            try testArgs(@Vector(16, f32), @Vector(16, i128), .{
                -1e25, -1e24, -1e23, -1e22, -1e21, -1e20, -1e19, -1e18,
                -1e17, -1e16, -1e15, -1e14, -1e13, -1e12, -1e11, -1e10,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i128), .{
                -1e9, -1e8, -1e7, -1e6, -1e5, -1e4, -1e3, -1e2,
                -1e1, -1e0, 0,    1e0,  1e1,  1e2,  1e3,  1e4,
                1e5,  1e6,  1e7,  1e8,  1e9,  1e10, 1e11, 1e12,
                1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i128), .{
                1e21,  1e22,  1e23,           1e24,       1e25,           1e26,           1e27,  1e28,
                1e29,  1e30,  1e31,           1e32,       1e33,           1e34,           1e35,  1e36,
                1e37,  1e38,  imax(i128) - 1, imax(i128), imin(i128) + 2, imin(i128) + 3, -2e37, -2e36,
                -2e35, -2e34, -2e33,          -2e32,      -2e31,          -2e30,          -2e29, -2e28,
                -2e27, -2e26, -2e25,          -2e24,      -2e23,          -2e22,          -2e21, -2e20,
                -2e19, -2e18, -2e17,          -2e16,      -2e15,          -2e14,          -2e13, -2e12,
                -2e11, -2e10, -2e9,           -2e8,       -2e7,           -2e6,           -2e5,  -2e4,
                -2e3,  -2e2,  -2e1,           -2e0,       2e0,            2e1,            2e2,   2e3,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i128), .{
                2e4,   2e5,   2e6,            2e7,            2e8,            2e9,            2e10,  2e11,
                2e12,  2e13,  2e14,           2e15,           2e16,           2e17,           2e18,  2e19,
                2e20,  2e21,  2e22,           2e23,           2e24,           2e25,           2e26,  2e27,
                2e28,  2e29,  2e30,           2e31,           2e32,           2e33,           2e34,  2e35,
                2e36,  2e37,  imax(i128) - 3, imax(i128) - 2, imin(i128) + 4, imin(i128) + 5, -3e37, -3e36,
                -3e35, -3e34, -3e33,          -3e32,          -3e31,          -3e30,          -3e29, -3e28,
                -3e27, -3e26, -3e25,          -3e24,          -3e23,          -3e22,          -3e21, -3e20,
                -3e19, -3e18, -3e17,          -3e16,          -3e15,          -3e14,          -3e13, -3e12,
                -3e11, -3e10, -3e9,           -3e8,           -3e7,           -3e6,           -3e5,  -3e4,
                -3e3,  -3e2,  -3e1,           -3e0,           3e0,            3e1,            3e2,   3e3,
                3e4,   3e5,   3e6,            3e7,            3e8,            3e9,            3e10,  3e11,
                3e12,  3e13,  3e14,           3e15,           3e16,           3e17,           3e18,  3e19,
                3e20,  3e21,  3e22,           3e23,           3e24,           3e25,           3e26,  3e27,
                3e28,  3e29,  3e30,           3e31,           3e32,           3e33,           3e34,  3e35,
                3e36,  3e37,  imax(i128) - 5, imax(i128) - 4, imin(i128) + 6, imin(i128) + 7, -4e37, -4e36,
                -4e35, -4e34, -4e33,          -4e32,          -4e31,          -4e30,          -4e29, -4e28,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u128), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u128), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u128), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u128), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u128), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u128), .{
                1e30, 1e31,           1e32,       1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, imax(u128) - 1, imax(u128), 2e0,  2e1,  2e2,  2e3,  2e4,
                2e5,  2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12,
                2e13, 2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u128), .{
                2e21, 2e22, 2e23,           2e24,           2e25,           2e26, 2e27, 2e28,
                2e29, 2e30, 2e31,           2e32,           2e33,           2e34, 2e35, 2e36,
                2e37, 2e38, imax(u128) - 3, imax(u128) - 2, 3e0,            3e1,  3e2,  3e3,
                3e4,  3e5,  3e6,            3e7,            3e8,            3e9,  3e10, 3e11,
                3e12, 3e13, 3e14,           3e15,           3e16,           3e17, 3e18, 3e19,
                3e20, 3e21, 3e22,           3e23,           3e24,           3e25, 3e26, 3e27,
                3e28, 3e29, 3e30,           3e31,           3e32,           3e33, 3e34, 3e35,
                3e36, 3e37, 3e38,           imax(u128) - 5, imax(u128) - 4, 4e0,  4e1,  4e2,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u128), .{
                4e3,  4e4,  4e5,  4e6,             4e7,             4e8,  4e9,  4e10,
                4e11, 4e12, 4e13, 4e14,            4e15,            4e16, 4e17, 4e18,
                4e19, 4e20, 4e21, 4e22,            4e23,            4e24, 4e25, 4e26,
                4e27, 4e28, 4e29, 4e30,            4e31,            4e32, 4e33, 4e34,
                4e35, 4e36, 4e37, imax(u128) - 7,  imax(u128) - 6,  5e0,  5e1,  5e2,
                5e3,  5e4,  5e5,  5e6,             5e7,             5e8,  5e9,  5e10,
                5e11, 5e12, 5e13, 5e14,            5e15,            5e16, 5e17, 5e18,
                5e19, 5e20, 5e21, 5e22,            5e23,            5e24, 5e25, 5e26,
                5e27, 5e28, 5e29, 5e30,            5e31,            5e32, 5e33, 5e34,
                5e35, 5e36, 5e37, imax(u128) - 9,  imax(u128) - 8,  6e0,  6e1,  6e2,
                6e3,  6e4,  6e5,  6e6,             6e7,             6e8,  6e9,  6e10,
                6e11, 6e12, 6e13, 6e14,            6e15,            6e16, 6e17, 6e18,
                6e19, 6e20, 6e21, 6e22,            6e23,            6e24, 6e25, 6e26,
                6e27, 6e28, 6e29, 6e30,            6e31,            6e32, 6e33, 6e34,
                6e35, 6e36, 6e37, imax(u128) - 11, imax(u128) - 10, 7e0,  7e1,  7e2,
                7e3,  7e4,  7e5,  7e6,             7e7,             7e8,  7e9,  7e10,
            });

            try testArgs(@Vector(1, f32), @Vector(1, i256), .{
                imin(i256),
            });
            try testArgs(@Vector(2, f32), @Vector(2, i256), .{
                imin(i256) + 1, -1e76,
            });
            try testArgs(@Vector(4, f32), @Vector(4, i256), .{
                -1e75, -1e74, -1e73, -1e72,
            });
            try testArgs(@Vector(8, f32), @Vector(8, i256), .{
                -1e71, -1e70, -1e69, -1e68, -1e67, -1e66, -1e65, -1e64,
            });
            try testArgs(@Vector(16, f32), @Vector(16, i256), .{
                -1e63, -1e62, -1e61, -1e60, -1e59, -1e58, -1e57, -1e56,
                -1e55, -1e54, -1e53, -1e52, -1e51, -1e50, -1e49, -1e48,
            });
            try testArgs(@Vector(32, f32), @Vector(32, i256), .{
                -1e47, -1e46, -1e45, -1e44, -1e43, -1e42, -1e41, -1e40,
                -1e39, -1e38, -1e37, -1e36, -1e35, -1e34, -1e33, -1e32,
                -1e31, -1e30, -1e29, -1e28, -1e27, -1e26, -1e25, -1e24,
                -1e23, -1e22, -1e21, -1e20, -1e19, -1e18, -1e17, -1e16,
            });
            try testArgs(@Vector(64, f32), @Vector(64, i256), .{
                -1e15, -1e14, -1e13, -1e12, -1e11, -1e10, -1e9, -1e8,
                -1e7,  -1e6,  -1e5,  -1e4,  -1e3,  -1e2,  -1e1, -1e0,
                0,     1e0,   1e1,   1e2,   1e3,   1e4,   1e5,  1e6,
                1e7,   1e8,   1e9,   1e10,  1e11,  1e12,  1e13, 1e14,
                1e15,  1e16,  1e17,  1e18,  1e19,  1e20,  1e21, 1e22,
                1e23,  1e24,  1e25,  1e26,  1e27,  1e28,  1e29, 1e30,
                1e31,  1e32,  1e33,  1e34,  1e35,  1e36,  1e37, 1e38,
                1e39,  1e40,  1e41,  1e42,  1e43,  1e44,  1e45, 1e46,
            });
            try testArgs(@Vector(128, f32), @Vector(128, i256), .{
                1e47,           1e48,           1e49,  1e50,  1e51,  1e52,  1e53,           1e54,
                1e55,           1e56,           1e57,  1e58,  1e59,  1e60,  1e61,           1e62,
                1e63,           1e64,           1e65,  1e66,  1e67,  1e68,  1e69,           1e70,
                1e71,           1e72,           1e73,  1e74,  1e75,  1e76,  imax(i256) - 1, imax(i256),
                imin(i256) + 2, imin(i256) + 3, -2e76, -2e75, -2e74, -2e73, -2e72,          -2e71,
                -2e70,          -2e69,          -2e68, -2e67, -2e66, -2e65, -2e64,          -2e63,
                -2e62,          -2e61,          -2e60, -2e59, -2e58, -2e57, -2e56,          -2e55,
                -2e54,          -2e53,          -2e52, -2e51, -2e50, -2e49, -2e48,          -2e47,
                -2e46,          -2e45,          -2e44, -2e43, -2e42, -2e41, -2e40,          -2e39,
                -2e38,          -2e37,          -2e36, -2e35, -2e34, -2e33, -2e32,          -2e31,
                -2e30,          -2e29,          -2e28, -2e27, -2e26, -2e25, -2e24,          -2e23,
                -2e22,          -2e21,          -2e20, -2e19, -2e18, -2e17, -2e16,          -2e15,
                -2e14,          -2e13,          -2e12, -2e11, -2e10, -2e9,  -2e8,           -2e7,
                -2e6,           -2e5,           -2e4,  -2e3,  -2e2,  -2e1,  -2e0,           2e0,
                2e1,            2e2,            2e3,   2e4,   2e5,   2e6,   2e7,            2e8,
                2e9,            2e10,           2e11,  2e12,  2e13,  2e14,  2e15,           2e16,
            });

            try testArgs(@Vector(1, f32), @Vector(1, u256), .{
                0,
            });
            try testArgs(@Vector(2, f32), @Vector(2, u256), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f32), @Vector(4, u256), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f32), @Vector(8, u256), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f32), @Vector(16, u256), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f32), @Vector(32, u256), .{
                1e30, 1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, 1e39, 1e40, 1e41, 1e42, 1e43, 1e44, 1e45,
                1e46, 1e47, 1e48, 1e49, 1e50, 1e51, 1e52, 1e53,
                1e54, 1e55, 1e56, 1e57, 1e58, 1e59, 1e60, 1e61,
            });
            try testArgs(@Vector(64, f32), @Vector(64, u256), .{
                1e62,           1e63,       1e64, 1e65, 1e66, 1e67, 1e68, 1e69,
                1e70,           1e71,       1e72, 1e73, 1e74, 1e75, 1e76, 1e77,
                imax(u256) - 1, imax(u256), 2e0,  2e1,  2e2,  2e3,  2e4,  2e5,
                2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12, 2e13,
                2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20, 2e21,
                2e22,           2e23,       2e24, 2e25, 2e26, 2e27, 2e28, 2e29,
                2e30,           2e31,       2e32, 2e33, 2e34, 2e35, 2e36, 2e37,
                2e38,           2e39,       2e40, 2e41, 2e42, 2e43, 2e44, 2e45,
            });
            try testArgs(@Vector(128, f32), @Vector(128, u256), .{
                2e46,           2e47, 2e48, 2e49, 2e50, 2e51, 2e52,           2e53,
                2e54,           2e55, 2e56, 2e57, 2e58, 2e59, 2e60,           2e61,
                2e62,           2e63, 2e64, 2e65, 2e66, 2e67, 2e68,           2e69,
                2e70,           2e71, 2e72, 2e73, 2e74, 2e75, 2e76,           imax(u256) - 3,
                imax(u256) - 2, 3e0,  3e1,  3e2,  3e3,  3e4,  3e5,            3e6,
                3e7,            3e8,  3e9,  3e10, 3e11, 3e12, 3e13,           3e14,
                3e15,           3e16, 3e17, 3e18, 3e19, 3e20, 3e21,           3e22,
                3e23,           3e24, 3e25, 3e26, 3e27, 3e28, 3e29,           3e30,
                3e31,           3e32, 3e33, 3e34, 3e35, 3e36, 3e37,           3e38,
                3e39,           3e40, 3e41, 3e42, 3e43, 3e44, 3e45,           3e46,
                3e47,           3e48, 3e49, 3e50, 3e51, 3e52, 3e53,           3e54,
                3e55,           3e56, 3e57, 3e58, 3e59, 3e60, 3e61,           3e62,
                3e63,           3e64, 3e65, 3e66, 3e67, 3e68, 3e69,           3e70,
                3e71,           3e72, 3e73, 3e74, 3e75, 3e76, imax(u256) - 5, imax(u256) - 4,
                4e0,            4e1,  4e2,  4e3,  4e4,  4e5,  4e6,            4e7,
                4e8,            4e9,  4e10, 4e11, 4e12, 4e13, 4e14,           4e15,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i8), .{
                imin(i8),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i8), .{
                imin(i8) + 1, -1e2,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i8), .{
                -1e1, -1e0, 0, 1e0,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i8), .{
                1e1, 1e2, imax(i8) - 1, imax(i8), imin(i8) + 2, imin(i8) + 3, -2e1, -2e0,
            });
            try testArgs(@Vector(16, f64), @Vector(16, i8), .{
                2e0, 2e1, imax(i8) - 3, imax(i8) - 2, imin(i8) + 4, imin(i8) + 5, -3e1, -3e0,
                3e0, 3e1, imax(i8) - 5, imax(i8) - 4, imin(i8) + 6, imin(i8) + 7, -4e1, -4e0,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i8), .{
                4e0, 4e1, imax(i8) - 7,  imax(i8) - 6,  imin(i8) + 8,  imin(i8) + 9,  -5e1, -5e0,
                5e0, 5e1, imax(i8) - 9,  imax(i8) - 8,  imin(i8) + 10, imin(i8) + 11, -6e1, -6e0,
                6e0, 6e1, imax(i8) - 11, imax(i8) - 10, imin(i8) + 12, imin(i8) + 13, -7e1, -7e0,
                7e0, 7e1, imax(i8) - 13, imax(i8) - 12, imin(i8) + 14, imin(i8) + 15, -8e1, -8e0,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i8), .{
                8e0,           8e1,           imax(i8) - 15, imax(i8) - 14, imin(i8) + 16, imin(i8) + 17, -9e1,          -9e0,
                9e0,           9e1,           imax(i8) - 17, imax(i8) - 16, imin(i8) + 18, imin(i8) + 19, -11e1,         -11e0,
                11e0,          11e1,          imax(i8) - 19, imax(i8) - 18, imin(i8) + 20, imin(i8) + 21, -12e1,         -12e0,
                12e0,          12e1,          imax(i8) - 21, imax(i8) - 20, imin(i8) + 22, imin(i8) + 23, -13e0,         13e0,
                imax(i8) - 23, imax(i8) - 22, imin(i8) + 24, imin(i8) + 25, -14e0,         14e0,          imax(i8) - 25, imax(i8) - 24,
                imin(i8) + 26, imin(i8) + 27, -15e0,         15e0,          imax(i8) - 27, imax(i8) - 26, imin(i8) + 28, imin(i8) + 29,
                -16e0,         16e0,          imax(i8) - 29, imax(i8) - 28, imin(i8) + 30, imin(i8) + 31, -17e0,         17e0,
                imax(i8) - 31, imax(i8) - 30, imin(i8) + 32, imin(i8) + 33, -18e0,         18e0,          imax(i8) - 33, imax(i8) - 32,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i8), .{
                imin(i8) + 34, imin(i8) + 35, -19e0,         19e0,          imax(i8) - 35, imax(i8) - 34, imin(i8) + 36, imin(i8) + 37,
                -21e0,         21e0,          imax(i8) - 37, imax(i8) - 36, imin(i8) + 38, imin(i8) + 39, -22e0,         22e0,
                imax(i8) - 39, imax(i8) - 38, imin(i8) + 40, imin(i8) + 41, -23e0,         23e0,          imax(i8) - 41, imax(i8) - 40,
                imin(i8) + 42, imin(i8) + 43, -24e0,         24e0,          imax(i8) - 43, imax(i8) - 42, imin(i8) + 44, imin(i8) + 45,
                -25e0,         25e0,          imax(i8) - 45, imax(i8) - 44, imin(i8) + 46, imin(i8) + 47, -26e0,         26e0,
                imax(i8) - 47, imax(i8) - 46, imin(i8) + 48, imin(i8) + 49, -27e0,         27e0,          imax(i8) - 49, imax(i8) - 48,
                imin(i8) + 50, imin(i8) + 51, -28e0,         28e0,          imax(i8) - 51, imax(i8) - 50, imin(i8) + 52, imin(i8) + 53,
                -29e0,         29e0,          imax(i8) - 53, imax(i8) - 52, imin(i8) + 54, imin(i8) + 55, -31e0,         31e0,
                imax(i8) - 55, imax(i8) - 54, imin(i8) + 56, imin(i8) + 57, -32e0,         32e0,          imax(i8) - 57, imax(i8) - 56,
                imin(i8) + 58, imin(i8) + 59, -33e0,         33e0,          imax(i8) - 59, imax(i8) - 58, imin(i8) + 60, imin(i8) + 61,
                -34e0,         34e0,          imax(i8) - 61, imax(i8) - 60, imin(i8) + 62, imin(i8) + 63, -35e0,         35e0,
                imax(i8) - 63, imax(i8) - 62, imin(i8) + 64, imin(i8) + 65, -36e0,         36e0,          imax(i8) - 65, imax(i8) - 64,
                imin(i8) + 66, imin(i8) + 67, -37e0,         37e0,          imax(i8) - 67, imax(i8) - 66, imin(i8) + 68, imin(i8) + 69,
                -38e0,         38e0,          imax(i8) - 69, imax(i8) - 68, imin(i8) + 70, imin(i8) + 71, -39e0,         39e0,
                imax(i8) - 71, imax(i8) - 70, imin(i8) + 72, imin(i8) + 73, -41e0,         41e0,          imax(i8) - 73, imax(i8) - 72,
                imin(i8) + 74, imin(i8) + 75, -42e0,         42e0,          imax(i8) - 75, imax(i8) - 74, imin(i8) + 76, imin(i8) + 77,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u8), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u8), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u8), .{
                1e2, imax(u8) - 1, imax(u8), 2e0,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u8), .{
                2e1, 2e2, imax(u8) - 3, imax(u8) - 2, 3e0, 3e1, imax(u8) - 5, imax(u8) - 4,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u8), .{
                imax(u8) - 7,  imax(u8) - 6,  5e0, 5e1, imax(u8) - 9,  imax(u8) - 8,  6e0, 6e1,
                imax(u8) - 11, imax(u8) - 10, 7e0, 7e1, imax(u8) - 13, imax(u8) - 12, 8e0, 8e1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u8), .{
                imax(u8) - 15, imax(u8) - 14, 9e0,  9e1,  imax(u8) - 17, imax(u8) - 16, 11e0, 11e1,
                imax(u8) - 19, imax(u8) - 18, 12e0, 12e1, imax(u8) - 21, imax(u8) - 20, 13e0, 13e1,
                imax(u8) - 23, imax(u8) - 22, 14e0, 14e1, imax(u8) - 25, imax(u8) - 24, 15e0, 15e1,
                imax(u8) - 27, imax(u8) - 26, 16e0, 16e1, imax(u8) - 29, imax(u8) - 28, 17e0, 17e1,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u8), .{
                imax(u8) - 31, imax(u8) - 30, 18e0,          18e1,          imax(u8) - 33, imax(u8) - 32, 19e0,          19e1,
                imax(u8) - 35, imax(u8) - 34, 21e0,          21e1,          imax(u8) - 37, imax(u8) - 36, 22e0,          22e1,
                imax(u8) - 39, imax(u8) - 38, 23e0,          23e1,          imax(u8) - 41, imax(u8) - 40, 24e0,          24e1,
                imax(u8) - 43, imax(u8) - 42, 25e0,          25e1,          imax(u8) - 45, imax(u8) - 44, 26e0,          imax(u8) - 47,
                imax(u8) - 46, 27e0,          imax(u8) - 49, imax(u8) - 48, 28e0,          imax(u8) - 51, imax(u8) - 50, 29e0,
                imax(u8) - 53, imax(u8) - 52, 31e0,          imax(u8) - 55, imax(u8) - 54, 32e0,          imax(u8) - 57, imax(u8) - 56,
                33e0,          imax(u8) - 59, imax(u8) - 58, 34e0,          imax(u8) - 61, imax(u8) - 60, 35e0,          imax(u8) - 63,
                imax(u8) - 62, 36e0,          imax(u8) - 65, imax(u8) - 64, 37e0,          imax(u8) - 67, imax(u8) - 66, 38e0,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u8), .{
                imax(u8) - 69,  imax(u8) - 68,  39e0,           imax(u8) - 71,  imax(u8) - 70,  41e0,           imax(u8) - 73,  imax(u8) - 72,
                42e0,           imax(u8) - 75,  imax(u8) - 74,  43e0,           imax(u8) - 77,  imax(u8) - 76,  44e0,           imax(u8) - 79,
                imax(u8) - 78,  45e0,           imax(u8) - 81,  imax(u8) - 80,  46e0,           imax(u8) - 83,  imax(u8) - 82,  47e0,
                imax(u8) - 85,  imax(u8) - 84,  48e0,           imax(u8) - 87,  imax(u8) - 86,  49e0,           imax(u8) - 89,  imax(u8) - 88,
                51e0,           imax(u8) - 91,  imax(u8) - 90,  52e0,           imax(u8) - 93,  imax(u8) - 92,  53e0,           imax(u8) - 95,
                imax(u8) - 94,  54e0,           imax(u8) - 97,  imax(u8) - 96,  55e0,           imax(u8) - 99,  imax(u8) - 98,  56e0,
                imax(u8) - 101, imax(u8) - 100, 57e0,           imax(u8) - 103, imax(u8) - 102, 58e0,           imax(u8) - 105, imax(u8) - 104,
                59e0,           imax(u8) - 107, imax(u8) - 106, 61e0,           imax(u8) - 109, imax(u8) - 108, 62e0,           imax(u8) - 111,
                imax(u8) - 110, 63e0,           imax(u8) - 113, imax(u8) - 112, 64e0,           imax(u8) - 115, imax(u8) - 114, 65e0,
                imax(u8) - 117, imax(u8) - 116, 66e0,           imax(u8) - 119, imax(u8) - 118, 67e0,           imax(u8) - 121, imax(u8) - 120,
                68e0,           imax(u8) - 123, imax(u8) - 122, 69e0,           imax(u8) - 125, imax(u8) - 124, 71e0,           imax(u8) - 127,
                imax(u8) - 126, 72e0,           imax(u8) - 129, imax(u8) - 128, 73e0,           imax(u8) - 131, imax(u8) - 130, 74e0,
                imax(u8) - 133, imax(u8) - 132, 75e0,           imax(u8) - 135, imax(u8) - 134, 76e0,           imax(u8) - 137, imax(u8) - 136,
                77e0,           imax(u8) - 139, imax(u8) - 138, 78e0,           imax(u8) - 141, imax(u8) - 140, 79e0,           imax(u8) - 143,
                imax(u8) - 142, 81e0,           imax(u8) - 145, imax(u8) - 144, 82e0,           imax(u8) - 147, imax(u8) - 146, 83e0,
                imax(u8) - 149, imax(u8) - 148, 84e0,           imax(u8) - 151, imax(u8) - 150, 85e0,           imax(u8) - 153, imax(u8) - 152,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i16), .{
                imin(i16),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i16), .{
                imin(i16) + 1, -1e4,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i16), .{
                -1e3, -1e2, -1e1, -1e0,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i16), .{
                0, 1e0, 1e1, 1e2, 1e3, 1e4, imax(i16) - 1, imax(i16),
            });
            try testArgs(@Vector(16, f64), @Vector(16, i16), .{
                imin(i16) + 2, imin(i16) + 3, -2e4, -2e3, -2e2,          -2e1,          -2e0,          2e0,
                2e1,           2e2,           2e3,  2e4,  imax(i16) - 3, imax(i16) - 2, imin(i16) + 4, imin(i16) + 5,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i16), .{
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0,          3e0,           3e1,           3e2,
                3e3,           3e4,           imax(i16) - 5, imax(i16) - 4, imin(i16) + 6, imin(i16) + 7, -4e3,          -4e2,
                -4e1,          -4e0,          4e0,           4e1,           4e2,           4e3,           imax(i16) - 7, imax(i16) - 6,
                imin(i16) + 8, imin(i16) + 9, -5e3,          -5e2,          -5e1,          -5e0,          5e0,           5e1,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i16), .{
                5e2,            5e3,            imax(i16) - 9,  imax(i16) - 8,  imin(i16) + 10, imin(i16) + 11, -6e3,           -6e2,
                -6e1,           -6e0,           6e0,            6e1,            6e2,            6e3,            imax(i16) - 11, imax(i16) - 10,
                imin(i16) + 12, imin(i16) + 13, -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,
                7e2,            7e3,            imax(i16) - 13, imax(i16) - 12, imin(i16) + 14, imin(i16) + 15, -8e3,           -8e2,
                -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            imax(i16) - 15, imax(i16) - 14,
                imin(i16) + 16, imin(i16) + 17, -9e3,           -9e2,           -9e1,           -9e0,           9e0,            9e1,
                9e2,            9e3,            imax(i16) - 17, imax(i16) - 16, imin(i16) + 18, imin(i16) + 19, -11e3,          -11e2,
                -11e1,          -11e0,          11e0,           11e1,           11e2,           11e3,           imax(i16) - 19, imax(i16) - 18,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i16), .{
                imin(i16) + 20, imin(i16) + 21, -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,
                12e2,           12e3,           imax(i16) - 21, imax(i16) - 20, imin(i16) + 22, imin(i16) + 23, -13e3,          -13e2,
                -13e1,          -13e0,          13e0,           13e1,           13e2,           13e3,           imax(i16) - 23, imax(i16) - 22,
                imin(i16) + 24, imin(i16) + 25, -14e3,          -14e2,          -14e1,          -14e0,          14e0,           14e1,
                14e2,           14e3,           imax(i16) - 25, imax(i16) - 24, imin(i16) + 26, imin(i16) + 27, -15e3,          -15e2,
                -15e1,          -15e0,          15e0,           15e1,           15e2,           15e3,           imax(i16) - 27, imax(i16) - 26,
                imin(i16) + 28, imin(i16) + 29, -16e3,          -16e2,          -16e1,          -16e0,          16e0,           16e1,
                16e2,           16e3,           imax(i16) - 29, imax(i16) - 28, imin(i16) + 30, imin(i16) + 31, -17e3,          -17e2,
                -17e1,          -17e0,          17e0,           17e1,           17e2,           17e3,           imax(i16) - 31, imax(i16) - 30,
                imin(i16) + 32, imin(i16) + 33, -18e3,          -18e2,          -18e1,          -18e0,          18e0,           18e1,
                18e2,           18e3,           imax(i16) - 33, imax(i16) - 32, imin(i16) + 34, imin(i16) + 35, -19e3,          -19e2,
                -19e1,          -19e0,          19e0,           19e1,           19e2,           19e3,           imax(i16) - 35, imax(i16) - 34,
                imin(i16) + 36, imin(i16) + 37, -12e3,          -21e2,          -21e1,          -21e0,          21e0,           21e1,
                21e2,           21e3,           imax(i16) - 37, imax(i16) - 36, imin(i16) + 38, imin(i16) + 39, -22e3,          -22e2,
                -22e1,          -22e0,          22e0,           22e1,           22e2,           22e3,           imax(i16) - 39, imax(i16) - 38,
                imin(i16) + 40, imin(i16) + 41, -23e3,          -23e2,          -23e1,          -23e0,          23e0,           23e1,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u16), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u16), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u16), .{
                1e2, 1e3, 1e4, imax(u16) - 1,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u16), .{
                imax(u16), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u16) - 3, imax(u16) - 2,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u16), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u16) - 5, imax(u16) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u16) - 7, imax(u16) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u16), .{
                5e2,            5e3,            5e4,            imax(u16) - 9,  imax(u16) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u16) - 11, imax(u16) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u16) - 13, imax(u16) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u16) - 15, imax(u16) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u16) - 17, imax(u16) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u16), .{
                11e2,           11e3,           imax(u16) - 19, imax(u16) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u16) - 21, imax(u16) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u16) - 23, imax(u16) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u16) - 25, imax(u16) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u16) - 27, imax(u16) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u16) - 29, imax(u16) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u16) - 31, imax(u16) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u16) - 33, imax(u16) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u16) - 35, imax(u16) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u16) - 37, imax(u16) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u16) - 39, imax(u16) - 38,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u16), .{
                23e0,           23e1,           23e2,           23e3,           imax(u16) - 41, imax(u16) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u16) - 43, imax(u16) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u16) - 45, imax(u16) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u16) - 47, imax(u16) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u16) - 49, imax(u16) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u16) - 51, imax(u16) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u16) - 53, imax(u16) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u16) - 55, imax(u16) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u16) - 57, imax(u16) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u16) - 59, imax(u16) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u16) - 61, imax(u16) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u16) - 63, imax(u16) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u16) - 65, imax(u16) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u16) - 67, imax(u16) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u16) - 69, imax(u16) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u16) - 71, imax(u16) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u16) - 73, imax(u16) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u16) - 75, imax(u16) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u16) - 77, imax(u16) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u16) - 79, imax(u16) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u16) - 81, imax(u16) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i32), .{
                imin(i32),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i32), .{
                imin(i32) + 1, -1e9,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i32), .{
                -1e8, -1e7, -1e6, -1e5,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i32), .{
                -1e4, -1e3, -1e2, -1e1, -1e0, 0, 1e0, 1e1,
            });
            try testArgs(@Vector(16, f64), @Vector(16, i32), .{
                1e2,           1e3,       1e4,           1e5,           1e6,  1e7,  1e8,  1e9,
                imax(i32) - 1, imax(i32), imin(i32) + 2, imin(i32) + 3, -2e9, -2e8, -2e7, -2e6,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i32), .{
                -2e5,          -2e4,          -2e3,          -2e2,          -2e1, -2e0, 2e0,  2e1,
                2e2,           2e3,           2e4,           2e5,           2e6,  2e7,  2e8,  2e9,
                imax(i32) - 3, imax(i32) - 2, imin(i32) + 4, imin(i32) + 5, -3e8, -3e7, -3e6, -3e5,
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0, 3e0,  3e1,  3e2,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i32), .{
                3e3,           3e4,           3e5,           3e6,           3e7,            3e8,            imax(i32) - 5, imax(i32) - 4,
                imin(i32) + 6, imin(i32) + 7, -4e8,          -4e7,          -4e6,           -4e5,           -4e4,          -4e3,
                -4e2,          -4e1,          -4e0,          4e0,           4e1,            4e2,            4e3,           4e4,
                4e5,           4e6,           4e7,           4e8,           imax(i32) - 7,  imax(i32) - 6,  imin(i32) + 8, imin(i32) + 9,
                -5e8,          -5e7,          -5e6,          -5e5,          -5e4,           -5e3,           -5e2,          -5e1,
                -5e0,          5e0,           5e1,           5e2,           5e3,            5e4,            5e5,           5e6,
                5e7,           5e8,           imax(i32) - 9, imax(i32) - 8, imin(i32) + 10, imin(i32) + 11, -6e8,          -6e7,
                -6e6,          -6e5,          -6e4,          -6e3,          -6e2,           -6e1,           -6e0,          6e0,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i32), .{
                6e1,            6e2,            6e3,            6e4,            6e5,            6e6,            6e7,            6e8,
                imax(i32) - 11, imax(i32) - 10, imin(i32) + 12, imin(i32) + 13, -7e8,           -7e7,           -7e6,           -7e5,
                -7e4,           -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,            7e2,
                7e3,            7e4,            7e5,            7e6,            7e7,            7e8,            imax(i32) - 13, imax(i32) - 12,
                imin(i32) + 14, imin(i32) + 15, -8e8,           -8e7,           -8e6,           -8e5,           -8e4,           -8e3,
                -8e2,           -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            8e4,
                8e5,            8e6,            8e7,            8e8,            imax(i32) - 15, imax(i32) - 14, imin(i32) + 16, imin(i32) + 17,
                -9e8,           -9e7,           -9e6,           -9e5,           -9e4,           -9e3,           -9e2,           -9e1,
                -9e0,           9e0,            9e1,            9e2,            9e3,            9e4,            9e5,            9e6,
                9e7,            9e8,            imax(i32) - 17, imax(i32) - 16, imin(i32) + 18, imin(i32) + 19, -11e8,          -11e7,
                -11e6,          -11e5,          -11e4,          -11e3,          -11e2,          -11e1,          -11e0,          11e0,
                11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,           11e8,
                imax(i32) - 19, imax(i32) - 18, imin(i32) + 20, imin(i32) + 21, -12e8,          -12e7,          -12e6,          -12e5,
                -12e4,          -12e3,          -12e2,          -12e1,          -12e0,          12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           imax(i32) - 21, imax(i32) - 20,
                imin(i32) + 22, imin(i32) + 23, -13e8,          -13e7,          -13e6,          -13e5,          -13e4,          -13e3,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u32), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u32), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u32), .{
                1e2, 1e3, 1e4, imax(u32) - 1,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u32), .{
                imax(u32), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u32) - 3, imax(u32) - 2,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u32), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u32) - 5, imax(u32) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u32) - 7, imax(u32) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u32), .{
                5e2,            5e3,            5e4,            imax(u32) - 9,  imax(u32) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u32) - 11, imax(u32) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u32) - 13, imax(u32) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u32) - 15, imax(u32) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u32) - 17, imax(u32) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u32), .{
                11e2,           11e3,           imax(u32) - 19, imax(u32) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u32) - 21, imax(u32) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u32) - 23, imax(u32) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u32) - 25, imax(u32) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u32) - 27, imax(u32) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u32) - 29, imax(u32) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u32) - 31, imax(u32) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u32) - 33, imax(u32) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u32) - 35, imax(u32) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u32) - 37, imax(u32) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u32) - 39, imax(u32) - 38,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u32), .{
                23e0,           23e1,           23e2,           23e3,           imax(u32) - 41, imax(u32) - 40, 24e0,           24e1,
                24e2,           24e3,           imax(u32) - 43, imax(u32) - 42, 25e0,           25e1,           25e2,           25e3,
                imax(u32) - 45, imax(u32) - 44, 26e0,           26e1,           26e2,           26e3,           imax(u32) - 47, imax(u32) - 46,
                27e0,           27e1,           27e2,           27e3,           imax(u32) - 49, imax(u32) - 48, 28e0,           28e1,
                28e2,           28e3,           imax(u32) - 51, imax(u32) - 50, 29e0,           29e1,           29e2,           29e3,
                imax(u32) - 53, imax(u32) - 52, 31e0,           31e1,           31e2,           31e3,           imax(u32) - 55, imax(u32) - 54,
                32e0,           32e1,           32e2,           32e3,           imax(u32) - 57, imax(u32) - 56, 33e0,           33e1,
                33e2,           33e3,           imax(u32) - 59, imax(u32) - 58, 34e0,           34e1,           34e2,           34e3,
                imax(u32) - 61, imax(u32) - 60, 35e0,           35e1,           35e2,           35e3,           imax(u32) - 63, imax(u32) - 62,
                36e0,           36e1,           36e2,           36e3,           imax(u32) - 65, imax(u32) - 64, 37e0,           37e1,
                37e2,           37e3,           imax(u32) - 67, imax(u32) - 66, 38e0,           38e1,           38e2,           38e3,
                imax(u32) - 69, imax(u32) - 68, 39e0,           39e1,           39e2,           39e3,           imax(u32) - 71, imax(u32) - 70,
                41e0,           41e1,           41e2,           41e3,           imax(u32) - 73, imax(u32) - 72, 42e0,           42e1,
                42e2,           42e3,           imax(u32) - 75, imax(u32) - 74, 43e0,           43e1,           43e2,           43e3,
                imax(u32) - 77, imax(u32) - 76, 44e0,           44e1,           44e2,           44e3,           imax(u32) - 79, imax(u32) - 78,
                45e0,           45e1,           45e2,           45e3,           imax(u32) - 81, imax(u32) - 80, 46e0,           46e1,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i64), .{
                imin(i64),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i64), .{
                imin(i64) + 1, -1e18,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i64), .{
                -1e17, -1e16, -1e15, -1e14,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i64), .{
                -1e13, -1e12, -1e11, -1e10, -1e9, -1e8, -1e7, -1e6,
            });
            try testArgs(@Vector(16, f64), @Vector(16, i64), .{
                -1e5, -1e4, -1e3, -1e2, -1e1, -1e0, 0,   1e0,
                1e1,  1e2,  1e3,  1e4,  1e5,  1e6,  1e7, 1e8,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i64), .{
                1e9,   1e10,  1e11,          1e12,      1e13,          1e14,          1e15,  1e16,
                1e17,  1e18,  imax(i64) - 1, imax(i64), imin(i64) + 2, imin(i64) + 3, -2e18, -2e17,
                -2e16, -2e15, -2e14,         -2e13,     -2e12,         -2e11,         -2e10, -2e9,
                -2e8,  -2e7,  -2e6,          -2e5,      -2e4,          -2e3,          -2e2,  -2e1,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i64), .{
                -2e0,  2e0,   2e1,   2e2,   2e3,           2e4,           2e5,           2e6,
                2e7,   2e8,   2e9,   2e10,  2e11,          2e12,          2e13,          2e14,
                2e15,  2e16,  2e17,  2e18,  imax(i64) - 3, imax(i64) - 2, imin(i64) + 4, imin(i64) + 5,
                -3e18, -3e17, -3e16, -3e15, -3e14,         -3e13,         -3e12,         -3e11,
                -3e10, -3e9,  -3e8,  -3e7,  -3e6,          -3e5,          -3e4,          -3e3,
                -3e2,  -3e1,  -3e0,  3e0,   3e1,           3e2,           3e3,           3e4,
                3e5,   3e6,   3e7,   3e8,   3e9,           3e10,          3e11,          3e12,
                3e13,  3e14,  3e15,  3e16,  3e17,          3e18,          imax(i64) - 5, imax(i64) - 4,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i64), .{
                imin(i64) + 6, imin(i64) + 7, -4e18,         -4e17,         -4e16,          -4e15,          -4e14,          -4e13,
                -4e12,         -4e11,         -4e10,         -4e9,          -4e8,           -4e7,           -4e6,           -4e5,
                -4e4,          -4e3,          -4e2,          -4e1,          -4e0,           4e0,            4e1,            4e2,
                4e3,           4e4,           4e5,           4e6,           4e7,            4e8,            4e9,            4e10,
                4e11,          4e12,          4e13,          4e14,          4e15,           4e16,           4e17,           4e18,
                imax(i64) - 7, imax(i64) - 6, imin(i64) + 8, imin(i64) + 9, -5e18,          -5e17,          -5e16,          -5e15,
                -5e14,         -5e13,         -5e12,         -5e11,         -5e10,          -5e9,           -5e8,           -5e7,
                -5e6,          -5e5,          -5e4,          -5e3,          -5e2,           -5e1,           -5e0,           5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,            5e6,            5e7,            5e8,
                5e9,           5e10,          5e11,          5e12,          5e13,           5e14,           5e15,           5e16,
                5e17,          5e18,          imax(i64) - 9, imax(i64) - 8, imin(i64) + 10, imin(i64) + 11, -6e18,          -6e17,
                -6e16,         -6e15,         -6e14,         -6e13,         -6e12,          -6e11,          -6e10,          -6e9,
                -6e8,          -6e7,          -6e6,          -6e5,          -6e4,           -6e3,           -6e2,           -6e1,
                -6e0,          6e0,           6e1,           6e2,           6e3,            6e4,            6e5,            6e6,
                6e7,           6e8,           6e9,           6e10,          6e11,           6e12,           6e13,           6e14,
                6e15,          6e16,          6e17,          6e18,          imax(i64) - 11, imax(i64) - 10, imin(i64) + 12, imin(i64) + 13,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u64), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u64), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u64), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u64), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u64), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, imax(u64) - 1, imax(u64),
                2e0,  2e1,  2e2,  2e3,  2e4,  2e5,  2e6,           2e7,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u64), .{
                2e8,  2e9,  2e10, 2e11,          2e12,          2e13, 2e14, 2e15,
                2e16, 2e17, 2e18, imax(u64) - 3, imax(u64) - 2, 3e0,  3e1,  3e2,
                3e3,  3e4,  3e5,  3e6,           3e7,           3e8,  3e9,  3e10,
                3e11, 3e12, 3e13, 3e14,          3e15,          3e16, 3e17, 3e18,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u64), .{
                imax(u64) - 5, imax(u64) - 4, 4e0,           4e1,           4e2,  4e3,           4e4,           4e5,
                4e6,           4e7,           4e8,           4e9,           4e10, 4e11,          4e12,          4e13,
                4e14,          4e15,          4e16,          4e17,          4e18, imax(u64) - 7, imax(u64) - 6, 5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,  5e6,           5e7,           5e8,
                5e9,           5e10,          5e11,          5e12,          5e13, 5e14,          5e15,          5e16,
                5e17,          5e18,          imax(u64) - 9, imax(u64) - 8, 6e0,  6e1,           6e2,           6e3,
                6e4,           6e5,           6e6,           6e7,           6e8,  6e9,           6e10,          6e11,
                6e12,          6e13,          6e14,          6e15,          6e16, 6e17,          6e18,          imax(u64) - 11,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u64), .{
                imax(u64) - 10, 7e0,            7e1,            7e2,            7e3,            7e4,            7e5,            7e6,
                7e7,            7e8,            7e9,            7e10,           7e11,           7e12,           7e13,           7e14,
                7e15,           7e16,           7e17,           7e18,           imax(u64) - 13, imax(u64) - 12, 8e0,            8e1,
                8e2,            8e3,            8e4,            8e5,            8e6,            8e7,            8e8,            8e9,
                8e10,           8e11,           8e12,           8e13,           8e14,           8e15,           8e16,           8e17,
                8e18,           imax(u64) - 15, imax(u64) - 14, 9e0,            9e1,            9e2,            9e3,            9e4,
                9e5,            9e6,            9e7,            9e8,            9e9,            9e10,           9e11,           9e12,
                9e13,           9e14,           9e15,           9e16,           9e17,           9e18,           imax(u64) - 17, imax(u64) - 16,
                11e0,           11e1,           11e2,           11e3,           11e4,           11e5,           11e6,           11e7,
                11e8,           11e9,           11e10,          11e11,          11e12,          11e13,          11e14,          11e15,
                11e16,          11e17,          11e18,          imax(u64) - 19, imax(u64) - 18, 12e0,           12e1,           12e2,
                12e3,           12e4,           12e5,           12e6,           12e7,           12e8,           12e9,           12e10,
                12e11,          12e12,          12e13,          12e14,          12e15,          12e16,          12e17,          12e18,
                imax(u64) - 21, imax(u64) - 20, 13e0,           13e1,           13e2,           13e3,           13e4,           13e5,
                13e6,           13e7,           13e8,           13e9,           13e10,          13e11,          13e12,          13e13,
                13e14,          13e15,          13e16,          13e17,          13e18,          imax(u64) - 23, imax(u64) - 22, 14e0,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i128), .{
                imin(i128),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i128), .{
                imin(i128) + 1, -1e38,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i128), .{
                -1e37, -1e36, -1e35, -1e34,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i128), .{
                -1e33, -1e32, -1e31, -1e30, -1e29, -1e28, -1e27, -1e26,
            });
            try testArgs(@Vector(16, f64), @Vector(16, i128), .{
                -1e25, -1e24, -1e23, -1e22, -1e21, -1e20, -1e19, -1e18,
                -1e17, -1e16, -1e15, -1e14, -1e13, -1e12, -1e11, -1e10,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i128), .{
                -1e9, -1e8, -1e7, -1e6, -1e5, -1e4, -1e3, -1e2,
                -1e1, -1e0, 0,    1e0,  1e1,  1e2,  1e3,  1e4,
                1e5,  1e6,  1e7,  1e8,  1e9,  1e10, 1e11, 1e12,
                1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i128), .{
                1e21,  1e22,  1e23,           1e24,       1e25,           1e26,           1e27,  1e28,
                1e29,  1e30,  1e31,           1e32,       1e33,           1e34,           1e35,  1e36,
                1e37,  1e38,  imax(i128) - 1, imax(i128), imin(i128) + 2, imin(i128) + 3, -2e37, -2e36,
                -2e35, -2e34, -2e33,          -2e32,      -2e31,          -2e30,          -2e29, -2e28,
                -2e27, -2e26, -2e25,          -2e24,      -2e23,          -2e22,          -2e21, -2e20,
                -2e19, -2e18, -2e17,          -2e16,      -2e15,          -2e14,          -2e13, -2e12,
                -2e11, -2e10, -2e9,           -2e8,       -2e7,           -2e6,           -2e5,  -2e4,
                -2e3,  -2e2,  -2e1,           -2e0,       2e0,            2e1,            2e2,   2e3,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i128), .{
                2e4,   2e5,   2e6,            2e7,            2e8,            2e9,            2e10,  2e11,
                2e12,  2e13,  2e14,           2e15,           2e16,           2e17,           2e18,  2e19,
                2e20,  2e21,  2e22,           2e23,           2e24,           2e25,           2e26,  2e27,
                2e28,  2e29,  2e30,           2e31,           2e32,           2e33,           2e34,  2e35,
                2e36,  2e37,  imax(i128) - 3, imax(i128) - 2, imin(i128) + 4, imin(i128) + 5, -3e37, -3e36,
                -3e35, -3e34, -3e33,          -3e32,          -3e31,          -3e30,          -3e29, -3e28,
                -3e27, -3e26, -3e25,          -3e24,          -3e23,          -3e22,          -3e21, -3e20,
                -3e19, -3e18, -3e17,          -3e16,          -3e15,          -3e14,          -3e13, -3e12,
                -3e11, -3e10, -3e9,           -3e8,           -3e7,           -3e6,           -3e5,  -3e4,
                -3e3,  -3e2,  -3e1,           -3e0,           3e0,            3e1,            3e2,   3e3,
                3e4,   3e5,   3e6,            3e7,            3e8,            3e9,            3e10,  3e11,
                3e12,  3e13,  3e14,           3e15,           3e16,           3e17,           3e18,  3e19,
                3e20,  3e21,  3e22,           3e23,           3e24,           3e25,           3e26,  3e27,
                3e28,  3e29,  3e30,           3e31,           3e32,           3e33,           3e34,  3e35,
                3e36,  3e37,  imax(i128) - 5, imax(i128) - 4, imin(i128) + 6, imin(i128) + 7, -4e37, -4e36,
                -4e35, -4e34, -4e33,          -4e32,          -4e31,          -4e30,          -4e29, -4e28,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u128), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u128), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u128), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u128), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u128), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u128), .{
                1e30, 1e31,           1e32,       1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, imax(u128) - 1, imax(u128), 2e0,  2e1,  2e2,  2e3,  2e4,
                2e5,  2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12,
                2e13, 2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u128), .{
                2e21, 2e22, 2e23,           2e24,           2e25,           2e26, 2e27, 2e28,
                2e29, 2e30, 2e31,           2e32,           2e33,           2e34, 2e35, 2e36,
                2e37, 2e38, imax(u128) - 3, imax(u128) - 2, 3e0,            3e1,  3e2,  3e3,
                3e4,  3e5,  3e6,            3e7,            3e8,            3e9,  3e10, 3e11,
                3e12, 3e13, 3e14,           3e15,           3e16,           3e17, 3e18, 3e19,
                3e20, 3e21, 3e22,           3e23,           3e24,           3e25, 3e26, 3e27,
                3e28, 3e29, 3e30,           3e31,           3e32,           3e33, 3e34, 3e35,
                3e36, 3e37, 3e38,           imax(u128) - 5, imax(u128) - 4, 4e0,  4e1,  4e2,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u128), .{
                4e3,  4e4,  4e5,  4e6,             4e7,             4e8,  4e9,  4e10,
                4e11, 4e12, 4e13, 4e14,            4e15,            4e16, 4e17, 4e18,
                4e19, 4e20, 4e21, 4e22,            4e23,            4e24, 4e25, 4e26,
                4e27, 4e28, 4e29, 4e30,            4e31,            4e32, 4e33, 4e34,
                4e35, 4e36, 4e37, imax(u128) - 7,  imax(u128) - 6,  5e0,  5e1,  5e2,
                5e3,  5e4,  5e5,  5e6,             5e7,             5e8,  5e9,  5e10,
                5e11, 5e12, 5e13, 5e14,            5e15,            5e16, 5e17, 5e18,
                5e19, 5e20, 5e21, 5e22,            5e23,            5e24, 5e25, 5e26,
                5e27, 5e28, 5e29, 5e30,            5e31,            5e32, 5e33, 5e34,
                5e35, 5e36, 5e37, imax(u128) - 9,  imax(u128) - 8,  6e0,  6e1,  6e2,
                6e3,  6e4,  6e5,  6e6,             6e7,             6e8,  6e9,  6e10,
                6e11, 6e12, 6e13, 6e14,            6e15,            6e16, 6e17, 6e18,
                6e19, 6e20, 6e21, 6e22,            6e23,            6e24, 6e25, 6e26,
                6e27, 6e28, 6e29, 6e30,            6e31,            6e32, 6e33, 6e34,
                6e35, 6e36, 6e37, imax(u128) - 11, imax(u128) - 10, 7e0,  7e1,  7e2,
                7e3,  7e4,  7e5,  7e6,             7e7,             7e8,  7e9,  7e10,
            });

            try testArgs(@Vector(1, f64), @Vector(1, i256), .{
                imin(i256),
            });
            try testArgs(@Vector(2, f64), @Vector(2, i256), .{
                imin(i256) + 1, -1e76,
            });
            try testArgs(@Vector(4, f64), @Vector(4, i256), .{
                -1e75, -1e74, -1e73, -1e72,
            });
            try testArgs(@Vector(8, f64), @Vector(8, i256), .{
                -1e71, -1e70, -1e69, -1e68, -1e67, -1e66, -1e65, -1e64,
            });
            try testArgs(@Vector(16, f64), @Vector(16, i256), .{
                -1e63, -1e62, -1e61, -1e60, -1e59, -1e58, -1e57, -1e56,
                -1e55, -1e54, -1e53, -1e52, -1e51, -1e50, -1e49, -1e48,
            });
            try testArgs(@Vector(32, f64), @Vector(32, i256), .{
                -1e47, -1e46, -1e45, -1e44, -1e43, -1e42, -1e41, -1e40,
                -1e39, -1e38, -1e37, -1e36, -1e35, -1e34, -1e33, -1e32,
                -1e31, -1e30, -1e29, -1e28, -1e27, -1e26, -1e25, -1e24,
                -1e23, -1e22, -1e21, -1e20, -1e19, -1e18, -1e17, -1e16,
            });
            try testArgs(@Vector(64, f64), @Vector(64, i256), .{
                -1e15, -1e14, -1e13, -1e12, -1e11, -1e10, -1e9, -1e8,
                -1e7,  -1e6,  -1e5,  -1e4,  -1e3,  -1e2,  -1e1, -1e0,
                0,     1e0,   1e1,   1e2,   1e3,   1e4,   1e5,  1e6,
                1e7,   1e8,   1e9,   1e10,  1e11,  1e12,  1e13, 1e14,
                1e15,  1e16,  1e17,  1e18,  1e19,  1e20,  1e21, 1e22,
                1e23,  1e24,  1e25,  1e26,  1e27,  1e28,  1e29, 1e30,
                1e31,  1e32,  1e33,  1e34,  1e35,  1e36,  1e37, 1e38,
                1e39,  1e40,  1e41,  1e42,  1e43,  1e44,  1e45, 1e46,
            });
            try testArgs(@Vector(128, f64), @Vector(128, i256), .{
                1e47,           1e48,           1e49,  1e50,  1e51,  1e52,  1e53,           1e54,
                1e55,           1e56,           1e57,  1e58,  1e59,  1e60,  1e61,           1e62,
                1e63,           1e64,           1e65,  1e66,  1e67,  1e68,  1e69,           1e70,
                1e71,           1e72,           1e73,  1e74,  1e75,  1e76,  imax(i256) - 1, imax(i256),
                imin(i256) + 2, imin(i256) + 3, -2e76, -2e75, -2e74, -2e73, -2e72,          -2e71,
                -2e70,          -2e69,          -2e68, -2e67, -2e66, -2e65, -2e64,          -2e63,
                -2e62,          -2e61,          -2e60, -2e59, -2e58, -2e57, -2e56,          -2e55,
                -2e54,          -2e53,          -2e52, -2e51, -2e50, -2e49, -2e48,          -2e47,
                -2e46,          -2e45,          -2e44, -2e43, -2e42, -2e41, -2e40,          -2e39,
                -2e38,          -2e37,          -2e36, -2e35, -2e34, -2e33, -2e32,          -2e31,
                -2e30,          -2e29,          -2e28, -2e27, -2e26, -2e25, -2e24,          -2e23,
                -2e22,          -2e21,          -2e20, -2e19, -2e18, -2e17, -2e16,          -2e15,
                -2e14,          -2e13,          -2e12, -2e11, -2e10, -2e9,  -2e8,           -2e7,
                -2e6,           -2e5,           -2e4,  -2e3,  -2e2,  -2e1,  -2e0,           2e0,
                2e1,            2e2,            2e3,   2e4,   2e5,   2e6,   2e7,            2e8,
                2e9,            2e10,           2e11,  2e12,  2e13,  2e14,  2e15,           2e16,
            });

            try testArgs(@Vector(1, f64), @Vector(1, u256), .{
                0,
            });
            try testArgs(@Vector(2, f64), @Vector(2, u256), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f64), @Vector(4, u256), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f64), @Vector(8, u256), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f64), @Vector(16, u256), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f64), @Vector(32, u256), .{
                1e30, 1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, 1e39, 1e40, 1e41, 1e42, 1e43, 1e44, 1e45,
                1e46, 1e47, 1e48, 1e49, 1e50, 1e51, 1e52, 1e53,
                1e54, 1e55, 1e56, 1e57, 1e58, 1e59, 1e60, 1e61,
            });
            try testArgs(@Vector(64, f64), @Vector(64, u256), .{
                1e62,           1e63,       1e64, 1e65, 1e66, 1e67, 1e68, 1e69,
                1e70,           1e71,       1e72, 1e73, 1e74, 1e75, 1e76, 1e77,
                imax(u256) - 1, imax(u256), 2e0,  2e1,  2e2,  2e3,  2e4,  2e5,
                2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12, 2e13,
                2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20, 2e21,
                2e22,           2e23,       2e24, 2e25, 2e26, 2e27, 2e28, 2e29,
                2e30,           2e31,       2e32, 2e33, 2e34, 2e35, 2e36, 2e37,
                2e38,           2e39,       2e40, 2e41, 2e42, 2e43, 2e44, 2e45,
            });
            try testArgs(@Vector(128, f64), @Vector(128, u256), .{
                2e46,           2e47, 2e48, 2e49, 2e50, 2e51, 2e52,           2e53,
                2e54,           2e55, 2e56, 2e57, 2e58, 2e59, 2e60,           2e61,
                2e62,           2e63, 2e64, 2e65, 2e66, 2e67, 2e68,           2e69,
                2e70,           2e71, 2e72, 2e73, 2e74, 2e75, 2e76,           imax(u256) - 3,
                imax(u256) - 2, 3e0,  3e1,  3e2,  3e3,  3e4,  3e5,            3e6,
                3e7,            3e8,  3e9,  3e10, 3e11, 3e12, 3e13,           3e14,
                3e15,           3e16, 3e17, 3e18, 3e19, 3e20, 3e21,           3e22,
                3e23,           3e24, 3e25, 3e26, 3e27, 3e28, 3e29,           3e30,
                3e31,           3e32, 3e33, 3e34, 3e35, 3e36, 3e37,           3e38,
                3e39,           3e40, 3e41, 3e42, 3e43, 3e44, 3e45,           3e46,
                3e47,           3e48, 3e49, 3e50, 3e51, 3e52, 3e53,           3e54,
                3e55,           3e56, 3e57, 3e58, 3e59, 3e60, 3e61,           3e62,
                3e63,           3e64, 3e65, 3e66, 3e67, 3e68, 3e69,           3e70,
                3e71,           3e72, 3e73, 3e74, 3e75, 3e76, imax(u256) - 5, imax(u256) - 4,
                4e0,            4e1,  4e2,  4e3,  4e4,  4e5,  4e6,            4e7,
                4e8,            4e9,  4e10, 4e11, 4e12, 4e13, 4e14,           4e15,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i8), .{
                imin(i8),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i8), .{
                imin(i8) + 1, -1e2,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i8), .{
                -1e1, -1e0, 0, 1e0,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i8), .{
                1e1, 1e2, imax(i8) - 1, imax(i8), imin(i8) + 2, imin(i8) + 3, -2e1, -2e0,
            });
            try testArgs(@Vector(16, f80), @Vector(16, i8), .{
                2e0, 2e1, imax(i8) - 3, imax(i8) - 2, imin(i8) + 4, imin(i8) + 5, -3e1, -3e0,
                3e0, 3e1, imax(i8) - 5, imax(i8) - 4, imin(i8) + 6, imin(i8) + 7, -4e1, -4e0,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i8), .{
                4e0, 4e1, imax(i8) - 7,  imax(i8) - 6,  imin(i8) + 8,  imin(i8) + 9,  -5e1, -5e0,
                5e0, 5e1, imax(i8) - 9,  imax(i8) - 8,  imin(i8) + 10, imin(i8) + 11, -6e1, -6e0,
                6e0, 6e1, imax(i8) - 11, imax(i8) - 10, imin(i8) + 12, imin(i8) + 13, -7e1, -7e0,
                7e0, 7e1, imax(i8) - 13, imax(i8) - 12, imin(i8) + 14, imin(i8) + 15, -8e1, -8e0,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i8), .{
                8e0,           8e1,           imax(i8) - 15, imax(i8) - 14, imin(i8) + 16, imin(i8) + 17, -9e1,          -9e0,
                9e0,           9e1,           imax(i8) - 17, imax(i8) - 16, imin(i8) + 18, imin(i8) + 19, -11e1,         -11e0,
                11e0,          11e1,          imax(i8) - 19, imax(i8) - 18, imin(i8) + 20, imin(i8) + 21, -12e1,         -12e0,
                12e0,          12e1,          imax(i8) - 21, imax(i8) - 20, imin(i8) + 22, imin(i8) + 23, -13e0,         13e0,
                imax(i8) - 23, imax(i8) - 22, imin(i8) + 24, imin(i8) + 25, -14e0,         14e0,          imax(i8) - 25, imax(i8) - 24,
                imin(i8) + 26, imin(i8) + 27, -15e0,         15e0,          imax(i8) - 27, imax(i8) - 26, imin(i8) + 28, imin(i8) + 29,
                -16e0,         16e0,          imax(i8) - 29, imax(i8) - 28, imin(i8) + 30, imin(i8) + 31, -17e0,         17e0,
                imax(i8) - 31, imax(i8) - 30, imin(i8) + 32, imin(i8) + 33, -18e0,         18e0,          imax(i8) - 33, imax(i8) - 32,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u8), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u8), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u8), .{
                1e2, imax(u8) - 1, imax(u8), 2e0,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u8), .{
                2e1, 2e2, imax(u8) - 3, imax(u8) - 2, 3e0, 3e1, imax(u8) - 5, imax(u8) - 4,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u8), .{
                imax(u8) - 7,  imax(u8) - 6,  5e0, 5e1, imax(u8) - 9,  imax(u8) - 8,  6e0, 6e1,
                imax(u8) - 11, imax(u8) - 10, 7e0, 7e1, imax(u8) - 13, imax(u8) - 12, 8e0, 8e1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u8), .{
                imax(u8) - 15, imax(u8) - 14, 9e0,  9e1,  imax(u8) - 17, imax(u8) - 16, 11e0, 11e1,
                imax(u8) - 19, imax(u8) - 18, 12e0, 12e1, imax(u8) - 21, imax(u8) - 20, 13e0, 13e1,
                imax(u8) - 23, imax(u8) - 22, 14e0, 14e1, imax(u8) - 25, imax(u8) - 24, 15e0, 15e1,
                imax(u8) - 27, imax(u8) - 26, 16e0, 16e1, imax(u8) - 29, imax(u8) - 28, 17e0, 17e1,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u8), .{
                imax(u8) - 31, imax(u8) - 30, 18e0,          18e1,          imax(u8) - 33, imax(u8) - 32, 19e0,          19e1,
                imax(u8) - 35, imax(u8) - 34, 21e0,          21e1,          imax(u8) - 37, imax(u8) - 36, 22e0,          22e1,
                imax(u8) - 39, imax(u8) - 38, 23e0,          23e1,          imax(u8) - 41, imax(u8) - 40, 24e0,          24e1,
                imax(u8) - 43, imax(u8) - 42, 25e0,          25e1,          imax(u8) - 45, imax(u8) - 44, 26e0,          imax(u8) - 47,
                imax(u8) - 46, 27e0,          imax(u8) - 49, imax(u8) - 48, 28e0,          imax(u8) - 51, imax(u8) - 50, 29e0,
                imax(u8) - 53, imax(u8) - 52, 31e0,          imax(u8) - 55, imax(u8) - 54, 32e0,          imax(u8) - 57, imax(u8) - 56,
                33e0,          imax(u8) - 59, imax(u8) - 58, 34e0,          imax(u8) - 61, imax(u8) - 60, 35e0,          imax(u8) - 63,
                imax(u8) - 62, 36e0,          imax(u8) - 65, imax(u8) - 64, 37e0,          imax(u8) - 67, imax(u8) - 66, 38e0,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i16), .{
                imin(i16),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i16), .{
                imin(i16) + 1, -1e4,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i16), .{
                -1e3, -1e2, -1e1, -1e0,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i16), .{
                0, 1e0, 1e1, 1e2, 1e3, 1e4, imax(i16) - 1, imax(i16),
            });
            try testArgs(@Vector(16, f80), @Vector(16, i16), .{
                imin(i16) + 2, imin(i16) + 3, -2e4, -2e3, -2e2,          -2e1,          -2e0,          2e0,
                2e1,           2e2,           2e3,  2e4,  imax(i16) - 3, imax(i16) - 2, imin(i16) + 4, imin(i16) + 5,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i16), .{
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0,          3e0,           3e1,           3e2,
                3e3,           3e4,           imax(i16) - 5, imax(i16) - 4, imin(i16) + 6, imin(i16) + 7, -4e3,          -4e2,
                -4e1,          -4e0,          4e0,           4e1,           4e2,           4e3,           imax(i16) - 7, imax(i16) - 6,
                imin(i16) + 8, imin(i16) + 9, -5e3,          -5e2,          -5e1,          -5e0,          5e0,           5e1,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i16), .{
                5e2,            5e3,            imax(i16) - 9,  imax(i16) - 8,  imin(i16) + 10, imin(i16) + 11, -6e3,           -6e2,
                -6e1,           -6e0,           6e0,            6e1,            6e2,            6e3,            imax(i16) - 11, imax(i16) - 10,
                imin(i16) + 12, imin(i16) + 13, -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,
                7e2,            7e3,            imax(i16) - 13, imax(i16) - 12, imin(i16) + 14, imin(i16) + 15, -8e3,           -8e2,
                -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            imax(i16) - 15, imax(i16) - 14,
                imin(i16) + 16, imin(i16) + 17, -9e3,           -9e2,           -9e1,           -9e0,           9e0,            9e1,
                9e2,            9e3,            imax(i16) - 17, imax(i16) - 16, imin(i16) + 18, imin(i16) + 19, -11e3,          -11e2,
                -11e1,          -11e0,          11e0,           11e1,           11e2,           11e3,           imax(i16) - 19, imax(i16) - 18,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u16), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u16), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u16), .{
                1e2, 1e3, 1e4, imax(u16) - 1,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u16), .{
                imax(u16), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u16) - 3, imax(u16) - 2,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u16), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u16) - 5, imax(u16) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u16) - 7, imax(u16) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u16), .{
                5e2,            5e3,            5e4,            imax(u16) - 9,  imax(u16) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u16) - 11, imax(u16) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u16) - 13, imax(u16) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u16) - 15, imax(u16) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u16) - 17, imax(u16) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u16), .{
                11e2,           11e3,           imax(u16) - 19, imax(u16) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u16) - 21, imax(u16) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u16) - 23, imax(u16) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u16) - 25, imax(u16) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u16) - 27, imax(u16) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u16) - 29, imax(u16) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u16) - 31, imax(u16) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u16) - 33, imax(u16) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u16) - 35, imax(u16) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u16) - 37, imax(u16) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u16) - 39, imax(u16) - 38,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i32), .{
                imin(i32),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i32), .{
                imin(i32) + 1, -1e9,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i32), .{
                -1e8, -1e7, -1e6, -1e5,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i32), .{
                -1e4, -1e3, -1e2, -1e1, -1e0, 0, 1e0, 1e1,
            });
            try testArgs(@Vector(16, f80), @Vector(16, i32), .{
                1e2,           1e3,       1e4,           1e5,           1e6,  1e7,  1e8,  1e9,
                imax(i32) - 1, imax(i32), imin(i32) + 2, imin(i32) + 3, -2e9, -2e8, -2e7, -2e6,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i32), .{
                -2e5,          -2e4,          -2e3,          -2e2,          -2e1, -2e0, 2e0,  2e1,
                2e2,           2e3,           2e4,           2e5,           2e6,  2e7,  2e8,  2e9,
                imax(i32) - 3, imax(i32) - 2, imin(i32) + 4, imin(i32) + 5, -3e8, -3e7, -3e6, -3e5,
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0, 3e0,  3e1,  3e2,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i32), .{
                3e3,           3e4,           3e5,           3e6,           3e7,            3e8,            imax(i32) - 5, imax(i32) - 4,
                imin(i32) + 6, imin(i32) + 7, -4e8,          -4e7,          -4e6,           -4e5,           -4e4,          -4e3,
                -4e2,          -4e1,          -4e0,          4e0,           4e1,            4e2,            4e3,           4e4,
                4e5,           4e6,           4e7,           4e8,           imax(i32) - 7,  imax(i32) - 6,  imin(i32) + 8, imin(i32) + 9,
                -5e8,          -5e7,          -5e6,          -5e5,          -5e4,           -5e3,           -5e2,          -5e1,
                -5e0,          5e0,           5e1,           5e2,           5e3,            5e4,            5e5,           5e6,
                5e7,           5e8,           imax(i32) - 9, imax(i32) - 8, imin(i32) + 10, imin(i32) + 11, -6e8,          -6e7,
                -6e6,          -6e5,          -6e4,          -6e3,          -6e2,           -6e1,           -6e0,          6e0,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u32), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u32), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u32), .{
                1e2, 1e3, 1e4, imax(u32) - 1,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u32), .{
                imax(u32), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u32) - 3, imax(u32) - 2,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u32), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u32) - 5, imax(u32) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u32) - 7, imax(u32) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u32), .{
                5e2,            5e3,            5e4,            imax(u32) - 9,  imax(u32) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u32) - 11, imax(u32) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u32) - 13, imax(u32) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u32) - 15, imax(u32) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u32) - 17, imax(u32) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u32), .{
                11e2,           11e3,           imax(u32) - 19, imax(u32) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u32) - 21, imax(u32) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u32) - 23, imax(u32) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u32) - 25, imax(u32) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u32) - 27, imax(u32) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u32) - 29, imax(u32) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u32) - 31, imax(u32) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u32) - 33, imax(u32) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u32) - 35, imax(u32) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u32) - 37, imax(u32) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u32) - 39, imax(u32) - 38,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i64), .{
                imin(i64),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i64), .{
                imin(i64) + 1, -1e18,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i64), .{
                -1e17, -1e16, -1e15, -1e14,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i64), .{
                -1e13, -1e12, -1e11, -1e10, -1e9, -1e8, -1e7, -1e6,
            });
            try testArgs(@Vector(16, f80), @Vector(16, i64), .{
                -1e5, -1e4, -1e3, -1e2, -1e1, -1e0, 0,   1e0,
                1e1,  1e2,  1e3,  1e4,  1e5,  1e6,  1e7, 1e8,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i64), .{
                1e9,   1e10,  1e11,          1e12,      1e13,          1e14,          1e15,  1e16,
                1e17,  1e18,  imax(i64) - 1, imax(i64), imin(i64) + 2, imin(i64) + 3, -2e18, -2e17,
                -2e16, -2e15, -2e14,         -2e13,     -2e12,         -2e11,         -2e10, -2e9,
                -2e8,  -2e7,  -2e6,          -2e5,      -2e4,          -2e3,          -2e2,  -2e1,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i64), .{
                -2e0,  2e0,   2e1,   2e2,   2e3,           2e4,           2e5,           2e6,
                2e7,   2e8,   2e9,   2e10,  2e11,          2e12,          2e13,          2e14,
                2e15,  2e16,  2e17,  2e18,  imax(i64) - 3, imax(i64) - 2, imin(i64) + 4, imin(i64) + 5,
                -3e18, -3e17, -3e16, -3e15, -3e14,         -3e13,         -3e12,         -3e11,
                -3e10, -3e9,  -3e8,  -3e7,  -3e6,          -3e5,          -3e4,          -3e3,
                -3e2,  -3e1,  -3e0,  3e0,   3e1,           3e2,           3e3,           3e4,
                3e5,   3e6,   3e7,   3e8,   3e9,           3e10,          3e11,          3e12,
                3e13,  3e14,  3e15,  3e16,  3e17,          3e18,          imax(i64) - 5, imax(i64) - 4,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u64), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u64), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u64), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u64), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u64), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, imax(u64) - 1, imax(u64),
                2e0,  2e1,  2e2,  2e3,  2e4,  2e5,  2e6,           2e7,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u64), .{
                2e8,  2e9,  2e10, 2e11,          2e12,          2e13, 2e14, 2e15,
                2e16, 2e17, 2e18, imax(u64) - 3, imax(u64) - 2, 3e0,  3e1,  3e2,
                3e3,  3e4,  3e5,  3e6,           3e7,           3e8,  3e9,  3e10,
                3e11, 3e12, 3e13, 3e14,          3e15,          3e16, 3e17, 3e18,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u64), .{
                imax(u64) - 5, imax(u64) - 4, 4e0,           4e1,           4e2,  4e3,           4e4,           4e5,
                4e6,           4e7,           4e8,           4e9,           4e10, 4e11,          4e12,          4e13,
                4e14,          4e15,          4e16,          4e17,          4e18, imax(u64) - 7, imax(u64) - 6, 5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,  5e6,           5e7,           5e8,
                5e9,           5e10,          5e11,          5e12,          5e13, 5e14,          5e15,          5e16,
                5e17,          5e18,          imax(u64) - 9, imax(u64) - 8, 6e0,  6e1,           6e2,           6e3,
                6e4,           6e5,           6e6,           6e7,           6e8,  6e9,           6e10,          6e11,
                6e12,          6e13,          6e14,          6e15,          6e16, 6e17,          6e18,          imax(u64) - 11,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i128), .{
                imin(i128),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i128), .{
                imin(i128) + 1, -1e38,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i128), .{
                -1e37, -1e36, -1e35, -1e34,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i128), .{
                -1e33, -1e32, -1e31, -1e30, -1e29, -1e28, -1e27, -1e26,
            });
            try testArgs(@Vector(16, f80), @Vector(16, i128), .{
                -1e25, -1e24, -1e23, -1e22, -1e21, -1e20, -1e19, -1e18,
                -1e17, -1e16, -1e15, -1e14, -1e13, -1e12, -1e11, -1e10,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i128), .{
                -1e9, -1e8, -1e7, -1e6, -1e5, -1e4, -1e3, -1e2,
                -1e1, -1e0, 0,    1e0,  1e1,  1e2,  1e3,  1e4,
                1e5,  1e6,  1e7,  1e8,  1e9,  1e10, 1e11, 1e12,
                1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i128), .{
                1e21,  1e22,  1e23,           1e24,       1e25,           1e26,           1e27,  1e28,
                1e29,  1e30,  1e31,           1e32,       1e33,           1e34,           1e35,  1e36,
                1e37,  1e38,  imax(i128) - 1, imax(i128), imin(i128) + 2, imin(i128) + 3, -2e37, -2e36,
                -2e35, -2e34, -2e33,          -2e32,      -2e31,          -2e30,          -2e29, -2e28,
                -2e27, -2e26, -2e25,          -2e24,      -2e23,          -2e22,          -2e21, -2e20,
                -2e19, -2e18, -2e17,          -2e16,      -2e15,          -2e14,          -2e13, -2e12,
                -2e11, -2e10, -2e9,           -2e8,       -2e7,           -2e6,           -2e5,  -2e4,
                -2e3,  -2e2,  -2e1,           -2e0,       2e0,            2e1,            2e2,   2e3,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u128), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u128), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u128), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u128), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u128), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u128), .{
                1e30, 1e31,           1e32,       1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, imax(u128) - 1, imax(u128), 2e0,  2e1,  2e2,  2e3,  2e4,
                2e5,  2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12,
                2e13, 2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u128), .{
                2e21, 2e22, 2e23,           2e24,           2e25,           2e26, 2e27, 2e28,
                2e29, 2e30, 2e31,           2e32,           2e33,           2e34, 2e35, 2e36,
                2e37, 2e38, imax(u128) - 3, imax(u128) - 2, 3e0,            3e1,  3e2,  3e3,
                3e4,  3e5,  3e6,            3e7,            3e8,            3e9,  3e10, 3e11,
                3e12, 3e13, 3e14,           3e15,           3e16,           3e17, 3e18, 3e19,
                3e20, 3e21, 3e22,           3e23,           3e24,           3e25, 3e26, 3e27,
                3e28, 3e29, 3e30,           3e31,           3e32,           3e33, 3e34, 3e35,
                3e36, 3e37, 3e38,           imax(u128) - 5, imax(u128) - 4, 4e0,  4e1,  4e2,
            });

            try testArgs(@Vector(1, f80), @Vector(1, i256), .{
                imin(i256),
            });
            try testArgs(@Vector(2, f80), @Vector(2, i256), .{
                imin(i256) + 1, -1e76,
            });
            try testArgs(@Vector(4, f80), @Vector(4, i256), .{
                -1e75, -1e74, -1e73, -1e72,
            });
            try testArgs(@Vector(8, f80), @Vector(8, i256), .{
                -1e71, -1e70, -1e69, -1e68, -1e67, -1e66, -1e65, -1e64,
            });
            try testArgs(@Vector(16, f80), @Vector(16, i256), .{
                -1e63, -1e62, -1e61, -1e60, -1e59, -1e58, -1e57, -1e56,
                -1e55, -1e54, -1e53, -1e52, -1e51, -1e50, -1e49, -1e48,
            });
            try testArgs(@Vector(32, f80), @Vector(32, i256), .{
                -1e47, -1e46, -1e45, -1e44, -1e43, -1e42, -1e41, -1e40,
                -1e39, -1e38, -1e37, -1e36, -1e35, -1e34, -1e33, -1e32,
                -1e31, -1e30, -1e29, -1e28, -1e27, -1e26, -1e25, -1e24,
                -1e23, -1e22, -1e21, -1e20, -1e19, -1e18, -1e17, -1e16,
            });
            try testArgs(@Vector(64, f80), @Vector(64, i256), .{
                -1e15, -1e14, -1e13, -1e12, -1e11, -1e10, -1e9, -1e8,
                -1e7,  -1e6,  -1e5,  -1e4,  -1e3,  -1e2,  -1e1, -1e0,
                0,     1e0,   1e1,   1e2,   1e3,   1e4,   1e5,  1e6,
                1e7,   1e8,   1e9,   1e10,  1e11,  1e12,  1e13, 1e14,
                1e15,  1e16,  1e17,  1e18,  1e19,  1e20,  1e21, 1e22,
                1e23,  1e24,  1e25,  1e26,  1e27,  1e28,  1e29, 1e30,
                1e31,  1e32,  1e33,  1e34,  1e35,  1e36,  1e37, 1e38,
                1e39,  1e40,  1e41,  1e42,  1e43,  1e44,  1e45, 1e46,
            });

            try testArgs(@Vector(1, f80), @Vector(1, u256), .{
                0,
            });
            try testArgs(@Vector(2, f80), @Vector(2, u256), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f80), @Vector(4, u256), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f80), @Vector(8, u256), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f80), @Vector(16, u256), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f80), @Vector(32, u256), .{
                1e30, 1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, 1e39, 1e40, 1e41, 1e42, 1e43, 1e44, 1e45,
                1e46, 1e47, 1e48, 1e49, 1e50, 1e51, 1e52, 1e53,
                1e54, 1e55, 1e56, 1e57, 1e58, 1e59, 1e60, 1e61,
            });
            try testArgs(@Vector(64, f80), @Vector(64, u256), .{
                1e62,           1e63,       1e64, 1e65, 1e66, 1e67, 1e68, 1e69,
                1e70,           1e71,       1e72, 1e73, 1e74, 1e75, 1e76, 1e77,
                imax(u256) - 1, imax(u256), 2e0,  2e1,  2e2,  2e3,  2e4,  2e5,
                2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12, 2e13,
                2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20, 2e21,
                2e22,           2e23,       2e24, 2e25, 2e26, 2e27, 2e28, 2e29,
                2e30,           2e31,       2e32, 2e33, 2e34, 2e35, 2e36, 2e37,
                2e38,           2e39,       2e40, 2e41, 2e42, 2e43, 2e44, 2e45,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i8), .{
                imin(i8),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i8), .{
                imin(i8) + 1, -1e2,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i8), .{
                -1e1, -1e0, 0, 1e0,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i8), .{
                1e1, 1e2, imax(i8) - 1, imax(i8), imin(i8) + 2, imin(i8) + 3, -2e1, -2e0,
            });
            try testArgs(@Vector(16, f128), @Vector(16, i8), .{
                2e0, 2e1, imax(i8) - 3, imax(i8) - 2, imin(i8) + 4, imin(i8) + 5, -3e1, -3e0,
                3e0, 3e1, imax(i8) - 5, imax(i8) - 4, imin(i8) + 6, imin(i8) + 7, -4e1, -4e0,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i8), .{
                4e0, 4e1, imax(i8) - 7,  imax(i8) - 6,  imin(i8) + 8,  imin(i8) + 9,  -5e1, -5e0,
                5e0, 5e1, imax(i8) - 9,  imax(i8) - 8,  imin(i8) + 10, imin(i8) + 11, -6e1, -6e0,
                6e0, 6e1, imax(i8) - 11, imax(i8) - 10, imin(i8) + 12, imin(i8) + 13, -7e1, -7e0,
                7e0, 7e1, imax(i8) - 13, imax(i8) - 12, imin(i8) + 14, imin(i8) + 15, -8e1, -8e0,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i8), .{
                8e0,           8e1,           imax(i8) - 15, imax(i8) - 14, imin(i8) + 16, imin(i8) + 17, -9e1,          -9e0,
                9e0,           9e1,           imax(i8) - 17, imax(i8) - 16, imin(i8) + 18, imin(i8) + 19, -11e1,         -11e0,
                11e0,          11e1,          imax(i8) - 19, imax(i8) - 18, imin(i8) + 20, imin(i8) + 21, -12e1,         -12e0,
                12e0,          12e1,          imax(i8) - 21, imax(i8) - 20, imin(i8) + 22, imin(i8) + 23, -13e0,         13e0,
                imax(i8) - 23, imax(i8) - 22, imin(i8) + 24, imin(i8) + 25, -14e0,         14e0,          imax(i8) - 25, imax(i8) - 24,
                imin(i8) + 26, imin(i8) + 27, -15e0,         15e0,          imax(i8) - 27, imax(i8) - 26, imin(i8) + 28, imin(i8) + 29,
                -16e0,         16e0,          imax(i8) - 29, imax(i8) - 28, imin(i8) + 30, imin(i8) + 31, -17e0,         17e0,
                imax(i8) - 31, imax(i8) - 30, imin(i8) + 32, imin(i8) + 33, -18e0,         18e0,          imax(i8) - 33, imax(i8) - 32,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u8), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u8), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u8), .{
                1e2, imax(u8) - 1, imax(u8), 2e0,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u8), .{
                2e1, 2e2, imax(u8) - 3, imax(u8) - 2, 3e0, 3e1, imax(u8) - 5, imax(u8) - 4,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u8), .{
                imax(u8) - 7,  imax(u8) - 6,  5e0, 5e1, imax(u8) - 9,  imax(u8) - 8,  6e0, 6e1,
                imax(u8) - 11, imax(u8) - 10, 7e0, 7e1, imax(u8) - 13, imax(u8) - 12, 8e0, 8e1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u8), .{
                imax(u8) - 15, imax(u8) - 14, 9e0,  9e1,  imax(u8) - 17, imax(u8) - 16, 11e0, 11e1,
                imax(u8) - 19, imax(u8) - 18, 12e0, 12e1, imax(u8) - 21, imax(u8) - 20, 13e0, 13e1,
                imax(u8) - 23, imax(u8) - 22, 14e0, 14e1, imax(u8) - 25, imax(u8) - 24, 15e0, 15e1,
                imax(u8) - 27, imax(u8) - 26, 16e0, 16e1, imax(u8) - 29, imax(u8) - 28, 17e0, 17e1,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u8), .{
                imax(u8) - 31, imax(u8) - 30, 18e0,          18e1,          imax(u8) - 33, imax(u8) - 32, 19e0,          19e1,
                imax(u8) - 35, imax(u8) - 34, 21e0,          21e1,          imax(u8) - 37, imax(u8) - 36, 22e0,          22e1,
                imax(u8) - 39, imax(u8) - 38, 23e0,          23e1,          imax(u8) - 41, imax(u8) - 40, 24e0,          24e1,
                imax(u8) - 43, imax(u8) - 42, 25e0,          25e1,          imax(u8) - 45, imax(u8) - 44, 26e0,          imax(u8) - 47,
                imax(u8) - 46, 27e0,          imax(u8) - 49, imax(u8) - 48, 28e0,          imax(u8) - 51, imax(u8) - 50, 29e0,
                imax(u8) - 53, imax(u8) - 52, 31e0,          imax(u8) - 55, imax(u8) - 54, 32e0,          imax(u8) - 57, imax(u8) - 56,
                33e0,          imax(u8) - 59, imax(u8) - 58, 34e0,          imax(u8) - 61, imax(u8) - 60, 35e0,          imax(u8) - 63,
                imax(u8) - 62, 36e0,          imax(u8) - 65, imax(u8) - 64, 37e0,          imax(u8) - 67, imax(u8) - 66, 38e0,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i16), .{
                imin(i16),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i16), .{
                imin(i16) + 1, -1e4,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i16), .{
                -1e3, -1e2, -1e1, -1e0,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i16), .{
                0, 1e0, 1e1, 1e2, 1e3, 1e4, imax(i16) - 1, imax(i16),
            });
            try testArgs(@Vector(16, f128), @Vector(16, i16), .{
                imin(i16) + 2, imin(i16) + 3, -2e4, -2e3, -2e2,          -2e1,          -2e0,          2e0,
                2e1,           2e2,           2e3,  2e4,  imax(i16) - 3, imax(i16) - 2, imin(i16) + 4, imin(i16) + 5,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i16), .{
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0,          3e0,           3e1,           3e2,
                3e3,           3e4,           imax(i16) - 5, imax(i16) - 4, imin(i16) + 6, imin(i16) + 7, -4e3,          -4e2,
                -4e1,          -4e0,          4e0,           4e1,           4e2,           4e3,           imax(i16) - 7, imax(i16) - 6,
                imin(i16) + 8, imin(i16) + 9, -5e3,          -5e2,          -5e1,          -5e0,          5e0,           5e1,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i16), .{
                5e2,            5e3,            imax(i16) - 9,  imax(i16) - 8,  imin(i16) + 10, imin(i16) + 11, -6e3,           -6e2,
                -6e1,           -6e0,           6e0,            6e1,            6e2,            6e3,            imax(i16) - 11, imax(i16) - 10,
                imin(i16) + 12, imin(i16) + 13, -7e3,           -7e2,           -7e1,           -7e0,           7e0,            7e1,
                7e2,            7e3,            imax(i16) - 13, imax(i16) - 12, imin(i16) + 14, imin(i16) + 15, -8e3,           -8e2,
                -8e1,           -8e0,           8e0,            8e1,            8e2,            8e3,            imax(i16) - 15, imax(i16) - 14,
                imin(i16) + 16, imin(i16) + 17, -9e3,           -9e2,           -9e1,           -9e0,           9e0,            9e1,
                9e2,            9e3,            imax(i16) - 17, imax(i16) - 16, imin(i16) + 18, imin(i16) + 19, -11e3,          -11e2,
                -11e1,          -11e0,          11e0,           11e1,           11e2,           11e3,           imax(i16) - 19, imax(i16) - 18,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u16), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u16), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u16), .{
                1e2, 1e3, 1e4, imax(u16) - 1,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u16), .{
                imax(u16), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u16) - 3, imax(u16) - 2,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u16), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u16) - 5, imax(u16) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u16) - 7, imax(u16) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u16), .{
                5e2,            5e3,            5e4,            imax(u16) - 9,  imax(u16) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u16) - 11, imax(u16) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u16) - 13, imax(u16) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u16) - 15, imax(u16) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u16) - 17, imax(u16) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u16), .{
                11e2,           11e3,           imax(u16) - 19, imax(u16) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u16) - 21, imax(u16) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u16) - 23, imax(u16) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u16) - 25, imax(u16) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u16) - 27, imax(u16) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u16) - 29, imax(u16) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u16) - 31, imax(u16) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u16) - 33, imax(u16) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u16) - 35, imax(u16) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u16) - 37, imax(u16) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u16) - 39, imax(u16) - 38,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i32), .{
                imin(i32),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i32), .{
                imin(i32) + 1, -1e9,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i32), .{
                -1e8, -1e7, -1e6, -1e5,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i32), .{
                -1e4, -1e3, -1e2, -1e1, -1e0, 0, 1e0, 1e1,
            });
            try testArgs(@Vector(16, f128), @Vector(16, i32), .{
                1e2,           1e3,       1e4,           1e5,           1e6,  1e7,  1e8,  1e9,
                imax(i32) - 1, imax(i32), imin(i32) + 2, imin(i32) + 3, -2e9, -2e8, -2e7, -2e6,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i32), .{
                -2e5,          -2e4,          -2e3,          -2e2,          -2e1, -2e0, 2e0,  2e1,
                2e2,           2e3,           2e4,           2e5,           2e6,  2e7,  2e8,  2e9,
                imax(i32) - 3, imax(i32) - 2, imin(i32) + 4, imin(i32) + 5, -3e8, -3e7, -3e6, -3e5,
                -3e4,          -3e3,          -3e2,          -3e1,          -3e0, 3e0,  3e1,  3e2,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i32), .{
                3e3,           3e4,           3e5,           3e6,           3e7,            3e8,            imax(i32) - 5, imax(i32) - 4,
                imin(i32) + 6, imin(i32) + 7, -4e8,          -4e7,          -4e6,           -4e5,           -4e4,          -4e3,
                -4e2,          -4e1,          -4e0,          4e0,           4e1,            4e2,            4e3,           4e4,
                4e5,           4e6,           4e7,           4e8,           imax(i32) - 7,  imax(i32) - 6,  imin(i32) + 8, imin(i32) + 9,
                -5e8,          -5e7,          -5e6,          -5e5,          -5e4,           -5e3,           -5e2,          -5e1,
                -5e0,          5e0,           5e1,           5e2,           5e3,            5e4,            5e5,           5e6,
                5e7,           5e8,           imax(i32) - 9, imax(i32) - 8, imin(i32) + 10, imin(i32) + 11, -6e8,          -6e7,
                -6e6,          -6e5,          -6e4,          -6e3,          -6e2,           -6e1,           -6e0,          6e0,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u32), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u32), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u32), .{
                1e2, 1e3, 1e4, imax(u32) - 1,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u32), .{
                imax(u32), 2e0, 2e1, 2e2, 2e3, 2e4, imax(u32) - 3, imax(u32) - 2,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u32), .{
                3e0, 3e1, 3e2, 3e3, 3e4,           imax(u32) - 5, imax(u32) - 4, 4e0,
                4e1, 4e2, 4e3, 4e4, imax(u32) - 7, imax(u32) - 6, 5e0,           5e1,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u32), .{
                5e2,            5e3,            5e4,            imax(u32) - 9,  imax(u32) - 8,  6e0,            6e1,            6e2,
                6e3,            6e4,            imax(u32) - 11, imax(u32) - 10, 7e0,            7e1,            7e2,            7e3,
                imax(u32) - 13, imax(u32) - 12, 8e0,            8e1,            8e2,            8e3,            imax(u32) - 15, imax(u32) - 14,
                9e0,            9e1,            9e2,            9e3,            imax(u32) - 17, imax(u32) - 16, 11e0,           11e1,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u32), .{
                11e2,           11e3,           imax(u32) - 19, imax(u32) - 18, 12e0,           12e1,           12e2,           12e3,
                imax(u32) - 21, imax(u32) - 20, 13e0,           13e1,           13e2,           13e3,           imax(u32) - 23, imax(u32) - 22,
                14e0,           14e1,           14e2,           14e3,           imax(u32) - 25, imax(u32) - 24, 15e0,           15e1,
                15e2,           15e3,           imax(u32) - 27, imax(u32) - 26, 16e0,           16e1,           16e2,           16e3,
                imax(u32) - 29, imax(u32) - 28, 17e0,           17e1,           17e2,           17e3,           imax(u32) - 31, imax(u32) - 30,
                18e0,           18e1,           18e2,           18e3,           imax(u32) - 33, imax(u32) - 32, 19e0,           19e1,
                19e2,           19e3,           imax(u32) - 35, imax(u32) - 34, 21e0,           21e1,           21e2,           21e3,
                imax(u32) - 37, imax(u32) - 36, 22e0,           22e1,           22e2,           22e3,           imax(u32) - 39, imax(u32) - 38,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i64), .{
                imin(i64),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i64), .{
                imin(i64) + 1, -1e18,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i64), .{
                -1e17, -1e16, -1e15, -1e14,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i64), .{
                -1e13, -1e12, -1e11, -1e10, -1e9, -1e8, -1e7, -1e6,
            });
            try testArgs(@Vector(16, f128), @Vector(16, i64), .{
                -1e5, -1e4, -1e3, -1e2, -1e1, -1e0, 0,   1e0,
                1e1,  1e2,  1e3,  1e4,  1e5,  1e6,  1e7, 1e8,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i64), .{
                1e9,   1e10,  1e11,          1e12,      1e13,          1e14,          1e15,  1e16,
                1e17,  1e18,  imax(i64) - 1, imax(i64), imin(i64) + 2, imin(i64) + 3, -2e18, -2e17,
                -2e16, -2e15, -2e14,         -2e13,     -2e12,         -2e11,         -2e10, -2e9,
                -2e8,  -2e7,  -2e6,          -2e5,      -2e4,          -2e3,          -2e2,  -2e1,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i64), .{
                -2e0,  2e0,   2e1,   2e2,   2e3,           2e4,           2e5,           2e6,
                2e7,   2e8,   2e9,   2e10,  2e11,          2e12,          2e13,          2e14,
                2e15,  2e16,  2e17,  2e18,  imax(i64) - 3, imax(i64) - 2, imin(i64) + 4, imin(i64) + 5,
                -3e18, -3e17, -3e16, -3e15, -3e14,         -3e13,         -3e12,         -3e11,
                -3e10, -3e9,  -3e8,  -3e7,  -3e6,          -3e5,          -3e4,          -3e3,
                -3e2,  -3e1,  -3e0,  3e0,   3e1,           3e2,           3e3,           3e4,
                3e5,   3e6,   3e7,   3e8,   3e9,           3e10,          3e11,          3e12,
                3e13,  3e14,  3e15,  3e16,  3e17,          3e18,          imax(i64) - 5, imax(i64) - 4,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u64), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u64), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u64), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u64), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u64), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, imax(u64) - 1, imax(u64),
                2e0,  2e1,  2e2,  2e3,  2e4,  2e5,  2e6,           2e7,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u64), .{
                2e8,  2e9,  2e10, 2e11,          2e12,          2e13, 2e14, 2e15,
                2e16, 2e17, 2e18, imax(u64) - 3, imax(u64) - 2, 3e0,  3e1,  3e2,
                3e3,  3e4,  3e5,  3e6,           3e7,           3e8,  3e9,  3e10,
                3e11, 3e12, 3e13, 3e14,          3e15,          3e16, 3e17, 3e18,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u64), .{
                imax(u64) - 5, imax(u64) - 4, 4e0,           4e1,           4e2,  4e3,           4e4,           4e5,
                4e6,           4e7,           4e8,           4e9,           4e10, 4e11,          4e12,          4e13,
                4e14,          4e15,          4e16,          4e17,          4e18, imax(u64) - 7, imax(u64) - 6, 5e0,
                5e1,           5e2,           5e3,           5e4,           5e5,  5e6,           5e7,           5e8,
                5e9,           5e10,          5e11,          5e12,          5e13, 5e14,          5e15,          5e16,
                5e17,          5e18,          imax(u64) - 9, imax(u64) - 8, 6e0,  6e1,           6e2,           6e3,
                6e4,           6e5,           6e6,           6e7,           6e8,  6e9,           6e10,          6e11,
                6e12,          6e13,          6e14,          6e15,          6e16, 6e17,          6e18,          imax(u64) - 11,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i128), .{
                imin(i128),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i128), .{
                imin(i128) + 1, -1e38,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i128), .{
                -1e37, -1e36, -1e35, -1e34,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i128), .{
                -1e33, -1e32, -1e31, -1e30, -1e29, -1e28, -1e27, -1e26,
            });
            try testArgs(@Vector(16, f128), @Vector(16, i128), .{
                -1e25, -1e24, -1e23, -1e22, -1e21, -1e20, -1e19, -1e18,
                -1e17, -1e16, -1e15, -1e14, -1e13, -1e12, -1e11, -1e10,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i128), .{
                -1e9, -1e8, -1e7, -1e6, -1e5, -1e4, -1e3, -1e2,
                -1e1, -1e0, 0,    1e0,  1e1,  1e2,  1e3,  1e4,
                1e5,  1e6,  1e7,  1e8,  1e9,  1e10, 1e11, 1e12,
                1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i128), .{
                1e21,  1e22,  1e23,           1e24,       1e25,           1e26,           1e27,  1e28,
                1e29,  1e30,  1e31,           1e32,       1e33,           1e34,           1e35,  1e36,
                1e37,  1e38,  imax(i128) - 1, imax(i128), imin(i128) + 2, imin(i128) + 3, -2e37, -2e36,
                -2e35, -2e34, -2e33,          -2e32,      -2e31,          -2e30,          -2e29, -2e28,
                -2e27, -2e26, -2e25,          -2e24,      -2e23,          -2e22,          -2e21, -2e20,
                -2e19, -2e18, -2e17,          -2e16,      -2e15,          -2e14,          -2e13, -2e12,
                -2e11, -2e10, -2e9,           -2e8,       -2e7,           -2e6,           -2e5,  -2e4,
                -2e3,  -2e2,  -2e1,           -2e0,       2e0,            2e1,            2e2,   2e3,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u128), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u128), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u128), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u128), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u128), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u128), .{
                1e30, 1e31,           1e32,       1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, imax(u128) - 1, imax(u128), 2e0,  2e1,  2e2,  2e3,  2e4,
                2e5,  2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12,
                2e13, 2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u128), .{
                2e21, 2e22, 2e23,           2e24,           2e25,           2e26, 2e27, 2e28,
                2e29, 2e30, 2e31,           2e32,           2e33,           2e34, 2e35, 2e36,
                2e37, 2e38, imax(u128) - 3, imax(u128) - 2, 3e0,            3e1,  3e2,  3e3,
                3e4,  3e5,  3e6,            3e7,            3e8,            3e9,  3e10, 3e11,
                3e12, 3e13, 3e14,           3e15,           3e16,           3e17, 3e18, 3e19,
                3e20, 3e21, 3e22,           3e23,           3e24,           3e25, 3e26, 3e27,
                3e28, 3e29, 3e30,           3e31,           3e32,           3e33, 3e34, 3e35,
                3e36, 3e37, 3e38,           imax(u128) - 5, imax(u128) - 4, 4e0,  4e1,  4e2,
            });

            try testArgs(@Vector(1, f128), @Vector(1, i256), .{
                imin(i256),
            });
            try testArgs(@Vector(2, f128), @Vector(2, i256), .{
                imin(i256) + 1, -1e76,
            });
            try testArgs(@Vector(4, f128), @Vector(4, i256), .{
                -1e75, -1e74, -1e73, -1e72,
            });
            try testArgs(@Vector(8, f128), @Vector(8, i256), .{
                -1e71, -1e70, -1e69, -1e68, -1e67, -1e66, -1e65, -1e64,
            });
            try testArgs(@Vector(16, f128), @Vector(16, i256), .{
                -1e63, -1e62, -1e61, -1e60, -1e59, -1e58, -1e57, -1e56,
                -1e55, -1e54, -1e53, -1e52, -1e51, -1e50, -1e49, -1e48,
            });
            try testArgs(@Vector(32, f128), @Vector(32, i256), .{
                -1e47, -1e46, -1e45, -1e44, -1e43, -1e42, -1e41, -1e40,
                -1e39, -1e38, -1e37, -1e36, -1e35, -1e34, -1e33, -1e32,
                -1e31, -1e30, -1e29, -1e28, -1e27, -1e26, -1e25, -1e24,
                -1e23, -1e22, -1e21, -1e20, -1e19, -1e18, -1e17, -1e16,
            });
            try testArgs(@Vector(64, f128), @Vector(64, i256), .{
                -1e15, -1e14, -1e13, -1e12, -1e11, -1e10, -1e9, -1e8,
                -1e7,  -1e6,  -1e5,  -1e4,  -1e3,  -1e2,  -1e1, -1e0,
                0,     1e0,   1e1,   1e2,   1e3,   1e4,   1e5,  1e6,
                1e7,   1e8,   1e9,   1e10,  1e11,  1e12,  1e13, 1e14,
                1e15,  1e16,  1e17,  1e18,  1e19,  1e20,  1e21, 1e22,
                1e23,  1e24,  1e25,  1e26,  1e27,  1e28,  1e29, 1e30,
                1e31,  1e32,  1e33,  1e34,  1e35,  1e36,  1e37, 1e38,
                1e39,  1e40,  1e41,  1e42,  1e43,  1e44,  1e45, 1e46,
            });

            try testArgs(@Vector(1, f128), @Vector(1, u256), .{
                0,
            });
            try testArgs(@Vector(2, f128), @Vector(2, u256), .{
                1e0, 1e1,
            });
            try testArgs(@Vector(4, f128), @Vector(4, u256), .{
                1e2, 1e3, 1e4, 1e5,
            });
            try testArgs(@Vector(8, f128), @Vector(8, u256), .{
                1e6, 1e7, 1e8, 1e9, 1e10, 1e11, 1e12, 1e13,
            });
            try testArgs(@Vector(16, f128), @Vector(16, u256), .{
                1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20, 1e21,
                1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29,
            });
            try testArgs(@Vector(32, f128), @Vector(32, u256), .{
                1e30, 1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37,
                1e38, 1e39, 1e40, 1e41, 1e42, 1e43, 1e44, 1e45,
                1e46, 1e47, 1e48, 1e49, 1e50, 1e51, 1e52, 1e53,
                1e54, 1e55, 1e56, 1e57, 1e58, 1e59, 1e60, 1e61,
            });
            try testArgs(@Vector(64, f128), @Vector(64, u256), .{
                1e62,           1e63,       1e64, 1e65, 1e66, 1e67, 1e68, 1e69,
                1e70,           1e71,       1e72, 1e73, 1e74, 1e75, 1e76, 1e77,
                imax(u256) - 1, imax(u256), 2e0,  2e1,  2e2,  2e3,  2e4,  2e5,
                2e6,            2e7,        2e8,  2e9,  2e10, 2e11, 2e12, 2e13,
                2e14,           2e15,       2e16, 2e17, 2e18, 2e19, 2e20, 2e21,
                2e22,           2e23,       2e24, 2e25, 2e26, 2e27, 2e28, 2e29,
                2e30,           2e31,       2e32, 2e33, 2e34, 2e35, 2e36, 2e37,
                2e38,           2e39,       2e40, 2e41, 2e42, 2e43, 2e44, 2e45,
            });
        }
    };
}

fn binary(comptime op: anytype, comptime opts: struct { compare: Compare = .relaxed }) type {
    return struct {
        // noinline so that `mem_lhs` and `mem_rhs` are on the stack
        noinline fn testArgKinds(
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Gpr,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            _: Sse,
            comptime Type: type,
            comptime imm_lhs: Type,
            mem_lhs: Type,
            comptime imm_rhs: Type,
            mem_rhs: Type,
        ) !void {
            const expected = comptime op(Type, imm_lhs, imm_rhs);
            var reg_lhs = mem_lhs;
            var reg_rhs = mem_rhs;
            _ = .{ &reg_lhs, &reg_rhs };
            try checkExpected(expected, op(Type, reg_lhs, reg_rhs), opts.compare);
            try checkExpected(expected, op(Type, reg_lhs, mem_rhs), opts.compare);
            try checkExpected(expected, op(Type, reg_lhs, imm_rhs), opts.compare);
            try checkExpected(expected, op(Type, mem_lhs, reg_rhs), opts.compare);
            try checkExpected(expected, op(Type, mem_lhs, mem_rhs), opts.compare);
            try checkExpected(expected, op(Type, mem_lhs, imm_rhs), opts.compare);
            try checkExpected(expected, op(Type, imm_lhs, reg_rhs), opts.compare);
            try checkExpected(expected, op(Type, imm_lhs, mem_rhs), opts.compare);
        }
        // noinline for a more helpful stack trace
        noinline fn testArgs(comptime Type: type, comptime imm_lhs: Type, comptime imm_rhs: Type) !void {
            try testArgKinds(
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                undefined,
                Type,
                imm_lhs,
                imm_lhs,
                imm_rhs,
                imm_rhs,
            );
        }
        fn testInts() !void {
            try testArgs(i4, 0x3, 0x2);
            try testArgs(u4, 0xe, 0x6);
            try testArgs(i8, 0x48, 0x6c);
            try testArgs(u8, 0xbb, 0x43);
            try testArgs(i16, -0x0fdf, 0x302e);
            try testArgs(u16, 0xb8bf, 0x626d);
            try testArgs(i32, -0x6280178f, 0x6802c034);
            try testArgs(u32, 0x80d7a2c6, 0xbff6a402);
            try testArgs(i64, 0x0365a53b8ee0c987, -0x1bb6d3013500a7d2);
            try testArgs(u64, 0x71138bc6b4a38898, 0x1bc4043de9438c7b);
            try testArgs(i128, 0x76d428c46cdeaa2ac43de8abffb22f6d, 0x427f7545abe434a12544fdbe2a012889);
            try testArgs(u128, 0xe05fc132ef2cd8affee00a907f0a851f, 0x29f912a72cfc6a7c6973426a9636da9a);
            try testArgs(i256, -0x53d4148cee74ea43477a65b3daa7b8fdadcbf4508e793f4af113b8d8da5a7eb6, -0x30dcbaf7b9b7a3df033694e6795444d842fb0b8f79bc18b3ea8a6b7ccad3ea91);
            try testArgs(u256, 0xb7935f5c2f3b1ae7a422c0a7c446884294b7d5370bada307d2fe5a4c4284a999, 0x310e6e196ba4f143b8d285ca6addf7f3bb3344224aff221b27607a31e148be08);
            try testArgs(i258, -0x0eee283365108dbeea0bec82f5147418d8ffe86f9eed00e414b4eccd65c21239a, -0x122c730073fc29a24cd6e3e6263566879bc5325d8566b8db31fcb4a76f7ab95eb);
            try testArgs(u258, 0x186d5ddaab8cb8cb04e5b41e36f812e039d008baf49f12894c39e29a07796d800, 0x2072daba6ffad168826163eb136f6d28ca4360c8e7e5e41e29755e19e4753a4f5);
            try testArgs(i495, 0x2fe6bc5448c55ce18252e2c9d44777505dfe63ff249a8027a6626c7d8dd9893fd5731e51474727be556f757facb586a4e04bbc0148c6c7ad692302f46fbd, -0x016a358821ef8240172f3a08e8830c06e6bcf2225f5f4d41ed42b44d249385f55cc594e1278ecac31c73faed890e5054af1a561483bb1bb6fb1f753514cf);
            try testArgs(u495, 0x6eaf4e252b3bf74b75bac59e0b43ca5326bad2a25b3fdb74a67ef132ac5e47d72eebc3316fb2351ee66c50dc5afb92a75cea9b0e35160652c7db39eeb158, 0x49fbed744a92b549d8c05bb3512c617d24dd824f3f69bdf3923bc326a75674b85f5b828d2566fab9c86f571d12c2a63c9164feb0d191d27905533d09622a);
            try testArgs(i512, -0x3a6876ca92775286c6e1504a64a9b8d56985bebf4a1b66539d404e0e96f24b226f70c4bcff295fdc2043b82513b2052dc45fd78f7e9e80e5b3e101757289f054, 0x5080c516a819bd32a0a5f0976441bbfbcf89e77684f1f10eb326aeb28e1f8d593278cff60fc99b8ffc87d8696882c64728dd3c322b7142803f4341f85a03bc10);
            try testArgs(u512, 0xe5b1fedca3c77db765e517aabd05ffc524a3a8aff1784bbf67c45b894447ede32b65b9940e78173c591e56e078932d465f235aece7ad47b7f229df7ba8f12295, 0x8b4bb7c2969e3b121cc1082c442f8b4330f0a50058438fed56447175bb10178607ecfe425cb54dacc25ef26810f3e04681de1844f1aa8d029aca75d658634806);
        }
        fn testFloats() !void {
            @setEvalBranchQuota(21_700);

            try testArgs(f16, -nan(f16), -nan(f16));
            try testArgs(f16, -nan(f16), -inf(f16));
            try testArgs(f16, -nan(f16), -fmax(f16));
            try testArgs(f16, -nan(f16), -1e1);
            try testArgs(f16, -nan(f16), -1e0);
            try testArgs(f16, -nan(f16), -1e-1);
            try testArgs(f16, -nan(f16), -fmin(f16));
            try testArgs(f16, -nan(f16), -tmin(f16));
            try testArgs(f16, -nan(f16), -0.0);
            try testArgs(f16, -nan(f16), 0.0);
            try testArgs(f16, -nan(f16), tmin(f16));
            try testArgs(f16, -nan(f16), fmin(f16));
            try testArgs(f16, -nan(f16), 1e-1);
            try testArgs(f16, -nan(f16), 1e0);
            try testArgs(f16, -nan(f16), 1e1);
            try testArgs(f16, -nan(f16), fmax(f16));
            try testArgs(f16, -nan(f16), inf(f16));
            try testArgs(f16, -nan(f16), nan(f16));

            try testArgs(f16, -inf(f16), -nan(f16));
            try testArgs(f16, -inf(f16), -inf(f16));
            try testArgs(f16, -inf(f16), -fmax(f16));
            try testArgs(f16, -inf(f16), -1e1);
            try testArgs(f16, -inf(f16), -1e0);
            try testArgs(f16, -inf(f16), -1e-1);
            try testArgs(f16, -inf(f16), -fmin(f16));
            try testArgs(f16, -inf(f16), -tmin(f16));
            try testArgs(f16, -inf(f16), -0.0);
            try testArgs(f16, -inf(f16), 0.0);
            try testArgs(f16, -inf(f16), tmin(f16));
            try testArgs(f16, -inf(f16), fmin(f16));
            try testArgs(f16, -inf(f16), 1e-1);
            try testArgs(f16, -inf(f16), 1e0);
            try testArgs(f16, -inf(f16), 1e1);
            try testArgs(f16, -inf(f16), fmax(f16));
            try testArgs(f16, -inf(f16), inf(f16));
            try testArgs(f16, -inf(f16), nan(f16));

            try testArgs(f16, -fmax(f16), -nan(f16));
            try testArgs(f16, -fmax(f16), -inf(f16));
            try testArgs(f16, -fmax(f16), -fmax(f16));
            try testArgs(f16, -fmax(f16), -1e1);
            try testArgs(f16, -fmax(f16), -1e0);
            try testArgs(f16, -fmax(f16), -1e-1);
            try testArgs(f16, -fmax(f16), -fmin(f16));
            try testArgs(f16, -fmax(f16), -tmin(f16));
            try testArgs(f16, -fmax(f16), -0.0);
            try testArgs(f16, -fmax(f16), 0.0);
            try testArgs(f16, -fmax(f16), tmin(f16));
            try testArgs(f16, -fmax(f16), fmin(f16));
            try testArgs(f16, -fmax(f16), 1e-1);
            try testArgs(f16, -fmax(f16), 1e0);
            try testArgs(f16, -fmax(f16), 1e1);
            try testArgs(f16, -fmax(f16), fmax(f16));
            try testArgs(f16, -fmax(f16), inf(f16));
            try testArgs(f16, -fmax(f16), nan(f16));

            try testArgs(f16, -1e1, -nan(f16));
            try testArgs(f16, -1e1, -inf(f16));
            try testArgs(f16, -1e1, -fmax(f16));
            try testArgs(f16, -1e1, -1e1);
            try testArgs(f16, -1e1, -1e0);
            try testArgs(f16, -1e1, -1e-1);
            try testArgs(f16, -1e1, -fmin(f16));
            try testArgs(f16, -1e1, -tmin(f16));
            try testArgs(f16, -1e1, -0.0);
            try testArgs(f16, -1e1, 0.0);
            try testArgs(f16, -1e1, tmin(f16));
            try testArgs(f16, -1e1, fmin(f16));
            try testArgs(f16, -1e1, 1e-1);
            try testArgs(f16, -1e1, 1e0);
            try testArgs(f16, -1e1, 1e1);
            try testArgs(f16, -1e1, fmax(f16));
            try testArgs(f16, -1e1, inf(f16));
            try testArgs(f16, -1e1, nan(f16));

            try testArgs(f16, -1e0, -nan(f16));
            try testArgs(f16, -1e0, -inf(f16));
            try testArgs(f16, -1e0, -fmax(f16));
            try testArgs(f16, -1e0, -1e1);
            try testArgs(f16, -1e0, -1e0);
            try testArgs(f16, -1e0, -1e-1);
            try testArgs(f16, -1e0, -fmin(f16));
            try testArgs(f16, -1e0, -tmin(f16));
            try testArgs(f16, -1e0, -0.0);
            try testArgs(f16, -1e0, 0.0);
            try testArgs(f16, -1e0, tmin(f16));
            try testArgs(f16, -1e0, fmin(f16));
            try testArgs(f16, -1e0, 1e-1);
            try testArgs(f16, -1e0, 1e0);
            try testArgs(f16, -1e0, 1e1);
            try testArgs(f16, -1e0, fmax(f16));
            try testArgs(f16, -1e0, inf(f16));
            try testArgs(f16, -1e0, nan(f16));

            try testArgs(f16, -1e-1, -nan(f16));
            try testArgs(f16, -1e-1, -inf(f16));
            try testArgs(f16, -1e-1, -fmax(f16));
            try testArgs(f16, -1e-1, -1e1);
            try testArgs(f16, -1e-1, -1e0);
            try testArgs(f16, -1e-1, -1e-1);
            try testArgs(f16, -1e-1, -fmin(f16));
            try testArgs(f16, -1e-1, -tmin(f16));
            try testArgs(f16, -1e-1, -0.0);
            try testArgs(f16, -1e-1, 0.0);
            try testArgs(f16, -1e-1, tmin(f16));
            try testArgs(f16, -1e-1, fmin(f16));
            try testArgs(f16, -1e-1, 1e-1);
            try testArgs(f16, -1e-1, 1e0);
            try testArgs(f16, -1e-1, 1e1);
            try testArgs(f16, -1e-1, fmax(f16));
            try testArgs(f16, -1e-1, inf(f16));
            try testArgs(f16, -1e-1, nan(f16));

            try testArgs(f16, -fmin(f16), -nan(f16));
            try testArgs(f16, -fmin(f16), -inf(f16));
            try testArgs(f16, -fmin(f16), -fmax(f16));
            try testArgs(f16, -fmin(f16), -1e1);
            try testArgs(f16, -fmin(f16), -1e0);
            try testArgs(f16, -fmin(f16), -1e-1);
            try testArgs(f16, -fmin(f16), -fmin(f16));
            try testArgs(f16, -fmin(f16), -tmin(f16));
            try testArgs(f16, -fmin(f16), -0.0);
            try testArgs(f16, -fmin(f16), 0.0);
            try testArgs(f16, -fmin(f16), tmin(f16));
            try testArgs(f16, -fmin(f16), fmin(f16));
            try testArgs(f16, -fmin(f16), 1e-1);
            try testArgs(f16, -fmin(f16), 1e0);
            try testArgs(f16, -fmin(f16), 1e1);
            try testArgs(f16, -fmin(f16), fmax(f16));
            try testArgs(f16, -fmin(f16), inf(f16));
            try testArgs(f16, -fmin(f16), nan(f16));

            try testArgs(f16, -tmin(f16), -nan(f16));
            try testArgs(f16, -tmin(f16), -inf(f16));
            try testArgs(f16, -tmin(f16), -fmax(f16));
            try testArgs(f16, -tmin(f16), -1e1);
            try testArgs(f16, -tmin(f16), -1e0);
            try testArgs(f16, -tmin(f16), -1e-1);
            try testArgs(f16, -tmin(f16), -fmin(f16));
            try testArgs(f16, -tmin(f16), -tmin(f16));
            try testArgs(f16, -tmin(f16), -0.0);
            try testArgs(f16, -tmin(f16), 0.0);
            try testArgs(f16, -tmin(f16), tmin(f16));
            try testArgs(f16, -tmin(f16), fmin(f16));
            try testArgs(f16, -tmin(f16), 1e-1);
            try testArgs(f16, -tmin(f16), 1e0);
            try testArgs(f16, -tmin(f16), 1e1);
            try testArgs(f16, -tmin(f16), fmax(f16));
            try testArgs(f16, -tmin(f16), inf(f16));
            try testArgs(f16, -tmin(f16), nan(f16));

            try testArgs(f16, -0.0, -nan(f16));
            try testArgs(f16, -0.0, -inf(f16));
            try testArgs(f16, -0.0, -fmax(f16));
            try testArgs(f16, -0.0, -1e1);
            try testArgs(f16, -0.0, -1e0);
            try testArgs(f16, -0.0, -1e-1);
            try testArgs(f16, -0.0, -fmin(f16));
            try testArgs(f16, -0.0, -tmin(f16));
            try testArgs(f16, -0.0, -0.0);
            try testArgs(f16, -0.0, 0.0);
            try testArgs(f16, -0.0, tmin(f16));
            try testArgs(f16, -0.0, fmin(f16));
            try testArgs(f16, -0.0, 1e-1);
            try testArgs(f16, -0.0, 1e0);
            try testArgs(f16, -0.0, 1e1);
            try testArgs(f16, -0.0, fmax(f16));
            try testArgs(f16, -0.0, inf(f16));
            try testArgs(f16, -0.0, nan(f16));

            try testArgs(f16, 0.0, -nan(f16));
            try testArgs(f16, 0.0, -inf(f16));
            try testArgs(f16, 0.0, -fmax(f16));
            try testArgs(f16, 0.0, -1e1);
            try testArgs(f16, 0.0, -1e0);
            try testArgs(f16, 0.0, -1e-1);
            try testArgs(f16, 0.0, -fmin(f16));
            try testArgs(f16, 0.0, -tmin(f16));
            try testArgs(f16, 0.0, -0.0);
            try testArgs(f16, 0.0, 0.0);
            try testArgs(f16, 0.0, tmin(f16));
            try testArgs(f16, 0.0, fmin(f16));
            try testArgs(f16, 0.0, 1e-1);
            try testArgs(f16, 0.0, 1e0);
            try testArgs(f16, 0.0, 1e1);
            try testArgs(f16, 0.0, fmax(f16));
            try testArgs(f16, 0.0, inf(f16));
            try testArgs(f16, 0.0, nan(f16));

            try testArgs(f16, tmin(f16), -nan(f16));
            try testArgs(f16, tmin(f16), -inf(f16));
            try testArgs(f16, tmin(f16), -fmax(f16));
            try testArgs(f16, tmin(f16), -1e1);
            try testArgs(f16, tmin(f16), -1e0);
            try testArgs(f16, tmin(f16), -1e-1);
            try testArgs(f16, tmin(f16), -fmin(f16));
            try testArgs(f16, tmin(f16), -tmin(f16));
            try testArgs(f16, tmin(f16), -0.0);
            try testArgs(f16, tmin(f16), 0.0);
            try testArgs(f16, tmin(f16), tmin(f16));
            try testArgs(f16, tmin(f16), fmin(f16));
            try testArgs(f16, tmin(f16), 1e-1);
            try testArgs(f16, tmin(f16), 1e0);
            try testArgs(f16, tmin(f16), 1e1);
            try testArgs(f16, tmin(f16), fmax(f16));
            try testArgs(f16, tmin(f16), inf(f16));
            try testArgs(f16, tmin(f16), nan(f16));

            try testArgs(f16, fmin(f16), -nan(f16));
            try testArgs(f16, fmin(f16), -inf(f16));
            try testArgs(f16, fmin(f16), -fmax(f16));
            try testArgs(f16, fmin(f16), -1e1);
            try testArgs(f16, fmin(f16), -1e0);
            try testArgs(f16, fmin(f16), -1e-1);
            try testArgs(f16, fmin(f16), -fmin(f16));
            try testArgs(f16, fmin(f16), -tmin(f16));
            try testArgs(f16, fmin(f16), -0.0);
            try testArgs(f16, fmin(f16), 0.0);
            try testArgs(f16, fmin(f16), tmin(f16));
            try testArgs(f16, fmin(f16), fmin(f16));
            try testArgs(f16, fmin(f16), 1e-1);
            try testArgs(f16, fmin(f16), 1e0);
            try testArgs(f16, fmin(f16), 1e1);
            try testArgs(f16, fmin(f16), fmax(f16));
            try testArgs(f16, fmin(f16), inf(f16));
            try testArgs(f16, fmin(f16), nan(f16));

            try testArgs(f16, 1e-1, -nan(f16));
            try testArgs(f16, 1e-1, -inf(f16));
            try testArgs(f16, 1e-1, -fmax(f16));
            try testArgs(f16, 1e-1, -1e1);
            try testArgs(f16, 1e-1, -1e0);
            try testArgs(f16, 1e-1, -1e-1);
            try testArgs(f16, 1e-1, -fmin(f16));
            try testArgs(f16, 1e-1, -tmin(f16));
            try testArgs(f16, 1e-1, -0.0);
            try testArgs(f16, 1e-1, 0.0);
            try testArgs(f16, 1e-1, tmin(f16));
            try testArgs(f16, 1e-1, fmin(f16));
            try testArgs(f16, 1e-1, 1e-1);
            try testArgs(f16, 1e-1, 1e0);
            try testArgs(f16, 1e-1, 1e1);
            try testArgs(f16, 1e-1, fmax(f16));
            try testArgs(f16, 1e-1, inf(f16));
            try testArgs(f16, 1e-1, nan(f16));

            try testArgs(f16, 1e0, -nan(f16));
            try testArgs(f16, 1e0, -inf(f16));
            try testArgs(f16, 1e0, -fmax(f16));
            try testArgs(f16, 1e0, -1e1);
            try testArgs(f16, 1e0, -1e0);
            try testArgs(f16, 1e0, -1e-1);
            try testArgs(f16, 1e0, -fmin(f16));
            try testArgs(f16, 1e0, -tmin(f16));
            try testArgs(f16, 1e0, -0.0);
            try testArgs(f16, 1e0, 0.0);
            try testArgs(f16, 1e0, tmin(f16));
            try testArgs(f16, 1e0, fmin(f16));
            try testArgs(f16, 1e0, 1e-1);
            try testArgs(f16, 1e0, 1e0);
            try testArgs(f16, 1e0, 1e1);
            try testArgs(f16, 1e0, fmax(f16));
            try testArgs(f16, 1e0, inf(f16));
            try testArgs(f16, 1e0, nan(f16));

            try testArgs(f16, 1e1, -nan(f16));
            try testArgs(f16, 1e1, -inf(f16));
            try testArgs(f16, 1e1, -fmax(f16));
            try testArgs(f16, 1e1, -1e1);
            try testArgs(f16, 1e1, -1e0);
            try testArgs(f16, 1e1, -1e-1);
            try testArgs(f16, 1e1, -fmin(f16));
            try testArgs(f16, 1e1, -tmin(f16));
            try testArgs(f16, 1e1, -0.0);
            try testArgs(f16, 1e1, 0.0);
            try testArgs(f16, 1e1, tmin(f16));
            try testArgs(f16, 1e1, fmin(f16));
            try testArgs(f16, 1e1, 1e-1);
            try testArgs(f16, 1e1, 1e0);
            try testArgs(f16, 1e1, 1e1);
            try testArgs(f16, 1e1, fmax(f16));
            try testArgs(f16, 1e1, inf(f16));
            try testArgs(f16, 1e1, nan(f16));

            try testArgs(f16, fmax(f16), -nan(f16));
            try testArgs(f16, fmax(f16), -inf(f16));
            try testArgs(f16, fmax(f16), -fmax(f16));
            try testArgs(f16, fmax(f16), -1e1);
            try testArgs(f16, fmax(f16), -1e0);
            try testArgs(f16, fmax(f16), -1e-1);
            try testArgs(f16, fmax(f16), -fmin(f16));
            try testArgs(f16, fmax(f16), -tmin(f16));
            try testArgs(f16, fmax(f16), -0.0);
            try testArgs(f16, fmax(f16), 0.0);
            try testArgs(f16, fmax(f16), tmin(f16));
            try testArgs(f16, fmax(f16), fmin(f16));
            try testArgs(f16, fmax(f16), 1e-1);
            try testArgs(f16, fmax(f16), 1e0);
            try testArgs(f16, fmax(f16), 1e1);
            try testArgs(f16, fmax(f16), fmax(f16));
            try testArgs(f16, fmax(f16), inf(f16));
            try testArgs(f16, fmax(f16), nan(f16));

            try testArgs(f16, inf(f16), -nan(f16));
            try testArgs(f16, inf(f16), -inf(f16));
            try testArgs(f16, inf(f16), -fmax(f16));
            try testArgs(f16, inf(f16), -1e1);
            try testArgs(f16, inf(f16), -1e0);
            try testArgs(f16, inf(f16), -1e-1);
            try testArgs(f16, inf(f16), -fmin(f16));
            try testArgs(f16, inf(f16), -tmin(f16));
            try testArgs(f16, inf(f16), -0.0);
            try testArgs(f16, inf(f16), 0.0);
            try testArgs(f16, inf(f16), tmin(f16));
            try testArgs(f16, inf(f16), fmin(f16));
            try testArgs(f16, inf(f16), 1e-1);
            try testArgs(f16, inf(f16), 1e0);
            try testArgs(f16, inf(f16), 1e1);
            try testArgs(f16, inf(f16), fmax(f16));
            try testArgs(f16, inf(f16), inf(f16));
            try testArgs(f16, inf(f16), nan(f16));

            try testArgs(f16, nan(f16), -nan(f16));
            try testArgs(f16, nan(f16), -inf(f16));
            try testArgs(f16, nan(f16), -fmax(f16));
            try testArgs(f16, nan(f16), -1e1);
            try testArgs(f16, nan(f16), -1e0);
            try testArgs(f16, nan(f16), -1e-1);
            try testArgs(f16, nan(f16), -fmin(f16));
            try testArgs(f16, nan(f16), -tmin(f16));
            try testArgs(f16, nan(f16), -0.0);
            try testArgs(f16, nan(f16), 0.0);
            try testArgs(f16, nan(f16), tmin(f16));
            try testArgs(f16, nan(f16), fmin(f16));
            try testArgs(f16, nan(f16), 1e-1);
            try testArgs(f16, nan(f16), 1e0);
            try testArgs(f16, nan(f16), 1e1);
            try testArgs(f16, nan(f16), fmax(f16));
            try testArgs(f16, nan(f16), inf(f16));
            try testArgs(f16, nan(f16), nan(f16));

            try testArgs(f32, -nan(f32), -nan(f32));
            try testArgs(f32, -nan(f32), -inf(f32));
            try testArgs(f32, -nan(f32), -fmax(f32));
            try testArgs(f32, -nan(f32), -1e1);
            try testArgs(f32, -nan(f32), -1e0);
            try testArgs(f32, -nan(f32), -1e-1);
            try testArgs(f32, -nan(f32), -fmin(f32));
            try testArgs(f32, -nan(f32), -tmin(f32));
            try testArgs(f32, -nan(f32), -0.0);
            try testArgs(f32, -nan(f32), 0.0);
            try testArgs(f32, -nan(f32), tmin(f32));
            try testArgs(f32, -nan(f32), fmin(f32));
            try testArgs(f32, -nan(f32), 1e-1);
            try testArgs(f32, -nan(f32), 1e0);
            try testArgs(f32, -nan(f32), 1e1);
            try testArgs(f32, -nan(f32), fmax(f32));
            try testArgs(f32, -nan(f32), inf(f32));
            try testArgs(f32, -nan(f32), nan(f32));

            try testArgs(f32, -inf(f32), -nan(f32));
            try testArgs(f32, -inf(f32), -inf(f32));
            try testArgs(f32, -inf(f32), -fmax(f32));
            try testArgs(f32, -inf(f32), -1e1);
            try testArgs(f32, -inf(f32), -1e0);
            try testArgs(f32, -inf(f32), -1e-1);
            try testArgs(f32, -inf(f32), -fmin(f32));
            try testArgs(f32, -inf(f32), -tmin(f32));
            try testArgs(f32, -inf(f32), -0.0);
            try testArgs(f32, -inf(f32), 0.0);
            try testArgs(f32, -inf(f32), tmin(f32));
            try testArgs(f32, -inf(f32), fmin(f32));
            try testArgs(f32, -inf(f32), 1e-1);
            try testArgs(f32, -inf(f32), 1e0);
            try testArgs(f32, -inf(f32), 1e1);
            try testArgs(f32, -inf(f32), fmax(f32));
            try testArgs(f32, -inf(f32), inf(f32));
            try testArgs(f32, -inf(f32), nan(f32));

            try testArgs(f32, -fmax(f32), -nan(f32));
            try testArgs(f32, -fmax(f32), -inf(f32));
            try testArgs(f32, -fmax(f32), -fmax(f32));
            try testArgs(f32, -fmax(f32), -1e1);
            try testArgs(f32, -fmax(f32), -1e0);
            try testArgs(f32, -fmax(f32), -1e-1);
            try testArgs(f32, -fmax(f32), -fmin(f32));
            try testArgs(f32, -fmax(f32), -tmin(f32));
            try testArgs(f32, -fmax(f32), -0.0);
            try testArgs(f32, -fmax(f32), 0.0);
            try testArgs(f32, -fmax(f32), tmin(f32));
            try testArgs(f32, -fmax(f32), fmin(f32));
            try testArgs(f32, -fmax(f32), 1e-1);
            try testArgs(f32, -fmax(f32), 1e0);
            try testArgs(f32, -fmax(f32), 1e1);
            try testArgs(f32, -fmax(f32), fmax(f32));
            try testArgs(f32, -fmax(f32), inf(f32));
            try testArgs(f32, -fmax(f32), nan(f32));

            try testArgs(f32, -1e1, -nan(f32));
            try testArgs(f32, -1e1, -inf(f32));
            try testArgs(f32, -1e1, -fmax(f32));
            try testArgs(f32, -1e1, -1e1);
            try testArgs(f32, -1e1, -1e0);
            try testArgs(f32, -1e1, -1e-1);
            try testArgs(f32, -1e1, -fmin(f32));
            try testArgs(f32, -1e1, -tmin(f32));
            try testArgs(f32, -1e1, -0.0);
            try testArgs(f32, -1e1, 0.0);
            try testArgs(f32, -1e1, tmin(f32));
            try testArgs(f32, -1e1, fmin(f32));
            try testArgs(f32, -1e1, 1e-1);
            try testArgs(f32, -1e1, 1e0);
            try testArgs(f32, -1e1, 1e1);
            try testArgs(f32, -1e1, fmax(f32));
            try testArgs(f32, -1e1, inf(f32));
            try testArgs(f32, -1e1, nan(f32));

            try testArgs(f32, -1e0, -nan(f32));
            try testArgs(f32, -1e0, -inf(f32));
            try testArgs(f32, -1e0, -fmax(f32));
            try testArgs(f32, -1e0, -1e1);
            try testArgs(f32, -1e0, -1e0);
            try testArgs(f32, -1e0, -1e-1);
            try testArgs(f32, -1e0, -fmin(f32));
            try testArgs(f32, -1e0, -tmin(f32));
            try testArgs(f32, -1e0, -0.0);
            try testArgs(f32, -1e0, 0.0);
            try testArgs(f32, -1e0, tmin(f32));
            try testArgs(f32, -1e0, fmin(f32));
            try testArgs(f32, -1e0, 1e-1);
            try testArgs(f32, -1e0, 1e0);
            try testArgs(f32, -1e0, 1e1);
            try testArgs(f32, -1e0, fmax(f32));
            try testArgs(f32, -1e0, inf(f32));
            try testArgs(f32, -1e0, nan(f32));

            try testArgs(f32, -1e-1, -nan(f32));
            try testArgs(f32, -1e-1, -inf(f32));
            try testArgs(f32, -1e-1, -fmax(f32));
            try testArgs(f32, -1e-1, -1e1);
            try testArgs(f32, -1e-1, -1e0);
            try testArgs(f32, -1e-1, -1e-1);
            try testArgs(f32, -1e-1, -fmin(f32));
            try testArgs(f32, -1e-1, -tmin(f32));
            try testArgs(f32, -1e-1, -0.0);
            try testArgs(f32, -1e-1, 0.0);
            try testArgs(f32, -1e-1, tmin(f32));
            try testArgs(f32, -1e-1, fmin(f32));
            try testArgs(f32, -1e-1, 1e-1);
            try testArgs(f32, -1e-1, 1e0);
            try testArgs(f32, -1e-1, 1e1);
            try testArgs(f32, -1e-1, fmax(f32));
            try testArgs(f32, -1e-1, inf(f32));
            try testArgs(f32, -1e-1, nan(f32));

            try testArgs(f32, -fmin(f32), -nan(f32));
            try testArgs(f32, -fmin(f32), -inf(f32));
            try testArgs(f32, -fmin(f32), -fmax(f32));
            try testArgs(f32, -fmin(f32), -1e1);
            try testArgs(f32, -fmin(f32), -1e0);
            try testArgs(f32, -fmin(f32), -1e-1);
            try testArgs(f32, -fmin(f32), -fmin(f32));
            try testArgs(f32, -fmin(f32), -tmin(f32));
            try testArgs(f32, -fmin(f32), -0.0);
            try testArgs(f32, -fmin(f32), 0.0);
            try testArgs(f32, -fmin(f32), tmin(f32));
            try testArgs(f32, -fmin(f32), fmin(f32));
            try testArgs(f32, -fmin(f32), 1e-1);
            try testArgs(f32, -fmin(f32), 1e0);
            try testArgs(f32, -fmin(f32), 1e1);
            try testArgs(f32, -fmin(f32), fmax(f32));
            try testArgs(f32, -fmin(f32), inf(f32));
            try testArgs(f32, -fmin(f32), nan(f32));

            try testArgs(f32, -tmin(f32), -nan(f32));
            try testArgs(f32, -tmin(f32), -inf(f32));
            try testArgs(f32, -tmin(f32), -fmax(f32));
            try testArgs(f32, -tmin(f32), -1e1);
            try testArgs(f32, -tmin(f32), -1e0);
            try testArgs(f32, -tmin(f32), -1e-1);
            try testArgs(f32, -tmin(f32), -fmin(f32));
            try testArgs(f32, -tmin(f32), -tmin(f32));
            try testArgs(f32, -tmin(f32), -0.0);
            try testArgs(f32, -tmin(f32), 0.0);
            try testArgs(f32, -tmin(f32), tmin(f32));
            try testArgs(f32, -tmin(f32), fmin(f32));
            try testArgs(f32, -tmin(f32), 1e-1);
            try testArgs(f32, -tmin(f32), 1e0);
            try testArgs(f32, -tmin(f32), 1e1);
            try testArgs(f32, -tmin(f32), fmax(f32));
            try testArgs(f32, -tmin(f32), inf(f32));
            try testArgs(f32, -tmin(f32), nan(f32));

            try testArgs(f32, -0.0, -nan(f32));
            try testArgs(f32, -0.0, -inf(f32));
            try testArgs(f32, -0.0, -fmax(f32));
            try testArgs(f32, -0.0, -1e1);
            try testArgs(f32, -0.0, -1e0);
            try testArgs(f32, -0.0, -1e-1);
            try testArgs(f32, -0.0, -fmin(f32));
            try testArgs(f32, -0.0, -tmin(f32));
            try testArgs(f32, -0.0, -0.0);
            try testArgs(f32, -0.0, 0.0);
            try testArgs(f32, -0.0, tmin(f32));
            try testArgs(f32, -0.0, fmin(f32));
            try testArgs(f32, -0.0, 1e-1);
            try testArgs(f32, -0.0, 1e0);
            try testArgs(f32, -0.0, 1e1);
            try testArgs(f32, -0.0, fmax(f32));
            try testArgs(f32, -0.0, inf(f32));
            try testArgs(f32, -0.0, nan(f32));

            try testArgs(f32, 0.0, -nan(f32));
            try testArgs(f32, 0.0, -inf(f32));
            try testArgs(f32, 0.0, -fmax(f32));
            try testArgs(f32, 0.0, -1e1);
            try testArgs(f32, 0.0, -1e0);
            try testArgs(f32, 0.0, -1e-1);
            try testArgs(f32, 0.0, -fmin(f32));
            try testArgs(f32, 0.0, -tmin(f32));
            try testArgs(f32, 0.0, -0.0);
            try testArgs(f32, 0.0, 0.0);
            try testArgs(f32, 0.0, tmin(f32));
            try testArgs(f32, 0.0, fmin(f32));
            try testArgs(f32, 0.0, 1e-1);
            try testArgs(f32, 0.0, 1e0);
            try testArgs(f32, 0.0, 1e1);
            try testArgs(f32, 0.0, fmax(f32));
            try testArgs(f32, 0.0, inf(f32));
            try testArgs(f32, 0.0, nan(f32));

            try testArgs(f32, tmin(f32), -nan(f32));
            try testArgs(f32, tmin(f32), -inf(f32));
            try testArgs(f32, tmin(f32), -fmax(f32));
            try testArgs(f32, tmin(f32), -1e1);
            try testArgs(f32, tmin(f32), -1e0);
            try testArgs(f32, tmin(f32), -1e-1);
            try testArgs(f32, tmin(f32), -fmin(f32));
            try testArgs(f32, tmin(f32), -tmin(f32));
            try testArgs(f32, tmin(f32), -0.0);
            try testArgs(f32, tmin(f32), 0.0);
            try testArgs(f32, tmin(f32), tmin(f32));
            try testArgs(f32, tmin(f32), fmin(f32));
            try testArgs(f32, tmin(f32), 1e-1);
            try testArgs(f32, tmin(f32), 1e0);
            try testArgs(f32, tmin(f32), 1e1);
            try testArgs(f32, tmin(f32), fmax(f32));
            try testArgs(f32, tmin(f32), inf(f32));
            try testArgs(f32, tmin(f32), nan(f32));

            try testArgs(f32, fmin(f32), -nan(f32));
            try testArgs(f32, fmin(f32), -inf(f32));
            try testArgs(f32, fmin(f32), -fmax(f32));
            try testArgs(f32, fmin(f32), -1e1);
            try testArgs(f32, fmin(f32), -1e0);
            try testArgs(f32, fmin(f32), -1e-1);
            try testArgs(f32, fmin(f32), -fmin(f32));
            try testArgs(f32, fmin(f32), -tmin(f32));
            try testArgs(f32, fmin(f32), -0.0);
            try testArgs(f32, fmin(f32), 0.0);
            try testArgs(f32, fmin(f32), tmin(f32));
            try testArgs(f32, fmin(f32), fmin(f32));
            try testArgs(f32, fmin(f32), 1e-1);
            try testArgs(f32, fmin(f32), 1e0);
            try testArgs(f32, fmin(f32), 1e1);
            try testArgs(f32, fmin(f32), fmax(f32));
            try testArgs(f32, fmin(f32), inf(f32));
            try testArgs(f32, fmin(f32), nan(f32));

            try testArgs(f32, 1e-1, -nan(f32));
            try testArgs(f32, 1e-1, -inf(f32));
            try testArgs(f32, 1e-1, -fmax(f32));
            try testArgs(f32, 1e-1, -1e1);
            try testArgs(f32, 1e-1, -1e0);
            try testArgs(f32, 1e-1, -1e-1);
            try testArgs(f32, 1e-1, -fmin(f32));
            try testArgs(f32, 1e-1, -tmin(f32));
            try testArgs(f32, 1e-1, -0.0);
            try testArgs(f32, 1e-1, 0.0);
            try testArgs(f32, 1e-1, tmin(f32));
            try testArgs(f32, 1e-1, fmin(f32));
            try testArgs(f32, 1e-1, 1e-1);
            try testArgs(f32, 1e-1, 1e0);
            try testArgs(f32, 1e-1, 1e1);
            try testArgs(f32, 1e-1, fmax(f32));
            try testArgs(f32, 1e-1, inf(f32));
            try testArgs(f32, 1e-1, nan(f32));

            try testArgs(f32, 1e0, -nan(f32));
            try testArgs(f32, 1e0, -inf(f32));
            try testArgs(f32, 1e0, -fmax(f32));
            try testArgs(f32, 1e0, -1e1);
            try testArgs(f32, 1e0, -1e0);
            try testArgs(f32, 1e0, -1e-1);
            try testArgs(f32, 1e0, -fmin(f32));
            try testArgs(f32, 1e0, -tmin(f32));
            try testArgs(f32, 1e0, -0.0);
            try testArgs(f32, 1e0, 0.0);
            try testArgs(f32, 1e0, tmin(f32));
            try testArgs(f32, 1e0, fmin(f32));
            try testArgs(f32, 1e0, 1e-1);
            try testArgs(f32, 1e0, 1e0);
            try testArgs(f32, 1e0, 1e1);
            try testArgs(f32, 1e0, fmax(f32));
            try testArgs(f32, 1e0, inf(f32));
            try testArgs(f32, 1e0, nan(f32));

            try testArgs(f32, 1e1, -nan(f32));
            try testArgs(f32, 1e1, -inf(f32));
            try testArgs(f32, 1e1, -fmax(f32));
            try testArgs(f32, 1e1, -1e1);
            try testArgs(f32, 1e1, -1e0);
            try testArgs(f32, 1e1, -1e-1);
            try testArgs(f32, 1e1, -fmin(f32));
            try testArgs(f32, 1e1, -tmin(f32));
            try testArgs(f32, 1e1, -0.0);
            try testArgs(f32, 1e1, 0.0);
            try testArgs(f32, 1e1, tmin(f32));
            try testArgs(f32, 1e1, fmin(f32));
            try testArgs(f32, 1e1, 1e-1);
            try testArgs(f32, 1e1, 1e0);
            try testArgs(f32, 1e1, 1e1);
            try testArgs(f32, 1e1, fmax(f32));
            try testArgs(f32, 1e1, inf(f32));
            try testArgs(f32, 1e1, nan(f32));

            try testArgs(f32, fmax(f32), -nan(f32));
            try testArgs(f32, fmax(f32), -inf(f32));
            try testArgs(f32, fmax(f32), -fmax(f32));
            try testArgs(f32, fmax(f32), -1e1);
            try testArgs(f32, fmax(f32), -1e0);
            try testArgs(f32, fmax(f32), -1e-1);
            try testArgs(f32, fmax(f32), -fmin(f32));
            try testArgs(f32, fmax(f32), -tmin(f32));
            try testArgs(f32, fmax(f32), -0.0);
            try testArgs(f32, fmax(f32), 0.0);
            try testArgs(f32, fmax(f32), tmin(f32));
            try testArgs(f32, fmax(f32), fmin(f32));
            try testArgs(f32, fmax(f32), 1e-1);
            try testArgs(f32, fmax(f32), 1e0);
            try testArgs(f32, fmax(f32), 1e1);
            try testArgs(f32, fmax(f32), fmax(f32));
            try testArgs(f32, fmax(f32), inf(f32));
            try testArgs(f32, fmax(f32), nan(f32));

            try testArgs(f32, inf(f32), -nan(f32));
            try testArgs(f32, inf(f32), -inf(f32));
            try testArgs(f32, inf(f32), -fmax(f32));
            try testArgs(f32, inf(f32), -1e1);
            try testArgs(f32, inf(f32), -1e0);
            try testArgs(f32, inf(f32), -1e-1);
            try testArgs(f32, inf(f32), -fmin(f32));
            try testArgs(f32, inf(f32), -tmin(f32));
            try testArgs(f32, inf(f32), -0.0);
            try testArgs(f32, inf(f32), 0.0);
            try testArgs(f32, inf(f32), tmin(f32));
            try testArgs(f32, inf(f32), fmin(f32));
            try testArgs(f32, inf(f32), 1e-1);
            try testArgs(f32, inf(f32), 1e0);
            try testArgs(f32, inf(f32), 1e1);
            try testArgs(f32, inf(f32), fmax(f32));
            try testArgs(f32, inf(f32), inf(f32));
            try testArgs(f32, inf(f32), nan(f32));

            try testArgs(f32, nan(f32), -nan(f32));
            try testArgs(f32, nan(f32), -inf(f32));
            try testArgs(f32, nan(f32), -fmax(f32));
            try testArgs(f32, nan(f32), -1e1);
            try testArgs(f32, nan(f32), -1e0);
            try testArgs(f32, nan(f32), -1e-1);
            try testArgs(f32, nan(f32), -fmin(f32));
            try testArgs(f32, nan(f32), -tmin(f32));
            try testArgs(f32, nan(f32), -0.0);
            try testArgs(f32, nan(f32), 0.0);
            try testArgs(f32, nan(f32), tmin(f32));
            try testArgs(f32, nan(f32), fmin(f32));
            try testArgs(f32, nan(f32), 1e-1);
            try testArgs(f32, nan(f32), 1e0);
            try testArgs(f32, nan(f32), 1e1);
            try testArgs(f32, nan(f32), fmax(f32));
            try testArgs(f32, nan(f32), inf(f32));
            try testArgs(f32, nan(f32), nan(f32));

            try testArgs(f64, -nan(f64), -nan(f64));
            try testArgs(f64, -nan(f64), -inf(f64));
            try testArgs(f64, -nan(f64), -fmax(f64));
            try testArgs(f64, -nan(f64), -1e1);
            try testArgs(f64, -nan(f64), -1e0);
            try testArgs(f64, -nan(f64), -1e-1);
            try testArgs(f64, -nan(f64), -fmin(f64));
            try testArgs(f64, -nan(f64), -tmin(f64));
            try testArgs(f64, -nan(f64), -0.0);
            try testArgs(f64, -nan(f64), 0.0);
            try testArgs(f64, -nan(f64), tmin(f64));
            try testArgs(f64, -nan(f64), fmin(f64));
            try testArgs(f64, -nan(f64), 1e-1);
            try testArgs(f64, -nan(f64), 1e0);
            try testArgs(f64, -nan(f64), 1e1);
            try testArgs(f64, -nan(f64), fmax(f64));
            try testArgs(f64, -nan(f64), inf(f64));
            try testArgs(f64, -nan(f64), nan(f64));

            try testArgs(f64, -inf(f64), -nan(f64));
            try testArgs(f64, -inf(f64), -inf(f64));
            try testArgs(f64, -inf(f64), -fmax(f64));
            try testArgs(f64, -inf(f64), -1e1);
            try testArgs(f64, -inf(f64), -1e0);
            try testArgs(f64, -inf(f64), -1e-1);
            try testArgs(f64, -inf(f64), -fmin(f64));
            try testArgs(f64, -inf(f64), -tmin(f64));
            try testArgs(f64, -inf(f64), -0.0);
            try testArgs(f64, -inf(f64), 0.0);
            try testArgs(f64, -inf(f64), tmin(f64));
            try testArgs(f64, -inf(f64), fmin(f64));
            try testArgs(f64, -inf(f64), 1e-1);
            try testArgs(f64, -inf(f64), 1e0);
            try testArgs(f64, -inf(f64), 1e1);
            try testArgs(f64, -inf(f64), fmax(f64));
            try testArgs(f64, -inf(f64), inf(f64));
            try testArgs(f64, -inf(f64), nan(f64));

            try testArgs(f64, -fmax(f64), -nan(f64));
            try testArgs(f64, -fmax(f64), -inf(f64));
            try testArgs(f64, -fmax(f64), -fmax(f64));
            try testArgs(f64, -fmax(f64), -1e1);
            try testArgs(f64, -fmax(f64), -1e0);
            try testArgs(f64, -fmax(f64), -1e-1);
            try testArgs(f64, -fmax(f64), -fmin(f64));
            try testArgs(f64, -fmax(f64), -tmin(f64));
            try testArgs(f64, -fmax(f64), -0.0);
            try testArgs(f64, -fmax(f64), 0.0);
            try testArgs(f64, -fmax(f64), tmin(f64));
            try testArgs(f64, -fmax(f64), fmin(f64));
            try testArgs(f64, -fmax(f64), 1e-1);
            try testArgs(f64, -fmax(f64), 1e0);
            try testArgs(f64, -fmax(f64), 1e1);
            try testArgs(f64, -fmax(f64), fmax(f64));
            try testArgs(f64, -fmax(f64), inf(f64));
            try testArgs(f64, -fmax(f64), nan(f64));

            try testArgs(f64, -1e1, -nan(f64));
            try testArgs(f64, -1e1, -inf(f64));
            try testArgs(f64, -1e1, -fmax(f64));
            try testArgs(f64, -1e1, -1e1);
            try testArgs(f64, -1e1, -1e0);
            try testArgs(f64, -1e1, -1e-1);
            try testArgs(f64, -1e1, -fmin(f64));
            try testArgs(f64, -1e1, -tmin(f64));
            try testArgs(f64, -1e1, -0.0);
            try testArgs(f64, -1e1, 0.0);
            try testArgs(f64, -1e1, tmin(f64));
            try testArgs(f64, -1e1, fmin(f64));
            try testArgs(f64, -1e1, 1e-1);
            try testArgs(f64, -1e1, 1e0);
            try testArgs(f64, -1e1, 1e1);
            try testArgs(f64, -1e1, fmax(f64));
            try testArgs(f64, -1e1, inf(f64));
            try testArgs(f64, -1e1, nan(f64));

            try testArgs(f64, -1e0, -nan(f64));
            try testArgs(f64, -1e0, -inf(f64));
            try testArgs(f64, -1e0, -fmax(f64));
            try testArgs(f64, -1e0, -1e1);
            try testArgs(f64, -1e0, -1e0);
            try testArgs(f64, -1e0, -1e-1);
            try testArgs(f64, -1e0, -fmin(f64));
            try testArgs(f64, -1e0, -tmin(f64));
            try testArgs(f64, -1e0, -0.0);
            try testArgs(f64, -1e0, 0.0);
            try testArgs(f64, -1e0, tmin(f64));
            try testArgs(f64, -1e0, fmin(f64));
            try testArgs(f64, -1e0, 1e-1);
            try testArgs(f64, -1e0, 1e0);
            try testArgs(f64, -1e0, 1e1);
            try testArgs(f64, -1e0, fmax(f64));
            try testArgs(f64, -1e0, inf(f64));
            try testArgs(f64, -1e0, nan(f64));

            try testArgs(f64, -1e-1, -nan(f64));
            try testArgs(f64, -1e-1, -inf(f64));
            try testArgs(f64, -1e-1, -fmax(f64));
            try testArgs(f64, -1e-1, -1e1);
            try testArgs(f64, -1e-1, -1e0);
            try testArgs(f64, -1e-1, -1e-1);
            try testArgs(f64, -1e-1, -fmin(f64));
            try testArgs(f64, -1e-1, -tmin(f64));
            try testArgs(f64, -1e-1, -0.0);
            try testArgs(f64, -1e-1, 0.0);
            try testArgs(f64, -1e-1, tmin(f64));
            try testArgs(f64, -1e-1, fmin(f64));
            try testArgs(f64, -1e-1, 1e-1);
            try testArgs(f64, -1e-1, 1e0);
            try testArgs(f64, -1e-1, 1e1);
            try testArgs(f64, -1e-1, fmax(f64));
            try testArgs(f64, -1e-1, inf(f64));
            try testArgs(f64, -1e-1, nan(f64));

            try testArgs(f64, -fmin(f64), -nan(f64));
            try testArgs(f64, -fmin(f64), -inf(f64));
            try testArgs(f64, -fmin(f64), -fmax(f64));
            try testArgs(f64, -fmin(f64), -1e1);
            try testArgs(f64, -fmin(f64), -1e0);
            try testArgs(f64, -fmin(f64), -1e-1);
            try testArgs(f64, -fmin(f64), -fmin(f64));
            try testArgs(f64, -fmin(f64), -tmin(f64));
            try testArgs(f64, -fmin(f64), -0.0);
            try testArgs(f64, -fmin(f64), 0.0);
            try testArgs(f64, -fmin(f64), tmin(f64));
            try testArgs(f64, -fmin(f64), fmin(f64));
            try testArgs(f64, -fmin(f64), 1e-1);
            try testArgs(f64, -fmin(f64), 1e0);
            try testArgs(f64, -fmin(f64), 1e1);
            try testArgs(f64, -fmin(f64), fmax(f64));
            try testArgs(f64, -fmin(f64), inf(f64));
            try testArgs(f64, -fmin(f64), nan(f64));

            try testArgs(f64, -tmin(f64), -nan(f64));
            try testArgs(f64, -tmin(f64), -inf(f64));
            try testArgs(f64, -tmin(f64), -fmax(f64));
            try testArgs(f64, -tmin(f64), -1e1);
            try testArgs(f64, -tmin(f64), -1e0);
            try testArgs(f64, -tmin(f64), -1e-1);
            try testArgs(f64, -tmin(f64), -fmin(f64));
            try testArgs(f64, -tmin(f64), -tmin(f64));
            try testArgs(f64, -tmin(f64), -0.0);
            try testArgs(f64, -tmin(f64), 0.0);
            try testArgs(f64, -tmin(f64), tmin(f64));
            try testArgs(f64, -tmin(f64), fmin(f64));
            try testArgs(f64, -tmin(f64), 1e-1);
            try testArgs(f64, -tmin(f64), 1e0);
            try testArgs(f64, -tmin(f64), 1e1);
            try testArgs(f64, -tmin(f64), fmax(f64));
            try testArgs(f64, -tmin(f64), inf(f64));
            try testArgs(f64, -tmin(f64), nan(f64));

            try testArgs(f64, -0.0, -nan(f64));
            try testArgs(f64, -0.0, -inf(f64));
            try testArgs(f64, -0.0, -fmax(f64));
            try testArgs(f64, -0.0, -1e1);
            try testArgs(f64, -0.0, -1e0);
            try testArgs(f64, -0.0, -1e-1);
            try testArgs(f64, -0.0, -fmin(f64));
            try testArgs(f64, -0.0, -tmin(f64));
            try testArgs(f64, -0.0, -0.0);
            try testArgs(f64, -0.0, 0.0);
            try testArgs(f64, -0.0, tmin(f64));
            try testArgs(f64, -0.0, fmin(f64));
            try testArgs(f64, -0.0, 1e-1);
            try testArgs(f64, -0.0, 1e0);
            try testArgs(f64, -0.0, 1e1);
            try testArgs(f64, -0.0, fmax(f64));
            try testArgs(f64, -0.0, inf(f64));
            try testArgs(f64, -0.0, nan(f64));

            try testArgs(f64, 0.0, -nan(f64));
            try testArgs(f64, 0.0, -inf(f64));
            try testArgs(f64, 0.0, -fmax(f64));
            try testArgs(f64, 0.0, -1e1);
            try testArgs(f64, 0.0, -1e0);
            try testArgs(f64, 0.0, -1e-1);
            try testArgs(f64, 0.0, -fmin(f64));
            try testArgs(f64, 0.0, -tmin(f64));
            try testArgs(f64, 0.0, -0.0);
            try testArgs(f64, 0.0, 0.0);
            try testArgs(f64, 0.0, tmin(f64));
            try testArgs(f64, 0.0, fmin(f64));
            try testArgs(f64, 0.0, 1e-1);
            try testArgs(f64, 0.0, 1e0);
            try testArgs(f64, 0.0, 1e1);
            try testArgs(f64, 0.0, fmax(f64));
            try testArgs(f64, 0.0, inf(f64));
            try testArgs(f64, 0.0, nan(f64));

            try testArgs(f64, tmin(f64), -nan(f64));
            try testArgs(f64, tmin(f64), -inf(f64));
            try testArgs(f64, tmin(f64), -fmax(f64));
            try testArgs(f64, tmin(f64), -1e1);
            try testArgs(f64, tmin(f64), -1e0);
            try testArgs(f64, tmin(f64), -1e-1);
            try testArgs(f64, tmin(f64), -fmin(f64));
            try testArgs(f64, tmin(f64), -tmin(f64));
            try testArgs(f64, tmin(f64), -0.0);
            try testArgs(f64, tmin(f64), 0.0);
            try testArgs(f64, tmin(f64), tmin(f64));
            try testArgs(f64, tmin(f64), fmin(f64));
            try testArgs(f64, tmin(f64), 1e-1);
            try testArgs(f64, tmin(f64), 1e0);
            try testArgs(f64, tmin(f64), 1e1);
            try testArgs(f64, tmin(f64), fmax(f64));
            try testArgs(f64, tmin(f64), inf(f64));
            try testArgs(f64, tmin(f64), nan(f64));

            try testArgs(f64, fmin(f64), -nan(f64));
            try testArgs(f64, fmin(f64), -inf(f64));
            try testArgs(f64, fmin(f64), -fmax(f64));
            try testArgs(f64, fmin(f64), -1e1);
            try testArgs(f64, fmin(f64), -1e0);
            try testArgs(f64, fmin(f64), -1e-1);
            try testArgs(f64, fmin(f64), -fmin(f64));
            try testArgs(f64, fmin(f64), -tmin(f64));
            try testArgs(f64, fmin(f64), -0.0);
            try testArgs(f64, fmin(f64), 0.0);
            try testArgs(f64, fmin(f64), tmin(f64));
            try testArgs(f64, fmin(f64), fmin(f64));
            try testArgs(f64, fmin(f64), 1e-1);
            try testArgs(f64, fmin(f64), 1e0);
            try testArgs(f64, fmin(f64), 1e1);
            try testArgs(f64, fmin(f64), fmax(f64));
            try testArgs(f64, fmin(f64), inf(f64));
            try testArgs(f64, fmin(f64), nan(f64));

            try testArgs(f64, 1e-1, -nan(f64));
            try testArgs(f64, 1e-1, -inf(f64));
            try testArgs(f64, 1e-1, -fmax(f64));
            try testArgs(f64, 1e-1, -1e1);
            try testArgs(f64, 1e-1, -1e0);
            try testArgs(f64, 1e-1, -1e-1);
            try testArgs(f64, 1e-1, -fmin(f64));
            try testArgs(f64, 1e-1, -tmin(f64));
            try testArgs(f64, 1e-1, -0.0);
            try testArgs(f64, 1e-1, 0.0);
            try testArgs(f64, 1e-1, tmin(f64));
            try testArgs(f64, 1e-1, fmin(f64));
            try testArgs(f64, 1e-1, 1e-1);
            try testArgs(f64, 1e-1, 1e0);
            try testArgs(f64, 1e-1, 1e1);
            try testArgs(f64, 1e-1, fmax(f64));
            try testArgs(f64, 1e-1, inf(f64));
            try testArgs(f64, 1e-1, nan(f64));

            try testArgs(f64, 1e0, -nan(f64));
            try testArgs(f64, 1e0, -inf(f64));
            try testArgs(f64, 1e0, -fmax(f64));
            try testArgs(f64, 1e0, -1e1);
            try testArgs(f64, 1e0, -1e0);
            try testArgs(f64, 1e0, -1e-1);
            try testArgs(f64, 1e0, -fmin(f64));
            try testArgs(f64, 1e0, -tmin(f64));
            try testArgs(f64, 1e0, -0.0);
            try testArgs(f64, 1e0, 0.0);
            try testArgs(f64, 1e0, tmin(f64));
            try testArgs(f64, 1e0, fmin(f64));
            try testArgs(f64, 1e0, 1e-1);
            try testArgs(f64, 1e0, 1e0);
            try testArgs(f64, 1e0, 1e1);
            try testArgs(f64, 1e0, fmax(f64));
            try testArgs(f64, 1e0, inf(f64));
            try testArgs(f64, 1e0, nan(f64));

            try testArgs(f64, 1e1, -nan(f64));
            try testArgs(f64, 1e1, -inf(f64));
            try testArgs(f64, 1e1, -fmax(f64));
            try testArgs(f64, 1e1, -1e1);
            try testArgs(f64, 1e1, -1e0);
            try testArgs(f64, 1e1, -1e-1);
            try testArgs(f64, 1e1, -fmin(f64));
            try testArgs(f64, 1e1, -tmin(f64));
            try testArgs(f64, 1e1, -0.0);
            try testArgs(f64, 1e1, 0.0);
            try testArgs(f64, 1e1, tmin(f64));
            try testArgs(f64, 1e1, fmin(f64));
            try testArgs(f64, 1e1, 1e-1);
            try testArgs(f64, 1e1, 1e0);
            try testArgs(f64, 1e1, 1e1);
            try testArgs(f64, 1e1, fmax(f64));
            try testArgs(f64, 1e1, inf(f64));
            try testArgs(f64, 1e1, nan(f64));

            try testArgs(f64, fmax(f64), -nan(f64));
            try testArgs(f64, fmax(f64), -inf(f64));
            try testArgs(f64, fmax(f64), -fmax(f64));
            try testArgs(f64, fmax(f64), -1e1);
            try testArgs(f64, fmax(f64), -1e0);
            try testArgs(f64, fmax(f64), -1e-1);
            try testArgs(f64, fmax(f64), -fmin(f64));
            try testArgs(f64, fmax(f64), -tmin(f64));
            try testArgs(f64, fmax(f64), -0.0);
            try testArgs(f64, fmax(f64), 0.0);
            try testArgs(f64, fmax(f64), tmin(f64));
            try testArgs(f64, fmax(f64), fmin(f64));
            try testArgs(f64, fmax(f64), 1e-1);
            try testArgs(f64, fmax(f64), 1e0);
            try testArgs(f64, fmax(f64), 1e1);
            try testArgs(f64, fmax(f64), fmax(f64));
            try testArgs(f64, fmax(f64), inf(f64));
            try testArgs(f64, fmax(f64), nan(f64));

            try testArgs(f64, inf(f64), -nan(f64));
            try testArgs(f64, inf(f64), -inf(f64));
            try testArgs(f64, inf(f64), -fmax(f64));
            try testArgs(f64, inf(f64), -1e1);
            try testArgs(f64, inf(f64), -1e0);
            try testArgs(f64, inf(f64), -1e-1);
            try testArgs(f64, inf(f64), -fmin(f64));
            try testArgs(f64, inf(f64), -tmin(f64));
            try testArgs(f64, inf(f64), -0.0);
            try testArgs(f64, inf(f64), 0.0);
            try testArgs(f64, inf(f64), tmin(f64));
            try testArgs(f64, inf(f64), fmin(f64));
            try testArgs(f64, inf(f64), 1e-1);
            try testArgs(f64, inf(f64), 1e0);
            try testArgs(f64, inf(f64), 1e1);
            try testArgs(f64, inf(f64), fmax(f64));
            try testArgs(f64, inf(f64), inf(f64));
            try testArgs(f64, inf(f64), nan(f64));

            try testArgs(f64, nan(f64), -nan(f64));
            try testArgs(f64, nan(f64), -inf(f64));
            try testArgs(f64, nan(f64), -fmax(f64));
            try testArgs(f64, nan(f64), -1e1);
            try testArgs(f64, nan(f64), -1e0);
            try testArgs(f64, nan(f64), -1e-1);
            try testArgs(f64, nan(f64), -fmin(f64));
            try testArgs(f64, nan(f64), -tmin(f64));
            try testArgs(f64, nan(f64), -0.0);
            try testArgs(f64, nan(f64), 0.0);
            try testArgs(f64, nan(f64), tmin(f64));
            try testArgs(f64, nan(f64), fmin(f64));
            try testArgs(f64, nan(f64), 1e-1);
            try testArgs(f64, nan(f64), 1e0);
            try testArgs(f64, nan(f64), 1e1);
            try testArgs(f64, nan(f64), fmax(f64));
            try testArgs(f64, nan(f64), inf(f64));
            try testArgs(f64, nan(f64), nan(f64));

            try testArgs(f80, -nan(f80), -nan(f80));
            try testArgs(f80, -nan(f80), -inf(f80));
            try testArgs(f80, -nan(f80), -fmax(f80));
            try testArgs(f80, -nan(f80), -1e1);
            try testArgs(f80, -nan(f80), -1e0);
            try testArgs(f80, -nan(f80), -1e-1);
            try testArgs(f80, -nan(f80), -fmin(f80));
            try testArgs(f80, -nan(f80), -tmin(f80));
            try testArgs(f80, -nan(f80), -0.0);
            try testArgs(f80, -nan(f80), 0.0);
            try testArgs(f80, -nan(f80), tmin(f80));
            try testArgs(f80, -nan(f80), fmin(f80));
            try testArgs(f80, -nan(f80), 1e-1);
            try testArgs(f80, -nan(f80), 1e0);
            try testArgs(f80, -nan(f80), 1e1);
            try testArgs(f80, -nan(f80), fmax(f80));
            try testArgs(f80, -nan(f80), inf(f80));
            try testArgs(f80, -nan(f80), nan(f80));

            try testArgs(f80, -inf(f80), -nan(f80));
            try testArgs(f80, -inf(f80), -inf(f80));
            try testArgs(f80, -inf(f80), -fmax(f80));
            try testArgs(f80, -inf(f80), -1e1);
            try testArgs(f80, -inf(f80), -1e0);
            try testArgs(f80, -inf(f80), -1e-1);
            try testArgs(f80, -inf(f80), -fmin(f80));
            try testArgs(f80, -inf(f80), -tmin(f80));
            try testArgs(f80, -inf(f80), -0.0);
            try testArgs(f80, -inf(f80), 0.0);
            try testArgs(f80, -inf(f80), tmin(f80));
            try testArgs(f80, -inf(f80), fmin(f80));
            try testArgs(f80, -inf(f80), 1e-1);
            try testArgs(f80, -inf(f80), 1e0);
            try testArgs(f80, -inf(f80), 1e1);
            try testArgs(f80, -inf(f80), fmax(f80));
            try testArgs(f80, -inf(f80), inf(f80));
            try testArgs(f80, -inf(f80), nan(f80));

            try testArgs(f80, -fmax(f80), -nan(f80));
            try testArgs(f80, -fmax(f80), -inf(f80));
            try testArgs(f80, -fmax(f80), -fmax(f80));
            try testArgs(f80, -fmax(f80), -1e1);
            try testArgs(f80, -fmax(f80), -1e0);
            try testArgs(f80, -fmax(f80), -1e-1);
            try testArgs(f80, -fmax(f80), -fmin(f80));
            try testArgs(f80, -fmax(f80), -tmin(f80));
            try testArgs(f80, -fmax(f80), -0.0);
            try testArgs(f80, -fmax(f80), 0.0);
            try testArgs(f80, -fmax(f80), tmin(f80));
            try testArgs(f80, -fmax(f80), fmin(f80));
            try testArgs(f80, -fmax(f80), 1e-1);
            try testArgs(f80, -fmax(f80), 1e0);
            try testArgs(f80, -fmax(f80), 1e1);
            try testArgs(f80, -fmax(f80), fmax(f80));
            try testArgs(f80, -fmax(f80), inf(f80));
            try testArgs(f80, -fmax(f80), nan(f80));

            try testArgs(f80, -1e1, -nan(f80));
            try testArgs(f80, -1e1, -inf(f80));
            try testArgs(f80, -1e1, -fmax(f80));
            try testArgs(f80, -1e1, -1e1);
            try testArgs(f80, -1e1, -1e0);
            try testArgs(f80, -1e1, -1e-1);
            try testArgs(f80, -1e1, -fmin(f80));
            try testArgs(f80, -1e1, -tmin(f80));
            try testArgs(f80, -1e1, -0.0);
            try testArgs(f80, -1e1, 0.0);
            try testArgs(f80, -1e1, tmin(f80));
            try testArgs(f80, -1e1, fmin(f80));
            try testArgs(f80, -1e1, 1e-1);
            try testArgs(f80, -1e1, 1e0);
            try testArgs(f80, -1e1, 1e1);
            try testArgs(f80, -1e1, fmax(f80));
            try testArgs(f80, -1e1, inf(f80));
            try testArgs(f80, -1e1, nan(f80));

            try testArgs(f80, -1e0, -nan(f80));
            try testArgs(f80, -1e0, -inf(f80));
            try testArgs(f80, -1e0, -fmax(f80));
            try testArgs(f80, -1e0, -1e1);
            try testArgs(f80, -1e0, -1e0);
            try testArgs(f80, -1e0, -1e-1);
            try testArgs(f80, -1e0, -fmin(f80));
            try testArgs(f80, -1e0, -tmin(f80));
            try testArgs(f80, -1e0, -0.0);
            try testArgs(f80, -1e0, 0.0);
            try testArgs(f80, -1e0, tmin(f80));
            try testArgs(f80, -1e0, fmin(f80));
            try testArgs(f80, -1e0, 1e-1);
            try testArgs(f80, -1e0, 1e0);
            try testArgs(f80, -1e0, 1e1);
            try testArgs(f80, -1e0, fmax(f80));
            try testArgs(f80, -1e0, inf(f80));
            try testArgs(f80, -1e0, nan(f80));

            try testArgs(f80, -1e-1, -nan(f80));
            try testArgs(f80, -1e-1, -inf(f80));
            try testArgs(f80, -1e-1, -fmax(f80));
            try testArgs(f80, -1e-1, -1e1);
            try testArgs(f80, -1e-1, -1e0);
            try testArgs(f80, -1e-1, -1e-1);
            try testArgs(f80, -1e-1, -fmin(f80));
            try testArgs(f80, -1e-1, -tmin(f80));
            try testArgs(f80, -1e-1, -0.0);
            try testArgs(f80, -1e-1, 0.0);
            try testArgs(f80, -1e-1, tmin(f80));
            try testArgs(f80, -1e-1, fmin(f80));
            try testArgs(f80, -1e-1, 1e-1);
            try testArgs(f80, -1e-1, 1e0);
            try testArgs(f80, -1e-1, 1e1);
            try testArgs(f80, -1e-1, fmax(f80));
            try testArgs(f80, -1e-1, inf(f80));
            try testArgs(f80, -1e-1, nan(f80));

            try testArgs(f80, -fmin(f80), -nan(f80));
            try testArgs(f80, -fmin(f80), -inf(f80));
            try testArgs(f80, -fmin(f80), -fmax(f80));
            try testArgs(f80, -fmin(f80), -1e1);
            try testArgs(f80, -fmin(f80), -1e0);
            try testArgs(f80, -fmin(f80), -1e-1);
            try testArgs(f80, -fmin(f80), -fmin(f80));
            try testArgs(f80, -fmin(f80), -tmin(f80));
            try testArgs(f80, -fmin(f80), -0.0);
            try testArgs(f80, -fmin(f80), 0.0);
            try testArgs(f80, -fmin(f80), tmin(f80));
            try testArgs(f80, -fmin(f80), fmin(f80));
            try testArgs(f80, -fmin(f80), 1e-1);
            try testArgs(f80, -fmin(f80), 1e0);
            try testArgs(f80, -fmin(f80), 1e1);
            try testArgs(f80, -fmin(f80), fmax(f80));
            try testArgs(f80, -fmin(f80), inf(f80));
            try testArgs(f80, -fmin(f80), nan(f80));

            try testArgs(f80, -tmin(f80), -nan(f80));
            try testArgs(f80, -tmin(f80), -inf(f80));
            try testArgs(f80, -tmin(f80), -fmax(f80));
            try testArgs(f80, -tmin(f80), -1e1);
            try testArgs(f80, -tmin(f80), -1e0);
            try testArgs(f80, -tmin(f80), -1e-1);
            try testArgs(f80, -tmin(f80), -fmin(f80));
            try testArgs(f80, -tmin(f80), -tmin(f80));
            try testArgs(f80, -tmin(f80), -0.0);
            try testArgs(f80, -tmin(f80), 0.0);
            try testArgs(f80, -tmin(f80), tmin(f80));
   