// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"

	forgejo_sdk "code.forgejo.org/f3/gof3/v3/forges/forgejo/sdk"
)

type organization struct {
	common
	forgejoOrganization *forgejo_sdk.Organization
}

var _ f3_tree.ForgeDriverInterface = &organization{}

func newOrganization() generic.NodeDriverInterface {
	return &organization{}
}

func (o *organization) SetNative(organization any) {
	o.forgejoOrganization = organization.(*forgejo_sdk.Organization)
}

func (o *organization) GetNativeID() string {
	return fmt.Sprintf("%d", o.forgejoOrganization.ID)
}

func (o *organization) NewFormat() f3.Interface {
	node := o.GetNode()
	return node.GetTree().(f3_tree.TreeInterface).NewFormat(node.GetKind())
}

func (o *organization) ToFormat() f3.Interface {
	if o.forgejoOrganization == nil {
		return o.NewFormat()
	}
	return &f3.Organization{
		Common:   f3.NewCommon(fmt.Sprintf("%d", o.forgejoOrganization.ID)),
		Name:     o.forgejoOrganization.UserName,
		FullName: o.forgejoOrganization.FullName,
	}
}

func (o *organization) FromFormat(content f3.Interface) {
	organization := content.(*f3.Organization)
	o.forgejoOrganization = &forgejo_sdk.Organization{
		ID:       util.ParseInt(organization.GetID()),
		UserName: organization.Name,
		FullName: organization.FullName,
	}
}

func (o *organization) getName(ctx context.Context) string {
	node := o.GetNode()
	organizations := f3_tree.GetFirstNodeKind(node, f3_tree.KindOrganizations).GetDriver().(*organizations)
	return organizations.getNameFromID(ctx, node.GetID().Int64())
}

func (o *organization) Get(ctx context.Context) bool {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	name := o.getName(ctx)
	if name == "" {
		return false
	}

	organization, resp, err := o.getClient().GetOrg(name)
	if resp.StatusCode == 404 {
		return false
	}
	if err != nil {
		panic(fmt.Errorf("organization %v %w", o, err))
	}
	o.forgejoOrganization = organization
	return true
}

func (o *organization) Patch(context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())
	_, err := o.getClient().EditOrg(o.forgejoOrganization.UserName, forgejo_sdk.EditOrgOption{
		FullName: o.forgejoOrganization.FullName,
	})
	if err != nil {
		panic(err)
	}
}

func (o *organization) Put(context.Context) id.NodeID {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	organization, _, err := o.getClient().CreateOrg(forgejo_sdk.CreateOrgOption{
		Name:     o.forgejoOrganization.UserName,
		FullName: o.forgejoOrganization.FullName,
	})
	if err != nil {
		panic(err)
	}
	o.forgejoOrganization = organization
	o.Trace("%s", organization)
	return id.NewNodeID(o.GetNativeID())
}

func (o *organization) Delete(ctx context.Context) {
	node := o.GetNode()
	o.Trace("%s", node.GetID())

	_, err := o.getClient().DeleteOrg(o.forgejoOrganization.UserName)
	if err != nil {
		panic(err)
	}
}
