//===-- Range reduction for double precision sin/cos/tan --------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC___SUPPORT_MATH_RANGE_REDUCTION_DOUBLE_NOFMA_H
#define LLVM_LIBC_SRC___SUPPORT_MATH_RANGE_REDUCTION_DOUBLE_NOFMA_H

#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/double_double.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/nearest_integer.h"
#include "src/__support/common.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/optimization.h"
#include "src/__support/math/range_reduction_double_common.h"

namespace LIBC_NAMESPACE_DECL {

namespace math {

namespace range_reduction_double_internal {

using fputil::DoubleDouble;

LIBC_INLINE unsigned LargeRangeReduction::fast(double x, DoubleDouble &u) {
  using FPBits = typename fputil::FPBits<double>;
  FPBits xbits(x);

  int x_e_m62 = xbits.get_biased_exponent() - (FPBits::EXP_BIAS + 62);
  idx = static_cast<unsigned>((x_e_m62 >> 4) + 3);
  // Scale x down by 2^(-(16 * (idx - 3))
  xbits.set_biased_exponent((x_e_m62 & 15) + FPBits::EXP_BIAS + 62);
  // 2^62 <= |x_reduced| < 2^(62 + 16) = 2^78
  x_reduced = xbits.get_val();
  // x * c_hi = ph.hi + ph.lo exactly.
  DoubleDouble x_split = fputil::split(x_reduced);
  DoubleDouble ph = fputil::exact_mult<double, SPLIT>(
      x_split, x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][0]);
  // x * c_mid = pm.hi + pm.lo exactly.
  DoubleDouble pm = fputil::exact_mult<double, SPLIT>(
      x_split, x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][1]);
  // x * c_lo = pl.hi + pl.lo exactly.
  DoubleDouble pl = fputil::exact_mult<double, SPLIT>(
      x_split, x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][2]);
  // Extract integral parts and fractional parts of (ph.lo + pm.hi).
  double sum_hi = ph.lo + pm.hi;
  double kd = fputil::nearest_integer(sum_hi);

  // x * 128/pi mod 1 ~ y_hi + y_mid + y_lo
  y_hi = (ph.lo - kd) + pm.hi; // Exact
  y_mid = fputil::exact_add(pm.lo, pl.hi);
  y_lo = pl.lo;

  // y_l = x * c_lo_2 + pl.lo
  double y_l =
      fputil::multiply_add(x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][3], y_lo);
  DoubleDouble y = fputil::exact_add(y_hi, y_mid.hi);
  y.lo += (y_mid.lo + y_l);

  // Digits of pi/128, generated by Sollya with:
  // > a = round(pi/128, D, RN);
  // > b = round(pi/128 - a, D, RN);
  constexpr DoubleDouble PI_OVER_128_DD = {0x1.1a62633145c07p-60,
                                           0x1.921fb54442d18p-6};

  // Error bound: with {a} denote the fractional part of a, i.e.:
  //   {a} = a - round(a)
  // Then,
  //   | {x * 128/pi} - (y_hi + y_lo) | <=  ulp(ulp(y_hi)) <= 2^-105
  //   | {x mod pi/128} - (u.hi + u.lo) | < 2 * 2^-6 * 2^-105 = 2^-110
  u = fputil::quick_mult<SPLIT>(y, PI_OVER_128_DD);

  return static_cast<unsigned>(static_cast<int64_t>(kd));
}

// Lookup table for sin(k * pi / 128) with k = 0, ..., 255.
// Table is generated with Sollya as follow:
// > display = hexadecimal;
// > for k from 0 to 255 do {
//     a = round(sin(k * pi/128), 51, RN);
//     b = round(sin(k * pi/128) - a, D, RN);
//     print("{", b, ",", a, "},");
//   };
LIBC_INLINE constexpr DoubleDouble SIN_K_PI_OVER_128[] = {
    {0, 0},
    {0x1.f938a73db97fbp-58, 0x1.92155f7a3667cp-6},
    {-0x1.912bd0d569a9p-61, 0x1.91f65f10dd814p-5},
    {0x1.ccbeeeae8129ap-56, 0x1.2d52092ce19f4p-4},
    {-0x1.e2718d26ed688p-60, 0x1.917a6bc29b42cp-4},
    {-0x1.cbb1f71aca352p-56, 0x1.f564e56a9731p-4},
    {-0x1.dd9ffeaecbdc4p-55, 0x1.2c8106e8e613cp-3},
    {-0x1.ab3802218894fp-55, 0x1.5e214448b3fc8p-3},
    {-0x1.49b466e7fe36p-55, 0x1.8f8b83c69a60cp-3},
    {-0x1.035e2873ca432p-55, 0x1.c0b826a7e4f64p-3},
    {-0x1.50b7bbc4768b1p-55, 0x1.f19f97b215f1cp-3},
    {-0x1.3ed9efaa42ab3p-55, 0x1.111d262b1f678p-2},
    {0x1.a8b5c974ee7b5p-54, 0x1.294062ed59f04p-2},
    {0x1.4325f12be8946p-54, 0x1.4135c941766p-2},
    {0x1.fc2047e54e614p-55, 0x1.58f9a75ab1fdcp-2},
    {-0x1.512c678219317p-54, 0x1.7088530fa45ap-2},
    {-0x1.2e59dba7ab4c2p-54, 0x1.87de2a6aea964p-2},
    {-0x1.d24afdade848bp-54, 0x1.9ef7943a8ed8cp-2},
    {0x1.5b362cb974183p-57, 0x1.b5d1009e15ccp-2},
    {-0x1.e97af1a63c807p-54, 0x1.cc66e9931c46p-2},
    {-0x1.c3e4edc5872f8p-55, 0x1.e2b5d3806f63cp-2},
    {0x1.fb44f80f92225p-54, 0x1.f8ba4dbf89ab8p-2},
    {0x1.9697faf2e2fe5p-53, 0x1.073879922ffecp-1},
    {-0x1.7bc8eda6af93cp-53, 0x1.11eb3541b4b24p-1},
    {0x1.b25dd267f66p-55, 0x1.1c73b39ae68c8p-1},
    {-0x1.5769d0fbcddc3p-53, 0x1.26d054cdd12ep-1},
    {0x1.c20673b2116b2p-54, 0x1.30ff7fce17034p-1},
    {0x1.3c7c4bc72a92cp-53, 0x1.3affa292050b8p-1},
    {-0x1.e7f895d302395p-53, 0x1.44cf325091dd8p-1},
    {0x1.13c293edceb32p-53, 0x1.4e6cabbe3e5e8p-1},
    {-0x1.75720992bfbb2p-55, 0x1.57d69348cecap-1},
    {-0x1.24a366a5fe547p-53, 0x1.610b7551d2cep-1},
    {0x1.21165f626cdd5p-54, 0x1.6a09e667f3bccp-1},
    {-0x1.bcac43c389ba9p-53, 0x1.72d0837efff98p-1},
    {-0x1.21ea6f59be15bp-53, 0x1.7b5df226aafbp-1},
    {0x1.d217be0e2b971p-53, 0x1.83b0e0bff976cp-1},
    {0x1.69d0f6897664ap-54, 0x1.8bc806b15174p-1},
    {-0x1.615f32b6f907ap-54, 0x1.93a22499263fcp-1},
    {0x1.6788ebcc76dc6p-54, 0x1.9b3e047f3874p-1},
    {0x1.ddae89fd441d1p-53, 0x1.a29a7a046278p-1},
    {-0x1.f98273c5d2495p-54, 0x1.a9b66290ea1a4p-1},
    {-0x1.926da300ffccep-55, 0x1.b090a581502p-1},
    {0x1.90e58336c64a8p-53, 0x1.b728345196e3cp-1},
    {0x1.9f6963354e3fep-53, 0x1.bd7c0ac6f9528p-1},
    {0x1.a47d3a2a0dcbep-54, 0x1.c38b2f180bdbp-1},
    {0x1.ed0489e16b9ap-54, 0x1.c954b213411f4p-1},
    {-0x1.0f3db5dad5ac5p-53, 0x1.ced7af43cc774p-1},
    {0x1.ac42b5a8b6943p-53, 0x1.d4134d14dc938p-1},
    {-0x1.d75033dfb9ca8p-53, 0x1.d906bcf328d48p-1},
    {0x1.83c37c6107db3p-55, 0x1.ddb13b6ccc23cp-1},
    {0x1.7f59c49f6cd6dp-54, 0x1.e212104f686e4p-1},
    {0x1.ee94a90d7b88bp-53, 0x1.e6288ec48e11p-1},
    {-0x1.a27d3874701f9p-53, 0x1.e9f4156c62ddcp-1},
    {-0x1.85f4e1b8298dp-54, 0x1.ed740e7684964p-1},
    {-0x1.ab4e148e52d9ep-54, 0x1.f0a7efb9230d8p-1},
    {0x1.8a11412b82346p-54, 0x1.f38f3ac64e588p-1},
    {0x1.562172a361fd3p-56, 0x1.f6297cff75cbp-1},
    {0x1.3564acef1ff97p-53, 0x1.f8764fa714ba8p-1},
    {-0x1.5e82a3284d5c8p-53, 0x1.fa7557f08a518p-1},
    {-0x1.709bccb89a989p-54, 0x1.fc26470e19fd4p-1},
    {0x1.9e082721dfb8ep-53, 0x1.fd88da3d12524p-1},
    {-0x1.eade132f3981dp-53, 0x1.fe9cdad01883cp-1},
    {0x1.e3a843d1db55fp-53, 0x1.ff621e3796d7cp-1},
    {0x1.765595d548d9ap-54, 0x1.ffd886084cd0cp-1},
    {0, 1},
#ifndef LIBC_MATH_HAS_SMALL_TABLES
    {0x1.765595d548d9ap-54, 0x1.ffd886084cd0cp-1},
    {0x1.e3a843d1db55fp-53, 0x1.ff621e3796d7cp-1},
    {-0x1.eade132f3981dp-53, 0x1.fe9cdad01883cp-1},
    {0x1.9e082721dfb8ep-53, 0x1.fd88da3d12524p-1},
    {-0x1.709bccb89a989p-54, 0x1.fc26470e19fd4p-1},
    {-0x1.5e82a3284d5c8p-53, 0x1.fa7557f08a518p-1},
    {0x1.3564acef1ff97p-53, 0x1.f8764fa714ba8p-1},
    {0x1.562172a361fd3p-56, 0x1.f6297cff75cbp-1},
    {0x1.8a11412b82346p-54, 0x1.f38f3ac64e588p-1},
    {-0x1.ab4e148e52d9ep-54, 0x1.f0a7efb9230d8p-1},
    {-0x1.85f4e1b8298dp-54, 0x1.ed740e7684964p-1},
    {-0x1.a27d3874701f9p-53, 0x1.e9f4156c62ddcp-1},
    {0x1.ee94a90d7b88bp-53, 0x1.e6288ec48e11p-1},
    {0x1.7f59c49f6cd6dp-54, 0x1.e212104f686e4p-1},
    {0x1.83c37c6107db3p-55, 0x1.ddb13b6ccc23cp-1},
    {-0x1.d75033dfb9ca8p-53, 0x1.d906bcf328d48p-1},
    {0x1.ac42b5a8b6943p-53, 0x1.d4134d14dc938p-1},
    {-0x1.0f3db5dad5ac5p-53, 0x1.ced7af43cc774p-1},
    {0x1.ed0489e16b9ap-54, 0x1.c954b213411f4p-1},
    {0x1.a47d3a2a0dcbep-54, 0x1.c38b2f180bdbp-1},
    {0x1.9f6963354e3fep-53, 0x1.bd7c0ac6f9528p-1},
    {0x1.90e58336c64a8p-53, 0x1.b728345196e3cp-1},
    {-0x1.926da300ffccep-55, 0x1.b090a581502p-1},
    {-0x1.f98273c5d2495p-54, 0x1.a9b66290ea1a4p-1},
    {0x1.ddae89fd441d1p-53, 0x1.a29a7a046278p-1},
    {0x1.6788ebcc76dc6p-54, 0x1.9b3e047f3874p-1},
    {-0x1.615f32b6f907ap-54, 0x1.93a22499263fcp-1},
    {0x1.69d0f6897664ap-54, 0x1.8bc806b15174p-1},
    {0x1.d217be0e2b971p-53, 0x1.83b0e0bff976cp-1},
    {-0x1.21ea6f59be15bp-53, 0x1.7b5df226aafbp-1},
    {-0x1.bcac43c389ba9p-53, 0x1.72d0837efff98p-1},
    {0x1.21165f626cdd5p-54, 0x1.6a09e667f3bccp-1},
    {-0x1.24a366a5fe547p-53, 0x1.610b7551d2cep-1},
    {-0x1.75720992bfbb2p-55, 0x1.57d69348cecap-1},
    {0x1.13c293edceb32p-53, 0x1.4e6cabbe3e5e8p-1},
    {-0x1.e7f895d302395p-53, 0x1.44cf325091dd8p-1},
    {0x1.3c7c4bc72a92cp-53, 0x1.3affa292050b8p-1},
    {0x1.c20673b2116b2p-54, 0x1.30ff7fce17034p-1},
    {-0x1.5769d0fbcddc3p-53, 0x1.26d054cdd12ep-1},
    {0x1.b25dd267f66p-55, 0x1.1c73b39ae68c8p-1},
    {-0x1.7bc8eda6af93cp-53, 0x1.11eb3541b4b24p-1},
    {0x1.9697faf2e2fe5p-53, 0x1.073879922ffecp-1},
    {0x1.fb44f80f92225p-54, 0x1.f8ba4dbf89ab8p-2},
    {-0x1.c3e4edc5872f8p-55, 0x1.e2b5d3806f63cp-2},
    {-0x1.e97af1a63c807p-54, 0x1.cc66e9931c46p-2},
    {0x1.5b362cb974183p-57, 0x1.b5d1009e15ccp-2},
    {-0x1.d24afdade848bp-54, 0x1.9ef7943a8ed8cp-2},
    {-0x1.2e59dba7ab4c2p-54, 0x1.87de2a6aea964p-2},
    {-0x1.512c678219317p-54, 0x1.7088530fa45ap-2},
    {0x1.fc2047e54e614p-55, 0x1.58f9a75ab1fdcp-2},
    {0x1.4325f12be8946p-54, 0x1.4135c941766p-2},
    {0x1.a8b5c974ee7b5p-54, 0x1.294062ed59f04p-2},
    {-0x1.3ed9efaa42ab3p-55, 0x1.111d262b1f678p-2},
    {-0x1.50b7bbc4768b1p-55, 0x1.f19f97b215f1cp-3},
    {-0x1.035e2873ca432p-55, 0x1.c0b826a7e4f64p-3},
    {-0x1.49b466e7fe36p-55, 0x1.8f8b83c69a60cp-3},
    {-0x1.ab3802218894fp-55, 0x1.5e214448b3fc8p-3},
    {-0x1.dd9ffeaecbdc4p-55, 0x1.2c8106e8e613cp-3},
    {-0x1.cbb1f71aca352p-56, 0x1.f564e56a9731p-4},
    {-0x1.e2718d26ed688p-60, 0x1.917a6bc29b42cp-4},
    {0x1.ccbeeeae8129ap-56, 0x1.2d52092ce19f4p-4},
    {-0x1.912bd0d569a9p-61, 0x1.91f65f10dd814p-5},
    {0x1.f938a73db97fbp-58, 0x1.92155f7a3667cp-6},
    {0, 0},
    {-0x1.f938a73db97fbp-58, -0x1.92155f7a3667cp-6},
    {0x1.912bd0d569a9p-61, -0x1.91f65f10dd814p-5},
    {-0x1.ccbeeeae8129ap-56, -0x1.2d52092ce19f4p-4},
    {0x1.e2718d26ed688p-60, -0x1.917a6bc29b42cp-4},
    {0x1.cbb1f71aca352p-56, -0x1.f564e56a9731p-4},
    {0x1.dd9ffeaecbdc4p-55, -0x1.2c8106e8e613cp-3},
    {0x1.ab3802218894fp-55, -0x1.5e214448b3fc8p-3},
    {0x1.49b466e7fe36p-55, -0x1.8f8b83c69a60cp-3},
    {0x1.035e2873ca432p-55, -0x1.c0b826a7e4f64p-3},
    {0x1.50b7bbc4768b1p-55, -0x1.f19f97b215f1cp-3},
    {0x1.3ed9efaa42ab3p-55, -0x1.111d262b1f678p-2},
    {-0x1.a8b5c974ee7b5p-54, -0x1.294062ed59f04p-2},
    {-0x1.4325f12be8946p-54, -0x1.4135c941766p-2},
    {-0x1.fc2047e54e614p-55, -0x1.58f9a75ab1fdcp-2},
    {0x1.512c678219317p-54, -0x1.7088530fa45ap-2},
    {0x1.2e59dba7ab4c2p-54, -0x1.87de2a6aea964p-2},
    {0x1.d24afdade848bp-54, -0x1.9ef7943a8ed8cp-2},
    {-0x1.5b362cb974183p-57, -0x1.b5d1009e15ccp-2},
    {0x1.e97af1a63c807p-54, -0x1.cc66e9931c46p-2},
    {0x1.c3e4edc5872f8p-55, -0x1.e2b5d3806f63cp-2},
    {-0x1.fb44f80f92225p-54, -0x1.f8ba4dbf89ab8p-2},
    {-0x1.9697faf2e2fe5p-53, -0x1.073879922ffecp-1},
    {0x1.7bc8eda6af93cp-53, -0x1.11eb3541b4b24p-1},
    {-0x1.b25dd267f66p-55, -0x1.1c73b39ae68c8p-1},
    {0x1.5769d0fbcddc3p-53, -0x1.26d054cdd12ep-1},
    {-0x1.c20673b2116b2p-54, -0x1.30ff7fce17034p-1},
    {-0x1.3c7c4bc72a92cp-53, -0x1.3affa292050b8p-1},
    {0x1.e7f895d302395p-53, -0x1.44cf325091dd8p-1},
    {-0x1.13c293edceb32p-53, -0x1.4e6cabbe3e5e8p-1},
    {0x1.75720992bfbb2p-55, -0x1.57d69348cecap-1},
    {0x1.24a366a5fe547p-53, -0x1.610b7551d2cep-1},
    {-0x1.21165f626cdd5p-54, -0x1.6a09e667f3bccp-1},
    {0x1.bcac43c389ba9p-53, -0x1.72d0837efff98p-1},
    {0x1.21ea6f59be15bp-53, -0x1.7b5df226aafbp-1},
    {-0x1.d217be0e2b971p-53, -0x1.83b0e0bff976cp-1},
    {-0x1.69d0f6897664ap-54, -0x1.8bc806b15174p-1},
    {0x1.615f32b6f907ap-54, -0x1.93a22499263fcp-1},
    {-0x1.6788ebcc76dc6p-54, -0x1.9b3e047f3874p-1},
    {-0x1.ddae89fd441d1p-53, -0x1.a29a7a046278p-1},
    {0x1.f98273c5d2495p-54, -0x1.a9b66290ea1a4p-1},
    {0x1.926da300ffccep-55, -0x1.b090a581502p-1},
    {-0x1.90e58336c64a8p-53, -0x1.b728345196e3cp-1},
    {-0x1.9f6963354e3fep-53, -0x1.bd7c0ac6f9528p-1},
    {-0x1.a47d3a2a0dcbep-54, -0x1.c38b2f180bdbp-1},
    {-0x1.ed0489e16b9ap-54, -0x1.c954b213411f4p-1},
    {0x1.0f3db5dad5ac5p-53, -0x1.ced7af43cc774p-1},
    {-0x1.ac42b5a8b6943p-53, -0x1.d4134d14dc938p-1},
    {0x1.d75033dfb9ca8p-53, -0x1.d906bcf328d48p-1},
    {-0x1.83c37c6107db3p-55, -0x1.ddb13b6ccc23cp-1},
    {-0x1.7f59c49f6cd6dp-54, -0x1.e212104f686e4p-1},
    {-0x1.ee94a90d7b88bp-53, -0x1.e6288ec48e11p-1},
    {0x1.a27d3874701f9p-53, -0x1.e9f4156c62ddcp-1},
    {0x1.85f4e1b8298dp-54, -0x1.ed740e7684964p-1},
    {0x1.ab4e148e52d9ep-54, -0x1.f0a7efb9230d8p-1},
    {-0x1.8a11412b82346p-54, -0x1.f38f3ac64e588p-1},
    {-0x1.562172a361fd3p-56, -0x1.f6297cff75cbp-1},
    {-0x1.3564acef1ff97p-53, -0x1.f8764fa714ba8p-1},
    {0x1.5e82a3284d5c8p-53, -0x1.fa7557f08a518p-1},
    {0x1.709bccb89a989p-54, -0x1.fc26470e19fd4p-1},
    {-0x1.9e082721dfb8ep-53, -0x1.fd88da3d12524p-1},
    {0x1.eade132f3981dp-53, -0x1.fe9cdad01883cp-1},
    {-0x1.e3a843d1db55fp-53, -0x1.ff621e3796d7cp-1},
    {-0x1.765595d548d9ap-54, -0x1.ffd886084cd0cp-1},
    {0, -1},
    {-0x1.765595d548d9ap-54, -0x1.ffd886084cd0cp-1},
    {-0x1.e3a843d1db55fp-53, -0x1.ff621e3796d7cp-1},
    {0x1.eade132f3981dp-53, -0x1.fe9cdad01883cp-1},
    {-0x1.9e082721dfb8ep-53, -0x1.fd88da3d12524p-1},
    {0x1.709bccb89a989p-54, -0x1.fc26470e19fd4p-1},
    {0x1.5e82a3284d5c8p-53, -0x1.fa7557f08a518p-1},
    {-0x1.3564acef1ff97p-53, -0x1.f8764fa714ba8p-1},
    {-0x1.562172a361fd3p-56, -0x1.f6297cff75cbp-1},
    {-0x1.8a11412b82346p-54, -0x1.f38f3ac64e588p-1},
    {0x1.ab4e148e52d9ep-54, -0x1.f0a7efb9230d8p-1},
    {0x1.85f4e1b8298dp-54, -0x1.ed740e7684964p-1},
    {0x1.a27d3874701f9p-53, -0x1.e9f4156c62ddcp-1},
    {-0x1.ee94a90d7b88bp-53, -0x1.e6288ec48e11p-1},
    {-0x1.7f59c49f6cd6dp-54, -0x1.e212104f686e4p-1},
    {-0x1.83c37c6107db3p-55, -0x1.ddb13b6ccc23cp-1},
    {0x1.d75033dfb9ca8p-53, -0x1.d906bcf328d48p-1},
    {-0x1.ac42b5a8b6943p-53, -0x1.d4134d14dc938p-1},
    {0x1.0f3db5dad5ac5p-53, -0x1.ced7af43cc774p-1},
    {-0x1.ed0489e16b9ap-54, -0x1.c954b213411f4p-1},
    {-0x1.a47d3a2a0dcbep-54, -0x1.c38b2f180bdbp-1},
    {-0x1.9f6963354e3fep-53, -0x1.bd7c0ac6f9528p-1},
    {-0x1.90e58336c64a8p-53, -0x1.b728345196e3cp-1},
    {0x1.926da300ffccep-55, -0x1.b090a581502p-1},
    {0x1.f98273c5d2495p-54, -0x1.a9b66290ea1a4p-1},
    {-0x1.ddae89fd441d1p-53, -0x1.a29a7a046278p-1},
    {-0x1.6788ebcc76dc6p-54, -0x1.9b3e047f3874p-1},
    {0x1.615f32b6f907ap-54, -0x1.93a22499263fcp-1},
    {-0x1.69d0f6897664ap-54, -0x1.8bc806b15174p-1},
    {-0x1.d217be0e2b971p-53, -0x1.83b0e0bff976cp-1},
    {0x1.21ea6f59be15bp-53, -0x1.7b5df226aafbp-1},
    {0x1.bcac43c389ba9p-53, -0x1.72d0837efff98p-1},
    {-0x1.21165f626cdd5p-54, -0x1.6a09e667f3bccp-1},
    {0x1.24a366a5fe547p-53, -0x1.610b7551d2cep-1},
    {0x1.75720992bfbb2p-55, -0x1.57d69348cecap-1},
    {-0x1.13c293edceb32p-53, -0x1.4e6cabbe3e5e8p-1},
    {0x1.e7f895d302395p-53, -0x1.44cf325091dd8p-1},
    {-0x1.3c7c4bc72a92cp-53, -0x1.3affa292050b8p-1},
    {-0x1.c20673b2116b2p-54, -0x1.30ff7fce17034p-1},
    {0x1.5769d0fbcddc3p-53, -0x1.26d054cdd12ep-1},
    {-0x1.b25dd267f66p-55, -0x1.1c73b39ae68c8p-1},
    {0x1.7bc8eda6af93cp-53, -0x1.11eb3541b4b24p-1},
    {-0x1.9697faf2e2fe5p-53, -0x1.073879922ffecp-1},
    {-0x1.fb44f80f92225p-54, -0x1.f8ba4dbf89ab8p-2},
    {0x1.c3e4edc5872f8p-55, -0x1.e2b5d3806f63cp-2},
    {0x1.e97af1a63c807p-54, -0x1.cc66e9931c46p-2},
    {-0x1.5b362cb974183p-57, -0x1.b5d1009e15ccp-2},
    {0x1.d24afdade848bp-54, -0x1.9ef7943a8ed8cp-2},
    {0x1.2e59dba7ab4c2p-54, -0x1.87de2a6aea964p-2},
    {0x1.512c678219317p-54, -0x1.7088530fa45ap-2},
    {-0x1.fc2047e54e614p-55, -0x1.58f9a75ab1fdcp-2},
    {-0x1.4325f12be8946p-54, -0x1.4135c941766p-2},
    {-0x1.a8b5c974ee7b5p-54, -0x1.294062ed59f04p-2},
    {0x1.3ed9efaa42ab3p-55, -0x1.111d262b1f678p-2},
    {0x1.50b7bbc4768b1p-55, -0x1.f19f97b215f1cp-3},
    {0x1.035e2873ca432p-55, -0x1.c0b826a7e4f64p-3},
    {0x1.49b466e7fe36p-55, -0x1.8f8b83c69a60cp-3},
    {0x1.ab3802218894fp-55, -0x1.5e214448b3fc8p-3},
    {0x1.dd9ffeaecbdc4p-55, -0x1.2c8106e8e613cp-3},
    {0x1.cbb1f71aca352p-56, -0x1.f564e56a9731p-4},
    {0x1.e2718d26ed688p-60, -0x1.917a6bc29b42cp-4},
    {-0x1.ccbeeeae8129ap-56, -0x1.2d52092ce19f4p-4},
    {0x1.912bd0d569a9p-61, -0x1.91f65f10dd814p-5},
    {-0x1.f938a73db97fbp-58, -0x1.92155f7a3667cp-6},
#endif // !LIBC_MATH_HAS_SMALL_TABLES
};

} // namespace range_reduction_double_internal

} // namespace math

} // namespace LIBC_NAMESPACE_DECL

#endif // LLVM_LIBC_SRC___SUPPORT_MATH_RANGE_REDUCTION_DOUBLE_NOFMA_H
